#ifndef lint
static char SCCSid[] = "@(#) ./comm/mesh2d.c 07/23/93";
#endif

/*
    This file contains routines to map a 2-d mesh onto various architectures.
    The mesh is not assumed to be periodic, though we may add that later
    (the Intel Delta/Paragon machines have this kind of mesh).

    These routines attempt to preserve neighbor relationships

    All routines here have names of the form PIMesh...

    For these routines to work, they need to know the sizes of the 
    "physical" mesh.

    These routines are used to COMPUTE the neighbors for a given
    topology.  Users should use the topology-independent routines
    when possible (PSNbr...).  Primarily, these are for use by PSCompile
    when compiling a processor subset.
 */    

#include "tools.h"
#include "comm/comm.h"
#include "comm/procset.h"

static int PNX = 0, PNY = 1;

void PIMeshSetSizes( nx, ny )
int nx, ny;
{
PNX = nx;
PNY = ny;
}

void PIiMeshInit( )
{
if (PNX <= 0) PNX = NUMNODES;
}

/*
    Algorithm:
    We describe a multiple s curve through the mesh.  For 16 nodes, the
    ring is
    +-+-+-+
    |     |
    + +-+-+
    | |
    + +-+-+
    |     |
    +-+-+-+
    If there are an odd number of nodes, then there will be one non-adjacent
    link.  This version will generate optimal mappings only for even numbers
    of COLUMNS.

    If there are an odd number of rows, we need to switch to the pattern

    +-+ +-+
    | | | |
    + +-+ +
    |     |
    + +-+-+
    | |   
    + +-+-+
    |     |
      ...

    (basically, do an s-mapping along the top two rows.
 */    
int PIMeshRingIn2d( procset, myid, np, offset, wrap )
ProcSet *procset;
int     myid, np, offset, wrap;
{
int i, j;
int nx, ny, loff;

PIiMeshInit( );
if (procset) {
    nx   = procset->nx;
    ny   = procset->ny;
    }
else {
    nx   = PNX;
    ny   = PNY;
    }

/* get our location in the mesh */
i = myid % nx;
j = myid / nx;
/* This is a very simple minded method that moves along the imbedding until
   we use up the offset */
while (offset) {
    /* Handle the special cases first */    
    if (i == 0) {
	if (j > 0 || offset < 0) {
	    /* positive offsets are in decreasing j */
	    if (offset > 0) 
		loff = (offset > j) ? j : offset;
	    else 
		loff = (-offset < (ny-j)) ? offset : -(ny-j);
	    j    -= loff;
	    myid -= loff * nx;
	    }
	else {
	    /* j == 0 and i == 0 and offset > 0 */
	    loff = (offset > nx) ? nx : offset;
	    i    += loff;
	    myid += loff;
	    }
	}
    /* These last two cases are not correct (offset is +nx,-1 etc) */
    else if (i == nx - 1) {
	if (j & 0x1) {
	    loff = (offset > 0) ? 1 : -1;
	    }
	else {
	    loff = (offset > 0) ? -1 : 1;
	    }
	}    
    else if (i == 1) {
	/* Note that this must follow "i == nx-1" in case nx == 2. */
	if (j & 0x1) {
	    loff  = (offset > 0) ?  1 : -1;
	    }
	else {
	    loff  = (offset > 0) ? -1 :  1;
	    }
	myid += loff * nx;
	j    += loff;
	}
    else {
	if (j & 0x1) {
	    /* positive offsets are in decreasing i */
	    if (offset > 0) 
		loff = (offset > i) ? i : offset;
	    else
		loff = (-offset > (nx-i)) ? -(nx-i) : offset;
	    myid -= loff;
	    i    -= loff;
	    }
	else {
	    /* positive offsets are in increasing i */
	    if (offset > 0) 
		loff = (offset > (nx-i)) ? nx-i : offset;
	    else
		loff = (-offset > i) ? -i : offset;
	    myid += loff;
	    i    += loff;
	    }
	}
    /* Catch errors with this loop */
    if (loff == 0) break;
    offset -= loff;
    }
if (!wrap && (myid < 0 || myid >= np)) return -1;
while (myid < 0) myid += np;
while (myid >= np) myid -= np;
return myid;
}


int PIMeshMesh2d( procset, myid, np, nx, ny, offx, offy, wrapx, wrapy )
ProcSet *procset;
int     myid, np, nx, ny, offx, offy, wrapx, wrapy;
{
int li, lj, loff;    /* location in mesh */

PIiMeshInit( );

if (!wrapx) {
    li = (myid % nx) + offx;
    if (li >= procset->nx || li < 0) return -1;
    }
if (!wrapy) {
    li = (myid / nx) + offy;
    if (li >= ny || li < 0) return -1;
    }

myid += offx + offy * nx;
while (myid >= NUMNODES) 
    myid -= NUMNODES;
while (myid < 0) 
    myid += NUMNODES;
return myid;
}

/* This is a generic tree for now, though we do force the far link to
   be the left (first used in the collective tree codes)

   On possiblity would be to use the ring mapping and then subdivisions
   of that
 */
int PIMeshTree( procset, myid, np, nbr )
ProcSet *procset;
int     myid, np, nbr;
{
PIiMeshInit( );
return PIDefTree( procset, myid, np, nbr );
}	

void PSSetNbrMesh()
{
PISetNbrRoutines( PIMeshTree, PIMeshRingIn2d, PIMeshMesh2d );
}
