/*--------------------------------------------------------------------------
 Routines to convert signals from one type to another.
 Block floating point is not handled when it is difficult; see
 sig_normalize and sig_bfp.
--------------------------------------------------------------------------*/
#include <stdio.h>
#include <malloc.h>
#include <math.h>

#include "imath.h"
#include "sig.h"
#include "arith.h"	/* for badkind */

static char sccsid[] = "@(#)typecast.c	1.1 7/15/91";

/*--------------------------------------------------------------------------
 Convert a vector of whatever to char.
--------------------------------------------------------------------------*/
static void
typecast_int16_int8(n, a, b)
    int n;
    short *a;
    char *b;
{
    while (n--)
	*b++ = (char) *a++;
}
static void
typecast_float32_int8(n, a, b)
    int n;
    float *a;
    char *b;
{
    while (n--)
	*b++ = (char) *a++;
}

/*--------------------------------------------------------------------------
 Convert a signal to 1-byte integers.
 Does not change complexness of signal.
--------------------------------------------------------------------------*/
void
sig_int8(this)
    sig_t *this;
{
    int reclen = this->reclen;
    int nrec = this->nrec;

    int n;
    int offset;

    switch (this->kind) {

    case hdr_INT8:
    case hdr_CINT8:
	/* Already this type. */
	break;

    case hdr_BFP16:
    case hdr_CBFP16:
	for (n=0; n<nrec; n++) {
	    if (this->exp[n]) {
		fprintf(stderr, "sig_int8: block floating point exponant of signal %s, record %d, was nonzero; normalize before converting.\n", this->fname,
		this->exp[n]);
		exit(1);
	    }
	}
	free(this->exp);
	this->exp = NULL;
	/* FALL THRU to non-bfp case. */

    case hdr_CINT16:
	/* Handle both block-floating-point and non-bfp cases. */

	/* Result is smaller than original, so convert in place. */
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_int16_int8(reclen,
		((short *) this->im)+offset, ((char *) this->im)+offset);
	this->im = (short *)realloc(this->im, nrec * reclen * sizeof(char));
	/* FALL THRU to real case. */
    case hdr_INT16:
	/* Handle real part of both complex and real signals. */
	/* Result is smaller than original, so convert in place. */
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_int16_int8(reclen,
		((short *) this->re)+offset, ((char *) this->re)+offset);
	this->re = (short *)realloc(this->re, nrec * reclen * sizeof(char));
	break;

    case hdr_CFLOAT32:
	/* Result is smaller than original, so convert in place. */
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_float32_int8(reclen,
		((float *) this->im)+offset, ((char *) this->im)+offset);
	this->im = (short *)realloc(this->im, nrec * reclen * sizeof(char));
	/* FALL THRU to real case. */
    case hdr_FLOAT32:
	/* Handle real part of both complex and real signals. */
	/* Result is smaller than original, so convert in place. */
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_float32_int8(reclen,
		((float *) this->re)+offset, ((char *) this->re)+offset);
	this->re = (short *)realloc(this->re, nrec * reclen * sizeof(char));
	break;
    default:
	badkind("sig_int8", this->kind);
    }

    this->kind = hdr_INT8 | (this->kind & hdr_KIND_COMPLEX);
}

/*--------------------------------------------------------------------------
 Convert a vector of whatever to short.
--------------------------------------------------------------------------*/
static void
typecast_int8_int16(n, a, b)
    int n;
    char *a;
    short *b;
{
    while (n--)
	*b++ = (short) *a++;
}
static void
typecast_float32_int16(n, a, b)
    int n;
    float *a;
    short *b;
{
    while (n--)
	*b++ = (short) *a++;
}

/*--------------------------------------------------------------------------
 Convert a signal to short integers.
 Does not change complexness of signal.
--------------------------------------------------------------------------*/
void
sig_int16(this)
    sig_t *this;
{
    int reclen = this->reclen;
    int nrec = this->nrec;

    int n;
    int offset;

    switch (this->kind) {

    case hdr_CINT8: {
	/* Result is larger than original, so make a new buffer. */
	char *tmp = malloc(sizeof(short) * reclen * nrec);
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_int8_int16(reclen,
		((char *) this->im)+offset, ((short *) tmp)+offset);
	free(this->im);
	this->im = (short *)tmp;
	/* FALL THRU to real case. */
	}
    case hdr_INT8: {
	/* Handle real part of both real and complex signals. */
	/* Result is larger than original, so make a new buffer. */
	char *tmp = malloc(sizeof(short) * reclen * nrec);
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_int8_int16(reclen,
		((char *) this->re)+offset, ((short *) tmp)+offset);
	free(this->re);
	this->re = (short *)tmp;
	} break;

    case hdr_INT16:
    case hdr_CINT16:
	/* Already this type. */
	break;

    case hdr_BFP16:
    case hdr_CBFP16:
	for (n=0; n<nrec; n++) {
	    if (this->exp[n]) {
		fprintf(stderr, "sig_int16: block floating point exponant of signal %s, record %d, was nonzero; normalize before converting.\n", this->fname,
		this->exp[n]);
		exit(1);
	    }
	}
	free(this->exp);
	this->exp = NULL;
	break;

    case hdr_CFLOAT32:
	/* Result is smaller than original, so convert in place. */
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_float32_int16(reclen,
		((float *) this->im)+offset, ((short *) this->im)+offset);
	this->im = (short *)realloc(this->im, nrec * reclen * sizeof(short));
	/* FALL THRU to real case. */
    case hdr_FLOAT32:
	/* Handle real part of both complex and real signals. */
	/* Result is smaller than original, so convert in place. */
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_float32_int16(reclen,
		((float *) this->re)+offset, ((short *) this->re)+offset);
	this->re = (short *)realloc(this->re, nrec * reclen * sizeof(short));
	break;
    default:
	badkind("sig_int16", this->kind);
    }

    this->kind = hdr_INT16 | (this->kind & hdr_KIND_COMPLEX);
}

/*--------------------------------------------------------------------------
 Convert a vector of whatever to float.
--------------------------------------------------------------------------*/
static void
typecast_int8_float32(n, a, b)
    int n;
    char *a;
    float *b;
{
    while (n--)
	*b++ = (float) *a++;
}
static void
typecast_int16_float32(n, a, b)
    int n;
    short *a;
    float *b;
{
    while (n--)
	*b++ = (float) *a++;
}
void
typecast_bfp16_float32(n, a, ae, b)
    int n;
    short *a, ae;
    float *b;
{
    if (ae > 0) {
	while (n--)
	    *b++ = (float) ((int)*a++ << ae);
    } else if (ae < 0) {
	while (n--)
	    *b++ = scalbn((float) *a++, ae);
    } else {
	while (n--)
	    *b++ = (float) *a++;
    }
}

/*--------------------------------------------------------------------------
 Convert array of complex float to bfp16 such that there is no overflow.
 Return exponant.
--------------------------------------------------------------------------*/
static int
typecast_cfloat32_cbfp16(n, ar, ai, br, bi)
    int n;
    float *ar, *ai;
    short *br, *bi;
{
    float m = 1.0;
    int i;
    int scale;

    /* Find max(max |r|, max |i|) */
    for (i=n; i--; ) {
	float t;
	t = ar[i]; if (t<0.0) t = -t; if (t > m) m = t;
	t = ai[i]; if (t<0.0) t = -t; if (t > m) m = t;
    }
    /* Get IEEE exponant of m.  ilogb(1)=0, ilogb(2)=1, etc.
     * If this is greater than 14, data needs to be scaled down
     * by (ilogb(m)-14) bits to avoid overflow.
     */
    scale = 14-ilogb(m + 0.5);
    if (scale > 0) scale = 0;

    while (n--) {
	*br++ = (short) scalbn(*ar++, scale);
	*bi++ = (short) scalbn(*ai++, scale);
    }

    return -scale;
}

/*--------------------------------------------------------------------------
 Convert array of float to bfp16 such that there is no overflow.
 Return exponant.
--------------------------------------------------------------------------*/
static int
typecast_float32_bfp16(n, ar, br)
    int n;
    float *ar;
    short *br;
{
    float m = 1.0;
    int i;
    int scale;

    /* Find max |r| */
    for (i=n; i--; ) {
	float t;
	t = ar[i]; if (t<0.0) t = -t; if (t > m) m = t;
    }
    /* Get IEEE exponant of m.  ilogb(1)=0, ilogb(2)=1, etc.
     * If this is greater than 14, data needs to be scaled down
     * by (ilogb(m)-14) bits to avoid overflow.
     */
    scale = 14-ilogb(m + 0.5);
    if (scale > 0) scale = 0;

    while (n--) {
	*br++ = (short) scalbn(*ar++, scale);
    }

    return -scale;
}

/*--------------------------------------------------------------------------
 Convert signal to block floating point.
 Does not change complexness of signal.
--------------------------------------------------------------------------*/
void
sig_bfp16(this)
    sig_t *this;
{
    int reclen = this->reclen;
    int nrec = this->nrec;

    int n;
    int offset;

    switch (this->kind) {
    case hdr_CBFP16:
    case hdr_BFP16:
	break;

    case hdr_CFLOAT32:
	this->exp = (short *)malloc(this->nrec * sizeof(short));

	/* Result is smaller than original, so convert in place. */
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    this->exp[n] = typecast_cfloat32_cbfp16(reclen,
		((float *) this->re)+offset, ((float *) this->im)+offset, 
		((short *) this->re)+offset, ((short *) this->im)+offset);

	this->im = (short *)realloc(this->im, nrec * reclen * sizeof(short));
	this->re = (short *)realloc(this->re, nrec * reclen * sizeof(short));
	break;

    case hdr_FLOAT32:
	this->exp = (short *)malloc(this->nrec * sizeof(short));

	/* Result is smaller than original, so convert in place. */
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    this->exp[n] = typecast_float32_bfp16(reclen,
		((float *) this->re)+offset, ((short *) this->re)+offset);

	this->re = (short *)realloc(this->re, nrec * reclen * sizeof(short));
	break;

    default:
	sig_int16(this);
	this->exp = (short *)calloc(this->nrec, sizeof(short));
    }

    this->kind = hdr_BFP16 | (this->kind & hdr_KIND_COMPLEX);
}

/*--------------------------------------------------------------------------
 Convert signal to float.
 Does not change complexness of signal.
--------------------------------------------------------------------------*/
void
sig_float32(this)
    sig_t *this;
{
    int reclen = this->reclen;
    int nrec = this->nrec;

    int n;
    int offset;
    float *tmp;

    /* Copy and convert original */
    switch (this->kind) {

    case hdr_CINT8:
	tmp = (float *)malloc(nrec * reclen * sizeof(float));
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_int8_float32(reclen,
		((char *) this->im)+offset, tmp+offset);
	free(this->im);
	this->im = (short *)tmp;
	/* FALL THRU to real case. */
    case hdr_INT8:
	/* Handle real part of both real and complex signals. */
	tmp = (float *)malloc(nrec * reclen * sizeof(float));
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_int8_float32(reclen,
		((char *) this->re)+offset, tmp+offset);
	free(this->re);
	this->re = (short *)tmp;
	break;

    case hdr_CINT16:
	tmp = (float *)malloc(nrec * reclen * sizeof(float));
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_int16_float32(reclen,
		((short *) this->im)+offset, tmp+offset);
	free(this->im);
	this->im = (short *)tmp;
	/* FALL THRU to real case. */
    case hdr_INT16:
	/* Handle real part of both real and complex signals. */
	tmp = (float *)malloc(nrec * reclen * sizeof(float));
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_int16_float32(reclen,
		((short *) this->re)+offset, tmp+offset);
	free(this->re);
	this->re = (short *)tmp;
	break;

    case hdr_CBFP16:
	tmp = (float *)malloc(nrec * reclen * sizeof(float));
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_bfp16_float32(reclen,
		((short *) this->im)+offset, this->exp[n], tmp+offset);
	free(this->im);
	this->im = (short *)tmp;
	/* Keep exponant until real case is done. */
	/* FALL THRU to real case. */
    case hdr_BFP16:
	/* Handle real part of both real and complex signals. */
	tmp = (float *)malloc(nrec * reclen * sizeof(float));
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    typecast_bfp16_float32(reclen,
		((short *) this->re)+offset, this->exp[n], tmp+offset);
	free(this->exp);
	this->exp = NULL;
	free(this->re);
	this->re = (short *)tmp;
	break;

    case hdr_CFLOAT32:
    case hdr_FLOAT32:
	break;

    default:
	badkind("sig_float32", this->kind);
    }
    this->kind = hdr_FLOAT32 | (this->kind & hdr_KIND_COMPLEX);
}
