#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include <values.h>
#include <assert.h>

#include "sig.h"
#include "imath.h"
#include "arith.h"

static char sccsid[] = "@(#)plot.c	1.3 7/23/91";

#define PLOT_INCH 512
#define PLOT_WIDTH    (8 * PLOT_INCH)
#define PLOT_CHARX    (PLOT_INCH/8)
#define PLOT_CHARY    (PLOT_INCH/4)		/* /3 for Tek? */
#define PLOT_MARGINX  (PLOT_CHARX * 8)
#define PLOT_MARGINYTOP (PLOT_CHARY * 3)
#define PLOT_MARGINYBOT (PLOT_CHARY * 2)
#define PLOT_TICK_WIDTH PLOT_CHARX/2
#define PLOT_TICK_HEIGHT PLOT_CHARY/3

/*--------------------------------------------------------------------------
 Plot the given array.
 Output is a set of axes, labelled at the min and max ends,
 plus a connect-the-dots graph of the data.

 The output device is a plot(5) file.  The plot(5) readers I've seen tend
 to have the same scale factor for both X and Y axes.  Thus, for readable
 graphs, we must scale the data ourselves if it isn't nearly square.

 The scheme I take is to allow the caller to specify the height and
 offset of the graph, in units of 'inches'.  
 The width is always taken to be 8 inches (to match Postscript printers).

 Input record is now float for generality, but all I did to effect this
 was change the declarations and initializations of y, miny and maxy.
--------------------------------------------------------------------------*/
static void
plot_record(ofp, len, y, label, phys_y_offset, phys_y_height)
    FILE *ofp;
    int len;
    float *y;
    char *label;
    int phys_y_offset;	/* offset in inches from bottom of page */
    int phys_y_height;	/* Y span in inches */
{
    float miny, maxy;
    int minx, maxx;
    int i;
    char obuf[100];
    float xscale, yscale;	/* multiply data by this */
    int pxoffset, pyoffset;	/* add this to scaled data */
    int pybot, pytop;		/* Y range, including margins */
    int x;

    assert(ofp != NULL);
    if (phys_y_height <= 0) {
	fprintf(stderr, "plot_r: height must be >= 1, but is %d!\n", 
	    phys_y_height);
	exit(1);
    }

    minx = 0;
    maxx = len-1;

    /* Find min and max Y. */
    miny=MAXFLOAT;
    maxy=MINFLOAT;
    for (i=0; i<len; i++) {
	float yi = y[i];
	if (yi > maxy) maxy = yi;
	if (yi < miny) miny = yi;
    }

    /* Set X scaling so span (including left & right margin) is PLOT_WIDTH.
     * Set Y scaling so span (including top & bottom margin) is 
     * phys_y_height * PLOT_INCH. 
     * Set offsets to reflect desired margin(s).
     * All data output is in range 
     * 0 <= X < PLOT_WIDTH and 
     * phys_y_offset*PLOT_INCH+PLOT_MARGINYBOT <= Y 
     * and Y < (phys_y_offset+phys_y_height)*PLOT_INCH - PLOT_MARGINYTOP
     */
    pybot = phys_y_offset * PLOT_INCH;
    pytop = pybot + phys_y_height*PLOT_INCH;
    pxoffset = PLOT_MARGINX;
    pyoffset = PLOT_MARGINYBOT + pybot;
    xscale = (PLOT_WIDTH - 2*PLOT_MARGINX) / (float) (maxx-minx);
    if (maxy == miny)
	yscale = 1;
    else 
	yscale = (phys_y_height*PLOT_INCH - (PLOT_MARGINYTOP+PLOT_MARGINYBOT))
		/ (float)(maxy-miny);
    if (yscale < 0.000001) {
	fprintf(stderr, 
	    "plot_r: yscale(%f)<=0! (Height=%d <= margins=%f, miny=%f, maxy=%f)\n",
	    yscale,
	    phys_y_height, (PLOT_MARGINYTOP+PLOT_MARGINYBOT)/(float)PLOT_INCH,
	    miny, maxy);
	exit(1);
    }

    /* Caller has already set the space */
    /* pl_space(ofp, 0, pybot, PLOT_WIDTH, pytop); */

    /* Draw box around entire space */
    pl_move(ofp, 0, pybot);
    pl_cont(ofp, PLOT_WIDTH-1, pybot); 
    pl_cont(ofp, PLOT_WIDTH-1, pytop-1); 
    pl_cont(ofp, 0, pytop-1); 
    pl_cont(ofp, 0, pybot); 

    /* Label */
    pl_move(ofp, PLOT_WIDTH/2 - strlen(label)*PLOT_CHARX/2, pytop-2*PLOT_CHARY);
    pl_label(ofp, label);

#define scx(x) ((int)(x * xscale) + pxoffset)
#define scy(y) ((int)((y-miny) * yscale) + pyoffset)

    /* X axis */
    if (miny <= 0 && 0 <= maxy)
	pl_line(ofp, scx(minx), scy(0), scx(maxx), scy(0));

    /* miny */
    pl_line(ofp, scx(minx), scy(miny), scx(maxx), scy(miny));

    pl_move(ofp, scx(maxx)+PLOT_CHARX, scy(miny)-PLOT_CHARY);
    sprintf(obuf, "%d", len); 
    pl_label(ofp, obuf);

    /* Y axis */
    pl_line(ofp, scx(0), scy(miny), scx(0), scy(maxy));

    sprintf(obuf, "%g", maxy); 
    x = scx(0)-strlen(obuf)*PLOT_CHARX;
    if (x<0) x=0;
    pl_move(ofp, x, scy(maxy)); 
    pl_label(ofp, obuf);

    sprintf(obuf, "%g", miny); 
    x = scx(0)-strlen(obuf)*PLOT_CHARX;
    if (x<0) x=0;
    pl_move(ofp, x, scy(miny)); 
    pl_label(ofp, obuf);

    /* Draw ticks along the Y=miny axis at multiples of 1, 10, or 100, ...
     * such that there are at least 2 tick and no more than 20.
     */
    {
	float order, logorder;
	int iorder;
	logorder = floor(log10((double) maxx/2));
	order = exp10(logorder);
	iorder = (int) max(order, 1.0);
	
	for (i=0; i<maxx; i += iorder)
	    pl_line(ofp, scx(i), scy(miny)-PLOT_TICK_HEIGHT, 
		    scx(i), scy(miny));
    }
    /* Draw ticks along the Y axis at multiples of 1, 10, or 100, ...
     * such that there are at least 2 tick and no more than 20.
     */
    {
	float y, order, logorder;
	logorder = floor(log10((double) (maxy-miny)/2));
	order = exp10(logorder);

	if (order > 0.09) {
	    for (y=0.0; y<maxy; y += order)
		if (y >= miny)
		    pl_line(ofp, scx(minx)-PLOT_TICK_WIDTH, scy(y), 
			scx(minx), scy(y));
	    for (y=0.0; y>miny; y -= order)
		if (y <= maxy)
		    pl_line(ofp, scx(minx)-PLOT_TICK_WIDTH, scy(y), 
			scx(minx), scy(y));
	}
    }

    /* Connect the dots. */
    pl_move(ofp, scx(0), scy(y[0]));
    for (i=1; i<len; i++) {
	pl_cont(ofp, scx(i), scy(y[i]));
    }

    /* Get the lead out. */
    fflush(ofp);
}

/* Private variables (yech) */
static FILE *ofp;
int old_pytop;		/* where to start next plot */

/*--------------------------------------------------------------------------
 Prepare to plot.
 phys_y_height must unfortunately be how many inches total of paper
 will be used vertically, i.e. the sum of the phys_y_height args to 
 sig_plot_record.
 cmd should be xplot or lpr -g.
 margin should be zero for xplot, and 1 for lpr -g (yech).
 Uses plot(3) library, writes plot(5) data stream to given cmd.
--------------------------------------------------------------------------*/
void
plot_open(cmd, phys_y_height, margin)
    char *cmd;
    int phys_y_height;
    int margin;
{
    ofp = popen(cmd, "w");
    if (ofp == NULL) {
	fprintf(stderr, "Could not execute plot command %s.\n", cmd);
	exit(1);
    }
    pl_open(ofp);

    /* Clear screen (needed for real terminals) */
    pl_erase(ofp);

    /* leave extra margin because plot2ps screws up? */
    pl_space(ofp,margin*-PLOT_INCH/2,margin*-PLOT_INCH/4,
	    PLOT_WIDTH,phys_y_height*PLOT_INCH);
    old_pytop = 0;
}

/*--------------------------------------------------------------------------
 Finish plotting.
--------------------------------------------------------------------------*/
void
plot_close()
{
    assert(ofp != NULL);
    pl_close(ofp);
    pclose(ofp);
}

/*--------------------------------------------------------------------------
 Plot the real part of the given record of the signal.
--------------------------------------------------------------------------*/
void
sig_plot_r(this, n, phys_y_height)
    sig_t *this;
    int n;
    int phys_y_height;	/* Y span in inches [8 for square graph] */
{
    int offset;
    int i;
    char label[200];
    float *y;

    assert(this != NULL);
    assert(n < this->nrec);

    /* Convert record to float so all types can use same plot routine */
    y = (float *)malloc(sizeof(y[0]) * this->reclen);

    offset = n * this->reclen;

    sprintf(label, "%s record %d", this->fname, n);
    switch (this->kind) {
    case hdr_CINT8:
    case hdr_INT8:
	for (i=0; i<this->reclen; i++)
	    y[i] = (float) ((char *)this->re)[offset+i];
	plot_record(ofp, this->reclen, y, label, old_pytop, phys_y_height);
	break;

    case hdr_CINT16:
    case hdr_INT16:
	for (i=0; i<this->reclen; i++)
	    y[i] = (float) this->re[offset+i];
	plot_record(ofp, this->reclen, y, label, old_pytop, phys_y_height);
	break;

    case hdr_CBFP16: 
    case hdr_BFP16: {
	int exp = this->exp[n];
	typecast_bfp16_float32(this->reclen, this->re+offset, exp, y);
	plot_record(ofp, this->reclen, y, label, old_pytop, phys_y_height);
	} break;

    case hdr_CFLOAT32:
    case hdr_FLOAT32:
	plot_record(ofp, this->reclen, 
	    ((float *)this->re)+offset, label, old_pytop, phys_y_height);
	break;

    default:
	badkind("sig_plot_r", this->kind);
    }

    free(y);

    old_pytop += phys_y_height;
}

/*--------------------------------------------------------------------------
 Plot the imaginary part of the given record of the signal.
--------------------------------------------------------------------------*/
void
sig_plot_i(this, n, phys_y_height)
    sig_t *this;
    int n;
    int phys_y_height;	/* Y span in inches [8 for square graph] */
{
    int offset;
    int i;
    char label[200];
    float *y;

    assert(this != NULL);
    assert(n < this->nrec);

    /* Convert record to float so all types can use same plot routine */
    y = (float *)malloc(sizeof(y[0]) * this->reclen);

    offset = n * this->reclen;

    sprintf(label, "%s record %d", this->fname, n);
    switch (this->kind) {
    case hdr_CINT8:
	for (i=0; i<this->reclen; i++)
	    y[i] = (float) ((char *)this->im)[offset+i];
	plot_record(ofp, this->reclen, y, label, old_pytop, phys_y_height);
	break;

    case hdr_CINT16:
	for (i=0; i<this->reclen; i++)
	    y[i] = (float) this->im[offset+i];
	plot_record(ofp, this->reclen, y, label, old_pytop, phys_y_height);
	break;

    case hdr_CBFP16:
	typecast_bfp16_float32(this->reclen, 
	    this->im + offset, this->exp[n], y);
	plot_record(ofp, this->reclen, y, label, old_pytop, phys_y_height);
	break;

    case hdr_CFLOAT32:
	offset = n * this->reclen;
	plot_record(ofp, this->reclen, 
	    ((float *)this->im)+offset, label, old_pytop, phys_y_height);
	break;

    default:
	badkind("sig_plot_i", this->kind);
    }

    free(y);

    old_pytop += phys_y_height;
}
