/*--------------------------------------------------------------------------
 I/O operators for RPN calculator which manipulates complex vectors.
--------------------------------------------------------------------------*/
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <malloc.h>

#include "file.h"
#include "sig.h"
#include "hdr.h"

static char sccsid[] = "@(#)file.c	1.2 7/22/91";

/*--------------------------------------------------------------------------
 Read the given file into the given sig_t.
 Signal is represented on disk as follows:
 file :==
    hdr
    record *
 record :==
    [exponant] real * [ imag * ]
--------------------------------------------------------------------------*/
sig_t *
sig_read(fname)
    char *fname;
{
    sig_t *this;
    FILE *fp;
    int n;
    hdr_t hdr;
    char *vr, *vi;

    if ((fp=fopen(fname, "r")) == NULL) {
	perror(fname);
	exit(1);
    }

    /* Read file's header, allocate space. */
    hdr_read(fp, &hdr, fname);
    this = sig_new(hdr.nrec, hdr.reclen, hdr.kind, fname);

    vr = (char *)this->re;
    vi = (char *)this->im;

    /* Read file body, depending on data type. */
    for (n=0; n<hdr.nrec; n++) {
	if (hdr.kind & hdr_KIND_BFP) {
	    /* Read exponant. */
	    short exp;
	    exp = getc(fp) << 8;
	    exp |= getc(fp);
	    this->exp[n] = exp;
	}
	/* Read reals */
	if (fread(vr, sig_ELBYTES(this), hdr.reclen, fp) != hdr.reclen) {
	    fprintf(stderr, "sig_read: ?: premature EOF in signal %s\n", fname);
	    break;
	}
	vr += sig_ELBYTES(this) * hdr.reclen;
	if (hdr.kind & hdr_KIND_COMPLEX) {
	    /* Read imag */
	    if (fread(vi, sig_ELBYTES(this), hdr.reclen, fp) != hdr.reclen) {
		fprintf(stderr, "sig_read: ?: premature EOF in signal %s\n", fname);
		break;
	    }
	    vi += sig_ELBYTES(this) * hdr.reclen;
	}
    }
    fclose(fp);

    if (hdr.nrec != n) {
	this->nrec = n;
	this->re = realloc(this->re, n * this->reclen * sig_ELBYTES(this));
	if (hdr.kind & hdr_KIND_COMPLEX)
	    this->im = realloc(this->im, n * this->reclen * sig_ELBYTES(this));
	if (hdr.kind & hdr_KIND_BFP)
	    this->exp = realloc(this->exp, n * sizeof(short));
    }


    return this;
}

/*--------------------------------------------------------------------------
 Write the given sig_t onto the given file.
 Signal is represented on disk as follows:
 file :==
    hdr
    record *
 record :==
    [exponant] real * [ imag * ]
--------------------------------------------------------------------------*/
void
sig_write(this, fname)
    sig_t *this;
    char *fname;
{
    FILE *fp;
    int n;
    char *vr, *vi;

    strcpy(this->fname, fname);
    if ((fp=fopen(fname, "w")) == NULL) {
	perror(fname);
	exit(1);
    }

    /* Write file's header. */
    hdr_write(fp, this->nrec, this->reclen, this->kind);

    vr = (char *)this->re;
    vi = (char *)this->im;

    /* Write file body, depending on data type. */
    for (n=0; n<this->nrec; n++) {
	if (this->kind & hdr_KIND_BFP) {
	    /* Write exponant. */
	    short exp;
	    exp = this->exp[n];
	    putc(exp>>8, fp);
	    putc(exp, fp);
	}
	/* Write reals */
	if (fwrite(vr, sig_ELBYTES(this), this->reclen, fp) != this->reclen) {
	    fprintf(stderr, "sig_write: ?: Can't write signal %s\n", fname);
	    exit(1);
	}
	vr += sig_ELBYTES(this) * this->reclen;
	if (this->kind & hdr_KIND_COMPLEX) {
	    /* Write imag */
	    if (fwrite(vi, sig_ELBYTES(this), this->reclen, fp) != this->reclen) {
		fprintf(stderr, "sig_write: ?: Can't write signal %s\n", fname);
		exit(1);
	    }
	    vi += sig_ELBYTES(this) * this->reclen;
	}
    }
    fclose(fp);
}
