// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)
    based on code by Marc Shapiro (shapiro@sor.inria.fr)

This file is part of the GNU C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _ThreadPtrXPlex_h
#ifdef __GNUG__
#pragma interface
#endif
#define _ThreadPtrXPlex_h 1

#include "ThreadPtr.Plex.h"

class ThreadPtrXPlex: public ThreadPtrPlex
{
  ThreadPtrIChunk*       ch;           // cached chunk

  void             make_initial_chunks(int up = 1);

  void             cache(int idx) const;
  void             cache(const ThreadPtr* p) const;

  ThreadPtr*             dopred(const ThreadPtr* p) const;
  ThreadPtr*             dosucc(const ThreadPtr* p) const;

  void             set_cache(const ThreadPtrIChunk* t) const; // logically, 
                                               // not physically const
public:
                   ThreadPtrXPlex();                 // set low = 0;
                                               // fence = 0;
                                               // csize = default

                   ThreadPtrXPlex(int ch_size);      // low = 0; 
                                               // fence = 0;
                                               // csize = ch_size

                   ThreadPtrXPlex(int lo,            // low = lo; 
                            int ch_size);      // fence=lo
                                               // csize = ch_size

                   ThreadPtrXPlex(int lo,            // low = lo
                            int hi,            // fence = hi+1
                            const ThreadPtr  initval,// fill with initval,
                            int ch_size = 0);  // csize= ch_size
                                               // or fence-lo if 0

                   ThreadPtrXPlex(const ThreadPtrXPlex&);
  
  void             operator= (const ThreadPtrXPlex&);

// virtuals


  ThreadPtr&             high_element ();
  ThreadPtr&             low_element ();

  const ThreadPtr&       high_element () const;
  const ThreadPtr&       low_element () const;

  Pix              first() const;
  Pix              last() const;
  void             prev(Pix& ptr) const;
  void             next(Pix& ptr) const;
  int              owns(Pix p) const;
  ThreadPtr&             operator () (Pix p);
  const ThreadPtr&       operator () (Pix p) const;

  int              low() const; 
  int              high() const;
  int              valid(int idx) const;
  void             prev(int& idx) const;
  void             next(int& x) const;
  ThreadPtr&             operator [] (int index);
  const ThreadPtr&       operator [] (int index) const;
    
  int              Pix_to_index(Pix p) const;
  Pix              index_to_Pix(int idx) const;    

  int              can_add_high() const;
  int              can_add_low() const;
  int              full() const;

  int              add_high(const ThreadPtr  elem);
  int              del_high ();
  int              add_low (const ThreadPtr  elem);
  int              del_low ();

  void             fill(const ThreadPtr  x);
  void             fill(const ThreadPtr  x, int from, int to);
  void             clear();
  void             reverse();
  void             append(const ThreadPtrXPlex& a); 
  void             prepend(const ThreadPtrXPlex& a);
    
  int              OK () const; 

};


inline void ThreadPtrXPlex::prev(int& idx) const
{
  --idx;
}

inline void ThreadPtrXPlex::next(int& idx) const
{
  ++idx;
}

inline  int ThreadPtrXPlex::full () const
{
  return 0;
}

inline int ThreadPtrXPlex::can_add_high() const
{
  return 1;
}

inline int ThreadPtrXPlex::can_add_low() const
{
  return 1;
}

inline  int ThreadPtrXPlex::valid (int idx) const
{
  return idx >= lo && idx < fnc;
}

inline int ThreadPtrXPlex::low() const
{
  return lo;
}

inline int ThreadPtrXPlex::high() const
{
  return fnc - 1;
}

inline ThreadPtr& ThreadPtrXPlex:: operator [] (int idx)
{
  if (!ch->actual_index(idx)) cache(idx);
  return *(ch->pointer_to(idx));
}

inline const ThreadPtr& ThreadPtrXPlex:: operator [] (int idx) const
{
  if (!ch->actual_index(idx)) cache(idx);
  return *((const ThreadPtr*)(ch->pointer_to(idx)));
}

inline  ThreadPtr& ThreadPtrXPlex::low_element ()
{
  if (empty()) index_error();
  return *(hd->pointer_to(lo));
}

inline  const ThreadPtr& ThreadPtrXPlex::low_element () const
{
  if (empty()) index_error();
  return *((const ThreadPtr*)(hd->pointer_to(lo)));
}

inline  ThreadPtr& ThreadPtrXPlex::high_element ()
{
  if (empty()) index_error();
  return *(tl()->pointer_to(fnc - 1));
}

inline const ThreadPtr& ThreadPtrXPlex::high_element () const
{
  if (empty()) index_error();
  return *((const ThreadPtr*)(tl()->pointer_to(fnc - 1)));
}

inline  int ThreadPtrXPlex::Pix_to_index(Pix px) const
{
  ThreadPtr* p = (ThreadPtr*)px;
  if (!ch->actual_pointer(p)) cache(p);
  return ch->index_of(p);
}

inline  Pix ThreadPtrXPlex::index_to_Pix(int idx) const
{
  if (!ch->actual_index(idx)) cache(idx);
  return (Pix)(ch->pointer_to(idx));
}

inline Pix ThreadPtrXPlex::first() const
{
  return Pix(hd->ThreadPtrIChunk::first_pointer());
}

inline Pix ThreadPtrXPlex::last() const
{
  return Pix(tl()->ThreadPtrIChunk::last_pointer());
}

inline void ThreadPtrXPlex::prev(Pix& p) const
{
  Pix q = Pix(ch->ThreadPtrIChunk::pred((ThreadPtr*) p));
  p = (q == 0)? Pix(dopred((const ThreadPtr*) p)) : q;
}

inline void ThreadPtrXPlex::next(Pix& p) const
{
  Pix q = Pix(ch->ThreadPtrIChunk::succ((ThreadPtr*) p));
  p = (q == 0)? Pix(dosucc((const ThreadPtr*)p)) : q;
}

inline ThreadPtr& ThreadPtrXPlex:: operator () (Pix p)
{
  return *((ThreadPtr*)p);
}

inline const ThreadPtr& ThreadPtrXPlex:: operator () (Pix p) const
{
  return *((const ThreadPtr*)p);
}

inline void ThreadPtrXPlex::set_cache(const ThreadPtrIChunk* t) const
{
  ((ThreadPtrXPlex*)(this))->ch = (ThreadPtrIChunk*)t;
}

#endif
