#ifndef _LPmsg_h_
#define _LPmsg_h_

#include <assert.h>
#include <bool.h>
#include "TimeUnit.h"
#include "LPmsgType.h"
#include "ClassId.h"

class LP;
class LPmsg;
typedef LPmsg *LPmsgptr;

class LPmsg {
protected:
    //
    // This information identifies the recipient and the time the message
    // should arrive.
    //
    TimeUnit pTime;
    int pPriority;   // priority of events occuring at the same time
    LP *pWho;

    //
    // This information identifies the sender of the information, and the
    // channel number the recipient should use to send things back to the
    // sender.
    //
    LP *pSender;
    int pReturnChannel;

    int pType;

public:

    //
    // for use by main process
    //
    LPmsg(LP* who, int type,  TimeUnit when, int prio = 0);
    LPmsg(int type,  TimeUnit when, int prio = 0);
    LPmsg(int type);
    LPmsg();

    static ClassId classId();
    virtual bool isA(ClassId);

    void simulatedTime( TimeUnit );
    TimeUnit simulatedTime();

    LP* who();
    void who( LP* );

    int type();
    void type(int );

    void priority( int );
    int priority( );

    void sender( LP* );
    LP* sender( );

    void returnChannel( int );
    int returnChannel( );

    int compare(LPmsg &p);
    operator==(LPmsg &p);
    operator<(LPmsg &p);
    operator<=(LPmsg &p);
    operator>(LPmsg &p);
    operator>=(LPmsg &p);
}; 

static inline
LPmsg::LPmsg()
{
    pWho = 0;
    pType = 0;

    pTime = 0;
    pPriority = 0;
}

static inline
LPmsg::LPmsg( LP* who, int type, TimeUnit w, int prio)
{
    pWho = who;
    pType = type;

    pTime = w;
    pPriority = prio;
}

static inline
LPmsg::LPmsg(int type, TimeUnit w, int prio)
{
    pType = type;

    pTime = w;
    pPriority = prio;
}

static inline
LPmsg::LPmsg(int type)
{
    pType = type;
    //
    // a null time means that the simulator will fill in with the
    // current time.
    //
    pTime = 0;
    pPriority = 0;
}

static inline LP*
LPmsg::who()
{
    return(pWho);
}

static inline void
LPmsg::who(LP* w)
{
    pWho = w;
}


static inline int
LPmsg::type()
{
    return(pType);
}

static inline void
LPmsg::type(int w)
{
    pType = w;
}


static inline int
LPmsg::priority()
{
    return(pPriority);
}

static inline void
LPmsg::priority(int w)
{
    pPriority = w;
}

static inline LP*
LPmsg::sender()
{
    return(pSender);
}

static inline void
LPmsg::sender(LP* w)
{
    pSender = w;
}

static inline int
LPmsg::returnChannel()
{
    return(pReturnChannel);
}

static inline void
LPmsg::returnChannel(int w)
{
    pReturnChannel = w;
}

static inline TimeUnit
LPmsg::simulatedTime()
{
    return(pTime);
}

static inline void
LPmsg::simulatedTime(TimeUnit w)
{
    pTime = w;
}

static inline int
LPmsg::compare(LPmsg &p)
{
    if ( pTime < p.pTime ) {
	return( -1 );
    } else if (pTime == p.pTime ) {
	return (pPriority - p.pPriority);
    } else return ( 1 );
}

static inline int
LPmsg::operator==(LPmsg &p)
{
    return( pPriority == p.pPriority && pTime == p.pTime );
}

static inline int
LPmsg::operator<(LPmsg &p)
{
    return( pTime < p.pTime 
	   || ( pTime == p.pTime && pPriority < p.pPriority ) );
}

static inline int
LPmsg::operator<=(LPmsg &p)
{
    return( pTime < p.pTime 
	   || ( pTime == p.pTime && pPriority <= p.pPriority ) );
}

static inline int
LPmsg::operator>(LPmsg &p)
{
    return( pTime > p.pTime 
	   || ( pTime == p.pTime && pPriority > p.pPriority ) );
}

static inline int
LPmsg::operator>=(LPmsg &p)
{
    return( pTime > p.pTime 
	   || ( pTime == p.pTime && pPriority >= p.pPriority ) );
}

#endif
