#include <stream.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <errno.h>

#include "Async.h"
#include "CpuMultiplexor.h"
#include "CpuMultiplexorP.h"
#include "assert.h"


#define IOSYSCALL( cc, type )						\
		if ( cc == -1 && errno == EWOULDBLOCK )			\
		{							\
			if ( ioWait( type ) == YES )			\
				ThisCpu->reserveByException( this ) ;	\
		}							\
		else							\
			return( cc )
			

/*
 * Returns YES if more bytes can be written, NO otherwise
 */
static inline Boolean AsyncIO::streamWrite( int cc, 
				unsigned& len, int& bufpos, int& written )
{
        if ( cc == -1 )
        {
                if ( errno == EWOULDBLOCK && ioWait( WRITE ) == YES )
		{
                        ThisCpu->reserveByException( this ) ;
			return( YES ) ;
		}
		else
			return( NO ) ;		// error
        }
        else if ( cc == 0 )
                return( NO ) ;
        else  // cc > 0
        {
                len -= cc ;
                bufpos += cc ;
                written += cc ;
                return( YES ) ;
        }
}

/*
 * The RETURN macro is used by calls that do writing (like write and send).
 * If an error happens in the middle of a write (for example,
 * the other side of a network connection dies) the write
 * will return the # of bytes written up to that point
 * and the next write call will return an error.
 * If no bytes have been written, an error is returned immediately
 */
#define RETURN( written, error )   return( ( written > 0 ) ? written : error )


AsyncIO::read( int fd, void *buf, unsigned nbytes )
{
	init( fd ) ;
	for ( ;; )
	{
		int bytes_in = syscall( SYS_read, fd, buf, nbytes ) ;
		IOSYSCALL( bytes_in, READ ) ;
	}
}


/*
 * We write out all the buffer, unless an error happens
 */
AsyncIO::write( int fd, const void *buf, unsigned nbytes )
{
	init( fd ) ;

	int bytes_written = 0 ;
	int bufpos = 0 ;
	int cc ;

	while ( nbytes > 0 )
	{
		cc = syscall( SYS_write, fd, &buf[ bufpos ], nbytes ) ;
		if ( streamWrite( cc, nbytes, bufpos, bytes_written ) == NO )
			break ;
	}
	RETURN( bytes_written, cc ) ;
}


AsyncIO::accept( int fd, struct sockaddr *name, int *namelen )
{
	init( fd ) ;
	for ( ;; )
	{
		int sd = syscall( SYS_accept, fd, name, namelen ) ;
		IOSYSCALL( sd, READ ) ;
	}
}


AsyncIO::connect( int fd, struct sockaddr *name, int namelen )
{
	init( fd ) ;
	for ( ;; )
	{
		int status = syscall( SYS_connect, fd, name, namelen ) ;
		if ( status == -1 )
		    switch ( errno )
		    {
			case EWOULDBLOCK:
			case EINPROGRESS:
			case EALREADY:
				if ( ioWait( WRITE ) == YES )
					ThisCpu->reserveByException( this ) ;
				break ;
			
			case EISCONN:
				status = 0 ;
						// fall through
			default:
				return( status ) ;
		    }
	}
}


/*
 * We send the whole message unless an error happens
*/
AsyncIO::send( int fd, char *msg, int len, int flags )
{
	init( fd ) ;

	int bytes_sent = 0 ;
	int msgpos = 0 ;
	int cc ;

	while ( len > 0 )
	{
	    cc = syscall( SYS_send, fd, &msg[ msgpos ], len, flags ) ;
	    if ( streamWrite( cc, unsigned( len ), msgpos, bytes_sent ) == NO )
			break ;
	}
	RETURN( bytes_sent, cc ) ;
}


AsyncIO::sendto( int fd, char *msg, int len, int flags,
					struct sockaddr *to, int tolen )
{
	init( fd ) ;

	int bytes_sent = 0 ;
	int msgpos = 0 ;
	int cc ;

	while ( len > 0 )
	{
	    int bytes_sent = syscall( SYS_sendto, fd, msg, len, flags,
							to, tolen ) ;
	    if ( streamWrite( cc, unsigned( len ), msgpos, bytes_sent ) == NO )
			break ;
	}
	RETURN( bytes_sent, cc ) ;
}


AsyncIO::sendmsg( int fd, struct msghdr *msg, int flags )
{
	init( fd ) ;
	for ( ;; )
	{
		int bytes_sent = syscall( SYS_sendmsg, fd, msg, flags ) ;
		IOSYSCALL( bytes_sent, WRITE ) ;
	}
}


AsyncIO::recv( int fd, char *buf, int len, int flags )
{
	init( fd ) ;
	for ( ;; )
	{
		int bytes_rcvd = syscall( SYS_recv, fd, buf, len, flags ) ;
		IOSYSCALL( bytes_rcvd, READ ) ;
	}
}


AsyncIO::recvfrom( int fd, char *buf, int len, int flags,
					struct sockaddr *from, int *fromlen )
{
	init( fd ) ;
	for ( ;; )
	{
		int bytes_rcvd = syscall( SYS_recvfrom, fd, buf, len, flags,
					from, fromlen ) ;
		IOSYSCALL( bytes_rcvd, READ ) ;
	}
}


AsyncIO::recvmsg( int fd, struct msghdr *msg, int flags )
{
	init( fd ) ;
	for ( ;; )
	{
		int bytes_rcvd = syscall( SYS_recvmsg, fd, msg, flags ) ;
		IOSYSCALL( bytes_rcvd, READ ) ;
	}
}


