#include <stream.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <signal.h>

#ifdef ultrix
#include <sys/smp_lock.h>
#endif

#include <fcntl.h>

#include "CpuMultiplexor.h"
#include "CpuMultiplexorP.h"
#include "Thread.h"
#include "assert.h"

#include "Async.h"

Boolean AsyncIO::initialized = NO ;

AsyncIO *asyncIO_array ;

static struct timeval no_wait = { 0, 0 } ;

typedef void (*finFunc)( int, void * ) ;

extern "C"
{
	int on_exit( finFunc, void * ) ;
} ;

extern int errno ;


#ifdef USE_SIGIO
/*
 * Invoked when a SIGIO happens
 */
void sigioHandler()
{
	ioCheck( &no_wait ) ;
}
#endif



/*
 * SIGIO must be blocked while ioCheck is running
 */
void ioCheck( struct timeval *tv )
{
	int owner = ThisCpu->cpuId() ;

	if ( AsyncIO::fdWidth[ owner ] == 0 )
		return ;

	fd_set readFds = AsyncIO::readMask[ owner ] ;
	fd_set writeFds = AsyncIO::writeMask[ owner ] ;

	int nready = select( AsyncIO::fdWidth[ owner ], &readFds, &writeFds,
						FD_SET_NULL, tv ) ;

	if ( nready == 0 )
		return ;

	/*
	 * Inspect the read set first
	 */
	int inReady = fdReschedule( &readFds,
		&AsyncIO::readMask[ owner ], AsyncIO::fdWidth[ owner ] ) ;
	if ( nready != inReady )
	{
		int outReady = fdReschedule( &writeFds,
		   &AsyncIO::writeMask[ owner ], AsyncIO::fdWidth[ owner ] ) ;
		ASSERT( inReady + outReady == nready ) ;
	}

	/*
	 * Calculate new AsyncIO::fdWidth
	 */
	int maxFd = -1 ;
	for ( int fd = 0 ; fd < AsyncIO::fdWidth[ owner ] ; fd++ )
		if ( FD_ISSET( fd, &AsyncIO::readMask[ owner ] ) ||
				FD_ISSET( fd, &AsyncIO::writeMask[ owner ] ) )
			maxFd = fd ;
	AsyncIO::fdWidth[ owner ] = maxFd + 1 ;
}



/*
 * Returns the number of fds found set in the mask.
 * These fd's are reset in the ioMask and threads blocked on
 * them are rescheduled
 */
static int fdReschedule( fd_set *activeFds, fd_set *ioMask, int width )
{
	int fdCount = 0 ;

	for ( int fd = 0 ; fd < width ; fd++ )
		if ( FD_ISSET( fd, activeFds ) )
		{
			asyncIO_array[ fd ].reschedule() ;
			FD_CLR( fd, ioMask ) ;
			fdCount++ ;
		}
	return( fdCount ) ;
}


void AsyncIO::ioInit()
{
	if ( initialized == NO )
	{
		syscall( SYS_write, 2, "Beginning initialization\n", 25 ) ;
		n_fds = getdtablesize() ;

		asyncIO_array = new AsyncIO[ n_fds ] ;

		FD_ZERO( &readMask[ 0 ] ) ;
		FD_ZERO( &writeMask[ 0 ] ) ;
		fdWidth[ 0 ] = 0 ;

#ifdef USE_SIGIO
		(void) signal( SIGIO, sigioHandler ) ;
#endif
#ifdef sparc
		(void) on_exit( AsyncIO::ioFin, 0 ) ;
#endif

		initialized = YES ;
	}
}


void AsyncIO::ioFin( int status, void *p )
{
	(void) syscall( SYS_write, 2, "IO: FINALIZED\n", 14 ) ;
	for ( int i = 0 ; i < n_fds ; i++ )
		asyncIO_array[i].fin() ;
}


/*
 * Try to set the fd for asynchronous I/O
 */
void AsyncIO::set_async()
{
	switch( fdState )
	{
		case ASYNC:
		case NO_ASYNC:
			break ;
		
		case READY:
			struct stat st ;

			if ( fstat( fd, &st ) == -1 )
				fdState = NO_ASYNC ;
			else
			{
				oldFdState = fcntl( fd, F_GETFL, 0 ) ;
				if ( oldFdState != -1 )
				{
#ifdef USE_SIGIO
					int flags = FASYNC + FNDELAY ;
#else
					int flags = FNDELAY ;
#endif
					if ( fcntl( fd, F_SETFL, flags ) == -1 )
						fdState = NO_ASYNC ;
					else
						fdState = ASYNC ;
				}
				else
					fdState = NO_ASYNC ;
			}
			break ;
		
		case UNUSED:
		default:
			cerr << "Bad fdState. fd = " << fd << "\n" ;
			exit( 1 ) ;
	}
}


void AsyncIO::fin()
{
	if ( fdState == ASYNC )
	{
		int status = fcntl( fd, F_SETFL, oldFdState ) ;
		ASSERT( status != -1 ) ;
	}
	fdState = UNUSED ;
}


/*
 * Returns YES if the thread was blocked, NO if it wasn't blocked
 */
Boolean AsyncIO::ioWait( IoType ioType )
{
	/*
	 * First thing we do is block SIGIO
	 * It will be renabled in reserveByException
	 *
	 * NOTE: it is possible that a SIGIO occurs between the operation
	 *	that invoked us and the block by sigsetmask.
	 *	The SIGIO handler moves the
	 *	threads waiting for I/O on a file descriptor to the
	 *	CPU queue. This thread which has decided to block itself
	 *	is not yet placed in the I/O queue of the file descriptor.
	 *	The solution is to check (using select) if I/O is available
	 *	on the fd before blocking ourselves.
	 */
	Boolean blocked ;

#ifdef USE_SIGIO
	fd_set mask ;

	oldSignalMask = sigblock( sigmask( SIGIO ) ) ;
	FD_ZERO( &mask ) ;
	FD_SET( fd, &mask ) ;
	int fdReady = select( fd+1, ( ioType == READ ) ? &mask : FD_SET_NULL,
				 ( ioType == WRITE ) ? &mask : FD_SET_NULL,
				 FD_SET_NULL,
				 &no_wait ) ;

	if ( fdReady > 0 )
	{
		(void) sigsetmask( oldSignalMask ) ;
		blocked = NO ;
	}
	else
#endif
	{
		blocked = YES ;
		int owner = ThisCpu->cpuId() ;
		FD_SET( fd, ( ioType == READ ) ? &readMask[ owner ]
					       : &writeMask[ owner ] ) ;
		if ( fd >= fdWidth[ owner ] )
			fdWidth[ owner ] = fd + 1 ;
	}
	return( blocked ) ;
}


void AsyncIO::reschedule()
{
	while ( ! pileOfThreads.isEmpty() )
	{
		Thread *p = (Thread *) pileOfThreads.remove() ;
		ThisCpu->add( p ) ;
	}
}


int AsyncIO::reserveByException( Thread *byWho )
{
	pileOfThreads.add( byWho ) ;
#ifdef USE_SIGIO
	(void) sigsetmask( oldSignalMask ) ;	// allow SIGIO again
#endif
	return( 1 ) ;				// always blocked
}

