// This may look like C code, but it is really -*- C++ -*-
// 
// Copyright (C) 1988 University of Illinois, Urbana, Illinois
//
// written by Dirk Grunwald (grunwald@cs.uiuc.edu)
//
#ifndef CpuMultiplexor_h
#define CpuMultiplexor_h

#ifdef __GNUG__
#  pragma once
#  pragma interface
#endif

#include <AwesimeConfig.h>
#include <HardwareContext.h>
#include <CpuMuxExceptions.h>
#include <SpinLock.h>

//
//	Define the abstractions for per UNIX process resources.
//

#include <ThreadContainer.h>

class Thread;
class ReserveByException;
class AwesimeHeap;
class SpinLock;
class SpinFetchAndOp;
class CpuMultiplexor;

//
//	Filled in by the single original UNIX process. This value may
//	change during execution
//

extern int CpuMultiplexors;
extern CpuMultiplexor *ThisCpu;
extern int CpuMuxDebugFlag;

extern SpinLock CpuCerrLock;

class CpuMultiplexor {
protected:

    friend class SimulationMultiplexor;
    friend class SingleSimMux;
    friend class MultiSimMux;
    friend class MonitorSimMux;

    Thread *currentThread;
    HardwareContext systemContext;

    ThreadContainer *myCurrentEvents;
    SpinLock *myCurrentEventsLock;
    int *myCurrentEventsCounter;

    int iYam;	// I yam what I yam
    int pid;	// ...except to UNIX

    char nameSpace[128];
    char *pNameTemplate;
    char *pName;

    volatile int *terminated;
    int enabled;

    virtual Thread *remove();
    //
    // Constructors of this class and all subclasses should call their
    // own allocateLocalEventStructures, because constructors automatically
    // chain the calls.
    //
    // Use AllocateEventStructures when allocating resources for a CPU and
    // you're not in the constructor. Use deallocateEventStructures for
    // removing those resources. These routines should call subclass
    // instances of the same names.
    //
    //void allocateLocalEventStructures(int,int);
    virtual void allocateEventStructures(int,int);
    virtual void deallocateEventStructures();

    //
    // Primarily provided so subclasses, e.g. SimulationMultiplexor,
    // can efficiently add threads without a lot of locking.
    // Note that these are non-virtuals.
    //
    void addUnlocked(Thread *);

    //
    //	Exception handlers.
    //

    friend class ExceptionReserve;
    friend class ExceptionTerminate;
    friend class ExceptionReschedule;
    friend class ExceptionIveSuspended;
    friend class ExceptionRelocate;
    friend class ExceptionEnrollDismissCpu;

    ExceptionClass *raisedBy;
    ExceptionReserve reserveException;
    ExceptionTerminate terminateException;
    ExceptionReschedule rescheduleException;
    ExceptionIveSuspended iveSuspendedException;

//
//	Public interfaces for exceptions
//
public:

    void raise(ExceptionClass *);
    void reserveByException( ReserveByException * );
    void threadTerminateException( Thread* );
    void flick();

public:
    CpuMultiplexor(int debug = 0);
    virtual ~CpuMultiplexor();

    virtual void fireItUp(int Cpus = 1, unsigned shared = (4196 * 500));
    virtual void warmThePot(int);
    virtual void stirItAround();
    virtual void coolItDown();

    virtual void terminateAll();

    virtual void add(Thread *);

    Thread * CurrentThread();

    void debug(int newdebug);
    int debug();
    const char *name();

    int cpuId();
};

static inline int
CpuMultiplexor::cpuId()
{
    return( iYam );
}

static inline const char *
CpuMultiplexor::name()
{
    return(pName);
}

static inline Thread *
CpuMultiplexor::CurrentThread()
{
    return( currentThread );
}

//
//	Expansions for exception handling
//
#include <Thread.h>

static inline void
CpuMultiplexor::flick()
{
    currentThread -> cpuAffinity = -1;
    raise( &rescheduleException ); 
}

static inline Thread *
CurrentThread()
{
    return( ThisCpu -> CurrentThread() );
}

static inline void
CheckCurrentStack()
{
    CurrentThread() -> checkStack();
}

//
//	These are things the user needs to define
//

extern ThreadContainer *AllocateHardwareCurrentEventsStructure();

#endif /* CpuMultiplexor_h */
