// This may look like C code, but it is really -*- C++ -*-
// 
// Copyright (C) 1988 University of Illinois, Urbana, Illinois
//
// written by Dirk Grunwald (grunwald@cs.uiuc.edu)
//

#ifdef __GNUG__
#  pragma implementation
#endif

#include "MonitorSimMux.h"
#include "CpuMuxP.h"
#include "SpinBarrier.h"
#include "Statistic.h"
#include "AwesimeConfig.h"
#include "SEv.PQ.h"
#include "OEv.h"
#include "Thread.h"

static Statistic ThreadsPerTick;

extern int CpuMuxDebugFlag;

MonitorSimMux *ThisMonitorSimMux;

MonitorSimMux::MonitorSimMux(bool debug) : (debug)
{
    ThisMonitorSimMux = this;

    pNameTemplate = "MonitorSimMux";
    iYam = 0;
    sprintf(nameSpace, "[%s-%d] ", pNameTemplate, iYam);
    pName = nameSpace;
}

void
MonitorSimMux::fireItUp(int, unsigned)
{
    if ( myEvents -> empty() ) {
	timeThisSet = myEvents -> front().time();
    } else {
	timeThisSet = SimMux::NullTime();
    }
    eventsThisSet = 0;

    pid = getpid();
    enabled = 1;
    
    while( ! *terminated ) {
	
	if ( myEvents -> empty() ) {
	    break;
	} else {
	    SEv ev = myEvents -> deq();
	    CurrentSimulatedTime = ev.time();
	    
	    
	    if ( CurrentSimulatedTime > timeThisSet ) {
		ThreadsPerTick += eventsThisSet;
		eventsThisSet = 1;
		timeThisSet = CurrentSimulatedTime;
	    } else {
		eventsThisSet++;
	    }
	    
	    //
	    // if it's not a thread event, execute the OE
	    //
	    if ( ev.isOtherEvent() ) {
		currentThread = 0;
		ev.oev() -> executeEvent();
	    }
	    else {
		currentThread = ev.thread();
		
		CERR_PRE;
		cerr << CpuMux::Cpu() -> name();
		cerr << " switchTo to " << currentThread -> name();
		cerr << " at " << CurrentSimulatedTime << "\n";
		CERR_POST;
		
		systemTransfer( currentThread );

		raisedBy -> handleException();
		raisedBy = 0;
	    }
	}
    }
    
    CERR_ALWAYS_PRE;
    cerr << name() << "Statistics for threads per tick\n";
    cerr << name() << ThreadsPerTick << "\n";
    cerr << name() << "Min = " << ThreadsPerTick.min() << "\n";
    cerr << name() << "Max = " << ThreadsPerTick.max() << "\n";
    CERR_POST;
}

Statistic *
MonitorSimMux::threadsPerTick()
{
    return (&ThreadsPerTick);
}

void
MonitorSimMux::await_(SimTimeUnit when)
{
    if (when > CurrentSimulatedTime) {
	
	if ( myEvents -> empty() ) {
	    CurrentSimulatedTime = when;
	} else {
	    
	    SEv& front = myEvents -> front();
	    //
	    // is the current one awaiting a time sooner than any other one?
	    //
	    
	    if ( when <= front.time() ) { 
		CurrentSimulatedTime = when;

		CERR_PRE;
		cerr << CpuMux::Cpu() -> name();
		cerr << " (await) bump time to " << when << "\n";
		CERR_POST;

	    } else {
		
		//
		// if it's an event, not a thread, suspend and let the
		// scheduler do it in the UNIX process context.
		//
		
		if  ( front.isOtherEvent() ) {
		    
		    myEvents -> enq( SEv(currentThread, when) );
		    CpuMux::Cpu() -> raise( &iveSuspendedException );

		} else { 
		    Thread *wuz = currentThread;
		    CurrentSimulatedTime = front.time();
		    currentThread = front.thread();
		    myEvents -> del_front();
		    
		    myEvents -> enq( SEv(wuz, when) );

		    if ( CurrentSimulatedTime > timeThisSet ) {
			ThreadsPerTick += eventsThisSet;
			eventsThisSet = 1;
			timeThisSet = CurrentSimulatedTime;
		    } else {
			eventsThisSet++;
		    }
	    

		    CERR_PRE;
		    cerr << CpuMux::Cpu() -> name();
		    cerr << "(await) transfer to " << currentThread -> name() << "\n";
		    CERR_POST;

		    threadTransfer(wuz, currentThread);

		}
	    }
	}
    }
}
