/*
 * jmp - trivial prime jump table
 *
 * If x is divisable by a trivial prime (2,3,5,7,11), then:
 *
 *		x + jmpindx[ (x>>1)%JMPMOD ]
 *
 * is the value of the smallest value > x that is not divisable by a
 * trivial prime.  JMPMOD is the product of the odd non-trivial primes.
 *
 * This table is useful for skipping values that are obviously not prime
 * by skipping values that are a multiple of trivial prime.
 *
 * If x is not divisable by a trivial prime, then:
 *
 *		x + jmp[ -jmpindx[(x>>1)%JMPMOD] ]
 *
 * Instead of testing successive odd values, this system allows us to
 * skip odd values divisable by trivial primes.  This is process on the
 * average reduces the values we need to test by a factor of at least 2.4.
 */

/*
 * trivial prime constants
 */
#define JMPMOD (3*5*7*11)	/* product of odd trivial primes */
#define JMPSIZE (2*4*6*10)	/* ints mod JMPMOD not div by trivial primes */
#define JPRIME (prime+4)	/* pointer to first non-trivial prime */

/* given x, return the index within jmpindx that applies */
#define jmpmod(x) (((x)>>1)%JMPMOD)

/* jmpindx table value */
#define jmpindxval(x) (jmpindx[jmpmod(x)])

/* return the smallest value >= x not divisable by a trivial prime */
#define firstjmp(x,tmp) ((tmp) = jmpindxval(x), ((tmp) > 0) ? ((x)+(tmp)) : (x))

/* given x not divisable by a trivial prime, return jmp[] index */
#define jmpptr(x) (-jmpindxval(x))

/* given a jmp pointer, return current jump increment and bump the pointer */
#define nxtjmp(p) ( *( ((p)<lastjmp) ? ((p)++) : (((p)=jmp),lastjmp) ) )

/* given a jmp pointer, return dec pointer and return previous jump increment */
#define prevjmp(p) ( *( ((p)>jmp) ? (--(p)) : ((p)=lastjmp) ) )

/*
 * external jump tables
 */
extern short jmpindx[];
extern unsigned char jmp[];
extern unsigned char *lastjmp;
