% SAVE.RED  F.J.Wright, FJW@Maths.QMW.AC.UK, London  27/3/92

% Commands for saving and displaying interactive
% input, optionally together with its output.
% See SAVE.DOC for further description.

%%module save;

% To allow re-reading source:
remprop('Save,      'stat);
remprop('Save,      'avalue);
remprop('SaveIO,    'avalue);
remprop('SaveQuery, 'avalue);
remprop('DisplayIn, 'avalue);
remprop('DisplayIO, 'avalue);


% SAVE user interface commands:
% ============================

% The main routine is "Save arglist", which can be
% controlled directly by the following switches:

switch SaveChain;
   % Chain back through input/ws statements if ON.
symbolic(!*SaveChain := t);  % ON is flagged ignore!

% The following two options can also be controlled using
% alternative interface commands.
% They MUST be fluid, which is the default for switches anyway.

switch SaveIO;
   % Save/Display input together with output.

switch SaveQuery;
   % Interactively query saving.


% These commands provide alternative interfaces:

symbolic procedure SaveIO arglist;
   % Save interactive input plus output.
   begin scalar !*SaveIO;  !*SaveIO := t;
      Save arglist
   end;

symbolic procedure SaveQuery arglist;
   % Save interactive input with interactive querying.
   begin scalar !*SaveIO, !*SaveQuery;  !*SaveQuery := t;
      !*SaveIO := Yesp "Display and save algebraic results also?";
      Save arglist
   end;


% DISPLAY user interface commands:
% ===============================

symbolic procedure DisplayIn arglist;
   % Display only interactive input with prompts,
   % like standard DISPLAY.
   begin scalar !*SaveIO, !*SaveQuery;
      Save('t . arglist)
   end;

symbolic procedure DisplayIO arglist;
   % Display interactive input with prompts plus output.
   begin scalar !*SaveIO, !*SaveQuery;
      !*SaveIO := t;
      Save('t . arglist)
   end;


% The internal routines that do the work:
% ======================================

fluid '(LastStmt SaveErrorFlag NumSaved);

%%%%% infix to;  % Already declared as power operator

symbolic procedure Save arglist;
   % Save/Display input together with output if !*SaveIO is true.
   % Interactively query saving if !*SaveQuery is true.
   % Chain back through input/ws statements if !*SaveChain is true.
   begin scalar LastStmt, file, NumSaved;
      if null getd 'editp then load cedit;  % Needed later
   % Always output this to the current output stream (terminal):
      if !*SaveIO then lprim list(
            "Algebraic results printed subject",
            "to display settings in effect NOW!");
      LastStmt := caar crbuflis!*;
   % Allow no arguments at all:
      if null arglist then arglist := '(nil);
      file := car arglist;  % Defaults to nil
   % Allow statement_sequence to be empty:
      if null(arglist := cdr arglist) then arglist := '(all)
   % or a single algebraic list:
      else if eqcar(car arglist, 'list) then arglist := cdar arglist;
      if !*SaveQuery then out 't else out file;
      if file eq 't or !*SaveQuery then terpri();
      NumSaved := 0;
      errorset!*(list('Save1, mkquote arglist, mkquote file), nil);
         % so that SaveQuery can quit early
      if not(file eq 't) then <<
         if !*SaveQuery then out file;
         prin2t ";END;  % of SAVEd file";
         shut file;
         lprim list(NumSaved, "statements saved to file", file)
      >>;
   end;  % Save

symbolic procedure Save1(arglist, file);
   % Save statement_sequence by prompt number.
   foreach arg in arglist do
      if arg eq 'all then
         Save_by_Prompts(list('difference, 1, LastStmt), file)
      else if arg eq 'last then Save_by_Prompts(LastStmt, file)
      else if idp arg then Save_by_Id(arg, file)
      else Save_by_Prompts(arg, file);

symbolic procedure SaveYesp(QueryString);
   % Query user WITHOUT saving response in crbuflis!*.
   begin scalar resp;
      terpri();
      prin2 QueryString;  prin2 " Y(es), N(o), Q(uit): ";
      resp := car explode read();
      terpri();
      if resp eq 'y then return t
      else if resp eq 'q then error(0, nil);
      % anything else treated as No and returns nil
   end;


% Save by prompt number:
% =====================

symbolic procedure Save_by_Prompts(arg, file);
   % Save statement range by prompt number.
   begin scalar arg1, arg2, x, argstep, SaveErrorFlag;
      if atom arg then arg2 := arg1 := SaveArgCheck arg
      else if (x := car arg) eq 'difference or x eq 'to then
      % Range: N1 - N2  or  N1 to N2:
      << arg2 := SaveArgCheck caddr arg;
         arg1 := SaveArgCheck cadr arg >>
      else if x eq 'minus or x eq 'last then
      % Range: -N == last N  or  -all == last all  etc:
      << arg2 := LastStmt;
         arg1 := if fixp(x := cadr arg) then
            SaveArgCheck add1(LastStmt - x) else 1 >>
      else SaveErrorFlag := t;
      if SaveErrorFlag then return
         lprim list ("Invalid argument", arg, "ignored");
      argstep := if arg1 > arg2 then -1 else 1;
      for n := arg1 step argstep until arg2 do <<
         if (x := assoc(n, crbuflis!*)) then <<
            Save_Stmt_by_Prompt(x, n,
               file eq 't or !*SaveQuery); %% or !*SaveIO
            if !*SaveQuery then
               (if SaveYesp("Save?") then <<
                  out file;
                  Save_Stmt_by_Prompt(x, n, nil);  % No prompts
                  NumSaved := add1 NumSaved;
                  out 't
               >>)
            else NumSaved := add1 NumSaved
         >>
         else lprim list("Input entry", n, "not found");
         % But this error should be impossible!
         if file eq 't then terpri()
      >>
   end;  % Save_by_Prompts

symbolic procedure SaveArgCheck arg;
   if fixp arg then
      if arg > LastStmt then LastStmt
      else if arg < 1 then 1 else arg
   else if arg eq 'last then LastStmt
   else SaveErrorFlag := t;

symbolic procedure Save_Stmt_by_Prompt(x, n, prompt);
   % Actually output one input statement, optionally preceded
   % by the prompt and optionally followed by its output.
   << 
      if prompt then
         << prin2 n; prin2 ": " >>;    % output prompt
      editp (x := cdr x);              % output input statement
      if !*SaveChain then SaveChain(1, x, n);
      if !*SaveIO and (x := assoc(n, resultbuflis!*)) then
         mathprint cdr x    % output result (cf. ws)
   >>;


%%% APPEAR NOT TO NEED BOTH CASES ANY MORE %%%
%%% MAY BE ABLE TO USE MATCHWORD HERE %%%

symbolic procedure SaveChain(depth, x, CurStmtNo);
   % Recursively chain back through embedded input expressions
   % of the form "INPUT N" or "WS N" or "WS" and display the chain.
   begin scalar n, l, y, ws!?;
   % Parse "INPUT" or "WS" out of input line x.
      % First find start of literal text:
   a: while x and not liter car x do x := cdr x;
      if null x then return;
      ws!? := nil;  % if true indicates "ws" parsed
      if car x memq '(!i !I) then << x := cdr x;
         l := '((!n !N)(!p !P)(!u !U)(!t !T));
         while l and x and car x memq car l do
            << l := cdr l;  x := cdr x >>;
         if null l then goto b  % input found
      >>
      else if car x memq '(!w !W) and
            (x := cdr x) and car x memq '(!s !S) then
         << x := cdr x;  ws!? := t;  goto b >>;  % ws found
      % Find end of literal text:
      while x and liter car x do x := cdr x;
      goto a;
   % Parse argument of input or ws:
   b: while x and (( seprp u or u eq '!( ) where u = car x) do
         x := cdr x;
      n := 0;  while x and (l := fixdigit car x) do
         << n := n*10 + l; x := cdr x >>;
      if n eq 0 and ws!? then <<  % ws used as variable
         n := CurStmtNo;
         repeat n := sub1 n
         until assoc(n, resultbuflis!*) or n eq 0
      >>
      else ws!? := nil;  % ws used as an operator
      if (y := assoc(n, crbuflis!*)) then  <<
         for i := 1 : depth do prin2 '!%;
         if ws!? then prin2 "(ws)";
         prin2 n;  prin2 ": ";
         editp(y := cdr y);  % output the input line
         SaveChain(depth+1, y, n)
      >>;
      goto a
   end;  % SaveChain

symbolic procedure fixdigit x;
   (if y then cadr y) where y =
      atsoc(x, '( (!0 0) (!1 1) (!2 2) (!3 3) (!4 4)
                  (!5 5) (!6 6) (!7 7) (!8 8) (!9 9) ) );


% Save by identifier:
% ==================

symbolic procedure Save_by_Id(arg, file);
   % Save statement by identifier.
   << 
      if Save_by_Id1(arg, file eq 't or !*SaveQuery) then
         if !*SaveQuery then
            (if SaveYesp("Save?") then <<
               out file;
               Save_by_Id1(arg, nil);  % off nat format
               NumSaved := add1 NumSaved;
               out 't
            >>)
         else NumSaved := add1 NumSaved
      else lprim list(
         "No algebraic value or procedure definition found for", arg);
      if file eq 't then terpri()
   >>;  % Save_by_Id

symbolic procedure Save_by_Id1(Id, !*nat);
   % Save the algebraic value or procedure definition
   % associated with a single identifier.
   % Returns true if found, nil otherwise.
   begin scalar FoundIt;
      if get(Id, 'avalue) then return
      begin scalar value;  % !*nat switch is fluid
         % Output assignment WITHOUT actually assigning:
         value := aeval Id;  % outputs current value.
         % value := cadr get(Id, 'avalue)  % uses last assigned value.
         varpri(value, list('setk, mkquote Id, mkquote value), 'only);
         return t
      end
      else if flagp(Id, 'opfn) and
      begin scalar buflis;  buflis := crbuflis!*;
         % Find the LAST matching procedure definition input,
         % which is FIRST in crbuflis!*.
         while buflis do
         begin scalar ProcDefn, x;
            ProcDefn := cdar buflis;  % Skip prompt number
            if not(x := MatchWord('algebraic, ProcDefn)) then
               x := ProcDefn;
            if (x := MatchWord('procedure, x)) and
                  MatchWord(Id, x) then
               << buflis := nil;  FoundIt := t;  editp ProcDefn >>
            else buflis := cdr buflis;
         end;
         return FoundIt
      end then return FoundIt
   end;  % Save_by_Id1

symbolic procedure MatchWord(Word, CharList);
   % Returns the tail of CharList immediately after the sequence
   % of characters comprising the atom Word, or nil if not found.
   <<
      Word := explode Word;
      % Skip blanks:
      while CharList and seprp car CharList do
         CharList := cdr CharList;
      while Word and CharList and car Word eq car CharList do
         << Word := cdr Word;  CharList := cdr CharList >>;
      if Word then nil else CharList
   >>;


% Allow access as algebraic-mode commands with no argument:
% ========================================================

% Note: the order of statements here is important.
% cf. algebraic (let cmd = cmd());
% Using SAVE commands with no filename only really makes
% sense with my default filename support installed.
put('Save,      'avalue, '(scalar (Save nil)));
put('SaveIO,    'avalue, '(scalar (SaveIO nil)));
put('SaveQuery, 'avalue, '(scalar (SaveQuery nil)));
put('DisplayIn, 'avalue, '(scalar (DisplayIn nil)));
put('DisplayIO, 'avalue, '(scalar (DisplayIO nil)));
symbolic operator Save, SaveIO, SaveQuery, DisplayIn, DisplayIO;

rlistat '(Save SaveIO SaveQuery DisplayIn DisplayIO);


%%endmodule;

end;
