module intpatch;  % Integrate dependent variables & rational powers.

% Author: Francis J. Wright <fjw@maths.qmw.ac.uk>.
% Date: 19 June 1992.
% REDUCE version: 3.4; PSL, Cambridge Lisp.

% This version fixes a bug that integrals that remained symbolic
% were not returned as unique kernels - I now use !*kk2q.
% This bug caused rather obscure symptoms, such as failures with
% on factor.

% This patch has two separate functions:

% 1: It allow integrals containing IMPLICITLY dependent variables,
% created using the DEPEND command, to remain symbolic rather than
% cause an error, whilst preserving other error handling as normal.
% ON FAILHARD turns this facility off.
% This facility was developed from a patch by Herbert Melenk,
% which this patch is intended to replace.

% 2: It integrates simple rational powers of the integration
% variable that the integrator currently fails to integrate.


%% fluid'(soft!-rerror!-number);
% Hope not necessary - it seems not to be.

put('int, 'simpfn, 'SimpIntPatch);

symbolic procedure SimpIntPatch u;
   % Driver for various patches:
   % 1: Catch errors from SimpInt, trap error number 7 only,
   % and pass on all other errors as normal hard REDUCE errors.
   % 2: Post-process unintegrated rational powers.
   begin scalar r, !*redefmsg, !*uncached; !*uncached := t;
      % !*redefmsg rebound to avoid PSL messages
      % about redefinition of rerror.
      %% integer soft!-rerror!-number;    % defaults to 0, not nil
      put('int, 'simpfn, 'SimpInt);       % assumed & reset by SimpInt
      copyd('rerror, 'intpatch!-rerror);  % redefine rerror
      r := errorset!*({'SimpInt, mkquote u}, nil);
      copyd('rerror, 'rerror!%);          % restore rerror
      put('int, 'simpfn, 'SimpIntPatch);  % reset INT interface
      if pairp r then <<
         % First call of SimpInt succeeded -
         % try to reprocess any integrals left:
         put('int, 'simpfn, 'SimpIntRatPow);
         u := resimp car r;  % this works ONLY with !*uncached := t;
         put('int, 'simpfn, 'SimpIntPatch);
         return u
      >>
      else if !*failhard or not(r eq 7) then
         rederr EMSG!*              % Error failure
      else return !*kk2q('int . u)  % Remain symbolic
   end;


% Integrator error trap patch to allow controlled error handling
% ==============================================================

% The error numbers generated by SIMPINT and the corresponding
% error message that would be output by INT are the following,
% collected from the INT source code:

%  1  =  "Improper number of arguments to INT"
%  2  =  "Improper number of arguments to INT"
%  3  =  "Too many arguments to INT"
%  4  =  "FAILHARD switch set"
%  5  =  "Invalid polynomial in int-quadterm"
%  6  =  "Empty list to mapply"
%  7  =  "Can't integrate in the presence of side-relations" (TRAPPED)
%  8  =  "Invalid exponent"
%  9  =  "FAILHARD switch set"

% If any other error number, such as 0, should occur then it
% corresponds to some other non-specific error.


symbolic procedure rerror!%(packagename,number,message);
   % This is precisely the definition of rerror in RLISP.RED,
   % but redefining it here makes sure it is loaded,
   % and also avoids the need to save it.
   % Precisely this procedure is also defined in SOFTSOLV.
   rederr message;

symbolic procedure intpatch!-rerror(packagename,number,message);
   %%   << soft!-rerror!-number := number; error1() >>;
   % The following will suffice provided errorsets
   % are not nested in the integrator.
   % It makes error message text available in EMSG!*.
   error(number, message);


% Integrator postprocessor patch to integrate simple rational
% powers that the integrator currently fails to integrate.
% =======================================================

symbolic procedure SimpIntRatPow u;  % u = (integ var)
   % Integrate integrands of the form var**(m/n),
   % which the integrator leaves in a rather bizarre form -
   % hence the precise form of the following code.
   % Returns original integral if it has the wrong form.
   begin scalar integ, var, power;
      integ := car u;  var := cadr u;
         % assumes true prefix forms, already evaluated by SimpInt.
%     power := errorset!*(
%        {'FindRatPow, mkquote integ, mkquote var}, nil);
%     errorset!*(u,v) == errorset(u,v,!*backtrace)
%     Backtrace from this is unlikely to be interesting, so ...
      power := errorset(
         {'FindRatPow, mkquote integ, mkquote var}, nil, nil);
      if errorp power then return !*kk2q('int . u);
      power := car power;  % correct form of integrand found.
      % integrand = var**power, so return integral:
      power := reval {'plus, power, 1};
      return simp!* {'quotient, {'expt, var, power}, power}
   end;

symbolic procedure FindRatPow(monom, var);
   % Return power of a monomial in var, as a
   % rational number in UNSIMPLIFIED prefix form
   % or cause error return to enclosing errorset.
   if eqcar(monom, 'quotient) then
      {'plus, FindRatPow(cadr monom, var), 
         {'minus, FindRatPow(caddr monom, var)}}
   else if eqcar(monom, 'times) then
      'plus . for each el in cdr monom collect FindRatPow1(el, var)
   else FindRatPow1(monom, var);

symbolic procedure FindRatPow1(monom, var);
   if monom eq 1 then 0  % only possible constant by linearity
   else if monom = var then 1
   else if eqcar(monom, 'expt) and
      cadr monom = var then caddr monom
   else error1();  % wrong form

endmodule;

end;
