

/***********************************************************************

   This software is for research and educational purposes only.

************************************************************************/



#ifndef zz_pX__H
#define zz_pX__H

#include "vector.h"
#include "zz_p.h"
#include "vec_zz_p.h"
#include "FFT.h"
#include "vec_long.h"

// some cross-over points

#define zz_pX_MOD_CROSSOVER (zz_pX_mod_crossover[zz_pInfo->PrimeCnt])
#define zz_pX_MUL_CROSSOVER (zz_pX_mul_crossover[zz_pInfo->PrimeCnt])
#define zz_pX_NEWTON_CROSSOVER (zz_pX_newton_crossover[zz_pInfo->PrimeCnt])
#define zz_pX_DIV_CROSSOVER (zz_pX_div_crossover[zz_pInfo->PrimeCnt])
#define zz_pX_HalfGCD_CROSSOVER (zz_pX_halfgcd_crossover[zz_pInfo->PrimeCnt])
#define zz_pX_GCD_CROSSOVER (zz_pX_gcd_crossover[zz_pInfo->PrimeCnt])
#define zz_pX_BERMASS_CROSSOVER (zz_pX_bermass_crossover[zz_pInfo->PrimeCnt])
#define zz_pX_TRACE_CROSSOVER (zz_pX_trace_crossover[zz_pInfo->PrimeCnt])

extern long zz_pX_mod_crossover[];
extern long zz_pX_mul_crossover[];
extern long zz_pX_newton_crossover[];
extern long zz_pX_div_crossover[];
extern long zz_pX_halfgcd_crossover[];
extern long zz_pX_gcd_crossover[];
extern long zz_pX_bermass_crossover[];
extern long zz_pX_trace_crossover[];



/************************************************************

                         zz_pX

The class zz_pX implements polynomial arithmetic modulo p.
Polynomials are represented as vector(zz_p)'s.
If f is a zz_pX, then f.rep is a vector(zz_p).
The zero polynomial is represented as a zero length vector.
Otherwise. f.rep[0] is the constant-term, and f.rep[f.rep.length()-1]
is the leading coefficient, which is always non-zero.
The member f.rep is public, so the vector representation is fully
accessible.
Use the member function normalize() to strip leading zeros.

**************************************************************/

class zz_pX {

public:

vector(zz_p) rep;

private:

static zz_pX _zero;

public:

/***************************************************************

          Constructors, Destructors, and Assignment

****************************************************************/


zz_pX()
//  initial value 0

   { }


zz_pX(INIT_SIZE_TYPE, long n) { rep.SetMaxLength(n); }

zz_pX(const zz_pX& a) : rep(a.rep) { }
// initial value is a

zz_pX(INIT_VAL_TYPE, long);
zz_pX(INIT_VAL_TYPE, zz_p);



void operator=(const zz_pX& a) 
   { rep = a.rep; }

~zz_pX() { }

void normalize();
// strip leading zeros

void SetMaxLength(long n) 
// pre-allocate space for n coefficients.
// Value is unchanged

   { rep.SetMaxLength(n); }


void kill() 
// free space held by this polynomial.  Value becomes 0.

   { rep.kill(); }

static const zz_pX& zero() { return _zero; }

};




/********************************************************************

                           input and output

I/O format:

   [a_0 a_1 ... a_n],

represents the polynomial a_0 + a_1*X + ... + a_n*X^n.

On output, all coefficients will be integers between 0 and p-1,
amd a_n not zero (the zero polynomial is [ ]).
On input, the coefficients are arbitrary integers which are
then reduced modulo p, and leading zeros stripped.

*********************************************************************/


istream& operator>>(istream& s, zz_pX& x);
ostream& operator<<(ostream& s, const zz_pX& a);




/**********************************************************

                   Some utility routines

***********************************************************/


inline long deg(const zz_pX& a) { return a.rep.length() - 1; }
// degree of a polynomial.
// note that the zero polynomial has degree -1.

zz_p coeff(const zz_pX& a, long i);
// zero if i not in range

void GetCoeff(zz_p& x, const zz_pX& a, long i);
// x = a[i], or zero if i not in range

zz_p LeadCoeff(const zz_pX& a);
// zero if a == 0

zz_p ConstTerm(const zz_pX& a);
// zero if a == 0

void SetCoeff(zz_pX& x, long i, zz_p a);
// x[i] = a, error is raised if i < 0

void SetCoeff(zz_pX& x, long i);
// x[i] = 1, error is raised if i < 0

void SetX(zz_pX& x);
// x is set to the monomial X

long IsX(const zz_pX& a);
// test if x = X

inline void clear(zz_pX& x) 
// x = 0

   { x.rep.SetLength(0); }

inline void set(zz_pX& x)
// x = 1

   { x.rep.SetLength(1); set(x.rep[0]); }

inline void swap(zz_pX& x, zz_pX& y)
// swap x & y (only pointers are swapped)

   { swap(x.rep, y.rep); }

void random(zz_pX& x, long n);
// generate a random polynomial of degree < n 

void trunc(zz_pX& x, const zz_pX& a, long m);
// x = a % X^m

void RightShift(zz_pX& x, const zz_pX& a, long n);
// x = a/X^n

void LeftShift(zz_pX& x, const zz_pX& a, long n);
// x = a*X^n

void diff(zz_pX& x, const zz_pX& a);
// x = derivative of a

void MakeMonic(zz_pX& x);
// makes x monic




/*******************************************************************

                        conversion routines

********************************************************************/



void operator<<(zz_pX& x, long a);
void operator<<(zz_pX& x, zz_p a);

void operator<<(zz_pX& x, const vector(zz_p)& a);




/*************************************************************

                        Comparison

**************************************************************/

long IsZero(const zz_pX& a); 

long IsOne(const zz_pX& a);

inline long operator==(const zz_pX& a, const zz_pX& b)
{
   return a.rep == b.rep;
}

inline long operator!=(const zz_pX& a, const zz_pX& b)
{
   return !(a.rep == b.rep);
}




/***************************************************************

                         Addition

****************************************************************/

void add(zz_pX& x, const zz_pX& a, const zz_pX& b);
// x = a + b

void sub(zz_pX& x, const zz_pX& a, const zz_pX& b);
// x = a - b

void negate(zz_pX& x, const zz_pX& a);
// x = -a

// scalar versions

void add(zz_pX & x, const zz_pX& a, zz_p b); // x = a + b
void add(zz_pX & x, const zz_pX& a, long b); // x = a + b
void add(zz_pX& x, zz_p a, const zz_pX& b); // x = a + b
void add(zz_pX& x, long a, const zz_pX& b); // x = a + b

void sub(zz_pX & x, const zz_pX& a, zz_p b); // x = a - b
void sub(zz_pX& x, const zz_pX& a, long b); // x = a - b
void sub(zz_pX& x, zz_p a, const zz_pX& b); // x = a - b
void sub(zz_pX& x, long a, const zz_pX& b); // x = a - b


/*****************************************************************

                        Multiplication

******************************************************************/


void mul(zz_pX& x, const zz_pX& a, const zz_pX& b);
// x = a * b

void sqr(zz_pX& x, const zz_pX& a);
// x = a^2

void mul(zz_pX & x, const zz_pX& a, zz_p b);
// mul by scalar

void mul(zz_pX& x, const zz_pX& a, long b);
void mul(zz_pX& x, zz_p a, const zz_pX& b); // x = a * b
void mul(zz_pX& x, long a, const zz_pX& b); // x = a * b


void PlainMul(zz_pX& x, const zz_pX& a, const zz_pX& b);
// always uses the "classical" algorithm

void PlainSqr(zz_pX& x, const zz_pX& a);
// always uses the "classical" algorithm


void FFTMul(zz_pX& x, const zz_pX& a, const zz_pX& b);
// always uses the FFT

void FFTSqr(zz_pX& x, const zz_pX& a);
// always uses the FFT

void MulTrunc(zz_pX& x, const zz_pX& a, const zz_pX& b, long n);
// x = a * b % X^n

void PlainMulTrunc(zz_pX& x, const zz_pX& a, const zz_pX& b, long n);
void FFTMulTrunc(zz_pX& x, const zz_pX& a, const zz_pX& b, long n);

void SqrTrunc(zz_pX& x, const zz_pX& a, long n);
// x = a^2 % X^n

void PlainSqrTrunc(zz_pX& x, const zz_pX& a, long n);
void FFTSqrTrunc(zz_pX& x, const zz_pX& a, long n);





// The following data structures and routines allow one
// to hand-craft various algorithms, using the FFT convolution
// algorithms directly.
// Look in the file zz_pX.c for examples.




// FFT representation of polynomials

class fftRep {
private:
   fftRep(const fftRep&); // disabled
   void operator=(const fftRep&); // disabled

public:
   long k;                // a 2^k point representation
   long MaxK;             // maximum space allocated
   long *tbl[4];
   long NumPrimes;

   void SetSize(long NewK);

   fftRep() { k = MaxK = -1; NumPrimes = zz_pInfo->NumPrimes; }
   fftRep(long InitK) 
   { k = MaxK = -1; NumPrimes = zz_pInfo->NumPrimes; SetSize(InitK); }
   ~fftRep();
};


void TofftRep(fftRep& y, const zz_pX& x, long k, long lo, long hi);
// computes an n = 2^k point convolution of x[lo..hi].

inline void TofftRep(fftRep& y, const zz_pX& x, long k)

   { TofftRep(y, x, k, 0, deg(x)); }

void RevTofftRep(fftRep& y, const vector(zz_p)& x,
                 long k, long lo, long hi, long offset);
// computes an n = 2^k point convolution of X^offset*x[lo..hi] mod X^n-1
// using "inverted" evaluation points.



void FromfftRep(zz_pX& x, fftRep& y, long lo, long hi);
// converts from FFT-representation to coefficient representation
// only the coefficients lo..hi are computed
// NOTE: this version destroys the data in y

// non-destructive versions of the above

void NDFromfftRep(zz_pX& x, const fftRep& y, long lo, long hi, fftRep& temp);
void NDFromfftRep(zz_pX& x, const fftRep& y, long lo, long hi);

void RevFromfftRep(vector(zz_p)& x, fftRep& y, long lo, long hi);

   // converts from FFT-representation to coefficient representation
   // using "inverted" evaluation points.
   // only the coefficients lo..hi are computed




void FromfftRep(zz_p* x, fftRep& y, long lo, long hi);
// convert out coefficients lo..hi of y, store result in x.
// no normalization is done.


// direct manipulation of FFT reps

void mul(fftRep& z, const fftRep& x, const fftRep& y);
void sub(fftRep& z, const fftRep& x, const fftRep& y);
void add(fftRep& z, const fftRep& x, const fftRep& y);

void reduce(fftRep& x, const fftRep& a, long k);
// reduces a 2^l point FFT-rep to a 2^k point FFT-rep

void AddExpand(fftRep& x, const fftRep& a);
//  x = x + (an "expanded" version of a)







/*************************************************************

                      Division

**************************************************************/

void DivRem(zz_pX& q, zz_pX& r, const zz_pX& a, const zz_pX& b);
// q = a/b, r = a%b

void div(zz_pX& q, const zz_pX& a, const zz_pX& b);
// q = a/b

void rem(zz_pX& r, const zz_pX& a, const zz_pX& b);
// r = a%b

long divide(zz_pX& q, const zz_pX& a, const zz_pX& b);
// if b | a, sets q = a/b and returns 1; otherwise returns 0

long divide(const zz_pX& a, const zz_pX& b);
// if b | a, sets q = a/b and returns 1; otherwise returns 0


void inv(zz_pX& x, const zz_pX& a, long m);
// computes x = a^{-1} % X^m 
// constant term must be non-zero



// These always use "classical" arithmetic
void PlainDivRem(zz_pX& q, zz_pX& r, const zz_pX& a, const zz_pX& b);
void PlainDiv(zz_pX& q, const zz_pX& a, const zz_pX& b);
void PlainRem(zz_pX& r, const zz_pX& a, const zz_pX& b);


// These always use FFT arithmetic
void FFTDivRem(zz_pX& q, zz_pX& r, const zz_pX& a, const zz_pX& b);
void FFTDiv(zz_pX& q, const zz_pX& a, const zz_pX& b);
void FFTRem(zz_pX& r, const zz_pX& a, const zz_pX& b);

void PlainInv(zz_pX& x, const zz_pX& a, long m);
// always uses "classical" algorithm
// ALIAS RESTRICTION: input may not alias output

void NewtonInv(zz_pX& x, const zz_pX& a, long m);
// uses a Newton Iteration with the FFT.
// ALIAS RESTRICTION: input may not alias output



/***********************************************************

                         GCD's

************************************************************/


void GCD(zz_pX& x, const zz_pX& a, const zz_pX& b);
// x = GCD(a, b),  x is always monic (or zero if a==b==0).

void XGCD(zz_pX& d, zz_pX& s, zz_pX& t, const zz_pX& a, const zz_pX& b);
// d = gcd(a,b), a s + b t = d 

void PlainXGCD(zz_pX& d, zz_pX& s, zz_pX& t, const zz_pX& a, const zz_pX& b);
// same as above, but uses classical algorithm


void PlainGCD(zz_pX& x, const zz_pX& a, const zz_pX& b);
// always uses "cdlassical" arithmetic


class zz_pXMatrix {
private:

   zz_pXMatrix(const zz_pXMatrix&);  // disable
   zz_pX elts[2][2];

public:

   zz_pXMatrix() { }
   ~zz_pXMatrix() { }

   void operator=(const zz_pXMatrix&);
   zz_pX& operator() (long i, long j) { return elts[i][j]; }
   const zz_pX& operator() (long i, long j) const { return elts[i][j]; }
};


void HalfGCD(zz_pXMatrix& M_out, const zz_pX& U, const zz_pX& V, long d_red);
// deg(U) > deg(V),   1 <= d_red <= deg(U)+1.
//
// This computes a 2 x 2 polynomial matrix M_out such that
//    M_out * (U, V)^T = (U', V')^T,
// where U', V' are consecutive polynomials in the Euclidean remainder
// sequence of U, V, and V' is the polynomial of highest degree
// satisfying deg(V') <= deg(U) - d_red.

void XHalfGCD(zz_pXMatrix& M_out, zz_pX& U, zz_pX& V, long d_red);

// same as above, except that U is replaced by U', and V by V'


/*************************************************************

             Modular Arithmetic without pre-conditioning

**************************************************************/

// arithmetic mod f.
// all inputs and outputs are polynomials of degree less than deg(f).
// ASSUMPTION: f is assumed monic, and deg(f) > 0.
// ALIAS RESTRICTIONS: f can not alias an output.
// NOTE: if you want to do many computations with a fixed f,
//       use the zz_pXModulus data structure and associated routines below.



void MulMod(zz_pX& x, const zz_pX& a, const zz_pX& b, const zz_pX& f);
// x = (a * b) % f

void SqrMod(zz_pX& x, const zz_pX& a, const zz_pX& f);
// x = a^2 % f

void MulByXMod(zz_pX& x, const zz_pX& a, const zz_pX& f);
// x = (a * X) mod f

void InvMod(zz_pX& x, const zz_pX& a, const zz_pX& f);
// x = a^{-1} % f, error is a is not invertible

long InvModStatus(zz_pX& x, const zz_pX& a, const zz_pX& f);
// if (a, f) = 1, returns 0 and sets x = a^{-1} % f
// otherwise, returns 1 and sets x = (a, f)

void PowerMod(zz_pX& x, const zz_pX& a, const ZZ& e, const zz_pX& f);
// x = a^e % f, e >= 0

void PowerMod(zz_pX& x, const zz_pX& a, long e, const zz_pX& f);
// x = a^e % f, e >= 0

void PowerXMod(zz_pX& x, const ZZ& e, const zz_pX& f);
//x = X^e mod f, e >= 0

void PowerXMod(zz_pX& x, long e, const zz_pX& f);
//x = X^e mod f, e >= 0

void PowerXPlusAMod(zz_pX& x, zz_p a, const ZZ& e, const zz_pX& f);
// x = (X + a)^e mod f, e >= 0

void PowerXPlusAMod(zz_pX& x, zz_p a, long e, const zz_pX& f);
// x = (X + a)^e mod f, e >= 0






/******************************************************************

        Modular Arithmetic with Pre-conditioning

*******************************************************************/


// If you need to do a lot of arithmetic modulo a fixed f,
// build zz_pXModulus F for f.  This pre-computes information about f
// that speeds up the computation a great deal.


class zz_pXModulus {
private:
   zz_pXModulus(const zz_pXModulus&);  // disabled
   void operator=(const zz_pXModulus&);  // disabled

public:
   zz_pXModulus() : UseFFT(0), n(-1)  { }
   ~zz_pXModulus() { }

   zz_pX f;   // the modulus
   long UseFFT;// flag indicating whether FFT should be used.
   long n;     // n = deg(f)
   long k;     // least k s/t 2^k >= n
   long l;     // least l s/t 2^l >= 2n-3
   fftRep FRep; // 2^k point rep of f
                // H = rev((rev(f))^{-1} rem X^{n-1})
   fftRep HRep; // 2^l point rep of H
   vector(zz_p) tracevec;  // mutable

   zz_pXModulus(const zz_pX& ff);
   operator const zz_pX& () const { return f; }

};

inline const zz_pX& rep(const zz_pXModulus& F) { return F.f; }

void build(zz_pXModulus& F, const zz_pX& f);
// deg(f) > 0


void rem21(zz_pX& x, const zz_pX& a, const zz_pXModulus& F);
// x = a % f
// deg(a) <= 2(n-1), where n = F.n = deg(f)

void rem(zz_pX& x, const zz_pX& a, const zz_pXModulus& F);
// x = a % f, no restrictions on deg(a);  makes repeated calls to rem21

void DivRem(zz_pX& q, zz_pX& r, const zz_pX& a, const zz_pXModulus& F);
void div(zz_pX& q, const zz_pX& a, const zz_pXModulus& F);


void MulMod(zz_pX& x, const zz_pX& a, const zz_pX& b, const zz_pXModulus& F);
// x = (a * b) % f
// deg(a), deg(b) < n

void SqrMod(zz_pX& x, const zz_pX& a, const zz_pXModulus& F);
// x = a^2 % f
// deg(a) < n

void PowerMod(zz_pX& x, const zz_pX& a, const ZZ& e, const zz_pXModulus& F);
// x = a^e % f, e >= 0

void PowerMod(zz_pX& x, const zz_pX& a, long e, const zz_pXModulus& F);
// x = a^e % f, e >= 0

void PowerXMod(zz_pX& x, const ZZ& e, const zz_pXModulus& F);
// x = X^e % f, e >= 0

void PowerXMod(zz_pX& x, long e, const zz_pXModulus& F);
// x = X^e % f, e >= 0

void PowerXPlusAMod(zz_pX& x, zz_p a, const ZZ& e, const zz_pXModulus& F);
// x = (X + a)^e % f


void PowerXPlusAMod(zz_pX& x, zz_p a, long e, const zz_pXModulus& F);
// x = (X + a)^e % f


// If you need to compute a * b % f for a fixed b, but for many a's
// (for example, computing powers of b modulo f), it is
// much more efficient to first build a zz_pXMultiplier B for b,
// and then use the routine below.

class zz_pXMultiplier {
private:
   zz_pXMultiplier(const zz_pXMultiplier&); // disabled
   void operator=(const zz_pXMultiplier&); // disabled

public:
   zz_pXMultiplier() : UseFFT(0)  { }
   ~zz_pXMultiplier() { }

   zz_pX b;   
   long UseFFT;
   fftRep B1; 
   fftRep B2; 

   operator const zz_pX& () const { return b; }
};

inline const zz_pX& rep(const zz_pXMultiplier& B) { return B.b; }


void build(zz_pXMultiplier& B, const zz_pX& b, const zz_pXModulus& F);



void MulMod(zz_pX& x, const zz_pX& a, const zz_pXMultiplier& B,
                                      const zz_pXModulus& F);

// x = (a * b) % f


/*******************************************************

              Evaluation and related problems

********************************************************/


void BuildFromRoots(zz_pX& x, const vector(zz_p)& a);
// computes the polynomial (X-a[0]) ... (X-a[n-1]), where n = a.length()


void eval(zz_p& b, const zz_pX& f, zz_p a);
// b = f(a)

void eval(vector(zz_p)& b, const zz_pX& f, const vector(zz_p)& a);
//  b[i] = f(a[i])


void interpolate(zz_pX& f, const vector(zz_p)& a, const vector(zz_p)& b);
// computes f such that f(a[i]) = b[i]


/*****************************************************************

                       vectors of zz_pX's

*****************************************************************/

vector_decl(zz_pX)

vector_eq_decl(zz_pX)

vector_io_decl(zz_pX)



/**********************************************************

         Modular Composition and Minimal Polynomials

***********************************************************/


// algorithms for computing g(h) mod f




void compose(zz_pX& x, const zz_pX& g, const zz_pX& h, const zz_pXModulus& F);
// x = g(h) mod f

void compose2(zz_pX& x1, zz_pX& x2, const zz_pX& g1, const zz_pX& g2,
              const zz_pX& h, const zz_pXModulus& F);
// xi = gi(h) mod f (i=1,2)
// ALIAS RESTRICTION:  xi may not alias gj, for i != j

void compose3(zz_pX& x1, zz_pX& x2, zz_pX& x3, 
              const zz_pX& g1, const zz_pX& g2, const zz_pX& g3,
              const zz_pX& h, const zz_pXModulus& F);
// xi = gi(h) mod f (i=1..3)
// ALIAS RESTRICTION:  xi may not alias gj, for i != j



// The routine build (see below) which is implicitly called
// by the various compose and UpdateMap routines builds a table
// of polynomials.  
// If zz_pXArgBound > 0, then the table is limited in
// size to approximamtely that many KB.
// If zz_pXArgBound <= 0, then it is ignored, and space is allocated
// so as to maximize speed.
// Initially, zz_pXArgBound = 0.


// If a single h is going to be used with many g's
// then you should build a zz_pXArgument for h,
// and then use the compose routine below.
// build computes and stores h, h^2, ..., h^m mod f.
// After this pre-computation, composing a polynomial of degree 
// roughly n with h takes n/m multiplies mod f, plus n^2
// scalar multiplies.
// Thus, increasing m increases the space requirement and the pre-computation
// time, but reduces the composition time.
// If zz_pXArgBound > 0, a table of size less than m may be built.

struct zz_pXArgument {
   vector(zz_pX) H;
};

extern long zz_pXArgBound;


void build(zz_pXArgument& H, const zz_pX& h, const zz_pXModulus& F, long m);

// m must be > 0, otherwise an error is raised

void compose(zz_pX& x, const zz_pX& g, const zz_pXArgument& H, 
             const zz_pXModulus& F);



void UpdateMap(vector(zz_p)& x, const vector(zz_p)& a,
               const zz_pXMultiplier& B, const zz_pXModulus& F);

/* computes (a, b), (a, (b*X)%f), ..., (a, (b*X^{n-1})%f),
   where ( , ) denotes the vector inner product.

   This is really a "transposed" MulMod by B.
*/

void UpdateMap(vector(zz_p)& x, const vector(zz_p)& a,
                    const zz_pX& b, const zz_pX& f);

// same as above, but uses only classical arithmetic


void ProjectPowers(vector(zz_p)& x, const vector(zz_p)& a, long k,
                   const zz_pX& h, const zz_pXModulus& F);

// computes (a, 1), (a, h), ..., (a, h^{k-1} % f)
// this is really a "transposed" compose.

void ProjectPowers(vector(zz_p)& x, const vector(zz_p)& a, long k,
                   const zz_pXArgument& H, const zz_pXModulus& F);

// same as above, but uses a pre-computed zz_pXArgument

void MinPoly(zz_pX& h, const vector(zz_p)& a, long m);
// computes the minimum polynomial of a linealy generated sequence;
// m is a bound on the degree of the polynomial;
// required: a.length() >= 2*m


void ProbMinPoly(zz_pX& h, const zz_pX& g, const zz_pXModulus& F, long m);
inline void ProbMinPoly(zz_pX& h, const zz_pX& g, const zz_pXModulus& F)
   { ProbMinPoly(h, g, F, F.n); }

// computes the monic minimal polynomial if (g mod f). 
// m = a bound on the degree of the minimal polynomial.
// If this argument is not supplied, it defaults to deg(f).
// The algorithm is probabilistic, always returns a divisor of
// the minimal polynomial, and returns a proper divisor with
// probability at most m/p.

void MinPoly(zz_pX& h, const zz_pX& g, const zz_pXModulus& F, long m);
inline void MinPoly(zz_pX& h, const zz_pX& g, const zz_pXModulus& F)
   { MinPoly(h, g, F, F.n); }

// same as above, but guarantees that result is correct

void IrredPoly(zz_pX& h, const zz_pX& g, const zz_pXModulus& F, long m);
inline void IrredPoly(zz_pX& h, const zz_pX& g, const zz_pXModulus& F)
   { IrredPoly(h, g, F, F.n); }

// same as above, but assumes that f is irreducible, 
// or at least that the minimal poly of g is itself irreducible.
// The algorithm is deterministic (and is always correct).



/*****************************************************************

                   Traces, norms, resultants

******************************************************************/

void TraceVec(vector(zz_p)& S, const zz_pX& f);

void FastTraceVec(vector(zz_p)& S, const zz_pX& f);
void PlainTraceVec(vector(zz_p)& S, const zz_pX& f);

void trace(zz_p& x, const zz_pX& a, const zz_pXModulus& F);

void TraceVec(const zz_pXModulus& F);

void norm(zz_p& x, const zz_pX& a, const zz_pX& f);
void resultant(zz_p& rres, const zz_pX& a, const zz_pX& b);

void CharPoly(zz_pX& g, const zz_pX& a, const zz_pX& f);
// g = char poly of (a mod f)
// only implemented for p >= deg(f)+1




#endif
