

/***********************************************************************

   This software is for research and educational purposes only.

************************************************************************/



#ifndef ZZ_p__H
#define ZZ_p__H

#include "ZZ.h"
#include "ZZVec.h"



// representation:  each ZZ_p is represented by a ZZ in the range 0..p-1.

// The constructor for a ZZ_p pre-allocates space for the underlying ZZ,
// and initializes it to zero.


class ZZ_pInfoT {
private:
   ZZ_pInfoT();                       // disabled
   ZZ_pInfoT(const ZZ_pInfoT&);   // disabled
   void operator=(const ZZ_pInfoT&);  // disabled
public:
   ZZ_pInfoT(const ZZ& NewP);
   ~ZZ_pInfoT();

   ZZ p;      // the modulus
   long size;  // p.size()
   long ExtendedModulusSize;

   // the following implement a "lazy" initialization strategy
   long initialized;  // flag if initialization really was done
   void init();
   void check() { if (!initialized) init(); }

   long NumPrimes;

   long MaxRoot;  

   long QuickCRT;

   ZZ MinusMModP;  //  -M mod p, M = product of primes

   long mmsize;     // size pre-computed for MultiMul

   // the following arrays are indexed 0..NumPrimes-1
   // q = FFTPrime[i]


   ZZVec CoeffModP;    // coeff mod p

   double *x;          // u/q, where u = (M/q)^{-1} mod q
   long *u;            // u, as above

   double **tbl;       // table used for MultiRem; only with SINGLE_MUL

   long **tbl1;        // table used for MultiRem; only with TBL_REM
};

extern ZZ_pInfoT *ZZ_pInfo; // info for current modulus, initially null



class ZZ_pBak {
private:
long MustRestore;
ZZ_pInfoT *ptr;

ZZ_pBak(const ZZ_pBak&);  // disabled
void operator=(const ZZ_pBak&);  // disabled

public:
void save();
void move();
void restore();

ZZ_pBak() { MustRestore = 0; ptr = 0; }
~ZZ_pBak() { if (MustRestore) restore(); else delete ptr; }


};


void ZZ_pInit(const ZZ& NewP);


struct ZZ_p_NoAlloc_type { ZZ_p_NoAlloc_type() { } };
const ZZ_p_NoAlloc_type ZZ_p_NoAlloc = ZZ_p_NoAlloc_type();



class ZZ_p {


private:

ZZ rep;

// static data

static ZZ_p _zero;

public:


typedef void (*DivHandlerPtr)(const ZZ_p& a);   // error-handler for division
static DivHandlerPtr DivHandler;


// ****** constructors and assignment

ZZ_p() : rep(INIT_SIZE, ZZ_pInfo->size) { }

ZZ_p(const ZZ_p& a) :  rep(INIT_SIZE, ZZ_pInfo->size) { rep = a.rep; }

ZZ_p(ZZ_p_NoAlloc_type) { }  // allocates no space

ZZ_p(INIT_VAL_TYPE, const ZZ&);  // initialization

ZZ_p(INIT_VAL_TYPE, long);

~ZZ_p() { } 

void operator=(const ZZ_p& a) { rep = a.rep; }

// read-only access to representation
friend const ZZ& rep(const ZZ_p& a) { return a.rep; }

// You can always access the representation directly...if you dare.
ZZ& LoopHole() { return rep; }



static const ZZ& modulus() { return ZZ_pInfo->p; }
static long ModulusSize() { return ZZ_pInfo->size; }

static const ZZ_p& zero() { return _zero; }

friend void clear(ZZ_p& x)
// x = 0
   { clear(x.rep); }

friend void set(ZZ_p& x)
// x = 1
   { set(x.rep); }

friend void swap(ZZ_p& x, ZZ_p& y)
// swap x and y

   { swap(x.rep, y.rep); }

// ****** addition

friend void add(ZZ_p& x, const ZZ_p& a, const ZZ_p& b)
// x = a + b

   { AddMod(x.rep, a.rep, b.rep, ZZ_p::modulus()); }

friend void sub(ZZ_p& x, const ZZ_p& a, const ZZ_p& b)
// x = a - b

   { SubMod(x.rep, a.rep, b.rep, ZZ_p::modulus()); }


friend void negate(ZZ_p& x, const ZZ_p& a)
// x = -a

   { NegateMod(x.rep, a.rep, ZZ_p::modulus()); }

friend void add(ZZ_p& x, const ZZ_p& a, long b)

   { AddMod(x.rep, a.rep, b, ZZ_p::modulus()); }

friend void add(ZZ_p& x, long a, const ZZ_p& b)

   { AddMod(x.rep, a, b.rep, ZZ_p::modulus()); }

friend void sub(ZZ_p& x, const ZZ_p& a, long b)

   { SubMod(x.rep, a.rep, b, ZZ_p::modulus()); }

friend void sub(ZZ_p& x, long a, const ZZ_p& b)

   { SubMod(x.rep, a, b.rep, ZZ_p::modulus()); }



// ****** multiplication

friend void mul(ZZ_p& x, const ZZ_p& a, const ZZ_p& b)
// x = a*b

   { MulMod(x.rep, a.rep, b.rep, ZZ_p::modulus()); }

friend void mul(ZZ_p& x, const ZZ_p& a, long b)
// x = a*b

   { MulMod(x.rep, a.rep, b, ZZ_p::modulus()); }

friend void mul(ZZ_p& x, long a, const ZZ_p& b)
// x = a*b

   { MulMod(x.rep, a, b.rep, ZZ_p::modulus()); }
   

friend void sqr(ZZ_p& x, const ZZ_p& a)
// x = a^2

   { SqrMod(x.rep, a.rep, ZZ_p::modulus()); }






// ****** division



friend void div(ZZ_p& x, const ZZ_p& a, const ZZ_p& b);
// x = a/b
// If b != 0 & b not invertible & DivHandler != 0,
// then DivHandler will be called with the offending b.
// In this case, of course, p is not really prime, and one
// can factor p by taking a gcd with rep(b).
// Otherwise, if b is not invertible, an error occurs.

friend void div(ZZ_p& x, const ZZ_p& a, long b);
friend void div(ZZ_p& x, long a, const ZZ_p& b);

friend void inv(ZZ_p& x, const ZZ_p& a);
// x = 1/a
// Error handling is the same as above.





// ****** exponentiation

friend void power(ZZ_p& x, const ZZ_p& a, long e)
// x = a^e

   { PowerMod(x.rep, a.rep, e, ZZ_p::modulus()); }

friend void power(ZZ_p& x, const ZZ_p& a, const ZZ& e)
// x = a^e

   { PowerMod(x.rep, a.rep, e, ZZ_p::modulus()); }





// ****** conversion

friend void operator<<(ZZ_p& x, const ZZ& a)
// x = (a mod p)

   { rem(x.rep, a, ZZ_p::modulus()); }

friend void operator<<(ZZ_p& x, long a);
// x = (a mod p)

friend void operator<<(ZZ& x, const ZZ_p& a)
  { x = a.rep; }



// ****** comparison

friend long IsZero(const ZZ_p& a)
   { return IsZero(a.rep); }

friend long IsOne(const ZZ_p& a)
   { return IsOne(a.rep); }

friend long operator==(const ZZ_p& a, const ZZ_p& b)
   { return a.rep == b.rep; }

friend long operator!=(const ZZ_p& a, const ZZ_p& b)
   { return a.rep != b.rep; }


// ****** random numbers

friend void random(ZZ_p& x)
// x = random element in ZZ_p

   { RandomBnd(x.rep, ZZ_p::modulus()); }


// ****** input/output

friend ostream& operator<<(ostream& s, const ZZ_p& a)
   { return s << a.rep; }
   
friend istream& operator>>(istream& s, ZZ_p& x);




// some other friends...not for general use

friend void BlockConstruct(ZZ_p*,long);
friend void BlockDestroy(ZZ_p*,long);


};

// conversion operator...for notational consistency


#endif
