#############################################################################
##
#A  Matrix Group and G-module library                   Derek Holt
#A                                                      Charles Leedham-Green
#A                                                      Eamonn O'Brien
#A                                                      Sarah Rees 
##
#A  @(#)$Id$
##
#Y  Copyright 1994 -- School of Mathematical Sciences, ANU   
##
#H  $Log$
##
#############################################################################
##
#F  InfoPrim (...)  . . . . . . . . . . . information function for package
#F  InfoPrim2 (...)  . . . . . . . . . . . additional information function 
##
##
if not IsBound(InfoPrim)  then InfoPrim := Ignore;  fi;
if not IsBound(InfoPrim2)  then InfoPrim2 := Ignore;  fi;

#############################################################################
##
#F  IsBlockSystem (SystemOfBlocks) . . . . . . . . is object a block system? 
##
IsBlockSystem := function (SystemOfBlocks)

   return IsRec (SystemOfBlocks) and IsBound (SystemOfBlocks.isBlockSystem)
      and SystemOfBlocks.isBlockSystem; 

end; #IsBlockSystem

#############################################################################
##
#F  LengthOfBlockSystem (SystemOfBlocks) .  return number of blocks in system 
##
LengthOfBlockSystem := function (SystemOfBlocks)

   return SystemOfBlocks.nmrBlocks;

end; #LengthOfBlockSystem

#############################################################################
##
#F  BlockSizes (M) . . . . . . . . . . . . return block sizes field of module
##
BlockSizes := function (M)

   if IsBound (M.blockSizes) = false then return "unknown"; fi;
   return M.blockSizes;

end; #BlockSizes

#############################################################################
##
#F  SetBlockSizes (M) . . . set block sizes field of module to SizesOfBlocks
##
SetBlockSizes := function (M, SizesOfBlocks)

   M.blockSizes := SizesOfBlocks;

end; #SetBlockSizes

#############################################################################
##
#F  BlockNumbers (M)  . . . . . . . . . return block numbers field of module
##
BlockNumbers := function (M)

   if IsBound (M.blockNumbers) = false then return "unknown"; fi;
   return M.blockNumbers;

end; #BlockNumbers

#############################################################################
##
#F  SetBlockNumbers (M) . . . set block numbers field of module to NmrBlocks
##
SetBlockNumbers := function (M, NmrOfBlocks)

   M.blockNumbers := NmrOfBlocks;

end; #SetBlockNumbers

#############################################################################
##
#F  DeleteComponents (M) . . . . . . . . . . . . . . delete components from M 
##
DeleteComponents := function (M)

   Unbind (M.blockNumbers);
   Unbind (M.blockSizes);

end; #DeleteComponents 

#############################################################################
##
#F  ConstructBlock (M, TensorFactor) . . . M has a tensor decomposition 
##  with first component TensorFactor which is imprimitive; 
##  use the stored block for its block system to write down 
##  a block under the action of M 
##
ConstructBlock := function (M, TensorFactor)

     local r, s, e, f, B, i, j, k, SmallBlockSystem, SmallBlock, 
           LargeBlock, el;

     #dimension of second factor
     e := DimFlag (TensorFactorsFlag (M)[2]);

     #dimension of first factor
     f := DimFlag (TensorFactor);

     #natural basis for the decomposition
     B := TensorBasisFlag (M);

     #extract block system for action of M on second tensor factor 
     SmallBlockSystem := BlockSystemFlag (TensorFactor);

     #the stored representative block for the small system
     SmallBlock := SmallBlockSystem.block; 

     #number of blocks in small system
     r := LengthOfBlockSystem (SmallBlockSystem);

     #size of each block in the small system
     s := f / r;

     #now use this small block to construct block for larger system
     LargeBlock := [];
     for i in [1..s] do
        for k in [1..e] do
           el := SmallBlock[i][1] * B[(1 - 1)* e + k];
           for j in [2..f] do
              el := el + SmallBlock[i][j] * B[(j - 1)* e + k];
           od;
           Add (LargeBlock, el);
        od;
     od;

     return LargeBlock;

end; #ConstructBlock 

#############################################################################
##
#F  CheckTensorProduct(M) . . . . . . . M has a tensor decomposition found by 
##  SmashGMod; check whether first component of tensor product is imprimitive;
##  if so, use the block found for the first component to construct a
##  block for M, hand this block to MinBlocks (this is strictly unnecessary 
##  but the call sets up proper fields in M), and return TRUE; 
##  if primitive return FALSE; if unknown return "unknown"
##
CheckTensorProduct := function (M)

    local R, Result, TensorFactor, LargeBlock;

    TensorFactor := TensorFactorsFlag (M)[1];
    InfoPrim ("\n*** About to call StartPrimitivityTest again ****\n");
    R := StartPrimitivityTest (TensorFactor, false);
    Result := R[1];

    if (Result = false) then

       InfoPrim2 ("CheckTensorProduct found component is imprimitive\n");

       #use block under first factor to construct a block under action of M  
       LargeBlock := ConstructBlock (M, TensorFactor);

       #now call MinBlocks and set appropriate flags for M 
       TriangulizeMat (LargeBlock);
       SetBlockSystemFlag (M, MinBlocks (M, LargeBlock));

       #temporary code -- remove later 
       if (LargeBlock <> BlockSystemFlag (M).block) then 
          Error ("Blocks are not equal in CheckTensorProduct\n");
       fi;

       SetBlockSizes (M, [LengthOfBlockSystem (BlockSystemFlag (M))]);
       SetBlockNumbers (M, InverseSet (DimFlag (M), BlockSizes (M)));

       return true;
    elif (Result = true) then 
       return false;
    else 
       return Result;
    fi;

end; #CheckTensorProduct
 
#############################################################################
##
#F  InverseSet (d, Set) . . compute the inverse mod d of each element of Set 
## 
InverseSet := function (d, Set)
   
    local i, InverseSet;

    InverseSet := [];
    for i in [1..Length (Set)] do
       InverseSet[i] := d / Set[i];
    od;

    Sort (InverseSet);
    return (InverseSet);

end; #InverseSet 

#############################################################################
##
#F  GcdSeq (A, B) . . . .          compute gcd of two numbers given by their
##                                 prime factorisations A and B 
## 
GcdSeq := function (A, B)

   local found, prime, C, D, i, j, lenA, lenB;

   D := Copy (A);

   lenA := Int (Length (A) / 2);
   lenB := Int (Length (B) / 2);

   for i in [1..lenA] do
      
      found := false;
      prime := D[2 * i - 1];
      j := 1;
      while (j <= lenB) and (not found) do
         if B[2 * j - 1] = prime then
            D[2 * i] := Minimum (D[2 * i], B[2 * j]); 
            found := true;
         fi;
         j := j + 1;
      od;
      if not found then 
         D[2 * i] := 0;
      fi;
   od;

   C := [1, 1];
   j := 1;
   for i in [1..lenA] do
      if D[2 * i] <> 0 then
         C[j] := D[2 * i - 1];
         C[j + 1] := D[2 * i];
         j := j + 2;
      fi; 
   od; 
  
   return C;

end; #GcdSeq

#############################################################################
##
#F  LcmSeq (A, B) . . . .          compute lcm of two numbers given by their
##                                 prime factorisations A and B 
## 
LcmSeq := function (A, B) 
   
   local found, D, i, j, prime, lenA, lenB;

   lenA := Int (Length (A) / 2);
   lenB := Int (Length (B) / 2);

   D := Copy (A);
   for i in [1..lenA] do
      found := false;
      prime := D[2 * i - 1];
      j := 1;
      while (j <= lenB) and not found do
         if B[2 * j - 1] = prime then
            D[2 * i] := Maximum (D[2 * i], B[2 * j]); 
            found := true;
         fi; 
         j := j + 1;
      od; 
   od;

   #add in any primes which occur in B but not in A

   for i in [1..lenB] do
      found := false;
      prime := B[2 * i - 1];
      j := 1;
      while (j <= lenA) and not found do
         found := (D[2 * j - 1] = prime);
         j := j + 1;
      od; 

      if (not found) then 
         D[2 * lenA + 1] := prime;
         D[2 * lenA + 2] := B[2 * i];
         lenA := lenA + 1;
      fi; 
   od; 

   return D;

end; #LcmSeq

#############################################################################
##
#F  ExponentGL (d, q) . . . . . . . . . . . .   compute exponent of GL (d, q) 
## 
ExponentGL := function (d, q) 

   local x, n, i, p, t, pow, ExpSeq;

   n := PrimePowersInt (q - 1);
   for i in [2..d] do
      n := LcmSeq (n, PrimePowersInt (q^i - 1));
   od;

   p :=  FactorsInt (q)[1];
   t := 1;
   pow := p;
   while (pow < d) do
      pow := pow * p;
      t := t + 1;
   od;
   
   ExpSeq := Concatenation ([p, t], n);

   return ExpSeq;

end; #ExponentGL

#############################################################################
##
#F  IsValidSymOrder (Order, r) . . . .         is Order a valid order for an
##                                                 element in Symmetric (r)?
## 
##  an element of order p1^n1 * p2^n2 * ... * pk^nk needs at least 
##  p1^n1 + p2^n2 + ... + pk^nk points 
## 
IsValidSymOrder := function (order, r) 
   
   local S, total, i;

   S := PrimePowersInt (order);
   
   total := 0;
   for i in [1..Int (Length (S) / 2)] do
      total := total + S[2 * i - 1]^S[2 * i];
   od;

   return total <= r;

end; #IsValidSymOrder

#############################################################################
##
#F  GcdOrderGL (d, q, Order) . . .      compute the gcd of the order of the 
##                                      element and the exponent of GL(d, q)
##
GcdOrderGL := function (d, q, Order)
   
   local i, Char, Factors, p, Result, power;

   Factors := Set (FactorsInt (Order));
   Char := FactorsInt (q)[1];

   Result := 1;
   for p in Factors do 
      if p <> Char then 
         i := 1;
         while (OrderMod (q, p^i) <= d) and (Order mod p^i = 0) do
            i := i + 1;
         od; 
         Result := Result * p^(i - 1);
      else
         power := 1;
         while (power < p * d) and (Order mod power = 0) do
            power := power * p;
         od;
         Result := Result * power / p;
      fi;
   od;
   
   return Result;

end; #GcdOrderGL    

#############################################################################
##
#F  IsOrderValid (d, q, r, Order) . . . 
##  given element in GL(d, q) of order 'Order'; 
##  if Order divides exponent of GL(s,q) wr Sym(r), return true, else false 
##
IsOrderValid := function (d, q, r, Order)

   local matord, permord;

   #compute the gcd of the order of the element and the exponent of GL (d/r, q)
   matord := GcdOrderGL (d / r, q, Order);
   InfoPrim2 ("value of matorder is ", matord, "\n");

   permord := Order / matord;
   InfoPrim2 ("value of permorder is ", permord, "\n");

   return IsValidSymOrder (permord, r);

end; #IsOrderValid

#############################################################################
##
#F OrderOfElement (M, Order) . . . . does an element of this Order
##                                eliminate any of the possible block sizes? 
##
OrderOfElement := function (M, Order)
 
    local d, q, r, j, NmrOfBlocks;

    if Order = 1 then return; fi;

    NmrOfBlocks := BlockNumbers (M);

    d := DimFlag (M);
    q := Length (Elements (FieldFlag (M)));

    for j in [1..Length (NmrOfBlocks)] do
       r := NmrOfBlocks[j];
       InfoPrim2 ("r is ", r, "\n");
       if IsOrderValid (d, q, r, Order) = false then 
          InfoPrim ("r = ", r, " is invalid\n");
          NmrOfBlocks[j] := 0;
       fi;
    od; 

    NmrOfBlocks := Difference (NmrOfBlocks, [0]);

    SetBlockNumbers (M, NmrOfBlocks);

end; #OrderOfElement

#############################################################################
##
#F  SetOfNullSpaces (g, factors, F) . . . . . . .   compute the null space of 
##  each polynomial in the set, factors
## 
SetOfNullSpaces := function (g, factors, F)

   local NS, Result, m, t;

   Result := [];

   for m in factors do
#t := Runtime ();
       NS := NullspaceMat (Value(m, g));
#Print ("Time taken for null space computation is ", Runtime () - t, "\n");
      if Length (NS) <> 0 then 
         Add (Result, NS);
      fi;
   od;

   return Result;

end; #SetOfNullSpaces

############################################################################
##
#F PolynomialQuotient (F, f, g) . . .
## return quotient of polynomials f by g as polynomial over F 
## if f is not divisible by g then return false
## 
PolynomialQuotient := function (F, f, g)

   local P;

   P := PolynomialRing (F);
   f := EmbeddedPolynomial (P, f);
   g := EmbeddedPolynomial (P, g);

   return Quotient (f, g);

end; #PolynomialQuotient


#############################################################################
##
#F PolynomialCoefficients (f) . . . . . . return coefficients of polynomial f 
## where coeff[i] is the power of x^i in f
##
## leading zeroes are suppressed in f.coefficients;
## f.valuation is the largest power of x which divides f;
## hence we prepend f.valuation zeros to the list of coefficients 
## 
PolynomialCoefficients := function (f)

   local coeffs, i, zero, NmrZeros;
   
   coeffs := [];
   zero := f.baseRing.zero;
   NmrZeros := f.valuation;
   for i in [1..NmrZeros] do
      coeffs[i] := zero;
   od;

   Append (coeffs, f.coefficients);

   return coeffs;

end; #PolynomialCoefficients

#############################################################################
##
#F PolynomialRemainder (F, f, g) . . . return remainder of polynomials f by g 
##                                     as a polynomial over F 
## 
PolynomialRemainder := function (F, f, g)

   local r;

   r := RemainderCoeffs (PolynomialCoefficients (f), 
                                   PolynomialCoefficients (g));
   return Polynomial (F, r);

end; #PolynomialRemainder

#############################################################################
##
#F ZeroPolynomial (F) . . . . . . . . . . . . . return zero polynomial over F
## 
ZeroPolynomial := function (F)

   return Polynomial (F, []);

end; #PolynomialRemainder

#############################################################################
##
#F  FindLargestPower (TestElement, f, p, n, a, F) . . .     
## 
## if p <> char of F 
## 
## find largest power of x^(p^n) - a in f 
## 
## else 
## 
## find Rank ((x - a)^(p^n - 1)) evaluated for TestElement
## 
FindLargestPower := function (TestElement, p, n, a, f, F)

   local ZeroPol, i, factor, t, quotient, remainder;
   
   ZeroPol := ZeroPolynomial (F);

   if p <> F.char then

      #set up x^(p^n) - a over F
      factor := SetupFactor (F, p^n, a);

      #find the largest power, t, of x^(p^n) - a which divides f 
      t := -1;
      repeat 
         quotient := PolynomialQuotient (F, f, factor);
         remainder := PolynomialRemainder (F, f, factor);
         if remainder = ZeroPol then 
            f := quotient;
         fi;
         t := t + 1;
      until remainder <> ZeroPol;

      return [t, f];

   else
      #set up f = (x - a)^(p^n - 1) 
      f := SetupFactor (F, 1, 1)^(p^n - 1); 

      #find d - dimension of NS (f(TestElement)) = rank of f(TestElement) 
      t := RankMat (Value (f, TestElement));

      return [t];
      
   fi; #p = Char
      
end; #FindLargestPower

#############################################################################
##
#F  FreeRank (TestElement, p, f, F) . . .     
##  find largest t such that V contains a free F[C_p] module of rank t
## 
FreeRank := function (TestElement, p, f, F)

   local Result, Residue, rank, i, one, factor, t, h, Excess;

   InfoPrim2 ("f = ", f, "\n");
   one := F.root^0;

   if p <> F.char then

      #find the largest power of x^p - 1 which divides f 
      #where f = (x^p - 1)^t * Residue

      Result := FindLargestPower (TestElement, p, 1, 1, f, F);
      t := Result[1];
      Residue := Result[2];
      InfoPrim2 ("Residue = ", Residue, "\n");

      #now check whether Residue is simply (x - 1)^Degree(Residue)
      factor := Polynomial (F, [-one, one])^Degree (Residue);
      InfoPrim2 ("factor = ", factor, "\n");
      Excess := (factor <> Residue);
      
      return [t, Excess];

   else

      #set up x - 1
      factor := Polynomial (F, [-one, one]);

      Result := FindLargestPower (TestElement, p, 1, 1, f, F);
      t := Result[1];

      #now find height, h, of residue
      #do this by finding smallest h such that Rank ((x - 1)^h) = t * (p - h)

      f := factor;
      h := 0;
      repeat 
         rank := RankMat (Value (f, TestElement));
         f := f * factor;
         h := h + 1;
      until rank = t * (p - h);

      return [t, h];
      
   fi; #p = Char
      
end; #FreeRank

#############################################################################
##
#F  ExamineSmashGModResult (M, report) . . . . . .  examine results of Smash 
## 
ExamineSmashGModResult := function (M, report)

   #the following are possible outcomes from the call to SmashGMod: 
   #
   #(a) all commutators of generators are scalar (so the group 
   #    must be central by abelian) and SmashGMod exits with an error 
   #(b) group is imprimitive
   #(c) group is a tensor product
   #(d) group is semilinear 
   #(e) the commutator subgroup acts absolutely irreducibly 
   #
   #(a) may result in an Error; 
   #if (b) we return true; 
   #if (c) we return false; 
   #if (d) we return "unknown"
   #(e) has no impact and we return false;

   if ImprimitiveFlag (M) = true then
      if report then 
         Print ("Matrix group is imprimitive\n");
      fi;
      return true;
   elif TensorProdFlag (M) = true then 
      if report then 
         Print ("Module is a tensor product\n");
      fi;
      return false;
   elif SemiLinearFlag (M) = true then 
      if report then 
         Print ("Module is semilinear\n");
      fi;
      return "unknown";
   fi;

   return false;

end; #ExamineSmashGModResult

#############################################################################
##
#F  AddSmashGModQueue (TestElement, t, Queue)  add TestElement to smash queue 
## 
AddSmashGModQueue := function (TestElement, t, Queue)

   local Len;

   if IsScalarMatrix (TestElement) = true then return; fi;

   #keep track of element and the rank of the free module
   Len := Length (Queue) + 1;
   Queue[Len] := [];
   Queue[Len][1] := TestElement;
   Queue[Len][2] := t;

end; #AddSmashGModQueue

#############################################################################
##
#F  IndexMinimumRankElement (Queue). index in Queue of element of smallest rank 
## 
IndexMinimumRankElement := function (Queue)

   local t, i;
 
   if Length (Queue) = 0 then return 0; fi;

   t := [];
   for i in [1..Length (Queue)] do
      t[i] := Queue[i][2];
   od; 

   return Position (t, Minimum (t));

end; #IndexMinimumRankElement

#############################################################################
##
#F  SmashGModElement (M, TestElement)  . . .  . . call Smash with TestElement 
##
##  if SmashGMod finds system of imprimitivity, return true;
##  if SmashGMod finds a tensor product, then it may discover that the 
##  first component is semilinear -- in this case return "unknown";
##  else return false
## 
SmashGModElement := function (M, TestElement) 

   local SizesOfBlocks, Result, pos;

   SmashGMod (M, [TestElement], "tensorprod");

   #SmashGMod may find system of imprimitivity or a tensor product 
   #or it may find that the module is semilinear and hence
   #we set the imprimitive flag to "unknown"

   if ImprimitiveFlag (M) = true then 
      SetBlockSizes (M, [LengthOfBlockSystem (BlockSystemFlag (M))]);
      return true;
   elif TensorProdFlag (M) = true then 
      Result := CheckTensorProduct (M);

      #if our recursive call to PrimitiveTest found that the second
      #component of the tensor decomposition is imprimitive, then 
      #our group is imprimitive; alternately, we may have obtained 
      #semilinear as the result; in either case, we set flag;

      if Result <> false then
         SetImprimitiveFlag (M, Result); 
      fi;

      return Result;
   fi;

   return false;

end; #SmashGModElement

#############################################################################
##
#F  CallSmashGMod (M, TestElement, t)  . . .  . . call Smash with TestElement 
##
##  if SmashGMod finds system of imprimitivity, return true;
##  if it finds component is semilinear, return "unknown";
##  else return false
## 
CallSmashGMod := function (M, TestElement, t) 

   local SizesOfBlocks, Result, pos;

   Result := SmashGModElement (M, TestElement);

   #SmashGMod may find system of imprimitivity or 
   #may not be able to settle case 

   if (Result <> false) then 
      return Result;
   fi;

   #if SmashGMod did not find system, we can rule out all block sizes > t 
   SizesOfBlocks := BlockSizes (M);
   pos := PositionProperty (SizesOfBlocks, n -> n > t); 
   if pos <> false then 
      SizesOfBlocks := Sublist (SizesOfBlocks, [1..pos - 1]);
      SetBlockSizes (M, SizesOfBlocks);
   fi;

   return false;

end; #CallSmashGMod

#############################################################################
##
#F  CharPolPrimeOrder (M, TestElement, p, Queue) . . .
## 
##  test characteristic polynomial structure of elements of prime order
## 
CharPolPrimeOrder := function (M, TestElement, p, Queue)
  
   local Remove, first, f, Result, q, t, h, i, s, OrdMod, F, 
         SizesOfBlocks, Excess, Char, x;

   F := FieldFlag (M);
   f := CharacteristicPolynomial (TestElement);

   SizesOfBlocks := BlockSizes (M);

#   x := Runtime ();
   Result := FreeRank (TestElement, p, f, F);
#   Print ("Time to run FreeRank is ", Runtime () - x, "\n");

   t := Result[1];

   Char := F.char;

   if p <> Char then

      Excess := Result[2];
      q := Length (Elements (F));
      #smallest integer m such that q^m - 1 is divisible by p
      OrdMod := OrderMod (q, p);

      InfoPrim ("in CharPolPrimeOrder, p = ", p, " Excess? ", Excess, 
                 " OrdMod = ", OrdMod, " t = ", t, "\n");

      h := 0;
   else 
      h := Result[2];
      Excess := false;
      OrdMod := 0;
      InfoPrim ("in CharPolPrimeOrder, p = ", p, " t = ", t, " h = ", h, "\n");
   fi;

   #run over and seek to eliminate block sizes; 
   #store eliminated block sizes in Remove  

   first := true; Remove := [];

   for s in SizesOfBlocks do 
      if (p <> Char and s < OrdMod and Excess) or (p = Char and s < h) then 
         InfoPrim2 ("in first clause for s = ", s, "\n");
         Add (Remove, s);

      elif p = Char and s < p and Mod (t, s) <> 0 then  	 
         InfoPrim2 ("in second clause for s = ", s, " \n");
         Add (Remove, s);

      elif s > t and first then 
         InfoPrim2 ("in third clause for s = ", s, " \n");
         AddSmashGModQueue (TestElement, t, Queue);
         first := false;
      fi;
   od;      

   SizesOfBlocks := Difference (SizesOfBlocks, Remove);
   
   SetBlockSizes (M, SizesOfBlocks);

end; #CharPolPrimeOrder

#############################################################################
##
#F  CharPolPrimePowerOrder (M, TestElement, p, n, Queue) . . .
## 
##  order of TestElement is p^n 
##  test characteristic polynomial structure of elements of prime power order
## 
##  compute projective order, p^m, of TestElement
##  TestElement^(p^m) is scalar in a, say
##  now find largest power, t, of x^(p^m) - a 
##  which occurs in char poly of TestElement 
##
CharPolPrimePowerOrder := function (M, TestElement, p, n, Queue)
  
   local F, f, Result, t, x, o, m, scalar, a, SizesOfBlocks;

   F := FieldFlag (M);
   SizesOfBlocks := BlockSizes (M);

   f := CharacteristicPolynomial (TestElement);

   #note both projective order of TestElement and the scalar a
   Result := MatrixProjectiveOrder (TestElement);
   o := PrimePowersInt (Result[1]);
   m := o[2];
   a := Result[2];
  
#   x := Runtime ();
   Result := FindLargestPower (TestElement, p, m, a, f, F);
#   Print ("Time to find t is ", Runtime () - x, "\n");

   t := Result[1];

   InfoPrim ("in CharPolPrimePowerOrder, p^n = ", p^n, " proj order = ", p^m, 
              " scalar = ", a, " t = ", t, "\n");

   #can we eliminate block sizes? 

   if t < Maximum (SizesOfBlocks) then 
      #keep track of element of prime order for possible later processing
      AddSmashGModQueue (TestElement^(p^(m - 1)), t, Queue);
   fi;
      
end; #CharPolPrimePowerOrder

#############################################################################
##
#F  CharPolStructure (M, g, Order, Queue) . . .     
## 
##  examine characteristic polynomial of elements of prime-power order which 
##  can be obtained as powers of g, an element of order Order
## 
CharPolStructure := function (M, g, Order, Queue)
   
   local F, factors, p, i, TestElement, powers, p, n, SizesOfBlocks;

   F := FieldFlag (M);

   SizesOfBlocks := BlockSizes (M);

   #consider elements of prime order 

   factors := Set (FactorsInt (Order));
   for p in factors do

      InfoPrim2 ("call CharPolPrimeOrder with element of order ", p, "\n");

      TestElement := g^(Order / p);

      if IsScalarMatrix (TestElement) = false then 

         InfoPrim2 ("before call, Blocksizes are ", SizesOfBlocks, "\n");
         CharPolPrimeOrder (M, TestElement, p, Queue);
         SizesOfBlocks := BlockSizes (M);

         InfoPrim ("Blocksizes after CharPolPrimeOrder: ", SizesOfBlocks, "\n");
         if SettleComputation (M, Queue) then return; fi;
         
      else 
         InfoPrim ("Element of order ", p, " is scalar\n");
      fi;

   od; 

   #now consider elements of prime-power order, p^n, where n > 1

   powers := PrimePowersInt (Order);
   for i in [1..Length (powers) / 2] do
      
      p := powers[2 * i - 1];
      n := powers[2 * i];

      if n > 1 then 

         InfoPrim2 ("call CharPolPrimePowerOrder with element of order ", p^n, "\n");

         TestElement := g^(Order / p^n);

         if IsScalarMatrix (TestElement) = false then 
            CharPolPrimePowerOrder (M, TestElement, p, n, Queue);
            if SettleComputation (M, Queue) then return; fi;

         else 
            InfoPrim ("Element of order ", p^n, " is scalar\n");
         fi;

      fi; #if n > 1

   od; 

end; #CharPolStructure

#############################################################################
##
#F  CompositeOrders (M, Elts, Orders) . . .     
## 
## given list of elements and their projective orders
## select an element x of composite order o
## if IsValidSymOrder (o, r) = false then
## the element x cannot act faithfully on all of the blocks
## let p run over the primes dividing o
## then one of x^(o / p) must fix all blocks
## hand each of these elements to SmashGMod
## if we do not find a decomposition we have ruled out 
## all number of blocks <= r 
## 
## hence, we find the element which fails IsValidSymOrder
## for the largest possible value of r and hand all of its
## prime constituents to SmashGMod
##
CompositeOrders := function (M, Elts, Orders)

   local Result, o, rem, l, i, j, best, pos, g, Order, p, x, r, NmrOfBlocks;

   #first find set of composite orders 
   o := Set (Filtered (Orders, x -> Length (Set (FactorsInt (x))) > 1));
   InfoPrim2 ("Projective orders for CompositeOrders  is ", Orders, "\n");
   InfoPrim ("Input orders for CompositeOrders test is ", o, "\n");

   NmrOfBlocks := BlockNumbers (M);

   #now find which values of r fail IsValidSymOrder 
   rem := [];
   for i in [1..Length (o)] do
      l := [];
      for j in [1..Length (NmrOfBlocks)] do
         r := NmrOfBlocks[j];
         if IsValidSymOrder (o[i], r) = false then
            Add (l, r);
         fi;
      od;
      InfoPrim ("o is ", o[i], " invalid r is ", l, "\n");
      if l <> [] then 
         rem[i] := Maximum (l);
      fi;
   od;

   InfoPrim2 ("rem is ", rem, "\n");

   if Length (rem) = 0 then return false; fi;

   #maximum r encountered which fails required test
   best := Maximum (rem);
   pos := Position (rem, best);
   pos := Position (Orders, o[pos]);

   #note the element g which fails for maximum r and its order 
   g := Elts[pos];
   Order := Orders[pos];

   InfoPrim ("Use element of order ", Order, " to rule out r <= ", best, "\n");

   #apply SmashGMod to all elements of prime order obtained from g
   for p in Set (FactorsInt (Order)) do
      InfoPrim ("Now calling SmashGMod with element of order ", p, "\n");
      x := g^(Order / p);
      Result := SmashGModElement (M, x);
      if Result <> false then
         return Result;
      fi;
   od;

   #we can now rule out all number of blocks <= best 
   SetBlockNumbers (M, Filtered (NmrOfBlocks, x -> x > best));
   
   return false;

end; #CompositeOrders
   
#############################################################################
##
#F  CoreOfSpace (W, S, F) . . . find core of the space W under the action of 
##  the matrices in S defined over field F  
##
CoreOfSpace := function (W, S, F)
  
   local New, NmrGens, i, count, LenW;
   
   NmrGens := Length (S);
   LenW := Length (W);

   if NmrGens = 0 or LenW = 0 then return W; fi;

   count := 0; i := 1;

   repeat 

      #do we need to use Base? is the collection of generators supplied
      #by Intersection a basis?

      #we also may repeat RowSpace (W, F) command

      New := Intersection (RowSpace (W, F), RowSpace (W * S[i], F));
      New := Base (New);
      if Length (New) = LenW then 
         count := count + 1;
      else 
         W := New;
         LenW := Length (W);
         count := 0;
      fi;
      i := i + 1;
      if i > NmrGens then i := i - NmrGens; fi;
   until count = NmrGens or LenW = 0;

   return W;

end; #CoreOfSpace 

#############################################################################
##
#F  FindGoodFactors (M, g, MaxDegree) . . .   
## 
##  find irreducible factors in the characteristic polynomial of g 
##  of degree at most MaxDegree
## 
FindGoodFactors := function (M, g, MaxDegree)

   local F, f, factors, copy;

   F := FieldFlag (M);

   f := CharacteristicPolynomial (g);
   factors := Set (Factors (EmbeddedPolynomial (PolynomialRing (F), f)));

   InfoPrim2 ("before culling there are ", Length (factors), " factors\n");

   #remove all factors with degree > MaxDegree
   copy := Copy (factors);
   for f in copy do
      if Degree (f) > MaxDegree then 
         factors := Difference (factors, [f]);
      fi;
   od;
   InfoPrim2 ("after culling there are ", Length (factors), " factors\n");

   return factors;

end; #FindGoodFactors

#############################################################################
##
#F  CommonIntersections (M, Elts) . . .     
##
##  test pairs of element hoping to find a subspace of dimension some element 
##  of SizesOfBlocks fixed by both elements; if so, hand it to MinBlocks 
## 
CommonIntersections := function (M, Elts)

   local g, h, X, Y, i, j, k, l, F, BT, MaxBlockSize, SizesOfBlocks, 
        GoodFactors, NS, t;
   
   F := FieldFlag (M);

   GoodFactors := [];
   NS := [];
   SizesOfBlocks := BlockSizes (M);
   MaxBlockSize := Maximum (SizesOfBlocks);

   #now test element k and element l hoping to find a subspace of 
   #dimension at most MaxBlockSize fixed by both elements; 
   #if such a subspace is found, then we hope that 
   #it is contained in a block; hence we call MinBlocks 

   for k in [1..Length (Elts)] do

      #compute necessary information
      g := Elts[k];
      GoodFactors[k] := FindGoodFactors (M, g, MaxBlockSize);

     #t := Runtime ();
      NS[k] := SetOfNullSpaces (g, GoodFactors[k], F);
     #Print ("time to compute nullspaces is ", Runtime () - t, "\n");

      for l in [1..k - 1] do

         InfoPrim ("Length of NS ", k, " & NS ", l, " are ", 
                Length (NS[k]), " & ", Length (NS[l]), "\n");

         h := Elts[l];

         #now test intersections 
         for i in [1..Length (NS[k])] do
            for j in [1..Length (NS[l])] do
             X := Intersection (RowSpace (NS[k][i], F), RowSpace (NS[l][j], F));
               Y := CoreOfSpace (X.generators, [g, h], F);
               if Length (Y) <> 0 then 
                  InfoPrim ("Intersection dim ", Length (X.generators), "\n");
                  InfoPrim ("Core dim is ", Length (Y), "\n");
               fi;
               if Length (Y) in [1..MaxBlockSize] then 
                  InfoPrim ("Now computing blocks\n");
                  BT := MinBlocks (M, Y);
                  if LengthOfBlockSystem (BT) > 1 then 
                     InfoPrim ("Elements nmr ", k, " and ", l, " worked\n");
                     return BT; 
                  fi;
               fi;
            od; #for j 
         od; #for i

     od; #for l
   od; #for k

   return false;

end; #CommonIntersections  

#############################################################################
##
#F  SemiLinearCheck (M)  . . . . . . . . .  check if the module is semi-linear 
## 
##  
SemiLinearCheck := function (M)

   local gens, S, AllScalar, i; 

   gens := MatricesFlag (M);

   S := Commutators (gens);
   
   #does S consist entirely of scalars?
   AllScalar := true;

   i := 0; 
   while (i < Length (S) and AllScalar = true) do
      i := i + 1;
      AllScalar := IsScalarMatrix (S[i]);
   od;

   #if so, add a non-scalar generator to S; if the group 
   #does not have a non-scalar generator, it is reducible 
   #and will be eliminated earlier in BasicReductionTests 
   i := 0;
   while (i < Length (gens) and AllScalar = true) do
      i := i + 1;
      AllScalar := IsScalarMatrix(gens[i]);
      if AllScalar = false then 
         Add (S, gens[i]);
      fi;
   od;
     
   if S <> [] then 
      SmashGMod (M, S, "tensorprod");
   fi;

end; #SemiLinearCheck

#############################################################################
##
#F  BasicReductionTests (M) . . .     
##  carry out tests for basic reductions of the module M 
##
BasicReductionTests := function (M)

   local Result;

   #is the module irreducible?
   if IsIrredGMod (M) = false then 
      Print ("Module is not irreducible\n");
      return true;
   fi;

   #is the module absolutely irreducible?
   if IsAbsIrredGMod (M) = false then 
      Print ("Module is not absolutely irreducible\n");
      return true;
   fi;

   #test for semilinearity
   SemiLinearCheck (M);

   Result := ExamineSmashGModResult (M, true);

   if TensorProdFlag (M) = true then 
      Result := CheckTensorProduct (M);

      #if our recursive call to PrimitiveTest found that a component of 
      #the tensor decomposition is imprimitive, we can deduce that our 
      #group is imprimitive; alternately, it may have found that the 
      #associated module is semilinear

      if Result <> false then
         SetImprimitiveFlag (M, Result); 
      fi;

   fi;

   return Result;

   #return ExamineSmashGModResult (M, true);

end; #BasicReductionTests

#############################################################################
##
#F  FinishComputation (M, Queue, ProcessLeast) . . . . . . . 
##
##  if there is an element of rank < the smallest remaining block size, 
##  then finish the computation by calling SmashGMod;
##  even if this is not so, if ProcessLeast is true, then call SmashGMod 
##  with minimum rank element from SmashGMod Queue;
##  set appropriate flags; return true if block system found or
##  primitivity proved, else false
## 
FinishComputation := function (M, Queue, ProcessLeast)

   local Index, Result, TestElement, t, SizesOfBlocks;

   SizesOfBlocks := BlockSizes (M);

   #do we already know the answer?
   Result := ImprimitiveFlag (M);
   if Result = true or Result = false then
      return Result;
   fi;

   if Length (SizesOfBlocks) <> 0 then 

      #note index position of element of least rank in SmashGMod queue
      Index := IndexMinimumRankElement (Queue);

      #if this is smaller than remaining valid block sizes or
      #ProcessLeast is true, call SmashGMod

      if Index <> 0 then 
         t := Queue[Index][2];
         if ProcessLeast or (Minimum (SizesOfBlocks) > t) then 

            InfoPrim ("\nCall SmashGMod with element of rank t = ", t, " ...\n");

            TestElement := Queue[Index][1];
            Result := CallSmashGMod (M, TestElement, t); 
            if (Result <> true) and (Result <> false) then
               return true;
            fi;
          
            SetBlockNumbers (M, InverseSet (DimFlag (M), BlockSizes (M)));
            if Result = true then 
               return true;
            fi;
         fi;
      fi;

   fi;

   if Length (BlockSizes (M)) = 0 then 
      SetImprimitiveFlag (M, false);
      return true; 
   fi;

   return false;

end; #FinishComputation 

#############################################################################
##
#F  SettleComputation (M, Queue) . . . . . . . 
##
##  have we eliminated all valid block sizes?
##  can we settle computation via a single call to SmashGMod?
##  if either is the case, return true, else return false
##
SettleComputation := function (M, Queue)

   local Index, SizesOfBlocks;

   SizesOfBlocks := BlockSizes (M);

   if Length (SizesOfBlocks) = 0 then return true; fi;

   #note index position of element of least rank in SmashGMod queue
   Index := IndexMinimumRankElement (Queue);

   #is this smaller than existing valid block sizes?
   if Index <> 0 then 
      return (Minimum (SizesOfBlocks) > Queue[Index][2]);
   fi;

   return false;

end; #SettleComputation 

#############################################################################
##
#F  PrimitiveTest (M, NmrElements, Seed) . . .  test module M for primitivity 
## 
##  NmrElements is the number of random elements to choose
##  Seed is the random element seed 
##  function sets ImprimitiveFlag (M) to be true or false
##  if true, BlockSystemFlag (M) contains a block system
##
PrimitiveTest := function (M, NmrElements, Seed)

   local Orders, MatOrd, ProjectiveOrders, R, r, t, MaxDegree, E, 
         i, d, g, Result, Elts, Queue, tt; 

   d := DimFlag (M);

   Elts := []; Orders := []; ProjectiveOrders := [];
   Queue := [];

   for i in [1..NmrElements] do 

      #select random element in group, compute its order 
      repeat 
         g := RandomElement (Seed);
         MatOrd := MatrixOrder (g);
         InfoPrim ("\nOrder of element is ", MatOrd, "\n");
      until MatOrd <> 1;

      #keep this element for later tests 
      Elts[i] := g;
      Orders[i] := MatOrd;
      ProjectiveOrders[i] := MatrixProjectiveOrder (g)[1];

      #seek to eliminate possible numbers of blocks 
      OrderOfElement (M, MatOrd);
      SetBlockSizes (M, InverseSet (d, BlockNumbers (M)));

      InfoPrim ("After order check, block sizes are ", BlockSizes (M), "\n");

      #are we finished?
      if FinishComputation (M, Queue, false) then return; fi;

      #seek to eliminate possible sizes of blocks 
      CharPolStructure (M, g, MatOrd, Queue);
      SetBlockNumbers (M, InverseSet (d, BlockSizes (M)));

      #are we finished?
      if FinishComputation (M, Queue, false) then return; fi;
   od;

   #eliminate all block sizes > minimum value of rank 
   if FinishComputation (M, Queue, true) then return; fi;

   #can we use composite elements to rule out certain block numbers?
   #tt := Runtime ();
   Result := CompositeOrders (M, Elts, ProjectiveOrders);

   #a call to SmashGMod may occur during the call to CompositeOrders  
   #resulting in a "unknown"
   if Result <> true and Result <> false then
      return Result;
   fi;

   SetBlockSizes (M, InverseSet (d, BlockNumbers (M)));

   #are we finished?
   if FinishComputation (M, Queue, false) then return; fi;

   #Print ("time in CompositeOrders is ", Runtime () - tt, " ms \n");

   InfoPrim ("Number of blocks after Order, CharPolStructure, CompositeOrder is ", 
          BlockNumbers (M), "\n");

   #if there are at least two block sizes remaining, it may be worth 
   #trying to try find blocks by a random strategy 
    
   if Length (BlockNumbers (M)) > 1000 then 

      tt := Runtime ();

      E := Sublist (Elts, [1..Int (NmrElements / 4)]);
      Result := CommonIntersections (M, E);

      #are we finished?
      if Result <> false then 
         SetBlockSystemFlag (M, Result);
         r := LengthOfBlockSystem (Result);
         SetBlockNumbers (M, [r]);
         SetBlockSizes (M, [d / r]);
         SetImprimitiveFlag (M, true);
         return;
      fi;

      InfoPrim2 ("Time in CommonIntersections is ", Runtime () - tt, " ms \n");

   fi; #more than 1 block size remaining 

   #for each remaining number r of blocks 
   #apply EliminateBlockNumber to either find blocks or 
   #show that there are no block systems of r blocks 

   for r in BlockNumbers (M) do

       InfoPrim ("\n*** Calling EliminateBlockNumber with r = ", r, "***\n");

       Result := EliminateBlockNumber (M, Seed, Elts, ProjectiveOrders, r);

       if Result = true then 
          InfoPrim ("Successfully eliminated r = ", r, "\n");
          SetBlockNumbers (M, Difference (BlockNumbers (M), [r]));
       elif Result = false then  
          r := LengthOfBlockSystem (BlockSystemFlag (M));
          SetBlockNumbers (M, [r]);
          SetBlockSizes (M, [d / r]);
          SetImprimitiveFlag (M, true);
          return;
       else 
          #SmashGMod call found module is semilinear 
          return;
       fi;
   od;

   SetBlockSizes (M, InverseSet (d, BlockNumbers (M)));

   FinishComputation (M, Queue, false);

end; #PrimitiveTest

#############################################################################
##
#F  ReportResult (M)  . . . . . . have we discovered that M is (im)primitive? 
##
ReportResult := function (M, PrintFlag)

   local Result;

   #is the group imprimitive?
   Result := ImprimitiveFlag (M);
   
   #do we know the answer?
   if Result <> true and Result <> false then return Result; fi;

   #is the group primitive?
   Result := not Result;

   if PrintFlag then 
      if Result = false then 
         Print ("Number of blocks is ", 
             LengthOfBlockSystem (BlockSystemFlag (M)),"\n");
      fi;
      Print ("\nGroup primitive? ", Result, "\n");
   fi;

   #delete those components now unnecessary 
   DeleteComponents (M);

   return Result;

end; #ReportResult

#############################################################################
##
#F  StartPrimitivityTest (M, PrintFlag)  . . . . . . . begin test on module M 
##    
StartPrimitivityTest := function (M, PrintFlag)

   local Result, Seed, NmrElements, ListLength, TotalMultiplications, d; 

   #can we reduce the problem?
   if BasicReductionTests (M) <> false then 
      Result := ReportResult (M, PrintFlag);
      return [Result, M]; 
   fi;

   #constants which determine random elements 
   ListLength := 10;
   TotalMultiplications := 50;
   NmrElements := 20;

   #initialise Seed for random element generator
   Seed := InitialiseSeed (MatricesFlag (M), ListLength, TotalMultiplications);

   d := DimFlag (M);

   #initialise these components 
   SetBlockNumbers (M, Difference (DivisorsInt (d), [1]));
   SetBlockSizes (M, Difference (DivisorsInt (d), [d]));

   #now test for (im)primitivity
   PrimitiveTest (M, NmrElements, Seed);
   Result := ReportResult (M, PrintFlag);

   return [Result, M]; 
   
end; #StartPrimitivityTest 

#############################################################################
##
#F  IsPrimitiveGMod (M) . . . . . . . . . . . . . .  is G-module M primitive? 
##    
##  the function returns a boolean "true" or "false"  
##  if false, then BlockSystemFlag (M) returns the block system
##
IsPrimitiveGMod := function (M)

   local R, Result, t, BadInput; 

   t := Runtime ();

   if IsRec (M) = true then 
      BadInput := IsGModule (M) <> true;
   else 
      BadInput := true;
   fi; 

   if (BadInput) then 
      Print ("You must supply a G-module to this function\n");
      return false;
   fi; 

   Print ("Input G-module has dimension ", DimFlag (M), " over ", 
           FieldFlag (M), "\n");

   R := StartPrimitivityTest (M, true);
   Result := R[1];
   M := R[2];

   Print ("Time taken is ", Int ((Runtime () - t) / 1000), " seconds\n");

   return Result;

end; #IsPrimitiveGMod

#############################################################################
##
#F  IsPrimitiveMatrixGroup (G)   . . . . . . . .  is matrix group primitive? 
##    
##  the function returns list containing a boolean and a module M 
##  if boolean is false, then BlockSystemFlag (M) returns the block system
##
IsPrimitiveMatrixGroup := function (G)

   local M, Result, BadInput; 

   if IsRec (G) = true then 
      BadInput := IsGroup (G) <> true; 
   else 
      BadInput := true;
   fi; 

   if (BadInput) then 
      Print ("You must supply a matrix group to this function\n");
      return false;
   fi; 

   M := GModule (G); 

   Result := IsPrimitiveGMod (M);

   return [Result, M];

end; #IsPrimitiveMatrixGroup
