#include <LiDIA/quadratic_form.h>

inline bigint qf_floor(bigint x, bigint y)
{ // private floor function
  bigint d,r;
  div_rem(d,r,x,y);
  if(x.sign()*y.sign()==-1 && !r.is_zero())
   dec(d);
  return d;
}

inline bigint qf_round(bigint rn, bigint rd)
{
  rn.multiply_by_2();
  add(rn, rn, rd);
  rd.multiply_by_2();
  return qf_floor(rn,rd);
}

quadratic_form::quadratic_form()
{
  debug_handler("quadratic_form", "constructor()");
}

quadratic_form::quadratic_form(const bigint & a1, const bigint & b1, const bigint & c1)
{
  debug_handler("quadratic_form", "constructor(a,b,c)");
  a.assign(a1);
  b.assign(b1);
  c.assign(c1);
  
  bigint tmp;
  
  square(delta, b);
  multiply(tmp, a, c);
  shift_left(tmp, tmp, 2);
  tmp.negate();
  add(delta, delta, tmp);
}
quadratic_form::quadratic_form(const long & a1, const long & b1, const long & c1)
{
  debug_handler("quadratic_form", "constructor(a,b,c)");
  a.assign(a1);
  b.assign(b1);
  c.assign(c1);
  
  bigint tmp;
  
  square(delta, b);
  multiply(tmp, a, c);
  shift_left(tmp, tmp, 2);
  tmp.negate();
  add(delta, delta, tmp);
}

quadratic_form::quadratic_form(const quadratic_form & f)
{
  debug_handler("quadratic_form", "constructor(quadratic_form)");
  a.assign(f.a);
  b.assign(f.b);
  c.assign(f.c);
  delta.assign(f.delta);
}

quadratic_form::~quadratic_form()
{
  debug_handler("quadratic_form", "destructor()");
}
int quadratic_form::is_regular() const
{
  bigint t(delta);  
  bigint s(delta);
  if(s.is_negative())
    return 1;
  if(s.is_zero())
    return 0;
  sqrt(s,t);
  multiply(s,s,s);
  if(s!=t)
    return 1;
  else
    return 0;
}
int quadratic_form::is_bin_quad() const
{
  debug_handler("quadratic_form", "is_bin_quad()");
  bigint tmp(3);
  and(tmp, tmp, delta);
  return (tmp.is_zero() || tmp.is_one());
}


int quadratic_form::definiteness()
{
  debug_handler("quadratic_form", "definiteness()");
 
  if(delta.is_negative() && a.is_positive()&& is_regular())
    return 1;
  else if(delta.is_negative() && a.is_negative()&& is_regular())
    return -1;
  else if(delta.is_positive()&&is_regular())
    return 0;
  
  return 2;
}
  


int quadratic_form::is_pos_def() const
{
  debug_handler("quadratic_form", "is_pos_def()");
  if(delta.is_negative() && a.is_positive()&& is_regular())
    return 1;
  else 
    return 0;
}

int quadratic_form::is_neg_def() const
{
  debug_handler("quadratic_form", "is_neg_def()");
  return (delta.is_negative() && a.is_negative()&& is_regular());
}

int quadratic_form::is_pos_semidef() const
{
  debug_handler("quadratic_form", "is_pos_semidef()");
  return (delta.is_le_zero() && (a.is_positive() || c.is_positive()));
}

int quadratic_form::is_neg_semidef() const
{
  debug_handler("quadratic_form", "is_neg_semidef()");
  return (delta.is_le_zero() && (a.is_negative() || c.is_negative()));
}

int quadratic_form::is_indef() const
{
  debug_handler("quadratic_form", "is_indef()");
  return (delta.is_positive()&&is_regular());
}

void quadratic_form::assign(const quadratic_form & f)
{
  debug_handler("quadratic_form", "assign(quadratic_form)");
  a.assign(f.a);
  b.assign(f.b);
  c.assign(f.c);
  delta.assign(f.delta);
}

int quadratic_form::abs_compare(const quadratic_form &f) const
{
  debug_handler("quadratic_form", "abs_compare()");
  int i = a.abs_compare(f.a);
  if (i == 0)
    i = b.abs_compare(f.b);
  if (i == 0)
    i = c.abs_compare(f.c);
  return i;
}

int quadratic_form::compare(const quadratic_form &f) const
{
  debug_handler("quadratic_form", "compare()");
  int i = a.compare(f.a);
  if (i == 0)
    i = b.compare(f.b);
  if (i == 0)
    i = c.compare(f.c);
  return i;
}

void quadratic_form::neutral(const bigint & d)
{
  debug_handler("quadratic_form", "neutral(bigint)");
  delta.assign(d);
  a.assign_one();
  bigint tmp(3);
  and(b, delta, tmp);
  square(tmp, b);
  subtract(c, tmp, delta);
  shift_right(c, c, 2);
}

void quadratic_form::inverse()
{
  debug_handler("quadratic_form", "inverse()");
  b.negate();
}

const bigint &quadratic_form::get_a() const
{
  debug_handler("quadratic_form", "get_a()");
  return a;
}

const bigint & quadratic_form::get_b() const
{
  debug_handler("quadratic_form", "get_b()");
  return b;
}

const bigint & quadratic_form::get_c() const
{
  debug_handler("quadratic_form", "get_c()");
  return c;
}

const bigint & quadratic_form::get_delta() const
{
  debug_handler("quadratic_form", "get_delta()");
  return delta;
}

quadratic_form&  quadratic_form::operator = (const quadratic_form & f)
{
  debug_handler("quadratic_form", "operator=");
  a.assign(f.a);
  b.assign(f.b);
  c.assign(f.c);
  delta.assign(f.delta);
  return *this;
}

bigint quadratic_form::operator () (const bigint &x, const bigint &y)
{
  bigint tmp1, tmp2;
  
  square(tmp1, x);
  multiply(tmp1, tmp1, a);
  square(tmp2, y);
  multiply(tmp2, tmp2, c);
  add(tmp1, tmp1, tmp2);
  multiply(tmp2, x, y);
  multiply(tmp2, tmp2, b);
  add(tmp1, tmp1, tmp2);
  return tmp1;
}

int operator < (quadratic_form f, quadratic_form g) //f and g have the same discriminants
{
  if( f.a < g.a)
    return 1;
  if( (f.a == g.a)&&(f.b<g.b))
    return 1;
  if((f.a==g.a)&&(f.b==g.b)&&(f.c<g.c))
    return 1;
  return 0;
  
}  
int operator == (quadratic_form f, quadratic_form g)
{
  if(f.compare(g))
    return 0;
  else
    return 1;
}
int operator <= (quadratic_form f, quadratic_form g)
{
  return((f<g)||(f==g));
}

int operator > (quadratic_form f, quadratic_form g)
{
  if(!(f<g)&&!(f==g))
    return 1;
  else
    return 0;
}
int operator >= (quadratic_form f, quadratic_form g)
{
  return((f>g)||(f==g));
}


bigint  quadratic_form:: eval(const bigint & X, const bigint & Y)
{
  bigint temp1,temp2,temp3;
  square(temp1,X);     
  multiply (temp2,temp1,a);     //temp2=a*X^2
  multiply (temp3,X,Y);
  multiply (temp1,b,temp3);     //temp1=b*X*Y
  add      (temp1,temp1,temp2);	//temp1=a*X^2+b*XY
  square   (temp2,Y);
  multiply (temp3,temp2,c);	//temp3=Y^2*c
  add      (temp1,temp3,temp1);
  return temp1;
}


void transform (quadratic_form & f, matrix_GL2Z U)
{
  bigint temp1,temp2,temp3;
  multiply(temp1,U.get_s(),U.get_u());
  multiply(temp1,temp1,f.a);	//asu
  
  multiply(temp2,U.get_t(),U.get_v());
  multiply(temp2,temp2,f.c);	//ctv
  add(temp1,temp1,temp2);
  temp1.multiply_by_2();	//2(asu+tvc)
  
  multiply(temp2,U.get_s(),U.get_v());
  multiply(temp3,U.get_t(),U.get_u());
  add(temp2,temp2,temp3);
  multiply(temp2,temp2,f.b);	//b(sv+tu)
  add (temp1,temp1,temp2);	//new b
  
  
  temp2.assign(f.eval(U.get_s(),U.get_t())); //new a
  temp3.assign(f.eval(U.get_u(),U.get_v())); //new c
  
  if(U.det()==-1)
    {
      temp2.negate();
      temp1.negate();
      temp3.negate();
    }
  
  f.a.assign(temp2);
  f.b.assign(temp1);
  f.c.assign(temp3);
}


void quadratic_form::t()
{
  debug_handler("quadratic_form", "t()");
  swap(a, c);
  b.negate();
}

void quadratic_form::u()
{
  debug_handler("quadratic_form", "u()");
  a.negate();
  c.negate();
}

bigint quadratic_form::norm_number_pos_def()
{
  debug_handler("quadratic_form", "norm_number_pos_def()");  
  
  bigint _b,a_2(a);
  
  a_2.multiply_by_2();
  
  negate(_b,b);
  bigint s(qf_round(_b,a_2));
  return s;
}

bigint quadratic_form::norm_number_indef()
{
  debug_handler("quadratic_form", "norm_number_indef()");
  
  
  bigint sqrt_delta,		// sqrt(delta)
  abs_a,			// abs(a)
  bn,				// b after normalization
  abs_a_mult_2,			// 2 * abs(a)
  tmp,				// local variable
  s,				//normalization number  
  _b(b);			// -b
  
  _b.negate();
  
  int sign_a;			// sign(a): -1, 0, +1
  

  /**
   ** sqrt_delta = sqrt(delta)
   **/
  
  sqrt(sqrt_delta, delta);
  
  /**
   ** sign_a = sign(a); abs_a = abs(a)
   **/
  
  sign_a = a.sign();
  abs_a.assign(a);
  if (sign_a == -1)
    abs_a.negate();
  
  
  multiply(abs_a_mult_2,abs_a,2);
  
  /**                              _ 
   **                  | - b        |
   **    q = sign(a) * | ---------- |
   **                  | 2 * abs(a) |
   **                  |_           |
   **
   ** or
   **
   **                  | sqrt(delta) - b |
   **    q = sign(a) * | --------------- |
   **                  |    2 * abs(a)   |
   **                  |_               _|
   **
   **/
  
  if (abs_a.compare(sqrt_delta) > 0)
    {
      s=qf_round(_b, abs_a_mult_2);
      multiply(s,s,sign_a);
      
    }
  else
    {
      s=qf_floor(sqrt_delta - b , abs_a_mult_2);
      multiply(s,s,sign_a);
      
    }
  return s;
}


bigint quadratic_form::norm_number()
{
  debug_handler("quadratic_form", "norm_number()");
  if (is_pos_def())
    return norm_number_pos_def();
  else if (is_indef())
    return norm_number_indef();
  else
    return bigint(0);
}



int quadratic_form:: is_reduced_pos_def()
{
  debug_handler("quadratic_form", "is_reduced_pos_def()");
  int isnormal,tmp;
  bigint _a(a);
  _a.negate();
  if((a>=b) && (b>_a))
    isnormal =1;
  else
    isnormal=0;
  
  
  
  if( (c.compare(a)>0) || ((c==a)&& b.is_ge_zero()==1))
    tmp=1;
  else
    tmp=0;
  
  tmp = tmp  * isnormal;
  return tmp;
}

int quadratic_form:: is_reduced_irregular()
{
  int x;
  if((a.is_zero())&&(b.is_zero()))
    x = 1;
  else if((a.is_zero())&&(b.is_positive())&&(c.is_ge_zero())&&(c < b))
    x = 1;
  else
    x = 0;
  return x;
}

int quadratic_form:: is_reduced_indef()
{
  debug_handler("quadratic_form", "is_reduced_indef()");
  
  bigint sqrt_delta,		// sqrt(delta)
  abs_a,			// abs(a)
  _abs_a,			// - abs(a)
  abs_a_2,			// 2* abs(a)
  bn;				// local varaibles
  
  int sign_a;			// sign(a): -1, 0, +1
  
  int isnormal,tmp;
  
  /**
   ** sqrt_delta = sqrt(delta)
   **/
  
  sqrt(sqrt_delta, delta);
  
  /**
   ** sign_a = sign(a); abs_a = abs(a)
   **/
  
  sign_a = a.sign();
  abs_a.assign(a);
  if (sign_a == -1)
    abs_a.negate();
  
  multiply(_abs_a,abs_a,-1);
  
  multiply(abs_a_2,abs_a,2);
  
  
  if (abs_a > sqrt_delta)
    {
      isnormal=((_abs_a<b) && ( b <= abs_a));
    }
  else
    {
      isnormal=(((sqrt_delta - abs_a_2)< b) && (b <=sqrt_delta));
    }
  
  subtract(bn,sqrt_delta,abs_a_2);
  
  /*
     I will need some exeption handling here
     one wants: |sqrt(delta)-2|a||< b
     if sqrt_delta - 2|a| == -b it's o.k. since Delta should not be 
     a perfect square
     if sqrt_delta - 2|a| == b then it's not.
     
     I THINK THE NEXT IF SHOULD TAKE CARE OF THIS
     
     */
  
  if (bn.is_negative())
    {
      add(bn,bn,1);
    }
  
  
  bn.absolute_value();
  tmp = ((bn<b) && (b <= sqrt_delta));
  
  return tmp;
  
}

int quadratic_form:: is_reduced()
{
  debug_handler("quadratic_form", "is_reduced()");
  if (is_pos_def())
    return is_reduced_pos_def();
  else if (is_indef())
    return is_reduced_indef();
  else if (!is_regular())
    return is_reduced_irregular();
  else
    return 1;
}






int quadratic_form::normalize()
{
  debug_handler("quadratic_form", "normalize_pos_def()");  
  bigint tmp1,tmp2,s;		//tmp1 becomes new b tmp2 new c
  s=norm_number();
  multiply (tmp1,s,a);
  tmp1.multiply_by_2();
  add(tmp1,b,tmp1);		//tmp1=b+2sa
  
  
  multiply(tmp2,a,s);
  add(tmp2,tmp2,b);
  multiply(tmp2,tmp2,s);
  add(tmp2,tmp2,c);		//tmp2=as^2 +bs +c
  
  b.assign(tmp1);
  c.assign(tmp2);
  
  return is_reduced();
}


int quadratic_form::normalize(matrix_GL2Z  &U )
{
  debug_handler("quadratic_form", "normalize()");  
  bigint tmp1,tmp2,s;		//tmp1 becomes new b tmp2 new c
  s=norm_number();
  
  //matrix_GL2Z S((1),s,
	//	(0),(1));
  
  multiply (tmp1,s,a);
  tmp1.multiply_by_2();
  add(tmp1,b,tmp1);		//tmp1=b+2sa
  
  
  multiply(tmp2,a,s);
  add(tmp2,tmp2,b);
  multiply(tmp2,tmp2,s);
  add(tmp2,tmp2,c);		//tmp2=as^2 +bs +c
  
  b.assign(tmp1);
  c.assign(tmp2);
  
  U.es(s);
  return is_reduced();
}

int quadratic_form:: rho()
{
  debug_handler("quadratic_form", "rho()");
  t();
  normalize();
  return is_reduced();
  
}

int quadratic_form::rho(matrix_GL2Z & U)
{
  debug_handler("quadratic_form", "rho()");
  //matrix_GL2Z V( (0), (-1),
	//	(1) , (0));  
  U.te();  
  t();
  normalize(U);
  return is_reduced();
}


void quadratic_form::reduce()
{
  debug_handler("quadratic_form", "reduce()");
  if (is_reduced())
    return;
  
  if(!is_regular())
    {
      reduce_irregular();
      return;
    }
  normalize();
  while(!is_reduced())
    rho();
}

void quadratic_form::almost_reduce_irregular(matrix_GL2Z & U)
{
  matrix_GL2Z V((1),(0),
		(0),(1));
  bigint d;
  if(!a.is_zero())
    {
      if(delta.is_zero())
	d.assign_zero();
      else
	sqrt(d,delta);
      bigint t(a);
      t.multiply_by_2();
      bigint s(b+d);
      s.negate();
      bigint u,v,g;
      g=xgcd(v,u,s,t);
      s=s/g;
      t=t/g;
      
      u.negate();
      V=matrix_GL2Z (s,u,
		     t,v);
      
      transform(*this,V);
      multiply(U,U,V);
    }
}
void quadratic_form::reduce_irregular(matrix_GL2Z &U)
{
  matrix_GL2Z W;
  matrix_GL2Z D((0),(-1),
		(1),(0));
  almost_reduce_irregular(U);
  
  if(b.is_zero())
    {
      return;
    }
  if(b.is_negative()&&c.is_zero())
    {
      b.negate();
      W=matrix_GL2Z((0),(-1),
		    (1),(0));
      multiply(U,U,W);
      return;
    }
  
  if(b.is_negative())
    {
      t();
      multiply(U,U,D);
      almost_reduce_irregular(U);
    }
  bigint s(qf_floor(c,b));
  s.negate();
  W=matrix_GL2Z ((1),s,
		 (0),(1));
  transform(*this, W);  
  multiply(U,U,W);
  return;
}

void quadratic_form::reduce_irregular()
{
  matrix_GL2Z U((1),(0),
		(0),(1));
  reduce_irregular(U);
}



void quadratic_form:: reduce (matrix_GL2Z & U)
{
  debug_handler("quadratic_form", "reduce()");
  
  if (is_reduced())
    return; 
  
  if(!is_regular())
    {
      reduce_irregular(U);
      return;
    }
  normalize(U);
  while(!is_reduced())
    rho(U);
}



int prop_equivalent(const quadratic_form & f, const quadratic_form & g)
{
  debug_handler("quadratic_form", "prop_equivalent()");
  
  if (g.is_pos_def())
    return prop_equivalent_pos_def(f, g);
  else if (!g.is_regular())
    return prop_equivalent_irregular(f, g);
  else if (g.is_indef())
    return prop_equivalent_indef(f, g);
  else if (g.is_neg_def())
    return prop_equivalent_neg_def(f, g);
}

int prop_equivalent_pos_def(const quadratic_form &f, const quadratic_form &g)
{
  debug_handler("quadratic_form", "prop_equivalent_pos_def()");
  
  quadratic_form rf(f), rg(g);
  
  if(f.is_indef())
    return 0;
  rg.reduce();
  rf.reduce();
  return (rf.compare(rg) == 0);
}

int prop_equivalent_neg_def(const quadratic_form &f, const quadratic_form &g)
{
  debug_handler("quadratic_form", "prop_equivalent_pos_def()");
  if(!f.is_neg_def())
    return 0;
  quadratic_form rf(f), rg(g);
  rf.u();
  rg.u();
  
  rg.reduce();
  rf.reduce();
  return (rf.compare(rg) == 0);
}



int prop_equivalent_indef(const quadratic_form &f, const quadratic_form &g)
{
  debug_handler("quadratic_form", "prop_equivalent_indef()");
  if(f.is_pos_def())
    return 0;
  
  quadratic_form rf(f), rg(g),root;
  
  rf.reduce();
  
  rg.reduce();
  
  if (rf.compare(rg) == 0)
    return 1;
  
  root.assign(rg);
  
  do 
    {
      rg.rho();
      if (rf.compare(rg) == 0)
	return 1;
    } while (root.compare(rg) != 0);
  return 0;
}

int prop_equivalent_irregular(const quadratic_form & f,const quadratic_form & g)
{
  quadratic_form rf(f),rg(g);
  rf.reduce();
  rg.reduce();
  if(!rf.compare(rg))
    return 1;
  else
    return 0;
}
int prop_equivalent(const quadratic_form & f, const quadratic_form & g, matrix_GL2Z & U)
{
  debug_handler("quadratic_form", "prop_equivalent()");
  
  if (g.is_pos_def())
    return prop_equivalent_pos_def(f, g,U);
  else if (g.is_indef())
    return prop_equivalent_indef(f, g,U);
  else if (g.is_neg_def())
    return prop_equivalent_neg_def(f, g,U);
  else if (!g.is_regular())
    return prop_equivalent_irregular(f, g,U);
}

int prop_equivalent_pos_def(const quadratic_form &f, const quadratic_form &g, matrix_GL2Z & U)
{
  debug_handler("quadratic_form", "prop_equivalent_pos_def()");
  
  quadratic_form rf(f), rg(g);
  matrix_GL2Z V((1), (0),
		(0),(1));
  matrix_GL2Z S(V);
  if(f.is_indef())
    return 0;
  rg.reduce(S);
  rf.reduce(V);
  S.invert();
  multiply (V,V,S);
  int i;
  if (rf.compare(rg) == 0)
    {
      multiply(U,U,V);
      i = 1;
    }
  else
    i = 0;
  return i;
}
int prop_equivalent_neg_def(const quadratic_form &f, const quadratic_form &g, matrix_GL2Z & U)
{
  debug_handler("quadratic_form", "prop_equivalent_pos_def()");
  
  quadratic_form rf(f), rg(g);
  matrix_GL2Z V((-1), (0),
		(0),(1));
  matrix_GL2Z S(V);
  if(!f.is_neg_def())
    return 0;
  rf.u();
  rg.u();
  rg.reduce(S);
  rf.reduce(V);
  S.invert();
  multiply (V,V,S);
  int i;
  if (rf.compare(rg) == 0)
    {
      multiply(U,U,V);
      i = 1;
    }
  else
    i = 0;
  return i;
}

int prop_equivalent_irregular(const quadratic_form &f, const quadratic_form &g,matrix_GL2Z & U)
{
  if(!((!f.is_regular())&&(!g.is_regular())))
    return 0;
  quadratic_form rf(f),rg(g);
  matrix_GL2Z V ((1),(0),
		 (0),(1));
  matrix_GL2Z W(V);
  rf.reduce(V);
  rg.reduce(W);
  int i;
  if(rf.compare(rg))
    i =  0;
  else 
    {
      W.invert();
      multiply(V,V,W);
      multiply(U,U,V);
      i =  1;
    }
  return i;
}

int prop_equivalent_indef(const quadratic_form &f, const quadratic_form &g,matrix_GL2Z & U)
{
  debug_handler("quadratic_form", "prop_equivalent_indef()");
  if(f.is_pos_def())
    return 0;
  
  matrix_GL2Z V((1), (0),
		(0),(1));
  matrix_GL2Z S(V);
  
  quadratic_form rf(f), rg(g),root;
  
  rf.reduce(S);
  
  rg.reduce(V);
  
  if (rf.compare(rg) == 0)
    {
      V.invert();
      multiply(S,S,V);
      multiply(U,U,S);
      return 1;
    }
  
  root.assign(rg);
  
  do 
    {
      rg.rho(V);
      if (rf.compare(rg) == 0)
	{
	  V.invert();
	  multiply(S,S,V);
	  multiply(U,U,S);
	  return 1;
	}
    } while (root.compare(rg) != 0);
  return 0;
}

int equivalent(const quadratic_form & f, const quadratic_form & g, matrix_GL2Z & U) 
{ 
  quadratic_form hf(f),hg(g); 
  matrix_GL2Z W((-1),(0), 
		(0),(1)); 
  matrix_GL2Z I((1),(0), 
		(0),(1)); 
  
  if(f.is_pos_def()&&g.is_pos_def()) 
    return prop_equivalent(f,g,U); 
  if(f.is_neg_def()&&g.is_neg_def()) 
    { 
      matrix_GL2Z H(I);
      transform (hf,W);
      transform (hg,W);
      int i;
      if(prop_equivalent_pos_def(hf,hg,H))
	{
	  multiply(H,W,H);
	  multiply(H,H,inverse(W));
	  multiply(U,U,H);
	  i =  1;
	}
      else
	i = 0;
      return i;
    }
  
  if(f.is_neg_def()&&g.is_pos_def()) 
    { 
      matrix_GL2Z H(I);
      transform (hf,W);
      int i;
      if(prop_equivalent_pos_def(hf,hg,H))
	{
	  multiply(H,W,H);
	  multiply(U,U,H);
	  i = 1;
	}
      else
	i = 0;
      return i;
    }
  
  if(f.is_pos_def()&&g.is_neg_def()) 
    { 
      matrix_GL2Z H(I);
      transform (hg,W);
      int i;
      if(prop_equivalent_pos_def(hf,hg,H))
	{
	  multiply(H,H,inverse(W));
	  multiply(U,U,H);
	  i = 1;
	}
      else
	i =  0;
      return i;
    }
  
  
  if(f.is_indef()&&g.is_indef())
    { 
      matrix_GL2Z V((1), (0),
		    (0),(1));
      matrix_GL2Z S(V);
      
      quadratic_form rf(f), rg(g),root;
      
      rf.reduce(S);
      
      rg.reduce(V);
      quadratic_form rff(rf);
      rff.u();
      
      if (rf.compare(rg) == 0)
	{
	  V.invert();
	  multiply(S,S,V);
	  multiply(U,U,S);
	  return 1;
	}
      if (rff.compare(rg) == 0)
	{
	  V.invert();
	  multiply(S,S,W);
	  multiply(S,S,V);
	  multiply(U,U,S);
	  return 1;
	}
      
      root.assign(rg);
      
      do 
	{
	  rg.rho(V);
	  if (rf.compare(rg) == 0)
	    {
	      V.invert();
	      multiply(S,S,V);
	      multiply(U,U,S);
	      return 1;
	    }
	  if (rff.compare(rg) == 0)
	    {
	      V.invert();
	      multiply(S,S,W);
	      multiply(S,S,V);
	      multiply(U,U,S);
	      return 1;
	    }
	} while (root.compare(rg) != 0);
      
      return 0;
    }
  
  
  if(!f.is_regular()&&!g.is_regular())
    {
      matrix_GL2Z Q(I),R(I);
      hf.reduce(Q);
      hg.reduce(R);
      int i;
      if(!hf.compare(hg))
	{
	  R.invert();
	  multiply(Q,Q,R);
	  multiply(U,U,Q);
	  i = 1;
	}
      else
	{
	  transform(hg,W);
	  multiply(R,R,W);
	  hg.reduce(R);
	  if(!hf.compare(hg))
	    {
	      R.invert();
	      multiply(Q,Q,R);
	      multiply(U,U,Q);
	       i = 1;
	    }
	    else
	       i = 0;
	}
        return i;
    }	 
  return 0;  
}

int equivalent(const quadratic_form & f, const quadratic_form & g) 
{ 
  quadratic_form hf(f),hg(g); 
  matrix_GL2Z W((-1),(0), 
		(0),(1)); 
  matrix_GL2Z I((1),(0), 
		(0),(1)); 
  
  if(f.is_pos_def()&&g.is_pos_def()) 
    return prop_equivalent(f,g); 
  if(f.is_neg_def() && g.is_neg_def()) 
    { 
      transform (hf,W);
      transform (hg,W);
      return prop_equivalent_pos_def(hf,hg);
    }
  
  if(f.is_neg_def()&&g.is_pos_def()) 
    { 
      
      hf.u();
      if(prop_equivalent_pos_def(hf,hg))
	{
	  return 1;
	}
      else
	return 0;
    }
  
  if(f.is_pos_def()&&g.is_neg_def()) 
    { 
      hg.u();
      if(prop_equivalent_pos_def(hf,hg))
	{
	  return 1;
	}
      else
	return 0;
    }
  
  
  if(f.is_indef()&&g.is_indef())
    {  
      quadratic_form rf(f), rg(g),root;
      
      rf.reduce();
      
      rg.reduce();
      quadratic_form rff(rf);
      rff.u();
      
      if (rf.compare(rg) == 0)
	{
	  return 1;
	}
      if (rff.compare(rg) == 0)
	{
	  
	  return 1;
	}
      
      root.assign(rg);
      
      do 
	{
	  rg.rho();
	  if (rf.compare(rg) == 0)
	    {
	      return 1;
	    }
	  if (rff.compare(rg) == 0)
	    {
	      return 1;
	    }
	} while (root.compare(rg) != 0);
      
      return 0;
    }
  
  
  if(!f.is_regular()&&!g.is_regular())
    {
      hf.reduce();
      hg.reduce();
      if(!hf.compare(hg))
	{
	  
	  return 1;
	}
      else
	{
	  hg.u();
	  hg.reduce();
	  if(!hf.compare(hg))
	    {
	      return 1;
	    }
	  return 0;
	}
    }	 
  return 0;  
}


void quadratic_form::fundamental_automorphism(matrix_GL2Z & U)
{
  matrix_GL2Z V((-1),(0),
		(0),(1));
  if(is_pos_def())
    return;
  
  quadratic_form g(*this);
  g.reduce();
  
  quadratic_form h(g);
  quadratic_form i(g);
  transform(i,V);
  do
    {
      g.rho(U);
      
    }while((g.compare(h)!=0)&&(g.compare(i)!=0));
}



istream & operator >> (istream & in, quadratic_form & f)
{
  debug_handler("quadratic_form", "operator>>");
  int n = 0, sz = 3;
  char c;
  bigint *ibuf;
  ibuf = new bigint[sz];
  in >> c;
  if (c != '(')
    error_handler("quadratic_form", "operator>>::( expected");
  in >> c;
  while (c != ')' && n != 3)
    {
      in.putback(c);
      in >> ibuf[n];
      n++;
      in >> c;
    }
  f = quadratic_form(ibuf[0], ibuf[1], ibuf[2]);
  delete[] ibuf;
  return in;
}

ostream & operator << (ostream & out, const quadratic_form & f)
{
  debug_handler("quadratic_form", "operator<<");
  out << "( " << f.a << " " << f.b << " " << f.c << " )";
  return out;
}
