#ifndef ALGEBRAIC
#define ALGEBRAIC

#include <stream.h>

#if defined(HAVE_MAC_DIRS) || defined(__MWERKS__)
#include <LiDIA:lidia.h>
#include <LiDIA:bigint.h>
#include <LiDIA:bigrational.h>
#include <LiDIA:bigfloat.h>

#include <LiDIA:math_vector.h>

#include <LiDIA:bigint_matrix.h>
#include <LiDIA:bigmod_matrix.h>


#include <LiDIA:rational_factorization.h>

#include <LiDIA:polynomial.h>
#else
#include <LiDIA/lidia.h>
#include <LiDIA/bigint.h>
#include <LiDIA/bigrational.h>
#include <LiDIA/bigfloat.h>

#include <LiDIA/math_vector.h>

#include <LiDIA/bigint_matrix.h>
#include <LiDIA/bigmod_matrix.h>


#include <LiDIA/rational_factorization.h>

#include <LiDIA/polynomial.h>
#endif

class number_field;
class order;
class alg_number;
class module;

extern number_field *current_number_field;
extern order *current_order;


class number_field{
#ifdef LIDIA_DEBUG
  static int count;
#endif
  polynomial <bigint> f;		// the irreducible polynomial
  lidia_size_t real_roots;		// number of real roots
  math_matrix <bigfloat> conjugates;	// column i contains the conjugates 
                                        // of x^i (0 <= i < degree()
  void compute_conjugates();

    
public:
  // Contructors & destructor:
  number_field(const polynomial < bigint > & p = bigint(0));
  number_field(const bigint *v, lidia_size_t deg);
  number_field(order *);
  number_field(const number_field &);
  ~number_field();

  // member functions:
  bigfloat get_conjugate(lidia_size_t, lidia_size_t);
  math_matrix <bigfloat> get_conjugates();

  const polynomial <bigint> which_polynomial() const
  {return f;}

  lidia_size_t degree() const
  {return f.degree();}

  lidia_size_t no_of_real_embeddings() const
  {return real_roots;}

#ifndef HEADBANGER
  friend number_field overfield(const number_field &, const number_field &);

  void assign(const number_field & K)
  {f  = K.f; real_roots = K.real_roots; conjugates = K.conjugates;}
#endif

  // Comparisions:
  // Since deciding whether fields are isomorphic is difficult,
  // we consider fields to be equal only, if they are generated by
  // the same polynomial.
  friend bool operator ==(const number_field &, const number_field &);
  friend bool operator !=(const number_field &, const number_field &);
  friend bool operator <=(const number_field &, const number_field &);
  friend bool operator <(const number_field &, const number_field &);
  friend bool operator >=(const number_field &, const number_field &);
  friend bool operator >(const number_field &, const number_field &);

  // friend functions:
  friend const polynomial < bigint > which_polynomial(const number_field &);
  friend lidia_size_t degree(const number_field &);
  friend lidia_size_t no_of_real_embeddings(const number_field &);
  // In-/Output:
  friend ostream& operator<<(ostream&, const number_field&); // jeweils Polynom
  friend istream& operator>>(istream&, number_field&);       // ein-, ausgeben
};

inline bool operator !=(const number_field & A, const number_field & B)
{ return !(A==B);}
  
inline bool operator <(const number_field & A, const number_field & B)
{ return (!(A==B) && (A<=B));}
  
inline bool operator >=(const number_field & A, const number_field & B)
{return B<=A;}  

inline bool operator >(const number_field & A, const number_field & B)
{ return B<A;}

inline const polynomial < bigint > which_polynomial(const number_field & F)
{return F.f;}

inline lidia_size_t degree(const number_field & F)
{return F.f.degree();}

inline lidia_size_t no_of_real_embeddings(const number_field & F)
{return F.real_roots;}


class order{
#ifdef LIDIA_DEBUG
  static int count;
#endif
#ifndef HEADBANGER
  // Components of the data - type:
  const number_field * K;			// pointer to quotient field
  mutable math_matrix <bigint> table;		// multiplication table
  bigint_matrix base;
  bigint den;

  // Internal information, that is frequently needed and so stored
  // with the order, as soon, as it is computed:
  mutable math_vector <bigint> One;		// Representation of 1
  mutable bigint_matrix trace_matrix;		// The matrix containing the
                                                // entries Tr(w_i w_j)
  math_matrix <bigfloat> conjugates;		// The conjugates of w_i

  // Functions for checking, whats already computed:
  bool table_computed() const
  {return (table.get_no_of_columns()!=1);}

  void compute_table() const;

  bool base_computed() const
  {return (base.get_no_of_columns()!=1);}

  void compute_base();

  void compute_conjugates();


  // Internal routine for the work involved in comparisons
  void compare(const order&, bool &, bool &) const;

  // whether to use MT or polynomial for multiplying.
  bool using_necessary() const
  {return ((K==NULL) || base_computed());}
#endif
public:
  // Contructors & destructor:
  order(const number_field * F = current_number_field);
  order(const number_field &);
  order(const base_matrix <bigint> &);		// table uebergeben
  order(const base_matrix <bigint> &, const bigint &d, 
	const number_field * F= current_number_field);	// base uebergeben
  // Important: no default for d, so we can distinguish between
  // initialisation by giving the table and initialisation by
  // giving the Z-module representation and the field!

  order(const order &);				// copy constructor

  ~order();

  // member functions:
       // Accessing Information:
#ifndef HEADBANGER
  const number_field * which_field() const	// pointer to quotient field
  {return K;}
#endif

  bigint_matrix base_numerator() const
  {return base;}

  bigint base_denominator() const
  {return den;}

  bigint MT(lidia_size_t,lidia_size_t,lidia_size_t);	// accessing MT

  bigfloat get_conjugate(lidia_size_t, lidia_size_t);
  math_matrix <bigfloat> get_conjugates();

        //get often needed information:
  const math_vector <bigint> & get_one() const;	// 1	
  bigint_matrix get_trace_matrix() const;	// "Trace matrix"
  lidia_size_t degree() const;			// degree of extension

  lidia_size_t no_of_real_embeddings() const
  { if (K != NULL) return K->no_of_real_embeddings();
    else return -1;
  }

#ifndef HEADBANGER
  void assign(const order &);
#endif

  // Cast operator:
  operator module() const;

  // Comparisions:
  // We are interested in comparing orders only if they are
  // over the same field! So nothing else is supported!!
  friend bool operator ==(const order&, const order&);
  friend bool operator !=(const order&, const order&);
  friend bool operator <=(const order&, const order&);
  friend bool operator <(const order&, const order&);
  friend bool operator >=(const order&, const order&);
  friend bool operator >(const order&, const order&);

  // friend functions:
  friend const number_field * which_field(const order &);
  friend bigint_matrix base_numerator(const order &);
  friend bigint base_denominator(const order &);
  friend void swap(order &, order &);

  // Number-theoretic functions:
  friend lidia_size_t degree(const order & O);	// degree of field extension
  friend lidia_size_t no_of_real_embeddings(const order & O);
  friend bigint disc(const order & O);		// discriminant
  module pseudo_radical(const bigint & p, bigint & factor) const;
                                       // computes pseudo-radical
  bool dedekind(const bigint & p, polynomial < bigint > & h2) const;
              // uses Dedekind criterion for prime p; returns true,
              // if p is an index divisor and sets extended to the new order.

  order maximize(const bigint & p, bigint & factor) const;
              // maximizes order at p.
  order maximize() const;

  // In-/Output:
  friend ostream& operator<<(ostream &, const order &);// table ausgeben
  friend istream& operator>>(istream &, order &);
           // tries to interpret input either as multiplication table
           // or as base transformation according to dimension.
                          
  // friends:
  friend void multiply(alg_number &, const alg_number &, const alg_number &);
  friend bigint_matrix rep_matrix(const alg_number &);
  friend number_field::number_field(order *);
  friend class module;
};

inline const number_field * which_field(const order & O)
{return O.K;}

inline bigint_matrix base_numerator(const order & O)
{return O.base;}

inline bigint base_denominator(const order & O)
{return O.den;}

inline lidia_size_t degree(const order & O)
{return O.degree();}  

inline lidia_size_t no_of_real_embeddings(const order & O)
{return O.no_of_real_embeddings();}

inline bigint disc(const order & O)
{return (O.get_trace_matrix()).det();}

class alg_number{
#ifdef LIDIA_DEBUG
  static int count;
#endif
  bigint den;
  math_vector <bigint> coeff;
  const order * O;

public:
  // Constructors & destructor:
  alg_number(const order * O1 = current_order);
  alg_number(const bigint &, const order * O1 = current_order);
  alg_number(const base_vector <bigint> &, const bigint & i,
	     const order * O1 = current_order);
  alg_number(const bigint *, const bigint & i = 1,
	     const order * O1 = current_order);
  alg_number(const alg_number & a);
  ~alg_number();
    
  // member-functions
  bigint denominator() const
  {return den;}

  math_vector <bigint> coeff_vector() const
  {return coeff;}

  alg_number numerator() const
  {return alg_number(coeff, bigint(1), O);}

  const order * which_order() const
  {return O;}

  lidia_size_t degree() const
  { return O->degree();}

  bigfloat get_conjugate(lidia_size_t);
  math_vector <bigfloat> get_conjugates();

#ifndef HEADBANGER
  bool is_zero() const;
  bool is_one() const;

  void normalize();
  void negate();
  void invert();
  void multiply_by_2();
  void divide_by_2();

  void assign_zero();			// Remains member of same order
  void assign_one();			// Remains member of same order
  void assign(const bigint &);		// Remains member of same order
  void assign(const alg_number &);	// Becomes member of same order as a!!

  // Procedural versions of arithmetic operations:
  friend void add(alg_number &, const alg_number &, const alg_number &);
  friend void subtract(alg_number &, const alg_number &, const alg_number &);
  friend void multiply(alg_number &, const alg_number &, const alg_number &);
  friend void divide(alg_number &, const alg_number &, const alg_number &);

  friend void add(alg_number &, const alg_number &, const bigint &);
  friend void subtract(alg_number &, const alg_number &, const bigint &);
  friend void multiply(alg_number &, const alg_number &, const bigint &);
  friend void divide(alg_number &, const alg_number &, const bigint &);

  friend void add(alg_number &, const bigint &, const alg_number &);
  friend void subtract(alg_number &, const bigint &, const alg_number &);
  friend void multiply(alg_number &, const bigint &, const alg_number &);
  friend void divide(alg_number &, const bigint &, const alg_number &);

  friend void power(alg_number &, const alg_number &, const bigint &);
#endif
  friend void power_mod_p(alg_number &, const alg_number &,
			  const bigint &, const bigint &);

  // arithmetic operators:
  friend alg_number operator -(const alg_number &);
  friend alg_number operator +(const alg_number &, const alg_number &);
  friend alg_number operator +(const alg_number &, const bigint &);
  friend alg_number operator +(const bigint &, const alg_number &);
  friend alg_number operator -(const alg_number &, const alg_number &);
  friend alg_number operator -(const alg_number &, const bigint &);
  friend alg_number operator -(const bigint &, const alg_number &);
  friend alg_number operator *(const alg_number &, const alg_number &);
  friend alg_number operator *(const alg_number &, const bigint &);
  friend alg_number operator *(const bigint &, const alg_number &);
  friend alg_number operator /(const alg_number &, const alg_number &);
  friend alg_number operator /(const alg_number &, const bigint &);
  friend alg_number operator /(const bigint &, const alg_number &);

  alg_number & operator +=(const alg_number & a)
  { add(*this, *this, a); return *this;}

  alg_number & operator +=(const bigint & a)
  { add(*this, *this, a); return *this;}

  alg_number & operator -=(const alg_number & a)
  { subtract(*this, *this, a); return *this;}

  alg_number & operator -=(const bigint & a)
  { subtract(*this, *this, a); return *this;}

  alg_number & operator *=(const alg_number & a)
  { multiply(*this, *this, a); return *this;}

  alg_number & operator *=(const bigint & a)
  { multiply(*this, *this, a); return *this;}

  alg_number & operator /=(const alg_number & a)
  { divide(*this, *this, a); return *this;}

  alg_number & operator /=(const bigint & a)
  { divide(*this, *this, a); return *this;}

  // Comparisions:
  // By now, only comparision of numbers over the same order is implemented.
  friend bool operator ==(const alg_number&, const alg_number&);
  friend bool operator !=(const alg_number&, const alg_number&);

  bool operator ! () const
  {return is_zero();}

  // Some number-theoretic functions:
  friend lidia_size_t degree(const alg_number &);
  friend bigint_matrix rep_matrix(const alg_number &);
  friend bigrational norm(const alg_number &);	// Norm
  friend bigrational trace(const alg_number &);	// Trace
  friend polynomial <bigint> charpoly(const alg_number &); 
                     // characteristic polynomial

  // other functions:
#ifndef HEADBANGER
  friend void negate(alg_number &, const alg_number &);
  friend void invert(alg_number &, const alg_number &);
  friend alg_number inverse(const alg_number &);
#endif
  friend bigint denominator(const alg_number &);
  friend math_vector <bigint> coeff_vector(const alg_number &);
  friend alg_number numerator(const alg_number &);
  friend const order * which_order(const alg_number &);
#ifndef HEADBANGER
  friend void square(alg_number &, const alg_number &);
#endif
  friend void swap(alg_number &, alg_number &);
    
  // random numbers
  void randomize(const bigint &);

  // In-/Output:
  friend ostream& operator<<(ostream&, const alg_number&);
  friend istream& operator>>(istream&, alg_number&);
    
  // friends:
  friend class module;
  //  friend module operator *(const module &, const module &);
  friend void multiply(module &, const module &, const alg_number &);
  friend void multiply(module &, const alg_number &, const module &);
  friend void divide(module &, const module &, const alg_number &);

};

inline alg_number operator -(const alg_number & a){
  alg_number c(-a.coeff, a.den, a.which_order());
  return c;
}

inline alg_number operator +(const alg_number & a, const alg_number & b){
  alg_number c(a.which_order());
  add(c,a,b);
  return c;
}

inline alg_number operator +(const alg_number & a, const bigint & b){
  alg_number c(a.which_order());
  add(c,a,b);
  return c;
}

inline alg_number operator +(const bigint & b, const alg_number & a){
  alg_number c(a.which_order());
  add(c,a,b);
  return c;
}

inline alg_number operator -(const alg_number & a, const alg_number & b){
  alg_number c(a.which_order());
  subtract(c,a,b);
  return c;
}

inline alg_number operator - (const alg_number & a, const bigint & b)
{
  alg_number c(a.which_order());
  subtract(c,a,b);
  return c;
}

inline alg_number operator - (const bigint & b, const alg_number & a)
{
  alg_number c(a.which_order());
  subtract(c,b,a);
  return c;
}

inline alg_number operator *(const alg_number & a, const alg_number & b)
{
  alg_number c(a.which_order());
  multiply(c,a,b);
  return c;
}

inline alg_number operator *(const alg_number & a, const bigint & b)
{
  alg_number c(a.which_order());
  multiply(c,a,b);
  return c;
}

inline alg_number operator *(const bigint & b, const alg_number & a)
{
  alg_number c(a.which_order());
  multiply(c,a,b);
  return c;
}

inline alg_number operator /(const alg_number & a, const alg_number & b)
{
  debug_handler("alg_number",
		"in operator /(const alg_number & a, const alg_number & b)");
  alg_number c(a.which_order());
  divide(c,a,b);
  return c;
}

inline alg_number operator /(const alg_number & a, const bigint & b)
{
  debug_handler("alg_number",
		"in operator /(const alg_number & a, const bigint & b)");
  alg_number c(a.which_order());
  divide(c,a,b);
  return c;
}

inline alg_number operator /(const bigint & a, const alg_number & b)
{
  debug_handler("alg_number",
		"in operator /(const bigint & a, const alg_number & b)");
  alg_number c(b.which_order());
  divide(c,a,b);
  return c;
}

inline bool operator !=(const alg_number & a, const alg_number & b)
{ return !(a==b);}

inline lidia_size_t degree(const alg_number & a)
{ return (a.O)->degree();}

inline bigint denominator(const alg_number & a)
{ return a.den;}

inline math_vector <bigint> coeff_vector(const alg_number & a)
{ return a.coeff;}

inline alg_number numerator(const alg_number & a)
{ return alg_number(a.coeff, bigint(1), a.O);}


inline const order * which_order(const alg_number & a)
{ return a.O;}


class module{

protected:
#ifdef LIDIA_DEBUG
  static int count;
#endif
  bigint_matrix base;
  bigint den;
  const order * O;

  void compare(const module&, bool &, bool &) const;
    // internal routine for comparisons
    
public:
  // Contructors & destructor:
  module(const order * O1= current_order);		   // zero module
  module(const alg_number & a,
	 const alg_number & b = alg_number(bigint(0)));
  module(const base_matrix <bigint> &, const bigint & d = 1,
	 const order * O1 = current_order);
  module(const module &);
  ~module();

  // member-functions
  bigint denominator() const
  {return den;}

  bigint_matrix coeff_matrix() const
  {return base;}

  module numerator() const
  {return module(base, bigint(1), O);}

  const order * which_order() const
  {return O;}

  lidia_size_t degree() const
  { return O->degree();}

#ifndef HEADBANGER
  bool is_zero() const;		// In the obvious sense
#endif

  bool is_whole_order() const
  { bool equal, in_O; compare(*O, equal, in_O); return equal;}

#ifndef HEADBANGER
  bool is_one() const		// i.e. is_whole_order
  { return is_whole_order();}
#endif

  void normalize();

#ifndef HEADBANGER
  void invert();

  void assign_zero();		// Remains in the same order

  void assign_whole_order()	// Remains in the same order
  {assign(*O);}

  void assign_one()		// Remains in the same order, is the same as:
  {assign_whole_order();}

  void assign(const bigint &);	// Remains in the same order
  void assign(const alg_number &);// Becomes subset of same order as a !
  void assign(const module &);	// Becomes subset of same order as a !

  // Procedural versions of arithmetic operations:
  friend void add(module &, const module &, const module &);
  friend void intersect(module &, const module &, const module &);
  friend void multiply(module &, const module &, const module &);
  friend void multiply(module &, const module &, const bigint &);
  friend void multiply(module &, const bigint &, const module &);
  friend void multiply(module &, const module &, const alg_number &);
  friend void multiply(module &, const alg_number &, const module &);
  friend void divide(module &, const module &, const module &); // Warning:
  // Division by an order is only guaranteed to produce correct results, if
  // you are in the maximal order!!
  friend void divide(module &, const module &, const bigint &);
  friend void divide(module &, const module &, const alg_number &);

  friend void remainder(module &, const module &, const bigint &);

  friend void power(module &, const module &, const bigint &);
#endif

  // arithmetic operators:
  friend module operator +(const module &, const module &); // sum or union
  friend module operator &(const module &, const module &); // intersection
  friend module operator *(const module &, const module &); // product
  friend module operator *(const module &, const bigint &); // product
  friend module operator *(const bigint &, const module &); // product
  friend module operator *(const module &, const alg_number &); // product
  friend module operator *(const alg_number &, const module &); // product
  friend module operator /(const module &, const bigint &); // quotient 
  friend module operator /(const module &, const alg_number &); // quotient 
  friend module operator /(const module &, const module &); // quotient (??):
  // Division by an order is only guaranteed to produce correct results, if
  // you are in the maximal order!!
  friend module operator %(const module &, const bigint &); // Reduce mod pO

  module& operator +=(const module & a)
  { add(*this, *this, a); return *this;}

  module& operator &=(const module & a)
  { intersect(*this, *this, a); return *this;}

  module& operator *=(const module & a)
  { multiply(*this, *this, a); return *this;}

  module& operator *=(const bigint & a)
  { multiply(*this, *this, a); return *this;}

  module& operator /=(const bigint & a)
  { divide(*this, *this, a); return *this;}

  module& operator /=(const module & a)	// Warning:
  { divide(*this, *this, a); return *this;}
  // Division by a module is only guaranteed to produce correct results, if
  // you are in the maximal order!!

  module& operator %=(const bigint & a)
  { remainder(*this, *this, a); return *this;}
    
  // Comparisions:
  // By now, only comparision of modules over the same order is implemented.
  friend bool operator ==(const module&, const module&);
  friend bool operator !=(const module&, const module&);
  friend bool operator <=(const module&, const module&);
  friend bool operator <(const module&, const module&);
  friend bool operator >=(const module&, const module&);
  friend bool operator >(const module&, const module&);

  bool operator ! () const
  {return is_zero();}

  // Some number-theoretic function:
  friend lidia_size_t degree(const module &);
  friend bigrational norm(const module &);	// Norm
  friend bigrational exp(const module &);	// exponent
  order ring_of_multipliers(const bigint &p, bigint & factor) const;

  // reduction:
  void reduce();
  friend module reduce(const module & a)
  {module c(a); c.reduce(); return c;}

  // other functions:
#ifndef HEADBANGER
  friend void invert(module &, const module &);
  friend module inverse (const module &);
#endif

  friend bigint denominator(const module &);
  friend bigint_matrix coeff_matrix(const module &);
  friend module numerator(const module &);
  friend const order * which_order(const module &);
#ifndef HEADBANGER
  friend void square(module &, const module &);
#endif
  friend void swap(module &, module &);

  // random numbers
  void randomize(const bigint &);

  // In-/Output:
  friend ostream& operator<<(ostream &, const module &);
  friend istream& operator>>(istream &, module &);
};

inline module operator +(const module & a, const module & b)
{ module c(a.which_order()); add(c,a,b); return c;}

inline module operator &(const module & a, const module & b)
{ module c(a.which_order()); intersect(c,a,b); return c;}

inline module operator *(const module & a, const module & b)
{ module c(a.which_order()); multiply(c,a,b); return c;}

inline module operator *(const module &a, const bigint &b)
{ module c(a.which_order()); multiply(c,a,b); return c;}

inline module operator *(const bigint &b, const module &a)
{ module c(a.which_order()); multiply(c,a,b); return c;}

inline module operator *(const module &a, const alg_number &b)
{ module c(a.which_order()); multiply(c,a,b); return c;}

inline module operator *(const alg_number &b, const module &a)
{ module c(a.which_order()); multiply(c,a,b); return c;}

inline module operator /(const module & a, const module & b)
{ module c(a.which_order()); divide(c,a,b); return c;}

inline module operator /(const module & a, const bigint & b)
{ module c(a.which_order()); divide(c,a,b); return c;}

inline module operator /(const module & a, const alg_number & b)
{ module c(a.which_order()); divide(c,a,b); return c;}

inline module operator %(const module &a, const bigint &p)
{ module c(a.which_order()); remainder(c,a,p); return c;}

inline lidia_size_t degree(const module & a)
{ return (a.O)->degree();}

inline bigint denominator(const module & a)
{ return a.den;}

inline bigint_matrix coeff_matrix(const module & a)
{ return a.base;}

inline module numerator(const module & a)
{ return module(a.base, bigint(1), a.O);}

inline const order * which_order(const module & a)
{return a.O;}


#endif //ALGEBRAIC
