//
// LiDIA - a library for computational number theory
//   Copyright (c) 1994, 1995 by the LiDIA Group
//
// File        : bm_linalg1.c
// Author      : Patrick Theobald (PT)
// Last change : PT, Jun 28, 1995, initial version
//

/*
$Id: bm_linalg1.c,v 1.7 1995/08/07 08:30:55 lidiaadm Exp $
*/

#include <LiDIA/bigint_matrix.h>
#include <stdlib.h>

/**
 ** BEGIN: modular arithmetic
 **/

/**
 ** base operations
 **/

inline void 
pos_div_rem(bigint &q, bigint &r, const bigint &a, const bigint &b)
{
  debug_handler("bigint_matrix","in inline pos_div_rem()");

  div_rem(q,r,a,b);
  if (r.is_lt_zero())
    if (b.is_lt_zero()) 
      {
	subtract(r,r,b);
	inc(q);
      }
    else 
      {
	add(r,r,b);
	dec(q);
      }
}

inline void 
pos_div_rem_long(long &q, long &r, long a, long b) 
{
  debug_handler("bigint_matrix","in inline pos_div_rem_long()");

  q = a / b; 
  r = a - q * b;
  if (r < 0)
    if (b < 0) 
      {
	r -= b;
	q++;
      }
    else 
      {
	r += b;
	q--;
      }
}

inline void 
best_remainder(bigint &a, const bigint &b, const bigint &mod) 
{
  debug_handler("bigint_matrix","in inline best_remainder()");

  if (mod.is_one() || b.is_zero()) 
    a.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      remainder(a,b,mod);
      if (a <= -mod2) 
	add(a, a, mod);
      if (a > mod2) 
	subtract(a, a, mod);
    }
}

inline void 
best_remainder_long(long &a, const bigint &b, long mod) 
{
  debug_handler("bigint_matrix","in inline best_remainder_long()");

  if (mod==1 || b.is_zero()) 
    a = 0;
  else 
    {
      register long mod2 = mod/2;
      remainder(a, b, mod);
      if (a <= -mod2) 
	a += mod;
      if (a > mod2) 
	a -= mod;
    }
}

inline void 
add_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline add_mod()");

  if (mod.is_one()) 
    c.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      add(c, a, b);
      if (c <= -mod2) 
	add(c, c, mod);
      if (c > mod2) 
	subtract(c, c, mod);
    }
}

inline void 
add_mod_long(long &d, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline add_mod_long()");

  if (mod==1) 
    d=0;
  else 
    {
      register long mod2 = mod / 2;
      double c = (double) a + (double) b;
      if (c <= (double) -mod2) 
	c += (double) mod;
      if (c > (double) mod2) 
	c -= (double) mod;
      d = (long) c;
    }
}

inline void 
sub_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline sub_mod()");

  if (mod.is_one()) 
    c.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      subtract(c, a, b);
      if (c <= -mod2) 
	add(c, c, mod);
      if (c > mod2) 
	subtract(c, c, mod);
    }
}

inline void 
sub_mod_long(long &d, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline sub_mod_long()");

  if (mod==1) 
    d=0;
  else 
    {
      register long mod2 = mod/2;
      double c = (double) a - (double) b;
      if (c <= (double) -mod2)
	c += (double) mod;
      if (c > (double) mod2) 
	c -= (double) mod;
      d = (long) c;
    }
}

inline void 
mult_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline mult_mod()");

  if (mod.is_one()) 
    c.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      multiply(c, a, b);
      best_remainder(c, c, mod);
    }
}

inline void 
mult_mod_long(long &d, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline mult_mod_long()");

  if (mod==1) 
    d=0;
  else 
    {
      register long mod2 = mod/2;
      double ab = ((double) a) * ((double) b);
      register long q = (long) (ab / ((double) mod));
      register long res = (long) (ab - (((double) q) * ((double) mod)));
      if (res > mod2) 
	res -= mod;
      if (res <= -mod2) 
	res += mod;
      d = res;
    }
}

inline void 
div_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline div_mod()");

  bigint u, v;
  bigint d = xgcd(u, v, b, mod);
  if (!d.is_one())
    lidia_error_handler("bigint_matrix", "div_mod - Version bigint :: "
		  "Inverse undefined");
  mult_mod(c, a, u, mod);
}

inline void 
div_mod_long(long &de, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline div_mod_long()");

  long u, v;
  register long d = xgcd(u, v, b, mod);
  if (d != 1)
    lidia_error_handler("bigint_matrix", "div_mod_long - Version long :: "
		  "Inverse undefined");
  mult_mod_long(de,a,u,mod);
}

inline void 
inv_mod(bigint &c, const bigint &a, const bigint &mod) 
{
  debug_handler("bigint_matrix","in inline inv_mod()");

  bigint d, v, mod2;
  shift_right(mod2,mod,1);
  d = xgcd(c, v, a, mod);
  if (!d.is_one())
    lidia_error_handler("bigint_matrix", "inv_mod - Version bigint :: "
		  "Inverse undefined");
  if (c <= -mod2) 
    add(c, c, mod);
  if (c > mod2) 
    subtract(c, c, mod);
}

inline void 
inv_mod_long(long &e, long a, long mod)
{
  debug_handler("bigint_matrix","in inline inv_mod_long()");

  long t, mod2 = mod/2;
  register long d = xgcd(e, t, a, mod);
  if (d != 1)
    lidia_error_handler("bigint_matrix", "inv_mod_long - Version long :: "
		  "Inverse undefined");
  if (e <= -mod2) 
    e += mod;
  if (e > mod2) 
    e -= mod;
}

/**
 ** END: Modular arithmetic
 **/

/**
 ** BEGIN: Linear algebra - field Fp
 **/
  
/**
 ** column step form
 **/

inline int
STF_intern(bigint ** value, long r, long c, const bigint & mod)
{
  /**
   ** INPUT: **value = values of matrix
   **              r = number of rows
   **              c = number of columns
   **            mod = modulus for Fp - class
   ** DESCRIPTION: ex = STF_intern(value,r,c,mod);
   **              => matrix (value,r,c) in column step form
   **              => ex = (-1)^Number_of_column_exchanges
   ** VERSION: bigint 1.8
   **/

  debug_handler("bigint_matrix", "in inline - function "
		"STF_intern(bigint **, long, long, const bigint &)");

  register long i, j, z, j0 = c -1;
  bigint TMP, TMP1, TMP2;
  bigint *tmp;
  
  /* Step 1 - 4 */
  register int exchange = 1;
  
  /* Step 5 - 8 */
  for (i = r - 1; i >= 0; i--)
    {
      tmp = value[i];
      
      /* Step 9 - 13 */
      for (j = j0; j >= 0 && tmp[j].is_zero(); j--);
      
      /* Step 14 - 26 */
      if (j != -1)
	{
	  if (j != j0)
	    {
	      exchange = -exchange;
	      
	      /* exchange column j0 with column j */
	      for (z = 0; z <= i; z++)
		swap(value[z][j0],value[z][j]);
	    }
	  inv_mod(TMP, tmp[j0], mod);
	  
	  /* Step 27 - 29 */
	  for (j = j0 - 1; j >= 0; j--)
	    {
	      
	      /* Step 30 - 40 */
	      mult_mod(TMP1, tmp[j], TMP, mod);
	      
	      for (z = 0; z <= i; z++)
		{
		  mult_mod(TMP2, value[z][j0], TMP1, mod);
		  sub_mod(value[z][j], value[z][j], TMP2, mod);
		}
	    }
	  
	  /* Step 41 - 48 */
	  j0--;
	}
    }
  return exchange;
}

inline int
STF_intern(long **value, long r, long c, long mod) 
{
  /**
   ** INPUT: **value = values of matrix
   **              r = number of rows
   **              c = number of columns
   **            mod = modulus for Fp - class
   ** DESCRIPTION: ex = STF_intern(value,r,c,mod);
   **              => matrix (value,r,c) in column step form
   **              => ex = (-1)^Number_of_column_exchanges
   ** VERSION: long 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"STF_intern(long **, long, long, long)");
  
  register long i, j, z, j0 = c - 1;
  long TMP, TMP1, TMP2;
  long *tmp;
  
  /* Step 1 - 4 */
  register int exchange = 1;
  
  /* Step 5 - 8 */
  for (i = r - 1; i >= 0; i--)
    {
      tmp = value[i];

      /* Step 9 - 13 */
      for (j = j0; j >= 0 && tmp[j] == 0; j--);
      
      /* Step 14 - 26 */
      if (j != -1)
	{
	  if (j != j0)
	    {
	      exchange = -exchange;
	      for (z = 0; z <= i; z++)
		{
		  /* exchange column j0 with column j */
		  tmp = value[z];
		  TMP = tmp[j0];
		  tmp[j0] = tmp[j];
		  tmp[j] = TMP;
		}
	    }
	  inv_mod_long(TMP, tmp[j0], mod);
	  
	  /* Step 27 - 29 */
	  for (j = j0 - 1; j >= 0; j--)
	    {
	      
	      /* Step 30 - 40 */
	      mult_mod_long(TMP1, tmp[j], TMP, mod);
	      
	      for (z = 0; z <= i; z++)
		{
		  tmp = value[z];
		  mult_mod_long(TMP2, tmp[j0], TMP1, mod);
		  sub_mod_long(tmp[j], tmp[j], TMP2, mod);
		}
	    }
	  
	  /* Step 41 - 48 */
	  j0--;
	}
    }
  return exchange;
}

/**
 ** rank
 **/

inline long 
rank_intern(bigint ** Avalue, long r, long c, const bigint & Gmod)
{
  /**
   ** INPUT: **Avalue = values of matrix
   **               r = number of rows
   **               c = number of columns
   **            Gmod = modulus for Fp - class
   ** DESCRIPTION: rank_intern(Avalue,r,c,Gmod) = rank of matrix (Avalue,r,c)
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in inline - function "
		"rank_intern(bigint **, long, long, const bigint &)");
  
  register long i, j, No=0;
  const bigint *Atmp;

  if (Gmod.bit_length() > bigint::bits_per_digit())
    {
      /* bigint part */
      bigint **value = new bigint *[r];
      memory_handler(value, "bigint_matrix", "rank_intern - bigint part ::"
		     "Error in memory allocation (value)");
      bigint *tmp;
      
      for (i = 0; i < r; i++)
	{
	  tmp = new bigint[c];
	  Atmp = Avalue[i];
	  memory_handler(tmp, "bigint_matrix", "rank_intern - bigint part :: "
			 "Error in memory allocation (tmp)");
	  for (j = 0; j < c; j++)
	    best_remainder(tmp[j], Atmp[j], Gmod);
	  value[i] = tmp;
	}
      
      /* Step 1,2 */
      STF_intern(value, r, c, Gmod);
      
      /* Step 3 - 12 */
      for (j = 0; j < c; j++)
	{
	  for (i = r - 1; i >= 0 && value[i][j].is_zero(); i--);
	  if (i == -1)
	    No++;
	}
      
      for (j = 0; j < r; j++)
	delete[] value[j];
      delete[] value;
    }
  else
    {
      /* long part */
      long mod;
      Gmod.longify(mod);
      
      long **value = new long *[r];
      memory_handler(value, "bigint_matrix", "rank_intern - long part :: "
		     "Error in memory allocation (value)");
      long *tmp;
      
      for (i = 0; i < r; i++)
	{
	  tmp = new long[c];
	  Atmp = Avalue[i];
	  memory_handler(tmp, "bigint_matrix", "rank_intern - long part :: "
			 "Error in memory allocation (tmp)");
	  for (j = 0; j < c; j++)
	    best_remainder_long(tmp[j], Atmp[j], mod);
	  value[i] = tmp;
	}
      
      /* Step 1,2 */
      STF_intern(value, r, c, mod);
      
      /* Step 3 - 12 */
      for (j = 0; j < c; j++)
	{
	  for (i = r - 1; i >= 0 && value[i][j] == 0; i--);
	  if (i == -1)
	    No++;
	}
      for (j = 0; j < r; j++)
	delete[] value[j];
      delete[] value;
    }
  
  /* Step 13 - 24 */
  return (c - No);
}

/**
 ** rank and linearly independent rows or columns
 **/

long *
lininr_intern(bigint ** Avalue, long r, long c, const bigint & Gmod)
{
  /**
   ** INPUT: **Avalue = values of matrix
   **               r = number of rows
   **               c = number of columns
   **            Gmod = modulus for Fp - class
   ** DESCRIPTION: IND = lininr_intern(Avalue,r,c,Gmod);
   **              => IND[0] = Rank of matrix (Avalue,r,c)
   **              => IND[1],...,IND[IND[0]], such that
   **                 raw(IND[1]),...,raw(IND[IND[0]])
   **                 of matrix (Avalue,r,c) are linearly independent.
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in inline - function "
		"lininr_intern(bigint **, long, long, const bigint &)");

  register long i, j;
  const bigint *Atmp;

  long *l = new long[c + 1];
  memory_handler(l, "bigint_matrix", "lininr_intern :: "
		 "Error in memory allocation (l)");

  if (Gmod.bit_length() > bigint::bits_per_digit())
    {
      /* bigint part */
      bigint **value = new bigint *[r];
      memory_handler(value, "bigint_matrix", "lininr_intern - bigint part ::"
		     "Error in memory allocation (value)");
      bigint *tmp;
      
      for (i = 0; i < r; i++)
	{
	  tmp = new bigint[c];
	  Atmp = Avalue[i];
	  memory_handler(tmp,"bigint_matrix", "lininr_intern - bigint part :: "
			 "Error in memory allocation (tmp)");
	  for (j = 0; j < c; j++)
	    best_remainder(tmp[j], Atmp[j], Gmod);
	  value[i] = tmp;
	}
      
      /* Step 1,2 */
      STF_intern(value, r, c, Gmod);
      
      /* Step 3 - 12 */
      for (j = 0; j < c; j++)
	{
	  for (i = r - 1; i >= 0 && value[i][j].is_zero(); i--);
	  l[j] = i;
	}
      
      for (j = 0; j < r; j++)
	delete[] value[j];
      delete[] value;
    }
  else
    {
      /* long part */
      long mod;
      Gmod.longify(mod);
      
      long **value = new long *[r];
      memory_handler(value, "bigint_matrix", "lininr_intern - long part :: "
		     "Error in memory allocation (value)");
      long *tmp;
      
      for (i = 0; i < r; i++)
	{
	  tmp = new long[c];
	  Atmp = Avalue[i];
	  memory_handler(tmp, "bigint_matrix", "lininr_intern - long part :: "
			 "Error in memory allocation (tmp)");
	  for (j = 0; j < c; j++)
	    best_remainder_long(tmp[j], Atmp[j], mod);
	  value[i] = tmp;
	}
      
      /* Step 1,2 */
      STF_intern(value, r, c, mod);
      
      /* Step 3 - 12 */
      for (j = 0; j < c; j++)
	{
	  for (i = r - 1; i >= 0 && value[i][j] == 0; i--);
	  l[j] = i;
	}
      for (j = 0; j < r; j++)
	delete[] value[j];
      delete[] value;
    }
  
  /* Step 13 - 24 */
  for (i = 0; i < c && l[i] == -1; i++);    /* i = number of zero-columns */
  
  long TMP = c - i; /* rank */
  long *IND = new long[TMP + 1];
  memory_handler(IND, "bigint_matrix", "lininr_intern - "
		 "Error in memory allocation (IND)");
  IND[0] = TMP; /* rank */
  for (j = 0; j < TMP; j++)
    IND[TMP - j] = l[j + i];
  delete[] l;
  return IND;
}

inline long *
lininc_intern(bigint ** Avalue, long r, long c, const bigint & Gmod)
{
  /**
   ** INPUT: **Avalue = values of matrix
   **               r = number of rows
   **               c = number of columns
   **            Gmod = modulus for Fp - class
   ** DESCRIPTION: IND = lininc_intern(Avalue,r,c,Gmod);
   **              => IND[0] = Rank of matrix (Avalue,r,c)
   **              => IND[1],...,IND[IND[0]], such that
   **                 column(IND[1]),...,column(IND[IND[0]])
   **                 of matrix (Avalue,r,c) are linearly independent.
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in inline - function "
		"lininc_intern(bigint **, long, long, const bigint &)");

  register long i, j;
  long *l = new long[r + 1];
  memory_handler(l, "bigint_matrix", "lininc_intern :: "
		 "Error in memory allocation (l)");
  if (Gmod.bit_length() > bigint::bits_per_digit())
    {
      /* bigint part */
      bigint **value = new bigint *[c];
      memory_handler(value, "bigint_matrix", "lininc_intern - bigint part ::"
		     "Error in memory allocation (value)");
      bigint *tmp;
      
      for (i = 0; i < c; i++)
	{
	  tmp = new bigint[r];
	  memory_handler(tmp,"bigint_matrix", "lininc_intern - bigint part :: "
			 "Error in memory allocation (tmp)");
	  for (j = 0; j < r; j++)
	    best_remainder(tmp[j], Avalue[j][i], Gmod);
	  value[i] = tmp;
	}
      
      /* Step 1,2 */
      STF_intern(value, c, r, Gmod);
      
      /* Step 3 - 12 */
      for (j = 0; j < r; j++)
	{
	  for (i = c - 1; i >= 0 && value[i][j].is_zero(); i--);
	  l[j] = i;
	}
      
      for (j = 0; j < c; j++)
	delete[] value[j];
      delete[] value;
    }
  else
    {
      /* long part */
      long mod;
      Gmod.longify(mod);
      
      long **value = new long *[c];
      memory_handler(value, "bigint_matrix", "lininc_intern - long part :: "
		     "Error in memory allocation (value)");
      long *tmp;
      
      for (i = 0; i < c; i++)
	{
	  tmp = new long[r];
	  memory_handler(tmp, "bigint_matrix", "lininc_intern - long part :: "
			 "Error in memory allocation (tmp)");
	  for (j = 0; j < r; j++)
	    best_remainder_long(tmp[j], Avalue[j][i], mod);
	  value[i] = tmp;
	}
      
      /* Step 1,2 */
      STF_intern(value, c, r, mod);
      
      /* Step 3 - 12 */
      for (j = 0; j < r; j++)
	{
	  for (i = c - 1; i >= 0 && value[i][j] == 0; i--);
	  l[j] = i;
	}
      for (j = 0; j < c; j++)
	delete[] value[j];
      delete[] value;
    }
  
  /* Step 13 - 24 */
  for (i = 0; i < r && l[i] == -1; i++);    /* i = number of zero-columns */
  
  long TMP = r - i;
  long *IND = new long[TMP + 1];
  memory_handler(IND, "bigint_matrix", "lininc_intern - "
		 "Error in memory allocation (IND)");
  IND[0] = TMP; /* rank */
  for (j = 0; j < TMP; j++)
    IND[TMP - j] = l[j + i];
  delete[] l;
  return IND;
}

/**
 ** adjoint matrix
 **/

inline void 
adj_intern(bigint ** value, long r, const bigint & mod) 
{
  /**
   ** INPUT: **value = values of matrix
   **              r = number of rows = number of columns
   **            mod = modulus for Fp - class
   ** DESCRIPTION: adj_intern(value,r,mod);
   **              => adjoint matrix (value,r,r)
   ** VERSION: bigint 1.8
   **/
  
  debug_handler("bigint_matrix", "in inline - function "
		"adj_intern(bigint **, long, const bigint &)");

  register long i, j, z;
  bigint TMP, TMP1, TMP2;
  bigint *tmp, *tmp1, *Btmp, *Btmp1;
  long exchange = 1;
  bigint DET = 1;
  
  /* Step 1,2 */
  bigint **Bvalue = new bigint *[r];
  memory_handler(Bvalue, "bigint_matrix", "adj_intern - bigint part :: "
		 "Error in memory allocation (Bvalue)");
  for (i = 0; i < r; i++)
    {
      Btmp = new bigint[r];
      tmp = value[i];
      memory_handler(Btmp, "bigint_matrix", "adj_intern - bigint part :: "
		     "Error in memory allocation (Btmp)");
      for (j = 0; j < r; j++)
	{
	  Btmp[j].assign(tmp[j]);
	  if (i == j)
	    tmp[j].assign_one();
	  else
	    tmp[j].assign_zero();
	}
      Bvalue[i] = Btmp;
    }
  
  /* Step 3 - 5 */
  for (i = r - 1; i >= 0; i--)
    {
      Btmp = Bvalue[i];

      /* Step 6 - 11 */
      for (j = i; j >= 0 && Btmp[j].is_zero(); j--);
      
      /* Step 12 - 19 */
      if (j != i)
	{
	  exchange = -exchange;
	  for (z=0;z<r;z++)
	    {
	      Btmp1 = Bvalue[z];
	      tmp1 = value[z];
	      
	      /* A.swap_columns(i, j); */
	      TMP.assign(Btmp1[j]);
	      Btmp1[j].assign(Btmp1[i]);
	      Btmp1[i].assign(TMP);
	      
	      /* B.swap_columns(i, j); */
	      TMP.assign(tmp1[i]);
	      tmp1[i].assign(tmp1[j]);
	      tmp1[j].assign(TMP);
	    }
	}
      inv_mod(TMP1, Btmp[i], mod);
      
      /* Step 20 - 32 */
      for (j = 0; j < r; j++)
	{
	  if (j != i)
	    {
	      mult_mod(TMP2, Btmp[j], TMP1, mod);
	      for (z = 0; z < i; z++)
		{
		  Btmp1 = Bvalue[z];
		  
		  mult_mod(TMP, Btmp1[i], TMP2, mod);
		  sub_mod(Btmp1[j], Btmp1[j], TMP, mod);
		}
	      for (z = 0; z< r; z++)
		{
		  tmp1 = value[z];
	  
		  mult_mod(TMP, tmp1[i], TMP2, mod);
		  sub_mod(tmp1[j], tmp1[j], TMP, mod);
		}
	    }
	}
      mult_mod(DET, DET, Btmp[i], mod);
      for (z = 0; z < i; z++)
	{
	  Btmp1 = Bvalue[z]; 
	  mult_mod(Btmp1[i], Btmp1[i], TMP1, mod);
	}
      for (z = 0; z < r; z++)
	{
	  tmp1 = value[z];
	  mult_mod(tmp1[i], tmp1[i], TMP1, mod);
	}
    }
  
  /* Step 33 - 43 */
  if (exchange < 0)
    DET.negate();
  for (j = 0; j < r; j++)
    {
      tmp = value[j];
      for (i = 0; i < r; i++)
	mult_mod(tmp[i], tmp[i], DET, mod);
    }
  for (j = 0; j < r; j++)
    delete[] Bvalue[j];
  delete[] Bvalue;
}

inline void 
adj_intern(long **value, long r, long mod)
{
  /**
   ** INPUT: **value = values of matrix
   **              r = number of rows = number of columns
   **            mod = modulus for Fp - class
   ** DESCRIPTION: adj_intern(value,r,mod);
   **              => adjoint matrix (value,r,r)
   ** VERSION: long 1.8
   **/

  debug_handler("bigint_matrix", "in inline - function "
		"adj_intern(long **, long, long)");

  register long i, j, z;
  long TMP, TMP1, TMP2;
  long *tmp, *tmp1, *Btmp, *Btmp1;
  long exchange = 1;
  long DET = 1;

  /* Step 1,2 */
  long **Bvalue = new long *[r];
  memory_handler(Bvalue, "bigint_matrix", "adj_intern - long part :: "
		 "Error in memory allocation (Bvalue)");
  for (i = 0; i < r; i++)
  {
    Btmp = new long[r];
    tmp = value[i];
    memory_handler(Btmp, "bigint_matrix", "adj_intern - long part :: "
		   "Error in memory allocation (Btmp)");
    for (j = 0; j < r; j++)
      {
	Btmp[j] = tmp[j];
	if (i == j)
	  tmp[j] = 1;
	else
	  tmp[j] = 0;
      }
    Bvalue[i] = Btmp;
  }
  
  /* Step 3 - 5 */
  for (i = r - 1; i >= 0; i--)
    {
      Btmp = Bvalue[i];

    /* Step 6 - 11 */
    for (j = i; j >= 0 && Btmp[j]== 0; j--);

    /* Step 12 - 19 */
    if (j != i)
    {
      exchange = -exchange;
      for (z=0;z<r;z++)
	{
	  Btmp1 = Bvalue[z];
	  tmp1 = value[z];
	  
	  /* A.swap_columns(i, j); */
	  TMP=Btmp1[j];
	  Btmp1[j]=Btmp1[i];
	  Btmp1[i]=TMP;
	  
	  /* B.swap_columns(i, j); */
	  TMP=tmp1[i];
	  tmp1[i]=tmp1[j];
	  tmp1[j]=TMP;
	}
    }
    
    inv_mod_long(TMP1, Btmp[i], mod);
    
    /* Step 20 - 32 */
    for (j = 0; j < r; j++)
    {
      if (j != i)
      {
	mult_mod_long(TMP2, Btmp[j], TMP1, mod);
	for (z = 0; z < i; z++)
	  {
	    Btmp1 = Bvalue[z];
		    
	    mult_mod_long(TMP, Btmp1[i], TMP2, mod);
	    sub_mod_long(Btmp1[j], Btmp1[j], TMP, mod);
	  }
	
	for (z = 0; z < r; z++)
	  {
	    tmp1 = value[z];
	    
	    mult_mod_long(TMP, tmp1[i], TMP2, mod);
	    sub_mod_long(tmp1[j], tmp1[j], TMP, mod);
	  }
      }
    }
    mult_mod_long(DET, DET, Btmp[i], mod);
    for (z = 0; z < i; z++)
      {
	Btmp1 = Bvalue[z];
	mult_mod_long(Btmp1[i], Btmp1[i], TMP1, mod);
      }
    for (z = 0; z < r; z++)
      {
	tmp1 = value[z];
	mult_mod_long(tmp1[i], tmp1[i], TMP1, mod);
      }
  }
  
  /* Step 33 - 43 */
  if (exchange < 0)
    DET = -DET;
  for (j = 0; j < r; j++)
  {
    tmp = value[j];
    for (i = 0; i < r; i++)
      mult_mod_long(tmp[i], tmp[i], DET, mod);
  }

  for (j = 0; j < r; j++)
    delete[] Bvalue[j];
  delete[] Bvalue;
}

/**
 ** determinant
 **/

inline void 
det_intern(bigint & ret, bigint ** value, long r, const bigint & mod) 
{
  /**
   ** INPUT: **value = values of matrix
   **              r = number of rows = number of columns
   **            mod = modulus for Fp - class
   ** DESCRIPTION: det_intern(ret,value,r,mod);
   **              => ret = determinant of matrix (value,r,r)
   ** VERSION: bigint 1.8
   **/

  debug_handler("bigint_matrix", "in inline - function "
	  "det_intern(bigint &, bigint **, long, const bigint &)");

  register long i, j, z;
  bigint TMP, TMP1, TMP2;
  bigint *tmp, *tmp1;
  
  /* Step 1 - 4 */
  long ex = 1;
  ret.assign_one();

  /* Step 5 - 8 */
  for (i = 0; i < r; i++)
    {
      
      /* Step 9 - 13 */
      for (j = i; j < r && value[j][i].is_zero(); j++);
      
      /* Step 14 - 26 */
      if (j == r)
	{
	  ret.assign_zero();
	return;
	}
      if (j != i)
	{
	  ex = -ex;
	  tmp1 = value[j];
	  value[j] = value[i];
	  value[i] = tmp1;
	}
      tmp = value[i];
      
      /* Step 27 - 29 */
      inv_mod(TMP1, tmp[i], mod);
      for (j = i + 1; j < r; j++)
	{
	  
	  /* Step 30 - 40 */
	  tmp1 = value[j];
	  mult_mod(TMP2, tmp1[i], TMP1, mod);
	  for (z = i + 1; z < r; z++)
	    {
	      mult_mod(TMP, tmp[z], TMP2, mod);
	      sub_mod(tmp1[z], tmp1[z], TMP, mod);
	    }
	}
      mult_mod(ret, ret, tmp[i], mod);
    }
  if (ex < 0)
    ret.negate();
}

inline long 
det_intern(long **value, long r, long mod)
{
  /**
   ** INPUT: **value = values of matrix
   **              r = number of rows = number of columns
   **            mod = modulus for Fp - class
   ** DESCRIPTION: det_intern(value,r,mod) = determinant of matrix (value,r,r);
   ** VERSION: long 1.8
   **/

  debug_handler("bigint_matrix", "in inline - function "
		"det_intern(long **, long, long)");

  register long i, j, z;
  long TMP, TMP1, TMP2;
  long *tmp, *tmp1;
  
  /* Step 1 - 4 */
  long ex = 1;
  long ret = 1;
  
  /* Step 5 - 8 */
  for (i = 0; i < r; i++)
    {
      
      /* Step 9 - 13 */
      for (j = i; j < r && value[j][i] == 0; j++);
      
      /* Step 14 - 26 */
      if (j == r)
	return 0;
      if (j != i)
	{
	  ex = -ex;
	  tmp1 = value[j];
	  value[j] = value[i];
	  value[i] = tmp1;
	}
      tmp = value[i];
      
      /* Step 27 - 29 */
      inv_mod_long(TMP1, tmp[i], mod);
      
      for (j = i + 1; j < r; j++)
	{
	  
	  /* Step 30 - 40 */
	  tmp1 = value[j];
	  mult_mod_long(TMP2, tmp1[i], TMP1, mod);
	  for (z = i + 1; z < r; z++)
	    {
	      mult_mod_long(TMP, tmp[z], TMP2, mod);
	      sub_mod_long(tmp1[z], tmp1[z], TMP, mod);
	    }
	}
      mult_mod_long(ret, ret, tmp[i], mod);
    }
  if (ex < 0)
    ret = -ret;
  return ret;
}

/**
 ** Hessenberg form
 **/

inline void 
HBF_intern(bigint ** value, long r, const bigint & mod) 
{
  /**
   ** INPUT: **value = values of matrix
   **              r = number of rows = number of columns
   **            mod = modulus for Fp - class
   ** DESCRIPTION: HBF_intern(value,r,mod);
   **              => matrix (value,r,r) in Hessenberg form
   ** VERSION: bigint 1.8
   **/

  debug_handler("bigint_matrix", "in inline - function "
		"HBF_intern(bigint **, long, const bigint &)");

  /* Step 1,2 */
  register long i, j, z;
  bigint TMP, TMP1, TMP2;
  bigint *tmp;

  /* Step 3 - 11 */
  for (i = r - 1; i >= 1; i--)
  {
    for (j = i - 1; j >= 0 && value[i][j].is_zero(); j--);
    if (j != -1)
    {

      /* Step 12,13 */
      if (j != i - 1)
      {

	/* Step 14 - 18 */
	/* exchange columns i-1 and j */
	for (z = 0; z < r; z++)
	  swap(value[z][i-1],value[z][j]);

	/* Step 19 - 24 */
	/* exchange rows i-1 and j */
	tmp = value[i - 1];
	value[i - 1] = value[j];
	value[j] = tmp;
      }
      tmp = value[i];

      /* Step 25 - 41 */
      inv_mod(TMP2, tmp[i - 1], mod);
      for (j = i - 2; j >= 0; j--)
      {
	mult_mod(TMP1, tmp[j], TMP2, mod);
	for (z = 0; z < r; z++)
	{
	  mult_mod(TMP, value[z][i - 1], TMP1, mod);
	  sub_mod(value[z][j], value[z][j], TMP, mod);
	}
	for (z = 0; z < r; z++)
	{
	  mult_mod(TMP, value[j][z], TMP1, mod);
	  add_mod(value[i - 1][z], value[i - 1][z], TMP, mod);
	}
      }
    }
  }
}

inline void
HBF_intern(long **value, long r, long mod) 
{
  /**
   ** INPUT: **value = values of matrix
   **              r = number of rows = number of columns
   **            mod = modulus for Fp - class
   ** DESCRIPTION: HBF_intern(value,r,mod);
   **              => matrix (value,r,r) in Hessenberg form
   ** VERSION: long 1.8
   **/

  debug_handler("bigint_matrix", "in inline - function "
		"HBF_intern(long **, long, long)");

  /* Step 1,2 */
  long i, j, z;
  long TMP, TMP1, TMP2;
  long *tmp;

  /* Step 3 - 11 */
  for (i = r - 1; i >= 1; i--)
  {
    for (j = i - 1; j >= 0 && value[i][j] == 0; j--);
    if (j != -1)
    {

      /* Step 12,13 */
      if (j != i - 1)
      {

	/* Step 14 - 18 */
	/* exchange columns i-1 and j */
	for (z = 0; z < r; z++)
	{
	  TMP = value[z][i - 1];
	  value[z][i - 1] = value[z][j];
	  value[z][j] = TMP;
	}

	/* Step 19 - 24 */
	/* exchange rows i-1 and j */
	tmp = value[i - 1];
	value[i - 1] = value[j];
	value[j] = tmp;
      }
      tmp = value[i];

      /* Step 25 - 41 */
      inv_mod_long(TMP2, tmp[i - 1], mod);
      for (j = i - 2; j >= 0; j--)
      {
	mult_mod_long(TMP1, tmp[j], TMP2, mod);
	for (z = 0; z < r; z++)
	{
	  mult_mod_long(TMP, value[z][i - 1], TMP1, mod);
	  sub_mod_long(value[z][j], value[z][j], TMP, mod);
	}
	for (z = 0; z < r; z++)
	{
	  mult_mod_long(TMP, value[j][z], TMP1, mod);
	  add_mod_long(value[i - 1][z], value[i - 1][z], TMP, mod);
	}
      }
    }
  }
}

/**
 ** characteristic polynomial
 **/

inline bigint *
charpoly_intern(bigint ** value, long r, const bigint & mod) 
{
  /**
   ** INPUT: **value = values of matrix
   **              r = number of rows = number of columns
   **            mod = modulus for Fp - class
   ** DESCRIPTION: RES = charpoly_intern(value,r,mod);
   **              => RES[0],...,RES[r] are the coefficients of the
   **                 characteristic polynomial of matrix (value,r,r)
   ** VERSION: bigint 1.8
   **/

  debug_handler("bigint_matrix", "in inline - function "
		"charpoly_intern(bigint **, long, const bigint &)");

  register long i, j, z;
  bigint TMP;
  long sign;

  /* Step 1 - 5 */
  HBF_intern(value, r, mod);

  bigint *K = new bigint[r];	/* size = c */
  memory_handler(K, "bigint_matrix", "charpoly_intern - Version bigint :: "
		 "Error in memory allocation (K)");

  for (i = 0; i < r; i++)
    K[i].assign_one();

  /* Step 6 - 8 */
  bigint *tmp;
  bigint **P = new bigint *[r+1];
  memory_handler(P, "bigint_matrix", "charpoly_intern :: "
		 "Error in memory allocation (P)");
  for (i = 0; i < r+1; i++)
    {
      tmp = new bigint[r+1];
      memory_handler(tmp, "bigint_matrix", "charpoly_intern :: "
		     "Error in memory allocation (tmp)");
      for (j = 0; j < r+1; j++)
	tmp[j].assign_zero();
      P[i] = tmp;
    }

  P[0][0].assign_one();

  /* Step 9 - 11 */
  for (z = 1; z <= r; z++)
    {
      
      /* Step 12 - 16 */
      for (j = 1; j <= z - 1; j++)
	mult_mod(K[j - 1], K[j - 1], value[z - 1][z - 2], mod);
      
    /* Step 17 - 23 */
      subtract(P[z][z], mod, P[z - 1][z - 1]);
      for (i = 1; i <= z - 1; i++)
	{
	  mult_mod(TMP, value[z - 1][z - 1], P[i][z - 1], mod);
	  sub_mod(P[i][z], TMP, P[i - 1][z - 1], mod);
	}
      mult_mod(P[0][z], value[z - 1][z - 1], P[0][z - 1], mod);
      
      /* Step 24 - 34 */
      sign = 1;
      for (j = z - 1; j >= 1; j--)
	{
	  sign = -sign;
	  for (i = 0; i <= j - 1; i++)
	    {
	      mult_mod(TMP, sign, P[i][j - 1], mod);
	      mult_mod(TMP, TMP, value[j - 1][z - 1], mod);
	      mult_mod(TMP, TMP, K[j - 1], mod);
	      add_mod(P[i][z], P[i][z], TMP, mod);
	    }
	}
    }
  
  /* Step 35 - 40 */
  bigint *RES = new bigint[r + 1];
  memory_handler(RES, "bigint_matrix", "charpoly_intern - Version bigint :: "
		 "Error in memory allocation (RES)");
  for (i = 0; i < r + 1; i++)
    RES[i].assign(P[i][r]);
  delete[] K;

  for (i = 0; i < r+1; i++)
    delete[] P[i];
  delete[] P;

  return RES;
}

inline long *
charpoly_intern(long **value, long r, long mod) 
{
  /**
   ** INPUT: **value = values of matrix
   **              r = number of rows = number of columns
   **            mod = modulus for Fp - class
   ** DESCRIPTION: RES = charpoly_intern(value,r,mod);
   **              => RES[0],...,RES[r] are the coefficients of
   **                 the characteristic polynomial of matrix (value,r,r)
   ** VERSION: long 1.8
   **/

  debug_handler("bigint_matrix", "in inline - function "
		"charpoly_intern(long **, long, long)");
  long i, j, z;
  long TMP;
  long *tmp;
  long sign;

  /* Step 1 - 5 */
  HBF_intern(value, r, mod);
  long *K = new long[r];		/* size = r */
  memory_handler(K, "bigint_matrix", "charpoly_intern - Version long ::"
		 "Error in memory allocation (K)");
  for (i = 0; i < r; i++)
    K[i] = 1;

  /* Step 6 - 8 */
  long **P = new long *[r + 1];
  memory_handler(P, "bigint_matrix", "charpoly_intern - Version long :: "
		 "Error in memory allocation (P)");
  for (i = 0; i < r + 1; i++)
  {
    tmp = new long[r + 1];
    memory_handler(tmp, "bigint_matrix", "charpoly_intern - Version long :: "
		   "Error in memory allocation (tmp)");
    for (j = 0; j < r + 1; j++)
      tmp[j] = 0;
    P[i] = tmp;
  }
  P[0][0] = 1;

  /* Step 9 - 11 */
  for (z = 1; z <= r; z++)
  {

    /* Step 12 - 16 */
    for (j = 1; j < z; j++)
      mult_mod_long(K[j - 1], K[j - 1], value[z - 1][z - 2], mod);

    /* Step 17 - 23 */
    P[z][z] = mod - P[z - 1][z - 1];
    for (i = 1; i < z; i++)
    {
      mult_mod_long(TMP, value[z - 1][z - 1], P[i][z - 1], mod);
      sub_mod_long(P[i][z], TMP, P[i - 1][z - 1], mod);
    }
    mult_mod_long(P[0][z], value[z - 1][z - 1], P[0][z - 1], mod);

    /* Step 24 - 34 */
    sign = 1;
    for (j = z - 1; j >= 1; j--)
    {
      sign = -sign;
      for (i = 0; i <= j - 1; i++)
      {
	mult_mod_long(TMP, sign, P[i][j - 1], mod);
	mult_mod_long(TMP, TMP, value[j - 1][z - 1], mod);
	mult_mod_long(TMP, TMP, K[j - 1], mod);
	add_mod_long(P[i][z], P[i][z], TMP, mod);
      }
    }
  }

  /* Step 35 - 40 */
  long *RES = new long[r + 1];
  memory_handler(RES, "bigint_matrix", "charpoly_intern - Version long :: "
		 "Error in memory allocation (RES)");
  for (i = 0; i <= r; i++)
    RES[i] = P[i][r];
  for (i = 0; i <= r; i++)
    delete[] P[i];
  delete[] P;
  delete[] K;
  return RES;
}

/**
 ** END: Linear algebra - field Fp
 **/

bigint * bigint_matrix::
mgcd(const bigint * a, long n)
{
  /**
   ** DESCRIPTION: RES = mgcd(a,n);
   **              => RES[0] = RES[1]*a[0] + ... + RES[n]*a[n-1]
   **              => RES[0] = gcd(a[0],...,a[n-1])
   ** ALGORITHM: Bradley
   ** VERSION: 1.8
   **/

  debug_handler("matrix_bigint", "in member - function "
		"mgcd(const bigint *, long)");
  bigint *RES = NULL;
  register long i;	

  if (n<=0)
    lidia_error_handler("matrix_bigint","mgcd :: Error in parameter !!");
  if (n>1)
    {
      /* Step 1 - 8 */
      bigint TMP, TMP1, TMP2;
      
      bigint *tmp = new bigint[n];
      memory_handler(tmp, "matrix_bigint", "mgcd :: "
		     "Error in memory allocation (tmp)");
      bigint *y = new bigint[n];
      memory_handler(y, "matrix_bigint", "mgcd :: "
		     "Error in memory allocation (y)");
      bigint *z = new bigint[n];
      memory_handler(z, "matrix_bigint", "mgcd :: "
		     "Error in memory allocation (z)");
      
      tmp[0].assign(a[0]);
      for (i = 1; i <n; i++)
	tmp[i] = xgcd(y[i], z[i], tmp[i - 1], a[i]);
      
      /* Step 9 - 19 */
      bigint *x = new bigint[n];
      memory_handler(x, "matrix_bigint", "mgcd :: "
		     "Error in memory allocation (x)");
      bigint *y1 = new bigint[n];
      memory_handler(y1, "matrix_bigint", "mgcd :: "
		     "Error in memory allocation (y1)");
      
      x[n-1].assign(z[n-1]);
      y1[n-1].assign(y[n-1]);
      
      bigint *G = new bigint[n];
      memory_handler(G, "matrix_bigint", "mgcd :: "
		     "Error in memory allocation (G)");
      bigint *S = new bigint[n];
      memory_handler(S, "matrix_bigint", "mgcd :: "
		     "Error in memory allocation (S)");
      for (i = n - 2; i >= 1; i--)
	{
	  div_rem(G[i], TMP, tmp[i - 1], tmp[i]);
	  if (G[i].is_zero())
	    S[i].assign_zero();
	  else
	    nearest(S[i], y1[i + 1] * z[i], G[i]);

	  /* y1[i] = y1[i+1]*y[i] + S[i]*(a[i]/tmp[i]); */
	  div_rem(TMP1, TMP, a[i], tmp[i]);
	  multiply(TMP1, S[i], TMP1);
	  multiply(TMP2, y1[i + 1], y[i]);
	  add(y1[i], TMP2, TMP1);
	  
	  /* x[i] = y1[i+1]*z[i] - S[i]*G[i]; */
	  multiply(TMP1, S[i], G[i]);
	  multiply(TMP2, y1[i + 1], z[i]);
	  subtract(x[i], TMP2, TMP1);
	}
      x[0].assign(y1[1]);
      
      /* Step 20,21 */
      RES = new bigint[n + 1];
      memory_handler(RES, "matrix_bigint", "mgcd :: "
		     "Error in memory allocation (RES)");
      RES[0].assign(tmp[n - 1]);
      for (i = 1; i <= n; i++)
	RES[i].assign(x[i - 1]);
      
      delete[] tmp;
      delete[] y;
      delete[] z;
      delete[] x;
      delete[] y1;
      delete[] G;
      delete[] S;
    }	
  if (n==1)
    {		
      RES = new bigint [2];
      RES[0].assign(a[0]);
      RES[1].assign_one();
    }
  if (RES[0].is_lt_zero())
     for (i=0;i<=n;i++)
       RES[i].negate();
  return RES;
}

bigint *bigint_matrix::
mgcd1(const bigint * aconst, long n, bigint_matrix & T)
{
  /**
   ** DESCRIPTION: RES = mgcd1(a,n,A);
   **              => RES[0] = RES[1]*a[0] + ... + RES[n]*a[n-1]
   **              => RES[0] = gcd(a[0],...,a[n-1])
   **              => T*a = RES
   ** ALGORITHM: Blankinship,PIVOT: MINIMUM
   ** VERSION: 1.8
   **/
  
  debug_handler("bigint_matrix", "in member - function "
		"mgcd1(const bigint *, long, bigint_matrix &)");

  register long i, j, index, bound;
  bigint MIN, TMP, q, r, *Ttmp1, *Ttmp2 = NULL;
  
  if (T.columns != n)
    T.set_no_of_columns(n);
  if (T.rows != n)
    T.set_no_of_rows(n);
  T.diag(1, 0);
  
  bigint *a = new bigint[n + 1];
  memory_handler(a, "bigint_matrix", "mgcd1 :: "
		 "Error in memory allocation (a)");

  for (i = 0; i < n; i++)
    a[i].assign(aconst[i]);

  /* Init */
  for (index=0; index<n && a[index].is_zero();index++);

  if (index==n)
    {
      delete[] a;
      return new bigint[n];
    }
  else
    bound = index;

  do
    {      
      /* Pivot search: MINIMUM */
      MIN.assign(a[index]);
      
      for (i = bound; i < n; i++)
	if ((abs(MIN) > abs(a[i])) && !a[i].is_zero())
	  {
	    MIN.assign(a[i]);
	    index = i;
	  }
      
      /* first element != 0 */
      for (i = bound; i < n && (a[i].is_zero() || i == index); i++);
      if (i < n)
	{
	  div_rem(q, r, a[i], MIN);
	  a[i].assign(r);
	  Ttmp1 = T.value[i];
	  Ttmp2 = T.value[index];
	  for (j = 0; j < n; j++)
	    {
	      multiply(TMP, q, Ttmp2[j]);
	      subtract(Ttmp1[j], Ttmp1[j], TMP);
	    }
	}
    }
  while (i<n);

  Ttmp2 = T.value[index];
  
  /* gcd <0 ? */
  if (a[index] < 0)
    {
      a[index].negate();
      for (i = 0; i < n; i++)
	Ttmp2[i].negate();
    }
  
  if (index != 0)
    a[0].assign(a[index]);
  for (i = 1; i <= n; i++)
    a[i].assign(Ttmp2[i - 1]);
  return a;
}

void bigint_matrix::
mgcd2(bigint & RES, const bigint * aconst, long n)
{
  /**
   ** DESCRIPTION: mgcd2(Res,a,n);
   **              => RES = gcd(a[0],...,a[n-1])
   ** ALGORITHM: Blankinship
   ** IMPROVEMENTS: Havas, Majewski, reduction of all elements, MIN assignments
   ** PAPER: Hermite normal form computation for integer matrices, Havas
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"mgcd2(bigint &, const bigint *, long)");

  register long i, index, SW, bound;
  bigint MIN, TMP, q, r;

  bigint *a = new bigint[n + 1];
  memory_handler(a, "bigint_matrix", "mgcd2 :: "
		 "Error in memory allocation (a)");

  for (i = 0; i < n; i++)
    a[i].assign(aconst[i]);
  
  /* init */
  for (index=0; index<n && a[index].is_zero();index++);
  
  if (index==n)
    {
      RES.assign_zero();
      return;
    }
  else
    bound = index;
  
  do
    {
      MIN.assign(a[index]);
      
      /* Pivot search: MINIMUM */
      for (i = bound; i < n; i++)
	if ((abs(MIN) > abs(a[i])) && !a[i].is_zero())
	  {
	    MIN.assign(a[i]);
	    index = i;
	  }
      
      /* all elements */
      SW=0;
      
      for (i = bound; i < n; i++)
	if ((i != index) && !a[i].is_zero())
	  {
	    SW=1;
	    div_rem(q, r, a[i], MIN);
	    a[i].assign(r);
	  }
    }
  while (SW==1);

  /* gcd < 0 ? */
  if (a[index] < 0)
    ::negate(RES,a[index]);
  else
    RES.assign(a[index]);
}

bigint *bigint_matrix::
mgcd2(const bigint * aconst, long n)
{
  /**
   ** DESCRIPTION: RES = mgcd2(a,n);
   **              => RES[0] = RES[1]*a[0] + ... + RES[n]*a[n-1]
   **              => RES[0] = gcd(a[0],...,a[n-1])
   ** ALGORITHM: Blankinship
   ** IMPROVEMENTS: Havas, Majewski, reduction of all elements, MIN assignments
   ** PAPER: Hermite normal form computation for integer matrices, Havas
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"mgcd2(const bigint *, long)");

  register long i, j, index, SW, bound;
  bigint MIN, TMP, q, r, *Ttmp1, *Ttmp2 = NULL;

  bigint_matrix T(n,n);
  T.diag(1,0);

  bigint *a = new bigint[n + 1];
  memory_handler(a, "bigint_matrix", "mgcd2 :: "
		 "Error in memory allocation (a)");

  for (i = 0; i < n; i++)
    a[i].assign(aconst[i]);

  /* init */
  for (index=0; index<n && a[index].is_zero();index++);

  if (index==n)
    {
      delete[] a;
      return new bigint[n];
    }
  else
    bound = index;

  do
    {
      MIN.assign(a[index]);
      
      /* Pivot search: MINIMUM */
      for (i = bound; i < n; i++)
	if ((abs(MIN) > abs(a[i])) && !a[i].is_zero())
	  {
	    MIN.assign(a[i]);
	    index = i;
	  }
      
      /* all elements */
      SW=0;

      Ttmp2 = T.value[index];
      for (i = bound; i < n; i++)
	if ((i != index) && !a[i].is_zero())
	  {
	    SW=1;
	    Ttmp1 = T.value[i];
	    div_rem(q, r, a[i], MIN);
	    a[i].assign(r);
	    for (j = 0; j < n; j++)
	      {
		multiply(TMP, q, Ttmp2[j]);
		subtract(Ttmp1[j], Ttmp1[j], TMP);
	      }
	  }
    }
  while (SW==1);

  Ttmp2 = T.value[index];
  
  /* gcd < 0 ? */
  if (a[index] < 0)
    {
      a[index].negate();
      for (i = 0; i < n; i++)
	Ttmp2[i].negate();
    }
  
  if (index != 0)
    a[0].assign(a[index]);
  for (i = 1; i <= n; i++)
    a[i].assign(Ttmp2[i - 1]);
  
  return a;
}

bigint *bigint_matrix::
mgcd2(const bigint * aconst, long n, bigint_matrix & T)
{
  /**
   ** DESCRIPTION: RES = mgcd2(a,n,A);
   **              => RES[0] = RES[1]*a[0] + ... + RES[n]*a[n-1]
   **              => RES[0] = gcd(a[0],...,a[n-1])
   **              => T*a = RES
   ** ALGORITHM: Blankinship
   ** IMPROVEMENTS: Havas, Majewski, reduction of all elements, MIN assignments
   ** PAPER: Hermite normal form computation for integer matrices, Havas
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"mgcd2(const bigint *, long, bigint_matrix &)");

  register long i, j, index, bound, SW;
  bigint MIN, TMP, q, r, *Ttmp1, *Ttmp2 = NULL;

  if (T.columns != n)
    T.set_no_of_columns(n);
  if (T.rows != n)
    T.set_no_of_rows(n);
  T.diag(1, 0);

  bigint *a = new bigint[n + 1];
  memory_handler(a, "bigint_matrix", "mgcd2 :: "
		 "Error in memory allocation (a)");

  for (i = 0; i < n; i++)
    a[i].assign(aconst[i]);

  /* init */
  for (index=0; index<n && a[index].is_zero();index++);

  if (index==n)
    {
       delete[] a;
       return new bigint[n];
    }
  else
    bound = index;

  do
    {
      MIN.assign(a[index]);
      
      /* Pivot search: MINIMUM */
      for (i = bound; i < n; i++)
	if ((abs(MIN) > abs(a[i])) && !a[i].is_zero())
	  {
	    MIN.assign(a[i]);
	    index = i;
	  }
      
      /* all elements */
      SW=0;
      Ttmp2 = T.value[index];
      for (i = bound; i < n; i++)
	if ((i != index) && !a[i].is_zero())
	  {
	    SW=1;
	    Ttmp1 = T.value[i];
	    div_rem(q, r, a[i], MIN);
	    a[i].assign(r);
	    for (j = 0; j < n; j++)
	      {
		multiply(TMP, q, Ttmp2[j]);
		subtract(Ttmp1[j], Ttmp1[j], TMP);
	      }
	  }
    }
  while (SW==1);

  Ttmp2 = T.value[index];
  
  /* gcd < 0 ? */
  if (a[index] < 0)
    {
      a[index].negate();
      for (i = 0; i < n; i++)
	Ttmp2[i].negate();
    }
  
  if (index != 0)
    a[0].assign(a[index]);
  for (i = 1; i <= n; i++)
    a[i].assign(Ttmp2[i - 1]);
  
  return a;
}

bigint *bigint_matrix::
mgcd3(const bigint * aconst, long n, bigint_matrix & T)
{
  /**
   ** DESCRIPTION: RES = mgcd3(a,n,A);
   **              => RES[0] = RES[1]*a[0] + ... + RES[n]*a[n-1]
   **              => RES[0] = gcd(a[0],...,a[n-1])
   **              => T*a = RES
   ** ALGORITHM: Blankinship
   ** IMPROVEMENTS: Havas, Majewski, reduction of all elements, 
   **               index assignments
   ** PAPER: Hermite normal form computation for longeger matrices, Havas
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"mgcd3(const bigint *, long, bigint_matrix &)");

  register long i, j, index, bound, SW;
  bigint MIN, TMP, q, r, *Ttmp2 = NULL, *Ttmp1;

  if (T.columns != n)
    T.set_no_of_columns(n);
  if (T.rows != n)
    T.set_no_of_rows(n);
  T.diag(1, 0);

  bigint *a = new bigint[n + 1];
  memory_handler(a, "bigint_matrix", "mgcd3 :: "
		 "Error in memory allocation (a)");
  
  for (i = 0; i < n; i++)
    a[i].assign(aconst[i]);
  
  /* init */
  for (index=0; index<n && a[index].is_zero();index++);
  
  if (index==n)
    {
       delete[] a;
       return new bigint[n];
    }
  else
    bound = index;
  
  do
    {
      /* Pivot search: MINIMUM */
      for (i = bound; i < n; i++)
	if ((abs(a[index]) > abs(a[i])) && !a[i].is_zero())
	  index = i;

      MIN.assign(a[index]);
      
      /* all elements */
      SW=0;
      Ttmp2 = T.value[index];
      
      for (i = bound; i < n; i++)
	if ((i != index) && !a[i].is_zero())
	  {
	    SW=1;
	    Ttmp1 = T.value[i];
	    div_rem(q, r, a[i], MIN);
	    a[i].assign(r);
	    for (j = 0; j < n; j++)
	      {
		multiply(TMP, q, Ttmp2[j]);
		subtract(Ttmp1[j], Ttmp1[j], TMP);
	      }
	  }
    }
  while (SW==1);

  Ttmp2 = T.value[index];
  
  if (a[index] < 0)
    {
      a[index].negate();
      for (i = 0; i < n; i++)
	Ttmp2[i].negate();
    }
  
  if (index != 0)
    a[0].assign(a[index]);
  for (i = 1; i <= n; i++)
    a[i].assign(Ttmp2[i - 1]);
  
  return a;
}

bigint *bigint_matrix::
mgcd4(const bigint * aconst, long n, bigint_matrix & T)
{
  /**
   ** DESCRIPTION: RES = mgcd4(a,n,A);
   **              => RES[0] = RES[1]*a[0] + ... + RES[n]*a[n-1]
   **              => RES[0] = gcd(a[0],...,a[n-1])
   **              => T*a = RES
   ** ALGORITHM: Blankinship
   ** IMPROVEMENTS: Havas, Majewski, reduction of all elements, 
   **               MIN assignments, best remainder
   ** PAPER: Hermite normal form computation for longeger matrices, 
   **        Havas + best remainder
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in function "
		"mgcd4(const bigint *, long, bigint_matrix &)");

  register long i, j, index, bound, SW;
  bigint MIN, TMP, q, r, *Ttmp1, *Ttmp2 = NULL;

  if (T.columns != n)
    T.set_no_of_columns(n);
  if (T.rows != n)
    T.set_no_of_rows(n);
  T.diag(1, 0);

  bigint *a = new bigint[n + 1];
  memory_handler(a, "bigint_matrix", "in function "
		 "mgcd4(const bigint *, long, bigint_matrix &)");

  for (i = 0; i < n; i++)
    a[i].assign(aconst[i]);

  /* init */
  for (index=0; index<n && a[index].is_zero();index++);
  
  if (index==n)
    {
       delete[] a;
       return new bigint[n];
    }
  else
    bound = index;

  do
    {
      
      MIN.assign(a[index]);
      
      /* Pivot search: MINIMUM */
      for (i = bound; i < n; i++)
	if ((abs(MIN) > abs(a[i])) && !a[i].is_zero())
	  {
	    MIN.assign(a[i]);
	    index = i;
	  }
      
      /* all elements */
      SW = 0;
      Ttmp2 = T.value[index];
      for (i = bound; i < n; i++)
	if ((i != index) && !a[i].is_zero())
	  {
	    SW = 1;

	    Ttmp1 = T.value[i];
	    pos_div_rem(q, r, a[i], MIN);
	    if (r * 2 > abs(MIN))
	      {
		if (!MIN.is_lt_zero())
		  {
		    inc(q);
		    subtract(r, r, MIN);
		  }
		else
		  {
		    dec(q);
		    add(r,r,MIN);
		  }
	      }
	    
	    a[i].assign(r);
	    for (j = 0; j < n; j++)
	      {
		multiply(TMP, q, Ttmp2[j]);
		subtract(Ttmp1[j], Ttmp1[j], TMP);
	      }
	  }
    }
  while (SW==1);
  
  Ttmp2 = T.value[index];
  
  if (a[index] < 0)
    {
      a[index].negate();
      for (i = 0; i < n; i++)
	Ttmp2[i].negate();
    }
  
  if (index != 0)
    a[0].assign(a[index]);
  for (i = 1; i <= n; i++)
    a[i].assign(Ttmp2[i - 1]);

  return a;
}

/**
 ** Chinese remaindering theorem
 **/

void
chinrest(bigint & RES, const bigint * values, const bigint * prim)
{
  /**
   ** DESCRIPTION: chinrest(Res,v,prim);
   **              => Res = solution of the Chinese remaindering theorem
   **                 with parameters v and prim
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in function "
		"chinrest(bigint &, const bigint *, const bigint *)");
  register long i;
  bigint M, X, mod;
  bigint TMP, TMP0, TMP1, TMP2;

  long len;
  prim[0].longify(len);

  bigint *e = new bigint[len];
  memory_handler(e, "bigint_matrix", "chinrest :: "
		 "Error in memory allocation (e)");

  /* Step 1 */
  bigint L = 1;
  for (i = 0; i < len; i++)
    multiply(L, L, prim[i + 1]);

  for (i = 0; i < len; i++)
  {
    mod.assign(prim[i + 1]);
    div_rem(M, TMP, L, mod);
    best_remainder(TMP, M, mod);
    xgcd(TMP0, TMP1, mod, TMP);
    multiply(e[i], TMP1, M);
  }
  X.assign_zero();
  for (i = 0; i < len; i++)
  {
    multiply(TMP, e[i], values[i]);
    add(X, X, TMP);
  }
  delete[] e;
  best_remainder(X, X, L);
  subtract(TMP0, L, bigint(1));
  shift_left(TMP1, X, 1);
  shift_left(TMP2, L, 1);

  if (TMP1 >= -TMP0 && TMP1 <= TMP0)
    RES.assign(X);
  else if (TMP1 - TMP2 <= TMP0 || TMP1 - TMP2 >= -TMP0)
    subtract(RES, X, L);
  else
    lidia_error_handler("bigint_matrix", "in function chinrest - No Solution");
}

bigint
chinrest(const bigint * values, const bigint * prim)
{
  debug_handler("bigint_matrix",
		"in function chinrest(const bigint *, const bigint *)");
  bigint RES;
  chinrest(RES, values, prim);
  return RES;
}

void bigint_matrix::
chinrest(const bigint_matrix * v, const bigint * prim)
{
  debug_handler("bigint_matrix", "in member - function "
		"chinrest(const bigint_matrix *, const bigint *)");
  register long i, j, l;

  long len;
  prim[0].longify(len);
  bigint M, X, mod;
  bigint TMP, TMP0, TMP1, TMP2;
  bigint dummy;

  bigint *e = new bigint[len];
  memory_handler(e, "bigint_matrix", "chinrest :: "
		 "Error in memory allocation (e)");
  long r = v[0].rows;
  long c = v[0].columns;

  /* new dimensions */
  if (columns != c)
    set_no_of_columns(c);
  if (rows != r)
    set_no_of_rows(r);

  bigint *m = new bigint[len];
  memory_handler(m, "bigint_matrix", "chinrest :: "
		 "Error in memory allocation (m)");

  /* Step 1 */
  bigint L = 1;
  for (i = 0; i < len; i++)
    multiply(L, L, prim[i + 1]);

  for (i = 0; i < len; i++)
  {
    mod.assign(prim[i + 1]);
    ::divide(M, L, mod);
    best_remainder(TMP, M, mod);
    dummy = xgcd(TMP0, TMP1, mod, TMP);
    multiply(e[i], TMP1, M);
  }

  for (i = 0; i < r; i++)
    for (j = 0; j < c; j++)
    {
      X.assign_zero();

      for (l = 0; l < len; l++)
	m[l].assign(v[l].value[i][j]);

      for (l = 0; l < len; l++)
      {
	multiply(TMP, e[l], m[l]);
	add(X, X, TMP);
      }
      best_remainder(X, X, L);
      subtract(TMP0, L, bigint(1));
      shift_left(TMP1, X, 1);
      shift_left(TMP2, L, 1);
      if (!(TMP1 >= -TMP0 && TMP1 <= TMP0))
	if (TMP1 - TMP2 <= TMP0 || TMP1 - TMP2 >= -TMP0)
	  subtract(X, X, L);
	else
	  lidia_error_handler("bigint_matrix", "chinrest :: No Solution !");
      value[i][j].assign(X);
    }
  delete[] e;
  delete[] m;
}

bigint_matrix
chinrest(const bigint_matrix * v, const bigint * prim)
{
  debug_handler("bigint_matrix", "in function "
		"chinrest(const bigint_matrix *, const bigint *)");

  bigint_matrix A(v[0].rows, v[0].columns);
  A.chinrest(v, prim);
  return A;
}

/**
 ** List of primes
 **/

bigint *
get_primes(const bigint & C, const bigint & m)
{
  /**
   ** DESCRIPTION: get_primes(C,m) = v;
   **              => v = List of primes
   **              => !(m | v[i]) for all i=1,...,v[0]
   **              => v[1]*...*v[v[0]] > C
   ** VERSION: 1.82
   **/

  debug_handler("bigint_matrix", "in function "
		"get_primes(bigint, bigint)");

  bigint *List = NULL;	

  /* GET ENV Variable */
  static long ANZAHL=0;
  static char *LIDIA_PRIMES_NAME = NULL;

  if (ANZAHL==0)
   {
     char *LIDIA_PRIMES_SIZE = getenv("LIDIA_PRIMES_SIZE");
     if (LIDIA_PRIMES_SIZE==NULL)
       ANZAHL=LIDIA_PRIMELIST_SIZE;
     else
       {
	   ANZAHL = atoi(LIDIA_PRIMES_SIZE);
        if (LIDIA_PRIMES_SIZE != getenv("LIDIA_PRIMES_SIZE"))
           delete[] LIDIA_PRIMES_SIZE;
        }
   }
   
  if (LIDIA_PRIMES_NAME==NULL)
   {
      LIDIA_PRIMES_NAME = getenv("LIDIA_PRIMES_NAME");
      if (LIDIA_PRIMES_NAME==NULL)
	  {
	      LIDIA_PRIMES_NAME = new char[256];
	      strcpy(LIDIA_PRIMES_NAME,LIDIA_PRIMELIST_NAME);
	  }
   }

  long i = 0, j, PRObits = 0;
  long Hbits = C.bit_length();
  bigint *p = new bigint[ANZAHL];
  memory_handler(p, "bigint_matrix", "get_primes :: "
		 "Error in memory allocation (p)");

  ifstream in(LIDIA_PRIMES_NAME);
  if (in.fail())
     lidia_error_handler("bigint_matrix","get_primes :: "
		   "FILE LIDIA_PRIMES NOT FOUND !!");
  while (PRObits < Hbits && i < ANZAHL)
  {
    in >> p[i];
    if (m % p[i] != 0)
    {
      PRObits += p[i].bit_length();	// multiply(PRO, PRO, bigint(p[i]));
      i++;
    }
  }
  if (PRObits < Hbits)
  {
    delete[] p;
    
    if (ANZAHL== 86416)
      lidia_error_handler("bigint_matrix", "get_primes :: Primlist too small !!");
    ANZAHL*=2;
    if (ANZAHL > 86416)
 	ANZAHL=86416;
    cout << "bigint_matrix" << ":: get_primes :: " <<
	"Size of primelist increased ! " << endl << flush;
    cout << "new size: " << ANZAHL << endl << flush;
    List = get_primes(C,m);
   }
  else
  {
    List = new bigint[i + 1];
    memory_handler(List, "bigint_matrix", "get_primes :: "
		   "Error in memory allocation (List)");
    List[0].assign(i);
    for (j = 0; j < i; j++)
      List[j + 1].assign(p[j]);
    delete[] p;
  }
  return List;
}

/**
 ** BEGIN: Linear algebra
 ** PART 1
 **/

/** 
 ** trace
 **/

void bigint_matrix::
trace(bigint &tr) const
{
  /**
   ** DESCRIPTION: A.trace(tr)
   **              => tr = trace of matrix A
   ** ERROR: columns != rows
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"trace(bigint &)");

  if (rows != columns)
    lidia_error_handler("bigint_matrix", "trace :: non square matrix");
  register long i;
  tr.assign(value[0][0]);
  for (i=1;i<rows;i++)
    add(tr,tr,value[i][i]);
}

bigint bigint_matrix::
trace() const
{
  debug_handler("bigint_matrix", "in member - function trace()");

  bigint tr;
  trace(tr);
  return tr;
}

bigint
trace(const bigint_matrix &A)
{
  debug_handler("bigint_matrix", "in function trace(const bigint_matrix &)");

  bigint tr;
  A.trace(tr);
  return tr;
}

/**
 ** diagonal function
 **/

void bigint_matrix::
diag(const bigint & a, const bigint & b)
{
  /**
   ** DESCRIPTION: A.diag(a,b);
   **              => A.value[i][i] = a, i=0,...,min(columns,rows)
   **              => A.value[i][j] = b, i=0,...,rows and j=0,...,columns 
   **              and i != j
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"diag(const bigint &, const bigint &)");

  register long i, j;
  bigint *tmp;
  for (i = 0; i < rows; i++)
  {
    tmp = value[i];
    for (j = 0; j < columns; j++)
      if (i == j)
	tmp[j].assign(a);
      else
	tmp[j].assign(b);
  }
}

void
diag(bigint_matrix & A, const bigint & a, const bigint & b)
{
  debug_handler("bigint_matrix", "in function "
		"diag(bigint_matrix &, const bigint &, const bigint &)");

  A.diag(a, b);
}

/**
 ** regular expansion
 **/

void bigint_matrix::
regexpansion(const long *v)
{
  /**
   ** DESCRIPTION: A.regexpansion(v);
   **              => A = Regular Expansion of the old matrix A relative v.
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"regexpansion(const long *)");

  register long k = v[0];
  if (columns > k)
    {
      register long i=0, j=0;
      long diff = columns - rows;
      bigint_matrix A(*this);
      set_no_of_rows(columns);
      set_no_of_columns(columns);
      bigint_matrix ERG(diff, columns);
      while (i < columns && k > 0 && j < diff)
	{
	  if (i != v[k])
	    {
	      ERG.value[j][i].assign_one();
	      j++;
	    }
	  else
	    k--;
	  i++;
	}
      compose_v(ERG, A);
    }
}

void
regexpansion(bigint_matrix & A, const long *v)
{
  debug_handler("bigint_matrix", "in function "
		"regexpansion(const bigint_matrix &, const long *)");

  A.regexpansion(v);
}

/**
 ** rank
 **/

long bigint_matrix::
rank() const
{
  /**
   ** DESCRIPTION: A.rank() = Rank of matrix A.
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function rank");
  register long i;

  /* Step 1 */
  bigint H;
  hadamard(H);

  /* Step 2 */
  bigint *PRIM = get_primes(2 * H, 1);
  long Number_of_primes;
  PRIM[0].longify(Number_of_primes);

  long RANG = 0;

  /* Step 3,4 */
  long v;
  for (i = 1; i <= Number_of_primes; i++)
    {
      v = rank_intern(value, rows, columns, PRIM[i]);
      if (RANG < v)
	RANG = v;
    }
  delete[] PRIM;
  
  /* Step 5,6 */
  return RANG;
}

long
rank(const bigint_matrix & B)
{
  debug_handler("bigint_matrix", "in function "
		"rank(const bigint_matrix &)");
  
  return B.rank();
}

/**
 ** rank and linearly independent rows
 **/

long *bigint_matrix::
lininr() const
{
  /**
   ** DESCRIPTION: RES[0] = Rank of matrix (Avalue,r,c).
   **              RES[1],...,RES[RES[0]], such that
   **              row(RES[1]),...,row(RES[RES[0]])
   **              of matrix *this are linearly independent.
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"lininr()");
  register long i, j;
  long *tmp;
  
  /* Step 1 */
  bigint H;
  hadamard(H);

  /* Step 2 */
  bigint *PRIM = get_primes(2 * H, 1);
  long Number_of_primes;
  PRIM[0].longify(Number_of_primes);
  long RANG = 0;
  
  /* Step 3,4 */
  long **A = new long *[Number_of_primes];
  memory_handler(A, "bigint_matrix", "lininr :: "
		 "Error in memory allocation (A)");
  for (i = 0; i < Number_of_primes; i++)
    {
      tmp = new long[rows + 1];
      memory_handler(tmp, "bigint_matrix", "lininr :: "
		     "Error in memory allocation (tmp)");
      for (j = 0; j <= rows; j++)
	tmp[j] = 0;
      A[i] = tmp;
    }
  long *v = NULL;
  
  for (i = 1; i <= Number_of_primes; i++)
    {
      v = lininr_intern(value, rows, columns, PRIM[i]);
      if (RANG < v[0])
	RANG = v[0];
      for (j = 0; j < v[0]; j++)
	A[i - 1][j] = v[j + 1];
      delete[] v;
    }
  delete[] PRIM;
  
  /* Step 5,6 */
  long *RES = new long[RANG + 1];
  memory_handler(RES, "bigint_matrix", "lininr :: "
		 "Error in memory allocation (RES)");
  RES[0] = RANG;
  long TMP1;
  for (i = 0; i < RANG; i++)
    {
      TMP1 = 0;
      for (j = 0; j < Number_of_primes; j++)
	if (TMP1 < A[j][i])
	  TMP1 = A[j][i];
      RES[i + 1] = TMP1;
    }
  for (i = 0; i < Number_of_primes; i++)
    delete[] A[i];
  delete[] A;
  return RES;
}

long *
lininr(const bigint_matrix & B)
{
  debug_handler("bigint_matrix", "in function "
		"lininr(const bigint_matrix &)");
  return B.lininr();
}

/**
 ** rank linearly independent columns
 **/

long *bigint_matrix::
lininc() const
{
  /**
   ** DESCRIPTION: RES[0] = Rank of matrix (Avalue,r,c).
   **              RES[1],...,RES[RES[0]], such that
   **              column(RES[1]),...,column(RES[RES[0]])
   **              of matrix *this are linearly independent.
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"lininc()");
  register long i, j;
  long *tmp;

  /* Step 1 */
  bigint H;
  hadamard(H);

  /* Step 2 */
  bigint *PRIM = get_primes(2 * H, 1);
  long Number_of_primes;
  PRIM[0].longify(Number_of_primes);

  long RANG = 0;

  /* Step 3,4 */
  long **A = new long *[Number_of_primes];
  memory_handler(A, "bigint_matrix", "lininc :: "
		 "Error in memory allocation (A)");
  for (i = 0; i < Number_of_primes; i++)
  {
    tmp = new long[columns + 1];
    memory_handler(tmp, "bigint_matrix", "lininc :: "
		   "Error in memory allocation (tmp)");
    for (j = 0; j <= columns; j++)
      tmp[j] = 0;
    A[i] = tmp;
  }
  long *v = NULL;
  for (i = 1; i <= Number_of_primes; i++)
  {
    v = lininc_intern(value, rows, columns, PRIM[i]);
    if (RANG < v[0])
      RANG = v[0];
    for (j = 0; j < v[0]; j++)
      A[i - 1][j] = v[j + 1];
  }
  delete[] v;
  delete[] PRIM;

  /* Step 5,6 */
  long *RES = new long[RANG + 1];
  memory_handler(RES, "bigint_matrix", "lininc :: "
		 "Error in memory allocation (RES)");
  RES[0] = RANG;
  long TMP1;
  for (i = 0; i < RANG; i++)
  {
    TMP1 = 0;
    for (j = 0; j < Number_of_primes; j++)
      if (TMP1 < A[j][i])
	TMP1 = A[j][i];
    RES[i + 1] = TMP1;
  }
  for (i = 0; i < Number_of_primes; i++)
    delete[] A[i];
  delete[] A;
  return RES;
}

long *
lininc(const bigint_matrix & B)
{
  debug_handler("bigint_matrix", "in function "
		"lininc(const bigint_matrix &)");
  return B.lininc();
}

/**
 ** adjoint matrix
 **/

void bigint_matrix::
adj(const bigint_matrix & A)
{
  /**
   ** DESCRIPTION: B.adj(A);
   **              => B = adjoint matrix of matrix A
   ** ERROR: A.columns != A.rows
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in function "
		"adj(const bigint_matrix &)");
  register long i, j, z1, z2;
  if (A.columns != A.rows)
    lidia_error_handler("bigint_matrix", "adj :: non square matrix");
  bigint **Bbigint = new bigint *[A.rows];
  long **Blong = new long *[A.rows];
  memory_handler(Bbigint, "bigint_matrix", "adj :: "
		 "Error in memory allocation (Bbigint)");
  memory_handler(Blong, "bigint_matrix", "adj :: "
		 "Error in memory allocation (Blong)");
  bigint *tmp, *Atmp;
  long *tmp1;
  for (i = 0; i < A.rows; i++)
  {
    tmp = new bigint[A.rows];
    tmp1 = new long[A.rows];
    memory_handler(tmp, "bigint_matrix", "adj :: "
		   "Error in memory allocation (tmp)");
    memory_handler(tmp1, "bigint_matrix", "adj :: "
		   "Error in memory allocation (tmp1)");
    for (j = 0; j < A.rows; j++)
    {
      tmp[j].assign_zero();
      tmp1[j] = 0;
    }

    Bbigint[i] = tmp;
    Blong[i] = tmp1;
  }

  /* Step 1 */
  bigint H = A.hadamard();

  /* Step 2,3 */
  bigint *PRIM = get_primes((bigint)2*H, A.det());
  long n;
  PRIM[0].longify(n);

  /* Step 4 */
  bigint MOD;
  long Modlong;
  bigint_matrix *chininput = new bigint_matrix[n];
  memory_handler(chininput, "bigint_matrix", "adj :: "
		 "Error in memory allocation (chininput)");
  for (i = 1; i <= n; i++)
    {
      MOD.assign(PRIM[i]);
      if (MOD.bit_length() > bigint::bits_per_digit())
	{
	  for (z1 = 0; z1 < A.rows; z1++)
	    {
	      tmp = Bbigint[z1];
	      Atmp = A.value[z1];
	      for (z2 = 0; z2 < A.columns; z2++)
		best_remainder(tmp[z2], Atmp[z2], MOD);
	    }
	  
	  adj_intern(Bbigint, A.rows, MOD);
	  
	  chininput[i - 1].set_no_of_columns(A.columns);
	  chininput[i - 1].set_no_of_rows(A.rows);
	  
	  for (z1 = 0; z1 < A.rows; z1++)
	    {
	      tmp = Bbigint[z1];
	      Atmp = chininput[i-1].value[z1];
	      for (z2 = 0; z2 < A.columns; z2++)
		Atmp[z2].assign(tmp[z2]);
	    }
	}
      else
	{
	  MOD.longify(Modlong);
	  for (z1 = 0; z1 < A.rows; z1++)
	    {
	      tmp1 = Blong[z1];
	      Atmp = A.value[z1];
	      for (z2 = 0; z2 < A.columns; z2++)
		best_remainder_long(tmp1[z2], Atmp[z2], Modlong);
	    }

	  adj_intern(Blong, A.rows, Modlong);
	  
	  chininput[i - 1].set_no_of_columns(A.columns);
	  chininput[i - 1].set_no_of_rows(A.rows);
	  
	  for (z1 = 0; z1 < A.rows; z1++)
	    {
	      tmp1 = Blong[z1];
	      Atmp = chininput[i-1].value[z1];
	      for (z2 = 0; z2 < A.columns; z2++)
		Atmp[z2].assign(tmp1[z2]);
	    }
	}
    }

  /* Step 5 */
  chinrest(chininput, PRIM);
  for (i = 0; i < A.rows; i++)
    {
      delete[] Bbigint[i];
      delete[] Blong[i];
    }
  delete[] PRIM;
  delete[] Bbigint;
  delete[] Blong;
  delete[] chininput;
}

bigint_matrix
adj(const bigint_matrix & A)
{
  debug_handler("bigint_matrix", "in function "
		"adj(const bigint_matrix &)");
  bigint_matrix B(A.rows, A.columns);
  B.adj(A);
  return B;
}

/**
 ** lattice determinant
 **/

void bigint_matrix::
latticedet(bigint & DET) const
{
  /**
   ** INPUT: *this,DET
   ** DESCRIPTION: A.latticedet(DET);
   **              => DET = lattice determinant
   **              of the lattice formed by the columns of matrix A
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"latticedet(bigint &)");

  register long i, j;
  bigint *tmp, *tmp1;

  /* Step 1 */
  bigint_matrix B(*this);
  long *linuz = B.lininr();
  long r = linuz[0];

  /* Step 2 */
  bigint_matrix C(r, columns);
  for (i = 0; i < r; i++)
    {
      tmp = C.value[i];
      tmp1 = value[linuz[i+1]];
      for (j = 0; j < columns; j++)
	tmp[j].assign(tmp1[j]);
    }
  
  /* Step 3 */
  if (r == columns)
    C.det(DET);
  else
    {
      bigint_matrix H1 = C.trans();
      
      long *linuz1 = H1.lininr();
      
      bigint_matrix D(r, r);
      for (i = 0; i < r; i++)
	for (j = 0; j < r; j++)
	  D.value[j][i].assign(C.value[j][linuz1[i + 1]]);
      D.det(DET);
      delete[] linuz1;
    }
  delete[] linuz;
  if (DET.is_lt_zero())
    DET.negate();
}

void bigint_matrix::
latticedet3(bigint & DET) const
{
  /**
   ** INPUT: *this,DET
   ** DESCRIPTION: A.latticedet(DET);
   **              => DET = lattice determinant
   **              of the lattice formed by the columns of matrix A
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"latticedet(bigint &)");

  register long i, j;
  bigint *tmp, *tmp1;

  /* Step 1 */
  long *linuz = lininr();
  long r = linuz[0];

  /* Step 2 */
  if (r == columns)
    {
      bigint_matrix C(r, columns);
      for (i = 0; i < r; i++)
	{
	  tmp = C.value[i];
	  tmp1 = value[linuz[i+1]];
	  for (j = 0; j < columns; j++)
	    tmp[j].assign(tmp1[j]);
	}
      C.det(DET);
    }
  else
    { 
      long *linuz1 = lininc();
      
      bigint_matrix D(r, r);
      for (i = 0; i < r; i++)
	for (j = 0; j < r; j++)
	  D.value[j][i].assign(value[linuz[j+1]][linuz1[i + 1]]);
      D.det(DET);
      delete[] linuz1;
    }
  delete[] linuz;
  if (DET.is_lt_zero())
    DET.negate();
}

void bigint_matrix::
latticedet2(bigint & DET) const
{
  /**
   ** INPUT: *this,DET
   ** DESCRIPTION: A.latticedet2(DET);
   **              => DET = lattice determinant
   **              of the lattice formed by the columns of matrix A
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"latticedet2(bigint &)");

  register long i, j;
  bigint TMP, TMP1, *tmp, *tmp1, *tmp2;

  /* Step 1 */
  long *linu = lininr();
  long r = linu[0];

  /* Step 2 */
  bigint_matrix C(r, columns);
  bigint_matrix C1(r, columns);

  for (i = 0; i < r; i++)
    {
      tmp = C.value[i];
      tmp1 = value[linu[i+1]];
      tmp2 = C1. value[i];
      for (j = 0; j < columns; j++)
	{
	  tmp[j].assign(tmp1[j]);
	  tmp2[columns - 1 - j].assign(tmp1[j]);
	}
    }
  delete[] linu;
  
  /* Step 3 */
  if (r == columns)
    C.det(DET);
  else
    {
      linu = C.lininc();
      bigint_matrix D(r, r);
      for (i = 0; i < r; i++)
	for (j = 0; j < r; j++)
	  D.value[j][i].assign(C.value[j][linu[i + 1]]);
      D.det(TMP);
      delete[] linu;
      
      linu = C1.lininc();
      for (i = 0; i < r; i++)
	for (j = 0; j < r; j++)
	  D.value[j][i].assign(C1.value[j][linu[i + 1]]);
      D.det(TMP1);
      delete[] linu;
      DET = gcd(TMP, TMP1);
    }
  if (DET.is_lt_zero())
    DET.negate();
}

void bigint_matrix::
latticedet4(bigint & DET) const
{
  /**
   ** INPUT: *this,DET
   ** DESCRIPTION: A.latticedet2(DET);
   **              => DET = lattice determinant
   **              of the lattice formed by the columns of matrix A
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"latticedet2(bigint &)");

  register long i, j;
  bigint TMP, TMP1, *tmp, *tmp1, *tmp2;

  /* Step 1 */
  long *linu = lininr();
  long r = linu[0];

  /* Step 2 */
  bigint_matrix C(r, columns);
  bigint_matrix C1(r, columns);

  for (i = 0; i < r; i++)
    {
      tmp = C.value[i];
      tmp1 = value[linu[i+1]];
      tmp2 = C1. value[i];
      for (j = 0; j < columns; j++)
	{
	  tmp[j].assign(tmp1[j]);
	  tmp2[columns - 1 - j].assign(tmp1[j]);
	}
    }
  delete[] linu;
  
  /* Step 3 */
  if (r == columns)
    C.det(DET);
  else
    {
      linu = C.lininc();
      bigint_matrix D(r, r);
      for (i = 0; i < r; i++)
	for (j = 0; j < r; j++)
	  D.value[j][i].assign(C.value[j][linu[i + 1]]);
      D.det(TMP);
      delete[] linu;
      
      linu = C1.lininc();
      for (i = 0; i < r; i++)
	for (j = 0; j < r; j++)
	  D.value[j][i].assign(C1.value[j][linu[i + 1]]);
      D.det(TMP1);
      delete[] linu;
      DET = gcd(TMP, TMP1);
    }
  if (DET.is_lt_zero())
    DET.negate();
}

bigint bigint_matrix::
latticedet() const
{
  debug_handler("bigint_matrix", "in member - function "
		"latticedet()");

  bigint DET;
  latticedet(DET);
  return DET;
}

bigint bigint_matrix::
latticedet2() const
{
  debug_handler("bigint_matrix", "in member - function "
		"latticedet2()");

  bigint DET;
  latticedet2(DET);
  return DET;
}

bigint
latticedet(const bigint_matrix & A)
{
  debug_handler("bigint_matrix", "in function "
		"latticedet(bigint_matrix &)");

  return A.latticedet();
}

bigint
latticedet2(const bigint_matrix & A)
{
  debug_handler("bigint_matrix", "in function "
		"latticedet2(bigint_matrix &)");

  return A.latticedet2();
}

/**
 ** determinant
 **/

void bigint_matrix::
det(bigint & DET) const
{
  /**
   ** DESCRIPTION: A.det(DET)
   **              => DET = determinant of matrix A
   ** ERROR: columns != rows
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"det(bigint &)");

  if (rows != columns)
    lidia_error_handler("bigint_matrix", "det :: non square matrix");
  register long i, j, z1, z2;
  bigint **Bbigint;
  long **Blong;

  Bbigint = new bigint *[rows];
  Blong = new long *[rows];
  memory_handler(Bbigint, "bigint_matrix", "det :: "
		 "Error in memory allocation (Bbigint)");
  memory_handler(Blong, "bigint_matrix", "det :: "
		 "Error in memory allocation (Blong)");

  const bigint *consttmp;
  bigint *tmp;
  long *tmp1;
  for (i = 0; i < rows; i++)
  {
    Bbigint[i] = new bigint[columns];
    Blong[i] = new long[columns];
    memory_handler(Bbigint[i], "bigint_matrix", "det :: "
		   "Error in memory allocation (Bbigint[i])");
    memory_handler(Blong[i], "bigint_matrix", "det :: "
		   "Error in memory allocation (Blong[i])");
  }

  /* Step 1 */
  bigint H = hadamard();
  shift_left(H, H, 1);

  /* Step 2 */
  bigint *PRIM = get_primes(H, bigint(1));
  long n;
  PRIM[0].longify(n);

  bigint MOD;
  long Modlong, Detlong;

  /* Step 3 */
  bigint *chininput = new bigint[n];
  memory_handler(chininput, "bigint_matrix", "det :: "
		 "Error in memory allocation (chininput)");
  for (i = 1; i <= n; i++)
    {
      MOD.assign(PRIM[i]);
      if (MOD.bit_length() >= bigint::bits_per_digit())
	{
	  for (z1 = 0; z1 < rows; z1++)
	    {
	      consttmp = value[z1];
	      tmp = Bbigint[z1];
	      for (z2 = 0; z2 < columns; z2++)
		best_remainder(tmp[z2], consttmp[z2], MOD);
	    }
	  det_intern(chininput[i - 1], Bbigint, rows, MOD);
	}
      else
	{
	  MOD.longify(Modlong);
	  for (z1 = 0; z1 < rows; z1++)
	    {
	      consttmp = value[z1];
	      tmp1 = Blong[z1];
	      for (z2 = 0; z2 < columns; z2++)
		best_remainder_long(tmp1[z2], consttmp[z2], Modlong);
	    }
	  Detlong = det_intern(Blong, rows, Modlong);
	  chininput[i - 1].assign(Detlong);
	}
    }
  for (j = 0; j < rows; j++)
    {
      delete[] Bbigint[j];
      delete[] Blong[j];
    }
  delete[] Bbigint;
  delete[] Blong;
  
  /* Step 4 */
  ::chinrest(DET, chininput, PRIM);
  delete[] chininput;
  delete[] PRIM;
}

bigint bigint_matrix::
det() const
{
  debug_handler("bigint_matrix", "in member - function det()");

  bigint DET;
  det(DET);
  return DET;
}

bigint
det(const bigint_matrix & A)
{
  debug_handler("bigint_matrix", "in function "
		"det(bigint_matrix &)");

  bigint DET;
  A.det(DET);
  return DET;
}

/**
 ** characteristic polynomial
 **/

bigint *bigint_matrix::
charpoly() const
{
  /**
   ** DESCRIPTION: RES = A.charpoly();
   **              => RES[0],...,RES[r] are the coefficients of
   **                 the characteristic polynomial of matrix A
   ** ERROR: rows != columns
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function charpoly()");

  if (columns != rows)
    lidia_error_handler("bigint_matrix", "charpoly :: non square matrix"); 
  
  register long i, j, z1, z2;
  bigint TMP;
  bigint **Bbigint, *tmp, *tmp2;
  long len, **Blong, *tmp1;

  bigint *RES = new bigint[columns + 1];
  memory_handler(RES, "bigint_matrix", "charpoly :: "
		 "Error in memory allocation (RES)");

  /* Step 1 */
  power(TMP, (bigint) columns * max(), (bigint) columns);

  /* Step 2 */
  shift_left(TMP, TMP, 1);
  bigint *PRIM = get_primes(TMP, 1);
  PRIM[0].longify(len);
  
  bigint *zwbigint;
  long *zwlong;
  
  /* Step 3 */
  bigint_matrix U(columns + 1, len);
  Bbigint = new bigint *[rows];
  Blong = new long *[rows];
  memory_handler(Bbigint, "bigint_matrix", "charpoly :: "
		 "Error in memory allocation (Bbigint)");
  memory_handler(Blong, "bigint_matrix", "charpoly :: "
		 "Error in memory allocation (Blong)");  
  for (i = 0; i < rows; i++)
    {
      tmp = new bigint[columns];
      tmp1 = new long[columns];
      memory_handler(tmp, "bigint_matrix", "charpoly :: "
		     "Error in memory allocation (tmp)");
      memory_handler(tmp1, "bigint_matrix", "charpoly :: "
		     "Error in memory allocation (tmp1)");
      for (j = 0; j < columns; j++)
	{
	  tmp[j].assign_zero();
	  tmp1[j] = 0;
	}
      
      Bbigint[i] = tmp;
      Blong[i] = tmp1;
    }
  
  bigint MOD;
  long Modlong;
  
  /* Step 3 */
  for (i = 1; i <= len; i++)
    {
      MOD.assign(PRIM[i]);
      if (MOD.bit_length() > bigint::bits_per_digit())
	{
	  for (z1 = 0; z1 < rows; z1++)
	    {
	      tmp = Bbigint[z1];
	      tmp2 = value[z1];
	      for (z2 = 0; z2 < columns; z2++)
		best_remainder(tmp[z2], tmp2[z2], MOD);
	    }
	  
	  zwbigint = charpoly_intern(Bbigint, rows, MOD);
	  for (j = 0; j < columns + 1; j++)
	    U.value[j][i - 1].assign(zwbigint[j]);
	  delete[] zwbigint;
	}
      else
	{
	  MOD.longify(Modlong);
	  for (z1 = 0; z1 < rows; z1++)
	    {
	      tmp1 = Blong[z1];
	      tmp2 = value[z1];
	      for (z2 = 0; z2 < columns; z2++)
		best_remainder_long(tmp1[z2], tmp2[z2], Modlong);
	    }
	  
	  zwlong = charpoly_intern(Blong, rows, Modlong);
	  for (j = 0; j < columns + 1; j++)
	    U.value[j][i - 1].assign(zwlong[j]);
	  delete[] zwlong;
	}
    }
  for (j = 0; j < rows; j++)
    {
      delete[] Bbigint[j];
      delete[] Blong[j];
    }
  delete[] Bbigint;
  delete[] Blong;
  
  /* Step 4,5 */
  for (i = 0; i < columns + 1; i++)
    {
      ::chinrest(RES[i], tmp = U.row(i), PRIM);
      delete[] tmp;
    }

  delete[] PRIM;
  return RES;
}

bigint *
charpoly(const bigint_matrix & A)
{
  debug_handler("bigint_matrix", "in function "
		"charpoly(bigint_matrix &)");

  return A.charpoly();
}

/**
 ** END: Linear algebra
 **/
















