//
// LiDIA - a library for computational number theory
//   Copyright (c) 1994, 1995 by the LiDIA Group
//
// File        : dlp.c 
// Author      : Damian Weber (DW)
// Last change : TP, Aug 8 1995, initial version
//

#include <LiDIA/dlp.h>
#include <LiDIA/timer.h>
#include <stdlib.h>
#include <strstream.h>

inline bigint chinese_remainder(const bigint &x1,const bigint &p1,
				const bigint &x2,const bigint &p2)
{
  bigint m, x0, u, v, up1x2, vp2x1, sum;

  multiply(m, p1, p2);
  xgcd(u, v, p1, p2);
  multiply(up1x2, p1, u);
  multiply(up1x2, up1x2, x2);
  multiply(vp2x1, p2, v);
  multiply(vp2x1, vp2x1, x1);
  add(sum, up1x2, vp2x1);
  remainder(x0, sum, m);
  if (x0.is_lt_zero())
    add(x0, x0, m);
  return x0;
}

bigint search_generator(const bigint &p, 
			const bigint &g0,
			rational_factorization &f)
// given the factorization of p-1 this function determines
// the least generator of Fp greater or equal g0
{
  bigint y1,g,q;
  int np=f.no_of_comp();
  int j,no_gen;

  g=g0;
  while (1)
    {
      no_gen=0;
      for (j=0;j<=np-1;j++)
	{
	  q=f.base(j);
	  power_mod(y1,g,(p-(bigint)1)/q,p);
	  if (y1.is_one()) no_gen++;
	}
      if (!no_gen) break;
      g+=1;
    }
  return g;
}

inline void recursion(bigint &x,bigint &a,bigint &b,
		 const bigint &p, const bigint &q,
		 const bigint &a0, const bigint &g0)

  {
    static unsigned long rem;
 
    rem = x.least_significant_digit() & 7;

    if (rem<=2)		// set S1
	{
          multiply(x,x,a0);
	  remainder(x,x,p);
          inc(a);
	  return;
	}
    
    if (rem<=5)		// set S2
	{
          multiply(x,x,g0);
	  remainder(x,x,p);
          inc(b);
	  return;
	}

    square(x,x);	// set S3
    remainder(x,x,p);
    a.multiply_by_2();
    b.multiply_by_2();
    if (a.compare(q)>=0)
       subtract(a,a,q);
    if (b.compare(q)>=0)
       subtract(b,b,q);

  }


bigint pohlig_hellman_shanks(const bigint &g,
			       const bigint &a,
			       const bigint &p,
			       const bigint &q)
  {
    static bigint m,g0,a0,l;
    static bigint ai,bi,xi,a2i,b2i,x2i;
    static bigint u,v,s,t;
    static bigint d,k,gd,gk,i;

    m=(p-(bigint)1)/q;
    power_mod(a0,a,m,p);
    power_mod(g0,g,m,p);
    
    xi=1;
    ai=0;
    bi=0;
    recursion(xi,ai,bi,p,q,a0,g0);
    
    x2i=xi;a2i=ai,b2i=bi;
    //    cout << " " << x2i;
    
    recursion(x2i,a2i,b2i,p,q,a0,g0);
    
    //    while (x2i.compare(xi)!=0)
    
    while (x2i!=xi)
      {
	recursion(x2i,a2i,b2i,p,q,a0,g0);
	//          cout << " " << x2i;
	recursion(x2i,a2i,b2i,p,q,a0,g0);
	//          cout << " " << x2i;
	//	  j+=2;
	//	  if (j%10==0) cout << "\n";
	recursion(xi,ai,bi,p,q,a0,g0);
      }
    
    //    cout << "\n";
    
    s=(ai-a2i) % q;
    if (s.is_lt_zero()) s+=q;
    
    t=(b2i-bi) % q;
    if (t.is_lt_zero()) t+=q;
    
    //    d=xgcd(u,v,s,q);     
    
    d=xgcd_left(u,s,q);     
    if (u.is_lt_zero()) u+=q;
    
    if (d.is_one())
      l=(u*t) % q;
    else
      {
	k=((u*t) % q)/d;
	if (k.is_le_zero()) k=k+q;
	power_mod(gd,g0,q/d,p);
	power_mod(gk,g0,k,p);
	
	i=0;
	
	while (gk!=a0)
	  {
	    i+=1;
	    gk=(gk*gd) % p;
	    
	  }
	l=(k+i*q/d) % q;
      }
    
    return l;
  }

bigint discrete_log
(const bigint &a,
 const bigint &b,
 const bigint &p,
 const rational_factorization &f)
{
  int j;
  int np=f.no_of_comp();
  bigint q,tq,xc,mc;
  
  timer ti;
  long  td;
  
  cout << "intermediate results" << "\n";
  
  for (j=0;j<=np-1;j++)
    {
      power(q,f.base(j),f.exponent(j));
      ti.start_timer();
      
      tq=pohlig_hellman_shanks(a,b,p,q);
      ti.stop_timer();
      td=ti.real_time()/100;
      //
      // ------------------------------------------
      //  output mod q^j
      // ------------------------------------------
      //
      cout << tq << " mod " << f.base(j);
      if (f.exponent(j)>1)
	cout << "^" << f.exponent(j);
      cout << "    (" << td/60 << ":" << td%60 << ")" << "\n";
      // ------------------------------------------
      
      if (j) 
	{
	  xc=chinese_remainder(tq,q,xc,mc);
	  mc*=q;
	}
      else 
	{
	  xc=tq;
	  mc=q;
	}
    }
  
  return xc;
}
