
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : base_vector.h 
// Author      : Frank Lehmann (FL), Markus Maurer (MM) 
//               Thomas Papanikolaou (TP)
// Last change : FL/MM, Feb 15 1995, initial version
//               FL/MM, Mar  8 1995
//               FL/MM, May 10 1995, added get_data_address 
//                                   clean-up
//		 FL/MM, May 15 1995, added typedef size_type and
//				     changed type of corresponding variables
//               FL/MM, May 23 1995, removed #define LiDIA_VECTOR_NO_RANGE_CHECK
//		 FL/MM, Jul  6 1995, changed internal size_type of base_vector
//				     to the global LiDIA-type lidia_size_t
//               FL/MM, Jul 24 1995, added 'const' to T* parameters of constructors


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// *
// *    File        :  base_vector.h
// *
// *    Description :  -) definition of template class 'base_vector< T >'
// *
// *    Notes       :  *) any class T  MUST support the operators
// *                                  '<<'  and  '>>' 
// *                      to allow its use in the template
// *
// *                   *) moreover, we recommend to explicitly define an assignment-
// *                      operator ('=') for any class T used in the template,
// *                      to avoid difficulties during compilation
// *
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *



#ifndef LIDIA_BASE_VECTOR_H
#define LIDIA_BASE_VECTOR_H

#include <LiDIA/lidia.h>

typedef int lidia_size_t ;



#ifndef LiDIA_INLINE

#ifdef __GNUG__
#define LiDIA_INLINE inline
#else
#define LiDIA_INLINE
#endif

#endif 



#ifndef LiDIA_VEC_VIRTUAL

#ifdef __GNUG__
#define LiDIA_VEC_VIRTUAL virtual
#else
#define LiDIA_VEC_VIRTUAL
#endif

#endif



#define  EXPAND  'E'
#define  FIXED   'F'
#define  DEFAULT_RATIO 2.0




template <class T> class base_vector

 {
   protected :

   T *data ;

   lidia_size_t length    ;
   lidia_size_t allocated ;

   char  mode           ;
   float dyn_exp_ratio  ;


   // The following variables are only used by sort_vector.
   // They were defined as members of base_vector to avoid
   // trouble with virtual functions, i.e. the swap function
   // of base_vector exchanges the values of these variables too.

   protected :

   char      sort_dir   ;                         // sort-direction
   int (*el_cmp) ( const T & a, const T & b ) ;   // compare-func. for vec.elements



   /* * * * * * * * *   class description  * * * * * * * * 

   protected :

      T * copy_data ( T * d, const T * vd, int l ) ;

   public :

   //   *****  constructor functions  *****

   base_vector (          ) ;
   base_vector ( char md  ) ;                              //   creates empty vector

   base_vector ( lidia_size_t all          ) ;
   base_vector ( lidia_size_t all, char md ) ;                //   creates vector of cap. "all"

   base_vector ( lidia_size_t all, lidia_size_t len          ) ;
   base_vector ( lidia_size_t all, lidia_size_t len, char md ) ; //   creates vector of cap. "all" and sz. len

   base_vector ( const base_vector < T > & v          ) ;
   base_vector ( const base_vector < T > & v, char md ) ;  //   initializes with vector v

   base_vector ( const T *v, lidia_size_t len          ) ;
   base_vector ( const T *v, lidia_size_t len, char md ) ;          //   initializes with first len elts. of T



   //   *****  destructor function  *****

   LiDIA_VEC_VIRTUAL ~base_vector ( ) ;


   //   *****  i/o - functions  *****

   void read  ( istream & in  )        ;
   void write ( ostream & out )  const ;

   friend istream & operator>> ( istream & s,       base_vector < T > &  x ) ;
   friend ostream & operator<< ( ostream & s, const base_vector < T > &  x ) ;



   //   ***** functions for reading and modifying the vector's variables *****

   lidia_size_t  capacity ( )  const      ;   //   return the vector's capacity
   void set_capacity ( lidia_size_t all ) ;   //   set the vector's capacity to all

   void kill () ;                          //   deallocate the elements of the vector

   lidia_size_t size ( )  const      ;        //   return the vector's size
   int set_size ( lidia_size_t len ) ;        //   set the vector's size to len

   float exp_ratio ( )  const          ;   //   return the vector's expansion ratio
   int   set_exp_ratio ( float ratio ) ;   //   set the vector's expansion ratio to ratio
   int   set_mode ( char md )          ;   //   set the mode of the vector to md

   void set_data ( T * d, lidia_size_t l ) ;  //   set vector = T
   T *  get_data ()  const        ;        //   returns a pointer to a copy of the elements of the vector


#ifdef LiDIA_VECTOR_INTERNAL_DATA_ACCESS
   T *  get_data_address () const ;        //   returns a pointer to the vector's internal elts.
#endif


   //   ***** access operator

         T & operator[] ( lidia_size_t i )       ;  //   returns the (i+1)-st element of the vector
   const T & operator[] ( lidia_size_t i ) const ;  //   returns the (i+1)-st element of the vector
                                                 //   only if i is less than size()
   T & member ( lidia_size_t i ) const ;            //   returns the (i+1)-st element of a vector
                                                 //   only if i is less than size()

   //    ***** assignment *****

   base_vector < T > & operator= ( const base_vector < T > &  v ) ;  //  set vector = v


   //   ***** miscellaneous *****

   int reverse ( ) ;                            //   reverse the order of the vector's elements
   int reverse ( const base_vector< T > & b ) ;

   friend void swap ( base_vector< T > & a, base_vector< T > & b ) ; // exchange information of two vectors

   int concat ( const base_vector< T > & b, const base_vector< T > & c ) ;

   int shift_left  ( lidia_size_t pos, lidia_size_t num ) ;
   int shift_right ( lidia_size_t pos, lidia_size_t num ) ;

   int assign ( lidia_size_t at, const base_vector< T > & v, lidia_size_t from, lidia_size_t to ) ;


   * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */



  // *****  implementation  *****
 
  protected :

  T * copy_data ( T * d , const T * vd , lidia_size_t l ) 
   {
      debug_handler ( "base_vector", "copy_data (T*,T*,lidia_size_t)" ) ;

      lidia_size_t i ;

      for ( i = 0 ; i < l ; i++ )
       {
   	  d[i] = vd[i] ;
       }   

      return d ;
   }	


   public :

   // *****  constructor functions  *****

    base_vector ()        // create empty vector
    {
       debug_handler ( "base_vector", "constructor ( void )" ) ;

       length    =  0    ;
       allocated =  0    ;
       mode      = FIXED ;

       dyn_exp_ratio = DEFAULT_RATIO ;

       data = nil ;
    }

    base_vector ( char md )        // create empty vector
    {
       debug_handler ( "base_vector", "constructor (  char )" ) ;

       length    =  0 ;
       allocated =  0 ;
       mode      = md ;

       dyn_exp_ratio = DEFAULT_RATIO ;

       data = nil ;
    }


    base_vector ( lidia_size_t all )  // create vector of cap. "all"
    {
       debug_handler ( "base_vector", "constructor ( lidia_size_t )" ) ;

       length    =   0    ;
       allocated = all    ;
       mode      =  FIXED ;

       dyn_exp_ratio = DEFAULT_RATIO ;

       data = nil ;

       if ( all > 0 )
        {
            data = new T [ all ]  ;

            if ( data == nil )
              lidia_error_handler ( "base_vector", "constructor ( lidia_size_t )::out of memory" ) ;
        }
       else if ( all < 0 )
        {
            lidia_error_handler ( "base_vector", "constructor(  lidia_size_t )::invalid size" ) ;
        }
    }

    base_vector (  lidia_size_t all ,  char md )  // create vector of cap. "all"
    {
       debug_handler ( "base_vector", "constructor ( lidia_size_t,  char)" ) ;

       length    =   0 ;
       allocated = all ;
       mode      =  md ;

       dyn_exp_ratio = DEFAULT_RATIO ;

       data = nil ;

       if ( all > 0 )
        {
            data = new T [ all ]  ;

            if ( data == nil )
              lidia_error_handler ( "base_vector", "constructor ( lidia_size_t,  char )::out of memory" ) ;
        }
       else if ( all < 0 )
        {
            lidia_error_handler ( "base_vector", "constructor ( lidia_size_t,  char)::invalid size" ) ;
        }
    }


    base_vector (  lidia_size_t all,  lidia_size_t len ) // create vector of cap. "all" and sz. "len"
    {
       debug_handler ( "base_vector", "constructor (lidia_size_t, lidia_size_t)" ) ;

       length    = ( len < all ) ? len : all  ;
       allocated = all   ;
       mode      = FIXED ;

       dyn_exp_ratio = DEFAULT_RATIO ;

       data = nil ;

       if ( all > 0 )
        {
            data = new T [ all ]  ;

            if ( data == nil )
              lidia_error_handler ( "base_vector", "constructor (lidia_size_t, lidia_size_t)::out of memory" ) ;
        }
       else if ( all < 0 )
        {
            lidia_error_handler ( "base_vector", "constructor (lidia_size_t, lidia_size_t)::invalid size" ) ;
        }
    }

    base_vector (  lidia_size_t all,  lidia_size_t len,  char md ) // create vector of cap. "all" and sz. "len"
    {
       debug_handler ( "base_vector", "constructor (lidia_size_t, lidia_size_t, char)" ) ;

       length    = ( len < all ) ? len : all  ;
       allocated = all ;
       mode      = md  ;

       dyn_exp_ratio = DEFAULT_RATIO ;

       data = nil ;

       if ( all > 0 )
        {
            data = new T [ all ]  ;

            if ( data == nil )
              lidia_error_handler ( "base_vector", "constructor (lidia_size_t, lidia_size_t, char)::out of memory" ) ;
        }
       else if ( all < 0 )
        {
            lidia_error_handler ( "base_vector", "constructor (lidia_size_t, lidia_size_t, char)::invalid size" ) ;
        }
    }


    base_vector ( const base_vector < T > & v )
    {
       debug_handler ( "base_vector", "constructor ( base_vector < T > &  )" ) ;

       length    = v.length ;
       allocated = v.length ;
       mode      = FIXED    ;

       dyn_exp_ratio = DEFAULT_RATIO ;

       if ( allocated > 0 )
        {
           data = new T [ allocated ] ;

           if ( data == nil )
              lidia_error_handler ( "base_vector", "constructor ( base_vector < T > &  )::out of memory" ) ;

           copy_data ( data, v.data, length ) ;
        }
       else
        {
           data = nil ;
        }
    }

    base_vector ( const base_vector < T > & v ,  char md )
    {
       debug_handler ( "base_vector", "constructor ( base_vector < T > &, char )" ) ;

       length    = v.length ;
       allocated = v.length ;
       mode      = md       ;

       dyn_exp_ratio = DEFAULT_RATIO ;

       if ( allocated > 0 )
        {
           data = new T [ allocated ] ;

           if ( data == nil )
             lidia_error_handler ( "base_vector", "constructor ( base_vector < T > &, char  )::out of memory" ) ;

           copy_data ( data, v.data, length ) ;

        }
       else
        {
           data = nil ;
        }
    }


    base_vector ( const  T *v ,  lidia_size_t len )
    {
       debug_handler ( "base_vector", "constructor (  const T * , lidia_size_t )" ) ;

       length    = len   ;
       allocated = len   ;
       mode      = FIXED ;

      dyn_exp_ratio = DEFAULT_RATIO ;

       data = nil ;

       if ( allocated > 0 )
        {
           data = new T [ allocated ] ;

           if ( data == nil )
             lidia_error_handler ( "base_vector", "constructor (  const T * , lidia_size_t  )::out of memory" ) ;

           copy_data ( data, v, length ) ;
        }
       else if ( allocated < 0 )
        {
           lidia_error_handler ( "base_vector" , "constructor( const T * , lidia_size_t )::invalid length" ) ;
        }
    }

    base_vector ( const  T *v ,  lidia_size_t len,  char md )
    {
       debug_handler ( "base_vector", "constructor ( const T * , lidia_size_t, char )" ) ;

       length    = len ;
       allocated = len ;
       mode      = md  ;

       dyn_exp_ratio = DEFAULT_RATIO ;

       data = nil ;

       if ( allocated > 0 )
        {
           data = new T [ allocated ] ;

           if ( data == nil )
             lidia_error_handler ( "base_vector", "constructor ( const T * , lidia_size_t, char  )::out of memory" ) ;

           copy_data ( data, v, length ) ;
        }
       else if ( allocated < 0 )
        {
           lidia_error_handler ( "base_vector" , "constructor ( const T * , lidia_size_t, char )::invalid length" ) ;
        }
    }



   // *****  definition of base_vector destructor-function  *****

   LiDIA_VEC_VIRTUAL ~base_vector()
    {
       debug_handler ( "base_vector", "destructor()" )  ;

       if ( allocated ) { delete[] data ; }

       data = nil ;

       length    = 0 ;
       allocated = 0 ;
    }



   // *****  reading and modifying the vector's variables  *****

   lidia_size_t capacity ( ) const
    {
        debug_handler ( "base_vector<T>", "capacity()" ) ;
        return allocated ;
    }


   int set_capacity ( lidia_size_t all )
    {
       debug_handler ( "base_vector<T>", "set_capacity (lidia_size_t)" ) ;       

       T *tmp ;
       int rc ;

       rc = 0 ;

       if ( all != allocated )
        {
          if ( all == 0 )
           {
             delete [ ] data ;
             data = nil ;

             length    = 0 ;
             allocated = 0 ;

             rc = 0 ;
           }
          else if ( all > 0 )
           {
             tmp = new T [ all ] ;

             if ( tmp == nil )
               lidia_error_handler ( "base_vector" , "set_capacity ( lidia_size_t )::out of memory" ) ;

             if ( tmp != nil ) 
              {     
                 length = ( all < length ) ? all : length ;

                 copy_data ( tmp, data, length ) ;

                 if ( data != nil ) delete [] data ;

                 data = tmp ;
                 allocated = all ;

                 rc = 0 ;
              }
             else
              { 
                 rc = 1 ;
              }
           }
          else
           {
              lidia_error_handler ( "base_vector" , "set_capacity ( lidia_size_t )::negative size" ) ;
              rc = 1 ;
           }
        }
      return rc ;
    }

   void kill ( )
    {
       debug_handler ( "base_vector", "kill ( void )" ) ;

       if ( allocated )
        {
            delete [] data ;
            data = nil ;

            length    = 0 ;
            allocated = 0 ;
        }
    }


   lidia_size_t  size ( ) const 
    {
        debug_handler ( "base_vector<T>", "size()" ) ;
        return length ;
    }

   int  set_size ( lidia_size_t len )
    {
        debug_handler ( "base_vector", "set_size ( lidia_size_t )" ) ;
        int rc ;

        if ( len < 0 )
         {
            lidia_error_handler ( "base_vector" , "set_size ( lidia_size_t )::negative size" ) ;
            rc = 1 ;
         }
        else if ( len <= allocated )
         {
            length = len ;
            rc = 0 ;
         }
        else if ( mode == EXPAND )
         {
             // ***  realloc 'data' to provide access to expanding vectors  ***

             rc = set_capacity ( (lidia_size_t) ( dyn_exp_ratio * len ) ) ;

             if ( !rc )
                length = len ;
             else
                warning_handler ( "base_vector" , "set_size ( lidia_size_t )::invalid size" ) ;
         }
        else
         {
             warning_handler ( "base_vector" , "set_size ( lidia_size_t )::invalid size" ) ;
             rc = 1 ;
         }       

        return rc ;
    }


   float exp_ratio ( ) const
    {
        debug_handler ( "dyn_vector", "exp_ratio ( void ) " ) ;
        return dyn_exp_ratio ;
    }

   int set_exp_ratio ( float ratio )
    {
       debug_handler ( "dyn_vector", "set_exp_ratio ( float )" ) ;

       int rc ;

       if ( ratio >= 1 )
        {
          dyn_exp_ratio = ratio ;
          rc = 0 ;
        }
       else
        {
          warning_handler ( "base_vector", "set_exp_ratio( float )::invalid ratio" ) ;
          rc = 1 ;
        }

       return rc ;
    }

   int  set_mode ( char md )
    {
        debug_handler ( "base_vector", "set_mode ( char )" ) ;

        if ( md == FIXED || md == EXPAND )
         {
             mode = md ;
             return 0  ;
         }
        else
         {
             warning_handler ( "base_vector", "set_mode( char )::invalid mode" ) ;
             return 1  ;
         }
    }


   int set_data ( T * d, lidia_size_t l ) 
    {
       debug_handler ( "base_vector", "set_data ( T*, lidia_size_t )" ) ;

       lidia_size_t i ;

       set_capacity ( l ) ;
       length = l  ;

       for ( i = 0 ; i < l ; i++ )
           data[i] = d[i] ;

       return 0 ;
    }


    T* get_data () const
     {
        debug_handler ( "base_vector", "get_data ()" ) ;

        T   * d ;
        lidia_size_t i   ;

        d = new T [ length ] ;

        if ( d == nil )
         {
            lidia_error_handler ( "base_vector", "get_data ()::memory exhausted" ) ;
            return nil ;
         } 

        for ( i = 0 ; i < length ; i++ )
            d[i] = data[i] ;

	return d ;
     }



#ifdef LiDIA_VECTOR_INTERNAL_DATA_ACCESS

    T* get_data_address () const
     {
        debug_handler ( "base_vector", "get_data_address ()" ) ;
        return data ;
     }
#endif




   // *****  vector - input/output  *****

   int read ( istream & in )
    {
       debug_handler ( "base_vector", "read ( istream & )" ) ;

       lidia_size_t  n = 0, sz = 4 ;
       char c             ;

       T *vbuf;

#ifdef __GNUG__

      lidia_size_t i ;

      if ( in.flags() & ios::bin )  // binary-mode for stream 'in'
       {
	   in >> sz ;

	   set_capacity ( sz ) ;
	   length = sz ;

	   for ( i = 0 ; i < length ; i++ )
	    {
	      in >> data[i]  ;
	    }	   

	   return 0 ;
       }
#endif

       vbuf = new T[sz] ;
       if ( vbuf == nil )
         lidia_error_handler ( "base_vector", "read ( istream & )::out of memory" ) ;

       in >> c;
       if ( c != '[' )

         lidia_error_handler ( "base_vector", "read( istream & )::[ expected" ) ;

       in >> c ;

       while ( c != ']' )
        {
          in.putback(c) ;
          in >> vbuf[n] ;
          n++ ;

          if ( n == sz )
           {
             debug_handler ( "base_vector", "read( istream & )::doubling input size" ) ;

             T  *nbuf     ;
             lidia_size_t osz = sz ;

             sz *= 2 ;

             nbuf = new T[sz] ;
             if ( nbuf == nil ) 
               lidia_error_handler ( "base_vector", "read( istream & )::out of memory" ) ;

             copy_data ( nbuf, vbuf, osz ) ;

             delete[] vbuf ;
             vbuf = nbuf ;
           }

          in >> c;
        }

       set_capacity ( n ) ;
       length = n  ;

       copy_data ( data, vbuf, n ) ;

       delete[] vbuf;

       return 0 ;
    }


   int  write ( ostream &  out )  const
    {
      debug_handler ( "base_vector", "write ( ostream & )" ) ;

      lidia_size_t i ;

#ifdef __GNUG__

      if ( out.flags() & ios::bin )  // binary-mode for stream 'out'
       {
	   out << length ;

	   for ( i = 0 ; i < length ; i++ )
	    {
	      out << data[i] << " ";
	    }	   

	   return 0 ;
       }

#endif
       
      out << "[ ";

      for ( i = 0 ; i < length ; i++ )
       {
         out << data[i] << " ";
       }

      out << "]" << flush ;

      // printf ( "(%d,%d,%p,%c:%0.2f) ", length , allocated , data , mode , dyn_exp_ratio ) ;
      // fflush ( stdout ) ;

      return 0 ;
    }


   friend LiDIA_INLINE istream & operator>> ( istream & s, base_vector < T > &  x )
    {
       debug_handler ( "base_vector", "operator>>" ) ;
       x.read ( s ) ;
       return s ;
    }


   friend LiDIA_INLINE ostream & operator<< ( ostream & s,  const base_vector < T > &  x ) 
    {
       debug_handler ( "base_vector", "operator<<" ) ;
       x.write ( s ) ;
       return s ;
    }




   // *****  access operator and function  *****

    T &  operator[] ( lidia_size_t i )
    {
      debug_handler ( "base_vector", "operator[](lidia_size_t)" ) ;

      if ( i < 0 )
       {
           lidia_error_handler ( "base_vector", "operator[](lidia_size_t)::out_of_range" ) ;
       }
      else
       {
           if ( i < allocated )
            {
                length = ( i+1 > length ) ? ( i + 1 ) : length ;
            }
           else if ( mode == EXPAND )
            {
                set_capacity ( (lidia_size_t) (dyn_exp_ratio * (i+1)) ) ;
                length =  i + 1 ;
            }
           else
            {
                lidia_error_handler ( "base_vector", "operator[](lidia_size_t)::out_of_range" ) ;
            }
       }

      return ( data[i] ) ;
    }



    const T &  operator [] ( lidia_size_t i ) const
    {
      debug_handler ( "base_vector", "operator[](lidia_size_t)const" ) ;

      if ( ( i < 0 ) || ( i >= length ) )
       {
           lidia_error_handler ( "base_vector", "operator[](lidia_size_t)const::out_of_range" ) ;
       }

      return  ( data [i] ) ;      
    }




    T &  member ( lidia_size_t i ) const
    {
      debug_handler ( "base_vector", "member ( lidia_size_t )" ) ;

      if ( ( i < 0 ) || ( i >= length ) )
       {
           lidia_error_handler ( "base_vector", "member::out_of_range" ) ;
       }

      return  ( data [i] ) ;      
    }




   // *****  assignment operator  *****

   base_vector< T > & operator= ( const base_vector < T > &  v )
    {
        debug_handler ( "base_vector", "operator=" ) ;

        if ( this != & v )
         {
            set_capacity ( v.length ) ;
            length = v.length ;

            copy_data ( data, v.data, length ) ;

            sort_dir = v.sort_dir ;
            el_cmp   = v.el_cmp   ;
         }

        return *this ;
    }



   // ***** reverse - function *****

   int reverse ( )
    {
      debug_handler ( "base_vector", "reverse()" ) ;

      T   c    ;
      lidia_size_t i, j ;

      for ( i = 0, j = length - 1 ; i < j ; i++, j-- )
       {
          c       = data[i] ;
          data[i] = data[j] ;
          data[j] = c       ;
       }

      return 0 ;
    }

   int reverse ( const base_vector< T > & b )
    {
      debug_handler ( "base_vector", "reverse( const base_vector< T > & )" ) ;

      T   c    ;
      lidia_size_t i, j ;
      
      if ( this == & b )
       {
	 for ( i = 0, j = length - 1 ; i < j ; i++, j-- )
	  {
	    c       = data[i] ;
	    data[i] = data[j] ;
	    data[j] = c       ;
	  }
       }
      else
       {	
	 set_capacity ( b.length ) ;
	 length = b.length ;

	 for ( i = 0 ; i < length ; i++ )
	  {
	     data[i] = b.data[length - i-1] ;
	  }
       }

      return 0 ;
    }



 friend void swap (  base_vector< T > & v, base_vector< T > & b )
    {
        lidia_size_t    l, a ;
        char   m    ;
        float  r    ;
        T     *d    ; 
	char   s    ;
	int  (*e) ( const T & a, const T & b ) ;	

        l = v.length    ;
        a = v.allocated ;
        m = v.mode ;
        r = v.dyn_exp_ratio ;
        d = v.data ;
	s = v.sort_dir ;
	e = v.el_cmp   ;

        v.length        = b.length    ;
        v.allocated     = b.allocated ;
        v.mode          = b.mode ;
        v.dyn_exp_ratio = b.dyn_exp_ratio ;
        v.data          = b.data ;
	v.sort_dir      = b.sort_dir ;
	v.el_cmp        = b.el_cmp   ;

        b.length        = l ;
        b.allocated     = a ;
        b.mode          = m ;
        b.dyn_exp_ratio = r ;
        b.data          = d ;
	b.sort_dir      = s ;
	b.el_cmp        = e ;
    }  


   int concat ( const base_vector< T > & b , const base_vector< T > & c )
    {
        lidia_size_t i, j, l  ;

	if ( ( this != &b ) &&  ( this != &c ) )
	 {
	    l = b.length + c.length ;
	    set_capacity ( l ) ;
	    length = l ;
	    
	    for ( i = 0 ; i < b.length ; i++ )
	     {
	        data[i] = b.data[i] ;
	     }
	    
	    j = b.length ;

	    for ( i = 0 ; i < c.length ; i++ , j++ )
	     {
	        data[j] = c.data[i] ;
	     }
	 }
	else // one of the arguments aliases 'this'
	 {
	    if ( this == & b )
	     {
	        j = length ;
	        l = length + c.length ;
		set_capacity ( l ) ;
		length = l ;

		for ( i = 0 ; i < c.length ; i++ , j++ )
		 {
		    data[j] = c.data[i] ;
		 }
	     }	
	    else // this == & c
	     {
	        T *p ;

	        base_vector< T > tmp ( b.length + c.length ) ;

		for ( i = 0 ; i < b.length ; i++ )
		 {
		    tmp.data[i] = b.data[i] ;
		 }

		j = b.length ;

		for ( i = 0 ; i < c.length ; i++ , j++ )
		 {
		    tmp.data[j] = c.data[i] ;
		 }

		// exchange pointers instead of copying

		p        = data     ;
		data     = tmp.data ;
		tmp.data = p        ;

		tmp.allocated = allocated ;

		allocated = length = b.length + c.length ;
	     }
	 }
	
	return 0 ;
      }


    
   int shift_left ( lidia_size_t pos , lidia_size_t num )
    {
      debug_handler ( "base_vector", "shift_left()" ) ;

      int       err ;
      lidia_size_t i ;
      lidia_size_t old_len = length ;

      if ( pos < num )
       {
           err = 1 ;
           lidia_error_handler ("base_vector", "shift_left(lidia_size_t,lidia_size_t)::invalid position" ) ;
       }
      else
       {
           err = set_size ( old_len - num ) ;

           if ( err )
            {
               lidia_error_handler ("base_vector", "shift_left(lidia_size_t,lidia_size_t)::invalid position" ) ;
            }
           else
            {
               for ( i = pos ; i < old_len ; i++ )
                   data[i-num] = data[i] ;
            }
       }

      return err ;
    } 


   int shift_right ( lidia_size_t pos , lidia_size_t num )
    {
      debug_handler ( "base_vector", "shift_right()" ) ;

      int       err ;
      lidia_size_t i ;
      lidia_size_t old_len = length ;

      if ( pos < 0 )
       {
           err = 1 ;
           lidia_error_handler ("base_vector", "shift_right(lidia_size_t,lidia_size_t)::invalid position" ) ;
       }
      else
       {
           err = set_size ( old_len + num ) ;

           if ( err )
            {
               lidia_error_handler ("base_vector", "shift_right(lidia_size_t,lidia_size_t)::invalid size" ) ;
            }
           else
            {
               for ( i = old_len-1 ; i >= pos ; i-- )
                   data[i+num] = data[i] ;
            }
       }

      return err ;
    } 



   int assign ( lidia_size_t at, const base_vector< T > & v , lidia_size_t from , lidia_size_t to )
    {
        debug_handler ( "base_vector", "assign(lidia_size_t,base_vector,lidia_size_t,lidia_size_t)" ) ;

        lidia_size_t old_len, n ;
        lidia_size_t new_len    ;
        lidia_size_t i, j ;
        int       err  ;

        T *tmp ;

        if (    ( at   < 0 ) 
             || ( from < 0 ) || ( from >= v.length )
             || ( to   < 0 ) || ( to   >= v.length ) || ( to < from ) )
         {
            lidia_error_handler ( "base_vector", "assign(lidia_size_t,base_vector,lidia_size_t,lidia_size_t)::invalid indices" ) ;
            return 1 ;
         }

        n = to - from + 1 ;  // this value is positive
        old_len = length  ;
        new_len = ( length < at + n ) ? ( at + n ) : length ;

        if ( this != &v )                    // v aliases current instance
         {
            err = set_size ( new_len ) ;

            if ( ! err )
             {
                for ( i = from, j = at ; i <= to ; i++, j++ )
                   data[j] = v.data[i] ;
             }
         }
        else
         {
            err = set_size ( new_len ) ;

            if ( ! err )
             {
                tmp = new T [ n ] ;
                if ( tmp == nil )
                  lidia_error_handler ( "base_vector", "assign(...)::out of memory" ) ;

                for ( i = from, j = 0 ; i <= to ; i++, j++ )
                   tmp[j] = v.data[i] ;

                for ( i = 0, j = at ; i < n ; i++, j++ )
                   data[j] = tmp[i] ;

                delete [ ] tmp ;
             }
         }

        return err ;
    }

 } ; 


#endif














