#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_mul_int		WITH_3_ARGS(
	mp_float,	x,
	mp_int,		iy,
	mp_float,	z
)
/*
Sets z = x * (int)iy and returns z.  This is faster than multiplication
of two mp numbers.  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), zp = mp_ptr(z);
    mp_base_type	b;
    mp_length		t, guard, total, i;
    mp_acc_index	acc_index;
    mp_digit_ptr_type	z_dig_p;
    mp_sign_type	result_sign;
    mp_int		j, c, ri;


    DEBUG_BEGIN(DEBUG_MULI);
    DEBUG_PRINTF_1("+mul_int {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_2("iy = %d\n", iy);

    /*
    Check for compatible parameters.
    */

    mp_check_2("mp_mul_int", xp, zp);


    /*
    Check for result zero.
    */
    
    result_sign = mp_sign(xp) * int_sign(iy);

    if (result_sign == 0)
    {
	DEBUG_PRINTF_1("-} z = 0\n");
	DEBUG_END();

	mp_set_sign(zp, 0);
	return z;
    }


    j = int_abs(iy);
    b = mp_b(zp);
    t = mp_t(zp);


    /*
    Check for multiplication by +-b.
    */

    if (j == b)
    {

	/*
	First check for overflow.
	*/

	if (mp_expt(xp) >= MAX_EXPT)
	    mp_overflow(z);


	/*
	All we need do is increment the exponent.
	*/

	mp_copy_ptr(xp, zp);
        mp_set_sign(zp, result_sign);
	mp_expt(zp) = mp_expt(xp) + 1;


	/*
	Update statistics of max exponent.
	*/

        mp_update(zp);

	DEBUG_1("-} left-shifted z =", zp);
	DEBUG_END();

	return z;
    }



    /*
    We have to genuine multiplication here.  Compute number of guard
    digits required to give exact product, allocate accumulator space, and
    reset mp pointers x and z.
    */

    guard = (iy == 1)? 1: mp_guard_digits(iy, b);
    total = t + guard;

    mp_change_up();

    mp_acc_digit_alloc(total, acc_index);
    z_dig_p = mp_acc_digit_ptr(acc_index);

    if (mp_has_changed())
    {
	xp = mp_ptr(x);
	zp = mp_ptr(z);
    }

#define z_dig(i)	z_dig_p[i]
#define z_ptr(i)	(z_dig_p + (i))


    /*
    Form product in accumulator.  If j*b is not representable as an integer
    we must simulate double-precision multiplication.
    */

    c = 0;

    if (j <= MAX_INT / b)
    {
	/*
	Inner loop for small int_abs(iy) case.
	*/

	DEBUG_PRINTF_1("small case\n");

	for (i = t - 1; i >= 0; i--)
        {
	    ri = j * mp_digit(xp, i) + c;
	    c = ri / b;
	    z_dig(i + guard) = ri - b * c;	/* i.e. ri % b */
	}

	/*
	Here Brent did a non-portable check to see if overflow had occurred
	by checking whether ri went below zero.  We have to treat the
	first guard words of the accumulator separately.
	*/

	if (guard > 0)
	    for (i = guard - 1; i >= 0; i--)
	    {
		z_dig(i) = c % b;
		c /= b;
	    }
    }

    else
    {

	/*
	Here j may be too large for single-precision multiplication.
	*/


        mp_int		xi, j1 = j / b, j2 = j % b;

	DEBUG_PRINTF_3("large case: j1 = %d, j2 = %d\n", j1, j2);

	for (i = total - 1; i >= 0; i--)
	{
	    xi = (i >= guard)? mp_digit(xp, i - guard): 0;
	    ri = j2 * xi + (c % b);

	    /*
	    It is important that in the following c/b + ri/b is not factorized.
	    The division distributive law does not hold in integer arithmetic.
	    */

	    c = j1 * xi + c / b + ri / b;

	    z_dig(i) = ri % b;
	}

    }


    /*
    c can only be zero if integer overflow occurred or b^t is too small.
    */

    if (c)
	mp_bug("mp_mul_int: product overflowed");


    /*
    Normalize and round or truncate result.
    */

    mp_nzr(result_sign, mp_expt(xp) + guard, zp, z_ptr(0), guard);


    /*
    Restore accumulator stack pointer and level.
    */

    mp_acc_delete(acc_index);
    mp_change_down();


    DEBUG_1("-} z = ", zp);
    DEBUG_END();

    return z;
}
