#include "defs.h"
#include "mp.e"
#include "mp.h"



mp_float
mp_cosh		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = cosh(x), x not too large.  No guard digits are used.
Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_acc_float	temp;

    mp_check_2("mp_cosh", xp, yp);

    if (mp_is_zero(xp))
    {
	mp_int_to_mp(1, y);
	return y;
    }

    mp_acc_float_alloc(mp_b(xp), mp_t(xp), temp);
    round = MP_TRUNC;

    mp_abs(x, temp);

    /*
    If int_abs(x) is too large, mp_exp() will print an error message.
    */

    mp_exp(temp, temp);
    mp_rec(temp, y);

    mp_add_eq(y, temp);
    mp_div_int_eq(y, 2);

    round = save_round;
    mp_acc_float_delete(temp);

    return y;
}


mp_float
mp_sinh		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = sinh(x), x not too large.  No guard digits are used.
Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_acc_float	temp1, temp2;
    mp_sign_type	x_sign = mp_sign(xp);


    mp_check_2("mp_sinh", xp, yp);

    if (mp_is_zero(xp))
    {
	mp_set_sign(yp, 0);
	return y;
    }

    mp_acc_float_alloc_2(mp_b(xp), mp_t(xp), temp1, temp2);

    round = MP_TRUNC;
    mp_abs(x, temp1);

    if (mp_expt(xp) > 0)
    {
	/*
	int_abs(x) >= 1; if int_abs(x) is too large, mp_exp() will print
	an error message.
	*/

	mp_exp(temp1, temp1);
	mp_rec(temp1, y);
	mp_sub(temp1, y, y);
    }
    else
    {
	/*
	int_abs(x) < 1, so use mp_exp1() to avoid cancellation.
	*/

	mp_exp1(temp1, temp2);
	mp_add_int(temp2, 2, temp1);
	mp_mul(temp1, temp2, y);
	mp_add_int(temp2, 1, temp1);
	mp_div_eq(y, temp1);
    }

    /*
    Divide by two and set appropriate sign.
    */

    mp_div_int_eq(y, 2 * x_sign);

    round = save_round;
    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);

    return y;
}



mp_float
mp_tanh		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = tanh(x).  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_sign_type	x_sign;
    mp_base_type	b;
    mp_length		t;
    mp_acc_float	temp1, temp2;

    mp_check_2("mp_tanh", xp, yp);

    x_sign = mp_sign(xp);

    if (x_sign == 0)
    {
	mp_set_sign(yp, 0);
	return y;
    }

    b = mp_b(xp);
    t = mp_t(xp);

    mp_acc_float_alloc_2(b, t + 1 + mp_extra_guard_digits(1, b), temp1, temp2);
    mp_move(x, temp1);

    mp_set_sign(mp_acc_float_ptr(temp1), 1);

    /*
    See if int_abs(x) is so large that result is +-1.
    */

    if (mp_cmp_q(temp1, 2 * mp_times_log2_b(t, b), 5) > 0)
    {
	mp_int_to_mp(x_sign, y);
	goto BYE;
    }

    round = MP_TRUNC;

    mp_mul_int_eq(temp1, 2);

    if (mp_acc_expt(temp1) > 0)
    {
	/*
	int_abs(x) >= 1/2, so use mp_exp().
	*/

	mp_exp(temp1, temp1);
	mp_add_int(temp1, -1, temp2);
	mp_add_int_eq(temp1, 1);
	mp_div_eq(temp2, temp1);
    }

    else
    {
	/*
	int_abs(x) < 1/2, so use mp_exp1() to avoid cancellation.
	*/

	mp_exp1(temp1, temp1);
	mp_add_int(temp1, 2, temp2);
	mp_div(temp1, temp2, temp2);
    }


    /*
    Set sign of result to sign of x.
    */

    if (x_sign < 0)
	mp_set_sign(mp_acc_float_ptr(temp2), -1);
    
    round = save_round;
    mp_move_round(temp2, y);

    /*
    Assure that the result is in the interval [-1, 1].
    */

    yp = mp_ptr(y);

    if (!mp_is_zero(yp) && mp_expt(yp) > 0)
	mp_int_to_mp(x_sign, y);

BYE:
    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);

    return y;
}
