/*  integer_quot_rem.c
*/

#include "defs.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"
 
/*
The following checks whether the sign of a remainder r is inappropriate for
the sign of a divisor b, that is, whether the remainder is nonzero with sign
opposite to that of the divisor.  It assumes that b is known to be non-zero,
so that if not positive it is negative.
*/

#define wrong_sign(b, r)	((b) > 0 ? (r) < 0 : (r) > 0)


void
integer_quot_rem    WITH_4_ARGS(
    integer_big,	aint,
    integer_big,	bint,
    integer_big  *,	qptr,
    integer_big  *,	rptr
)
/*
Given big integers a and b with b nonzero, calculate the quotient q and
remainder r as the unique integers such that

    a = q * b + r,  with    0 <= r <  b if b > 0
		    and     b <  r <= 0 if b < 0	.

This is equivalent to saying that q is the floor of (largest integer not
exceeding) a/b, and r = a - q * b.

Return q in *qptr and r in *rptr.  Signal an error via bh_error_general if
b is zero.
*/
{
    block_declarations;
    inthdl_handle   ahdl;
    inthdl_handle   bhdl;
    inthdl_handle   qhdl;
    inthdl_handle   rhdl;
    inthdl_length   alen;
    inthdl_length   blen;

    DEBUG_INTEGER_2("+integer_quot_rem", aint, bint);

    /*
    Note that bint represents zero only if bint == 0: a big integer in
    t_handle form always has absolute value not less than BETA, so may
    not be zero
    */

    DENY(bint == 0);

    if (integer_is_single(bint))
    {
	if (bint == 1)		/* single comparison */
	{
	    *qptr = integer_incref(aint);
	    *rptr = 0;
	}
	else if (integer_is_single(aint))
	{
	    /*
	    aint and bint are both single-precision
	    */
 
	    *qptr = aint / bint;
	    *rptr = aint - *qptr * bint;
 
	    if (wrong_sign(bint, *rptr))
	    {
		/*
		Adjust remainder and quotient to satisfy sign condition.  It
		is not hard to show that always abs(q) <= abs(a), so that
		in this case q like a is single-precision; hence subtracting
		1 cannot yield a result of -BETA.
		*/
 
		*rptr += bint;
		*qptr -= 1;
	    }
	}
	else
	{
	    /*
	    bint (divisor) is single-precision but aint is not
	    */
 
	    ahdl = inthdl_big_to_handle(aint);
	    qhdl = inthdl_buf_alloc(intbig_curr_size(ahdl));

	    inthdl_quot_rem_beta(ahdl, bint, qhdl, rptr);
	    /*  Note that *rptr is necessarily a beta-digit  */

	    *qptr = inthdl_standardize(qhdl);
	}
    }
    else if ((bhdl=inthdl_big_to_handle(bint)), integer_is_single(aint))
    {
	/*
	aint is single-precision, but bint is not, so aint has smaller
	absolute value than bint (this relies on bint being standardized).
	*/

	if (wrong_sign(intbig_sign(bhdl), aint))
	{
	    /*
	    Set q = -1 and r = a + b.
	    */

	    *qptr = -1;
	    rhdl = inthdl_alloc(intbig_curr_size(bhdl));
	    inthdl_add_beta(bhdl, aint, rhdl);
	    *rptr = inthdl_standardize(rhdl);
	}
	else
	{
	    *qptr = 0;
	    *rptr = aint;
	}
    } 
    else
    { 
	/*
	aint and bint are both multi-precision
	*/

	ahdl = inthdl_big_to_handle(aint);
	alen = intbig_curr_size(ahdl);
	blen = intbig_curr_size(bhdl);

	if (alen < blen &&
	  !wrong_sign(intbig_sign(bhdl), intbig_sign(ahdl)))
	{
	    /*
	    aint has absolute value less than bint and the same sign
	    */

	    *qptr = 0;
	    *rptr = aint;
	    inthdl_incref(ahdl);
	}
	else
	{ 
/* changed by MJ 	    qhdl = inthdl_buf_alloc(alen - blen + 2); */
 	    qhdl = inthdl_buf_alloc(int_max(alen - blen + 2,1)); 

	    rhdl = inthdl_buf_alloc(blen);
	
	    inthdl_quot_rem(ahdl, bhdl, qhdl, rhdl);
	
	    *qptr = inthdl_standardize(qhdl);
	    *rptr = inthdl_standardize(rhdl);
	}
    }

    DEBUG_INTEGER_2("-integer_quot_rem", *qptr, *rptr);
}


integer_big
integer_rem WITH_2_ARGS(
    integer_big, a,
    integer_big, b
)
/*
Given big integers a and b with b nonzero, return the remainder r, defined as
resulting from the call integer_quot_rem(a, b, &q, &r)  (see above; q is
discarded).  Signal an error via bh_error_general if b is zero.  Note that
if the remainder is nonzero, it has the same sign as b.

The common case b = 2 is optimized: the macro integer_is_even(a) uses this.
*/
{
    block_declarations;

    if (b == 2)
    {
	t_int	lowdig;
    
	if (integer_is_single(a))
	    lowdig = a;
	else
	    lowdig = intbig_digit(inthdl_big_to_handle(a), 0);
    
	return 1 & lowdig;
    }
    else
    {
	integer_big	q;
	integer_big	r;
    
	integer_quot_rem(a, b, &q, &r);

	integer_delref(q);

	return r;
    }
}

integer_big
integer_div	WITH_2_ARGS(
    integer_big, a,
    integer_big, b
)
/*
Given big integers a and b with b nonzero, return the quotient q, defined as
resulting from the call integer_quot_rem(a, b, &q, &r) (see above; r is
discarded).  Signal an error via bh_error_general if b is zero.
*/
{
    block_declarations;

    integer_big	q;
    integer_big r;

    integer_quot_rem(a, b, &q, &r);

    integer_delref(r);

    return q;
}
