#include "defs.h"
#include "integer.e"
#include "modint.e"

Void
i_pollard_rho_factor WITH_6_ARGS(
    integer_big,        n,
    integer_big,        linear,
    integer_big,        start,
    t_int *,    iter,
    integer_big *,      factor,
    integer_big *,      residue
)
/*
** Given an integer n, find a factor using the Pollard rho
** method.  The polynomial used is x^2 + linear, and the starting
** point is start.
** An upper bound for the number of iterations may be given in iter.
** If iter is -1, there is no upper bound.
** If a factor is found, then factor will be returned as
** the factor, and residue will be returned as n divided by the
** factor.
** If no factor is found, then factor will be returned as 1
** and residue will be returned as n.
*/
{
	block_declarations;

	Logical                   stepwise;
	register integer_big      x;
	register integer_big      xdash;
	register t_int    down;
	register t_int    half;
	register t_int    up;
	register integer_big      diff;
	register integer_big      t1;
	register integer_big      t2;
	register integer_big      q;
	register integer_big      g;
	register t_int	k;
	register t_int	l;
	register t_int    i;

	/*
	 * The algorithm is Algorithm B from Section 4.5.4 of D.E. Knuth,
	 * The Art of Computer Programming, Vol II, Seminumerical Algorithms.
	 *
	 * John Brownie, August 1989
	 *
	 * Brents's speed up is implemented, see:
	 * R.P. Brent, An improved Monte Carlo factorization algorithm,
	 * BIT v20, 1980, pp 176-184.
	 * Accumulated g.c.d.'s are computed after 2^k iterations,
 	 * (k = 1, 2, ...).
	 *
	 * Wieb Bosma, October 1989.
	 */

	/*
	 * Is n prime?
	 */
	if (i_miller_rabin(n, 10) == 1)
	{
		*factor = integer_incref(n);
		*residue = 1;
		return;
	}

	/*
	 * Initialise
	 * xdash = (start^2 + linear) mod n
	 * x = (xdash^2 + linear) mod n
	 */
	t1 = modint_exp(n, start, 2);
	xdash = modint_add(n, t1, linear);
	integer_delref( t1 );

	x = integer_incref(xdash);
	stepwise = FALSE;

	/*
	 * Down and up give lower and upper bound for next bunch of
	 * 2^k iterations; only for the upper half of the interval
	 * the gcd of the product of |x-xdash| will be taken.
	 * If stepwise is true, instead of  the gcd of the product,
	 * the gcd after each iteration is calculated.
	 */
	down = 1;
	half = 3;
	up = 3;
	for(;;)
	{
		if(*iter > 0 && *iter < (up-down))
		{
			/* 
			 * Take gcd's stepwise if we reach the iteration bound
			 * otherwise we might miss factors!
			 */
			up = *iter;
			half = 1;
			i = 1;
			q = 1;
			stepwise = TRUE;
			goto STEP;
		}
		for(i = down+1; i < half; i++)
		{
			/* 
			 * Here only the iterates of x need be computed.
			 */
			t1 = modint_exp(n, x, 2);
			integer_delref(x);
			x = modint_add(n, t1, linear);
			integer_delref(t1);

			if(*iter>0) *iter -= 1;
		}
		q = 1;
		for(i = half; i <= up; i++)
		{
STEP:		;
			/* 
			 * Here the iterates of x need be computed, as well
			 * as |x-xdash|, the values of which are multiplied
			 * modulo n.
			 */
			t1 = modint_exp(n, x, 2);
			integer_delref(x);
			x = modint_add(n, t1, linear);
			integer_delref(t1);

			if (integer_compare(x, xdash) > 0)
				diff = integer_subtract(x, xdash);
			else
				diff = integer_subtract(xdash, x);

			t1 = q;
			q = modint_mult(n, t1, diff ); 
			integer_delref(t1);
			integer_delref(diff); 

			if(*iter > 0)
				(*iter) -= 1;

			if (stepwise == TRUE || *iter == 0)
				goto GCD;
		}
GCD:		;
		g = integer_gcd(q, n);
		integer_delref(q);

		if (integer_compare(g, 1) > 0)
		{
			/*
			 * g is a factor.
			 * If g == n and we did not take gcd's in every
			 * step, maybe we can recover different factors
			 * by backtracking. If we did take gcd's all the
			 * time  - the algorithm failed!
			 */
			if (integer_compare(g, n) != 0)
			{
				integer_delref(x);
				integer_delref(xdash);
				*factor = g;
				*residue = integer_div(n, g);
				return;
			}
			else if (stepwise == TRUE || *iter == 0)
			{
					/* admit failure */
				*factor = integer_incref(n);   
				*residue = 1;
				integer_delref(x);
				integer_delref(g);    
				integer_delref(xdash);
				return;
			}
			else
			{
					/* try backtracking */
				stepwise = TRUE;
				integer_delref(g);    
				i = half ;
				q = 1;
				*iter += (up - half);
				goto STEP;
			}
		}
		else if (*iter == 0)
		{
				/* failure because we ran out of iterations */
			*factor = 1;
			*residue = integer_incref(n);
			integer_delref(g);
			integer_delref(x);
			integer_delref(xdash);
			return;
		}
		else if (stepwise == TRUE)
		{
			i++;
			integer_delref(g);
			q = 1;
			goto STEP;
		}
			/* nothing found - try next bunch of iterations */
		down = up;
		half = 3 * (up + 1) / 2;
		up = 2 * up + 1;
		integer_delref( g );
		integer_delref(xdash);
		xdash = integer_incref(x);
	}
}
