#include "defs.h"
#include "integer.e"
#include "prime_cert.e"

/*
 * This file contains two public routines:
 *	i_next_prime(n, proof)
 *	i_previous_prime(n, proof)
 * These generate the prime number following and preceding n respectively.
 * For both functions, n must be a non-negative integer, either single or
 * infinite precision.
 *
 * For numbers < IBETA, a fast test due to Selfridge and Wagstaff is used.
 * For large numbers, the Miller-Rabin primality test is used, if proof is
 * FALSE.  Note that this only guarantees probable primes.  If proof is TRUE,
 * then primality is proved.
 *
 * John Brownie, August 1989
 */

extern t_int	small_primes[];
extern t_int	iuz[];

t_int
i_next_prime(n, proof)
t_int	n;
Logical	proof;
{
	block_declarations;

	register t_int	i;
	register t_int	resno;
	register t_int	p;
	register t_int	base;
	register t_int	start;
	register t_int	t;
	prime_cert_handle	cert;

	/*
	 * Check the easy cases first: n < 2 and n < the largest prime stored
	 * in the table of small primes.
	 */

	if (integer_compare(n, 2) < 0)
		return 2;
	if (integer_compare(n, small_primes[NSPRIMES - 1]) < 0)
		for (i = 0; i < NSPRIMES; i++)
			if (small_primes[i] > n)
				return small_primes[i];

	/*
	 * For efficiency, process single and infinite precision numbers
	 * separately.
	 */

	if (integer_is_single(n))
	{
		start = n % 210;
		base = n - start;
		for (i = 0; i < NRESIDUES; i++)
			if (iuz[i] > start)
			{
				resno = i;
				break;
			}
		if (i >= NRESIDUES)
		{
			resno = 0;
			base += 210;
		}

		/*
		 * Generate and test possible primes until a true prime
		 * is found.
		 */

		p = base + iuz[resno];
		for (;;)
		{
			if (integer_is_prime_small(p))
				return p;
			if (resno < NRESIDUES - 1)
				resno++;
			else
			{
				base += 210;
				resno = 0;
			}
			p = base + iuz[resno];
		}
	}
	start = integer_rem(n, 210);
	for (i = 0; i < NRESIDUES; i++)
		if (iuz[i] > start)
		{
			base = integer_subtract(n, start);
			resno = i;
			break;
		}
	if (i >= NRESIDUES)
	{
		base = integer_add(n, 210 - start);
		resno = 0;
	}

	/*
	 * Generate and test possible primes until a probable prime is found.
	 */

	p = integer_add(base, iuz[resno]);
	for (;;)
	{
		if (i_miller_rabin(p, 20) == 1)
		{
			cert = prime_cert_alloc(3);
			if (!proof || integer_is_prime(p, cert, FALSE, 1000))
			{
				prime_cert_delete(&cert);
				integer_delref(base);
				return p;
			}
			prime_cert_delete(&cert);
		}
		if (resno < NRESIDUES - 1)
			resno++;
		else
		{
			t = base;
			base = integer_add(base, 210);
			integer_delref(t);
			resno = 0;
		}
		t = p;
		p = integer_add(base, iuz[resno]);
		integer_delref(t);
	}
}

t_int
i_previous_prime(n, proof)
t_int	n;
Logical	proof;
{
	block_declarations;

	register t_int	i;
	register t_int	resno;
	register t_int	p;
	register t_int	base;
	register t_int	start;
	register t_int	t;
	prime_cert_handle	cert;

	/*
	 * Check the easy case first: n < the largest prime stored in the table
	 * of small primes.
	 */

	if (integer_compare(n, small_primes[NSPRIMES - 1]) < 0)
		for (i = NSPRIMES - 1; i >= 0; i--)
			if (small_primes[i] < n)
				return small_primes[i];

	/*
	 * For efficiency, process single and infinite precision numbers
	 * separately.
	 */

	if (integer_is_single(n))
	{
		start = n % 210;
		base = n - start;
		for (i = NRESIDUES - 1; i >= 0; i--)
			if (iuz[i] < start)
			{
				resno = i;
				break;
			}
		if (i < 0)
		{
			resno = NRESIDUES - 1;
			base -= 210;
		}

		/*
		 * Generate and test possible primes until a true prime
		 * is found.
		 */

		p = base + iuz[resno];
		for (;;)
		{
			if (integer_is_prime_small(p))
				return p;
			if (resno > 0)
				resno--;
			else
			{
				base -= 210;
				resno = NRESIDUES - 1;
			}
			p = base + iuz[resno];
		}
	}
	start = integer_rem(n, 210);
	for (i = NRESIDUES - 1; i >= 0; i--)
		if (iuz[i] < start)
		{
			base = integer_subtract(n, start);
			resno = i;
			break;
		}
	if (i < 0)
	{
		base = integer_add(n, start - 210);
		resno = NRESIDUES - 1;
	}

	/*
	 * Generate and test possible primes until a probable prime is found.
	 */

	p = integer_add(base, iuz[resno]);
	for (;;)
	{
		if (i_miller_rabin(p, 20) == 1)
		{
			cert = prime_cert_alloc(3);
			if (!proof || integer_is_prime(p, cert, FALSE, 1000))
			{
				prime_cert_delete(&cert);
				integer_delref(base);
				return p;
			}
			prime_cert_delete(&cert);
		}
		if (resno > 0)
			resno--;
		else
		{
			t = base;
			base = integer_subtract(base, 210);
			integer_delref(t);
			resno = NRESIDUES - 1;
		}
		t = p;
		p = integer_add(base, iuz[resno]);
		integer_delref(t);
	}
}
