#include "mem.h"
 
t_int memi_slow_class(n)
t_int   n;
/*
Return the class for a block of size n.
*/
{
    double      log();
    t_int       c;
 
    c = log((double)(n - (MIN_SIZE - 1))) / CLASS_LOG_S;
    ASSERT(c >= 0 && c < CLASS_LIMIT);
    return c;
}
 
t_void memi_copy_block(bp, newp)
t_ptr   bp, newp;
/*
Copy block bp over block newp.
*/
{
    t_ptr       limit;
 
    limit = NEXT_BLOCK(bp) - HEADER_SIZE;
 
    while (bp < limit)
        *newp++ = *bp++;
}
 
t_void memi_copy_block_zero_tail(bp, newp)
t_ptr   bp, newp;
/*
Copy block bp over block newp and zero the tail.
*/
{
    t_ptr       limit, newlimit;
 
    limit = NEXT_BLOCK(bp) - HEADER_SIZE;
    newlimit = NEXT_BLOCK(newp) - HEADER_SIZE;
 
    while (bp < limit)
        *newp++ = *bp++;
 
    while (newp < newlimit)
        DEREF(newp++) = 0;
}
 
t_void memi_copy_block_maybe_zero(bp, newp, do_zero)
t_ptr   bp, newp;
t_int   do_zero;
{
    if (do_zero)
        memi_copy_block_zero_tail(bp, newp);
    else
        memi_copy_block(bp, newp);
}
 
t_void memi_remove_from_list(bp)
t_ptr   bp;
/*
Remove bp from its free list.
*/
{
    t_ptr       next, prev;
 
    next = NEXT_FREE(bp);
    prev = PREV_FREE(bp);
    DEREF(prev) = next;
    if (next)
        PREV_FREE(next) = prev;
}
 
t_ptr memi_get_block(n)
t_len   n;
/*
Return a block of size n.
*/
{
    t_ptr       fp, bp, next, next_fp, next_free_next, newfreep;
    t_ptr       next_free, last, datap, prev, big_limit;
    t_len       size, newsize;
 
    ASSERT(n >= MIN_SIZE);
 
    ENTER_CHECK();
    STATS(memi_stats_get_block++);
 
    IF_ANALYSIS(
        memi_analysis_get_block++;
        memi_analysis_total_get_block += n;
    )
 
    fp = (t_ptr)&memi_free_lists[MEMI_CLASS(n)];
    big_limit = memi_free_limit;
 
    for (;;)
    {
        bp = DEREF(fp);
        if (!bp || (size = (next = NEXT_BLOCK(bp)) - bp) < n)
        {
            IF_ANALYSIS(
                memi_analysis_list_count++;
            )
 
            if (++fp < big_limit)
                continue;
            else
                break;
        }
 
        /*
        Found a block big enough.  Now see if there's a block on this
        list which is still big enough but smaller than the first block.
        */
 
        for (;;)
        {
            ASSERT(PREV_FREE(bp) == fp);
 
            next_fp = (t_ptr)&NEXT_FREE(bp);
            next_free = DEREF(next_fp);
 
            if (size == n)
                break;
 
            if (
                !next_free ||
                (
                    size = (next_free_next = NEXT_BLOCK(next_free)) - next_free
                ) < n
            )
                break;
 
            IF_ANALYSIS(
                memi_analysis_block_count++;
            )
 
            fp = next_fp;
            bp = next_free;
            next = next_free_next;
        }
 
        /*
        Remove from its free list.
        */
 
        DEREF(fp) = next_free;
        if (next_free)
            PREV_FREE(next_free) = fp;
 
        newfreep = bp + n;
        newsize = next - newfreep;
 
        if (newsize >= MIN_SIZE)
        {
            /*
            Extra space is big enough to be a block.
            */
 
            IF_ANALYSIS(
                memi_analysis_spare_block++;
            )
 
            NEXT_BLOCK(bp) = newfreep;
            PREV_BLOCK(newfreep) = bp;
            NEXT_BLOCK(newfreep) = next;
            PREV_BLOCK(next) = newfreep;
 
            memi_free_block(newfreep);
        }
        return bp;
    }
 
    /*
    Need to get some more process space.
    */
 
    if (memi_high_mem == memi_base_mem)
        memi_more_data(n, n, &bp);
    else
    {
        /*
        If the last block is free, only get the needed excess.
        If a gap has occurred, however, we still have to get the full amount.
        */
 
        last = PREV_BLOCK(memi_high_mem);
        if (IS_FREE(BLOCK_HANDLE(last)))
            if (memi_more_data(n - (memi_high_mem - last), n, &datap))
                bp = datap;
            else
            {
                /*
                Take the last block of its free list and coalesce with the
                new data.
                */
 
                prev = PREV_FREE(last);
                next = NEXT_FREE(last);
                DEREF(prev) = next;
                if (next)
                    PREV_FREE(next) = prev;
                bp = last;
            }
        else
            memi_more_data(n, n, &bp);
    }
 
    newfreep = bp + n;
 
    if (newfreep + MIN_SIZE <= memi_high_mem)
    {
        NEXT_BLOCK(bp) = newfreep;
        PREV_BLOCK(newfreep) = bp;
 
        NEXT_BLOCK(newfreep) = memi_high_mem;
        PREV_BLOCK(memi_high_mem) = newfreep;
 
        NEXT_FREE(newfreep) = NULL;
        memi_free_block(newfreep);
    }
    else
    {
        NEXT_BLOCK(bp) = memi_high_mem;
        PREV_BLOCK(memi_high_mem) = bp;
    }
 
    return bp;
}
 
t_void memi_free_block(bp)
t_ptr   bp;
/*
Free block bp (put it on its appropriate free list).
*/
{
    t_ptr       fp, np;
    t_len       size;
 
    size = NEXT_BLOCK(bp) - bp;
    fp = (t_ptr)&memi_free_lists[MEMI_CLASS(size)];
 
    IF_ANALYSIS(
        memi_analysis_free_block++;
        memi_analysis_total_free_block += size;
    )
 
    while ((np = DEREF(fp)) && NEXT_BLOCK(np) - np > size)
    {
        IF_ANALYSIS(
            memi_analysis_free_block_count++;
        )
        fp = (t_ptr)&NEXT_FREE(np);
    }
 
    DEREF(fp) = bp;
    PREV_FREE(bp) = fp;
    NEXT_FREE(bp) = np;
    if (np)
        PREV_FREE(np) = (t_ptr)&NEXT_FREE(bp);
}
 
t_void memi_free_block_coalesce(bp)
t_ptr   bp;
/*
Coalesce the block bp with its neighbours if possible and then free it.
*/
{
    memi_free_block(memi_coalesce_block(bp));
}
 
t_ptr memi_extend_block(bp, n, do_zero)
t_ptr   bp;
t_len   n;
t_int   do_zero;
/*
Extend block bp to have size n.  If do_zero, zero the new space.
Return the new block.
*/
{
    t_ptr       newp, next, old_next, fp, big_limit;
    t_len       size;
 
    ASSERT(NEXT_BLOCK(bp) - bp < n);
 
    old_next = NEXT_BLOCK(bp);
    memi_coalesce_right(bp);
    next = NEXT_BLOCK(bp);
    size = BLOCK_SIZE(bp);
 
    if (next - bp >= n)
    {
        /*
        Coalescing with the right block has yielded enough space.
        */
 
        t_ptr   newfreep;
 
        newfreep = bp + n;
        if (newfreep + MIN_SIZE <= next)
        {
            /*
            The right block has enough space to make an extra free block.
            */
 
            NEXT_BLOCK(bp) = newfreep;
            PREV_BLOCK(newfreep) = bp;
            NEXT_BLOCK(newfreep) = next;
            PREV_BLOCK(next) = newfreep;
            next = newfreep;
 
            memi_free_block(newfreep);
        }
 
        if (do_zero)
        {
            old_next -= HEADER_SIZE;
            next -= HEADER_SIZE;
            while (old_next < next)
                DEREF(old_next++) = 0;
        }
        return bp;
    }
 
    if (do_zero)
    {
        t_ptr   copy_old_next, copy_next;
 
        copy_old_next = old_next - HEADER_SIZE;
        copy_next = next - HEADER_SIZE;
        while (copy_old_next < copy_next)
            DEREF(copy_old_next++) = 0;
    }
 
    if (bp > memi_base_mem)
    {
        t_ptr   prev;
        t_len   prev_size;
 
        prev = PREV_BLOCK(bp);
        prev_size = BLOCK_SIZE(prev);
        if (prev_size + size >= n && IS_FREE(BLOCK_HANDLE(prev)))
        {
            t_ptr       sp, dp, ep;
 
            /*
            The previous block is free and there's enough room, so move bp left.
            */
 
            memi_remove_from_list(prev);
 
            sp = bp;
            dp = prev;
            ep = prev + size - HEADER_SIZE;
            while (dp < ep)
                DEREF(dp++) = DEREF(sp++);
 
            ep = prev + n;
 
            if (ep + MIN_SIZE <= next)
            {
                /*
                Spare space can be made into another block.
                */
 
                NEXT_BLOCK(ep) = next;
                PREV_BLOCK(next) = ep;
                NEXT_BLOCK(prev) = ep;
                PREV_BLOCK(ep) = prev;
 
                memi_free_block(ep);
            }
            else
            {
                ep = next;
 
                NEXT_BLOCK(prev) = ep;
                PREV_BLOCK(ep) = prev;
            }
 
            if (do_zero)
            {
                ep -= HEADER_SIZE;
                while (dp < ep)
                    DEREF(dp++) = 0;
            }
 
            return prev;
        }
    }
 
    /*
    Check to see if memi_get_block() will work without getting more process
    data.
    */
 
    fp = (t_ptr)&memi_free_lists[MEMI_CLASS(n)];
    big_limit = memi_free_limit;
    while (fp < big_limit)
    {
        t_ptr   fbp;
 
        fbp = DEREF(fp);
        if (fbp && BLOCK_SIZE(fbp) >= n)
            break;
        fp++;
    }
 
    if (fp == big_limit && next == memi_high_mem)
    {
        /*
        There isn't a place big enough to move it but this is the last
        block.  Attempt to get just enought extra process data and merge with
        that.
        */
 
        t_ptr   datap;
 
        ASSERT(memi_high_mem > memi_base_mem);
 
        if (!memi_more_data(n - size, n, &datap))
        {
            next = bp + n;
            if (next + MIN_SIZE <= memi_high_mem)
            {
                /*
                Some more data was obtained than asked for and it's big
                enough to make into a free block.
                */
 
                NEXT_BLOCK(bp) = next;
                PREV_BLOCK(next) = bp;
                NEXT_BLOCK(next) = memi_high_mem;
                PREV_BLOCK(memi_high_mem) = next;
                memi_free_block(next);
            }
            else
            {
                /*
                No spare space.  This is now the last block.
                */
 
                next = memi_high_mem;
                NEXT_BLOCK(bp) = next;
                PREV_BLOCK(next) = bp;
            }
 
            if (do_zero)
            {
                old_next -= HEADER_SIZE;
                next -= HEADER_SIZE;
                while (old_next < next)
                    DEREF(old_next++) = NULL;
            }
 
            return bp;
        }
 
        /*
        Since there's a gap we can't do anything special.
        Make the new space free.
        */
 
        ASSERT(datap == next);
        NEXT_BLOCK(datap) = memi_high_mem;
        PREV_BLOCK(memi_high_mem) = datap;
        memi_free_block(datap);
    }
 
    /*
    We definitely have to move the block somewhere else.
    */
 
    newp = memi_get_block(n);
    memi_copy_block_maybe_zero(bp, newp, do_zero);
 
    BLOCK_HANDLE(newp) = BLOCK_HANDLE(bp);
    memi_free_block_coalesce(bp);
 
    return newp;
}
 
t_void memi_reduce_block(bp, n)
t_ptr   bp;
t_len   n;
{
    t_ptr       next, newfreep;
 
    next = NEXT_BLOCK(bp);
    newfreep = bp + n;
 
    ASSERT(newfreep <= next);
    if (newfreep + MIN_SIZE <= next)
    {
        NEXT_BLOCK(newfreep) = next;
        PREV_BLOCK(next) = newfreep;
        NEXT_BLOCK(bp) = newfreep;
        PREV_BLOCK(newfreep) = bp;
 
        memi_free_block_coalesce(newfreep);
    }
}
