/******************************************************************************
  order_unit_con_construct.c
******************************************************************************/
#include "kant.h"

anf_elt
order_unit_con_construct WITH_2_ARGS (
	order,		ord,
	vector,		veca

)
/******************************************************************************
 
Description:	Computes a unit eta satisfying
		    
		|eta(i)| <  1   if vec_entry(veca,i) = -1  (==> sign(log(|eta(i)|)) = -1) 
		|eta(i)| >= 1   if vec_entry(veca,i) =  1  (==> sign(log(|eta(i)|)) =  1) 
		|eta(i)| == ?	if vec_entry(veca,i) =  0

		where eta(j) denotes the j-th conjugate of eta,

Calling sequence:
 
	eta = order_unit_con_construct(ord,veca);

	eta	: anf_elt	= see above
	ord	: order		= t_handle of order
	veca	: vector	= vector of length r1+r2 with
				  coefficients in { -1,0,1 }

History:

	93-02-10 KW	wrong discriminant
	92-09-02 KW	Check results (anf_print_level != 0)
	92-08-10 KW	minor changes
	92-06-11 KW	written
 
******************************************************************************/
{
	block_declarations;

	anf_elt			alpha,beta,eta;
	dyn_arr_handle		gamma;
	dyn_arr_handle		hnf_trans;
	dyn_arr_handle		hnf_trans_den;
	t_handle		C,R,Z;
	integer_big		trans_den;
	integer_big		a,b;
	integer_small		i,j,k,n;
	integer_small		Ivecb,Jvecb;
	integer_small		q;
	integer_small		r1,r2,r12;
	t_logical		ready;
	matrix			m1,m2,m3,m4,m5,m6;
	matrix			trans;
	t_comp			ctemp;
	t_real			croof,ctilde,d;
	t_real			tempt,tempu,tempv,tempw,tempx,tempy,tempz;
	vector			lambda;
	vector			u,v,w;
	vector			vecb;

	n = order_abs_degree(ord);
	R = order_reals(ord);
	C = comp_create(real_dec_prec(R));
	Z = structure_z;

	r1  = order_r1(ord);
	r2  = order_r2(ord);
	r12 = r1+r2;

	order_disc_assure(ord);

/*
**	Compute:	croof  = 2^((n-1)/4)) * disc(ord)^(1/(2n))
**			ctilde = croof^n
*/
	tempt = conv_int_to_real(R,order_disc(ord));
	tempu = real_abs(R,tempt);
	tempv = real_root(R,tempu,n+n);
	tempw = conv_int_to_real(R,2);
	tempx = real_power(R,tempw,n-1);
	tempy = real_root(R,tempx,4);
	croof  = real_mult(R,tempv,tempy);
	ctilde = real_power(R,croof,n);
	real_delete(&tempt);
	real_delete(&tempu);
	real_delete(&tempv);
	real_delete(&tempw);
	real_delete(&tempx);
	real_delete(&tempy);

/*
**	Set vecb. vecb gets a vector of length r1+r2,
**	whose entries are Logicals with following properties:
**
**	(a)	vec_entry(veca,k) = -1 ==> vec_entry(vecb,k) = TRUE
**	(b)	vec_entry(veca,k) =  1 ==> vec_entry(vecb,k) = FALSE
**	(c)	|Ivecb - Jvecb| is minimal
**
**	where
**
**		Ivecb := #{ k | vec_entry(vecb,k) = TRUE }
**		Jvecb := #{ k | vec_entry(vecb,k) = FALSE }
*/
	vecb = vec_new(r12);
	Ivecb = Jvecb = 0;
	for (j=1;j<=r1;j++)
	{
		switch (vec_entry(veca,j))
		{
		case -1	:	Ivecb++;
				vec_entry(vecb,j) = TRUE;
				break;
		case 1	:	Jvecb++;
				vec_entry(vecb,j) = FALSE;
				break;
		default	:	
		}
	}
	for (j=r1+1;j<=r12;j++)
	{
		switch (vec_entry(veca,j))
		{
		case -1	:	Ivecb += 2;
				vec_entry(vecb,j) = TRUE;
				break;
		case 1	:	Jvecb += 2;
				vec_entry(vecb,j) = FALSE;
				break;
		default	:	
		}
	}
	for (j=1;j<=r1;j++)
	{
		if (!vec_entry(veca,j))
		{
			if (Ivecb > Jvecb)
			{
				vec_entry(vecb,j) = FALSE;
				Jvecb++;
			}
			else
			{
				vec_entry(vecb,j) = TRUE;
				Ivecb++;
			}
		}
	}
	for (j=r1+1;j<=r12;j++)
	{
		if (!vec_entry(veca,j))
		{
			if (Ivecb > Jvecb)
			{
				Jvecb += 2;
				vec_entry(vecb,j) = FALSE;
			}
			else
			{
				Ivecb += 2;
				vec_entry(vecb,j) = TRUE;
			}
		}
	}

/*
*/
	q = Ivecb;
	if (!q) error_internal("order_unit_con_construct: q = 0");
	if (q == n) error_internal("order_unit_con_construct: q = n");

/*
**	Show
*/
	if (anf_print_level > 0)
	{
		printf("conjunit: (");
		for (i=1;i<=r12;i++)
		{
			switch (vec_entry(veca,i))
			{
			case -1 :	printf("<1");
					break;
			case 0	:	printf("??");
					break;
			case 1	:	printf(">1");
					break;
			}
			if (i<r12) printf(",");
		}
		printf(").\n");

		printf("adjusted: (");
		for (i=1;i<=r12;i++)
		{
			if (vec_entry(vecb,i)) printf("<1"); else printf(">1");
			if (i<r12) printf(",");
		}
		printf(").\n");
	}

/*
**	
*/
	gamma         = dyn_arr_alloc(1);
	hnf_trans     = dyn_arr_alloc(1);
	hnf_trans_den = dyn_arr_alloc(1);

	k = 0;
	dyn_arr_element(gamma,k) = 1;
	dyn_arr_element(hnf_trans,k) = mat_ring_create_id(Z,n);
	dyn_arr_element(hnf_trans_den,k) = 1;

	m1 = mat_incref(order_basis_real(ord));
	m2 = mat_incref(m1);
	trans     = mat_ring_create_id(Z,n);
	trans_den = 1;

	do
	{
/*
**		Compute d
*/
		alpha = anf_elt_con(ord,dyn_arr_element(gamma,k));
		anf_norm(ord,alpha,&a,&b);
		anf_elt_delete(ord,&alpha);
		b = integer_abs(a);
		tempu = conv_int_to_real(R,b);
		tempv = real_root(R,tempu,n);
		tempw = real_divide(R,croof,tempv);
		tempx = real_power(R,tempw,q);
		d = real_root(R,tempx,n-q);
		integer_delref(a);
		integer_delref(b);
		real_delete(&tempu);
		real_delete(&tempv);
		real_delete(&tempw);
		real_delete(&tempx);

/*
**		Try new d.
*
		tempv = conv_int_to_real(R,n);
		tempw = d;
		d = real_divide(R,d,tempv);
		real_delete(&tempv);
		real_delete(&tempw);
*/

		beta = 0;
		do
		{
/*
**			Set lambda
*/
			lambda = vec_new(n);
			tempu = real_power(R,d,q-n);
			tempv = real_root(R,tempu,q);
			tempw = real_incref(tempv);
			tempx = real_inverse(R,tempw);
			tempy = real_inverse(R,d);
			for (j=1;j<=r12;j++)
			{
				vec_entry(lambda,j)	= (vec_entry(vecb,j))
   							? real_incref(tempx)
   							: real_incref(tempy);
			}
			for (j=r1+1;j<=r12;j++)
			{
				vec_entry(lambda,j+r2) = real_incref(vec_entry(lambda,j));
			}
			real_delete(&tempu);
			real_delete(&tempv);
			real_delete(&tempw);
			real_delete(&tempx);
			real_delete(&tempy);

			m3 = mat_new(n,n);
			for (i=1;i<=n;i++)	
			{		
				for (j=1;j<=n;j++)
				{
					mat_elt(m3,i,j) = real_mult(R,vec_entry(lambda,i),mat_elt(m2,i,j));
				}
			}

/**
			m4 = mat_ring_gram(R,m3);
			printf("Gram:\n");
			mat_ring_write(R,m4);
			printf("\n");
			mat_delref(R,&m4);
**/

/*
**			LLL-reduction
*/
			m4 = m5 = 0;
			lll_real_d_reduce(R,m3,0.75,&m4,&m5);
			m6 = mat_real_to_mat_z(R,m5);
			v = mat_ring_col_to_vector(Z,m6,1);
			w = mat_vector_col_mult(Z,trans,v);
/*
**			beta is the first basis element of a LLL-reduced basis of M(I,k)
*/
			anf_elt_delete(ord,&beta);
			anf_elt_alloc(beta,n);
			for (i=1;i<=n;i++)
			{
				anf_elt_coef(beta,i) = integer_incref(vec_entry(w,i));
			}
			anf_elt_den(beta) = integer_incref(trans_den);
			if (anf_print_level > 1)
			{
				printf("beta[%2d] : ",k+1);
				anf_elt_write(ord,beta);
				printf("\n");
			}

/*
**			Check results
*/
			if (anf_print_level)
			{
				u = mat_ring_col_to_vector(R,m4,1);
				tempt = vec_ring_dot_product(R,u,u);
				vec_delete(R,&u);

				alpha = anf_elt_con(ord,beta);
				tempz = ring_zero(R);
				for(i=1;i<=n;i++)
				{
					ctemp = anf_elt_ith_con(ord,alpha,i);
					tempu = comp_norm(C,ctemp);
					tempv = real_mult(R,vec_entry(lambda,i),vec_entry(lambda,i));
					tempw = real_mult(R,tempv,tempu);
					tempx = tempz;
					tempz = real_add(R,tempz,tempw);
					comp_elt_delete(C,&ctemp);
					real_delete(&tempu);
					real_delete(&tempv);
					real_delete(&tempw);
					real_delete(&tempx);
				}

				tempu = conv_double_to_real(R,0.001);
				tempv = real_subtract(R,tempz,tempt);
				if (!real_zero_eps(R,tempv,tempu))
				{
					printf("\norder_unit_con_construct: ");
					printf("WARNING -- Unsufficient real precision -- WARNING !!\n");
				}
				real_delete(&tempu);
				real_delete(&tempv);

				real_delete(&tempt);
				real_delete(&tempz);
				anf_elt_delete(ord,&alpha);			
			}

			vec_delete(R,&lambda);
			vec_delete(Z,&v);
			vec_delete(Z,&w);
			mat_delref(R,&m3);
			mat_delref(R,&m4);
			mat_delref(R,&m5);
			mat_delref(Z,&m6);

/*
**			Modify d (d <-- n*d)
*/
			tempu = d;
			tempv = conv_int_to_real(R,n);
			d = real_mult(R,d,tempv);
			real_delete(&tempu);
			real_delete(&tempv);

/*
**			Check beta
*/
			alpha = anf_elt_con(ord,beta);
			tempv = ring_one(R);
			ready = TRUE;
			for (j=1;(j<=r12) && (ready);j++)
			{
				if (vec_entry(veca,j) != 1) continue;
				ctemp = anf_elt_ith_con(ord,alpha,j);
				tempu = comp_norm(C,ctemp);
				ready = !(real_compare(R,tempu,tempv) == -1);
				comp_elt_delete(C,&ctemp);
				real_delete(&tempu);
			}
			anf_elt_delete(ord,&alpha);
			real_delete(&tempv);

		}
		while(!ready);

		real_delete(&d);
		mat_delref(R,&m1);
		mat_delref(R,&m2);
		mat_delref(Z,&trans);
		integer_delref(trans_den);

/*
**		Enlarge dynamical arrays 
*/
		dyn_arr_assure_space_fun(gamma,k+2,1);
		dyn_arr_assure_space_fun(hnf_trans,k+2,1);
		dyn_arr_assure_space_fun(hnf_trans_den,k+2,1);

/*
**		Set gamma(I,k+1);
*/
		dyn_arr_element(gamma,k+1)
			= anf_elt_mult(ord,beta,dyn_arr_element(gamma,k));;
		anf_elt_delete(ord,&beta);
		if (anf_print_level > 1)
		{
			printf("gamma[%2d]: ",k+1);
			anf_elt_write(ord,dyn_arr_element(gamma,k+1));
			printf("\n");
		}

/*
**		Compute transformation matrix (matrix and denominator)
**		from ord to M(I,k+1).
*/
		alpha = anf_div(ord,1,dyn_arr_element(gamma,k+1));
		trans     = anf_rep_mat(ord,alpha);
		trans_den = integer_incref(anf_elt_den(alpha));
		anf_elt_delete(ord,&alpha);

		mat_z_hnf_col_sub(Z,trans,&dyn_arr_element(hnf_trans,k+1),0,FALSE);
		dyn_arr_element(hnf_trans_den,k+1) = integer_incref(trans_den);
		mat_z_simplify(Z,dyn_arr_element(hnf_trans,k+1),
			&dyn_arr_element(hnf_trans_den,k+1));

/*
**		Test M(I,k+1) = M(I,m) for 0 <= m <= k
*/
		for (i=0;i<=k;i++)
		{
			if ( !integer_compare(dyn_arr_element(hnf_trans_den,k+1),
					dyn_arr_element(hnf_trans_den,i))
				&& mat_z_equal(Z,dyn_arr_element(hnf_trans,k+1),
					dyn_arr_element(hnf_trans,i)) )
			{
/*
**				M(I,k+1) = M(I,i) ==> gamma(I,k+1) and gamma(I,i) are associate
**				==> eta = gamma(I,k+1)/gamma(I,i) satiesfies the requests (see above)
*/
				eta = anf_div(ord,dyn_arr_element(gamma,k+1),dyn_arr_element(gamma,i));

/*
**				Delete for finishing
*/
				for (j=0;j<=k+1;j++)
				{
					anf_elt_delete(ord,&dyn_arr_element(gamma,j));
					mat_delref(Z,&dyn_arr_element(hnf_trans,j));
					integer_delref(dyn_arr_element(hnf_trans_den,j));
				}
				dyn_arr_delete(&gamma);
				dyn_arr_delete(&hnf_trans);
				dyn_arr_delete(&hnf_trans_den);

				mat_delref(Z,&trans);
				integer_delref(trans_den);
				real_delete(&croof);
				real_delete(&ctilde);
				vec_delete(Z,&vecb);
				ring_delete(&C);

				return(eta);
			}
		}
			
/*
**		Compute real basis of M(I,k+1)
*/
		tempu = conv_int_to_real(R,trans_den);
		tempv = real_inverse(R,tempu);
		m1 = mat_real_mat_z_mult(R,order_basis_real(ord),trans);
		m2 = mat_ring_scalar_left_mult(R,tempv,m1);
		real_delete(&tempu);
		real_delete(&tempv);

		k++;
	}
	while(1);
}
