/******************************************************************************
  mat_z_smith.c
******************************************************************************/
 
#include "stdio.h"        
#include "kant.h"
#include "mat.h"
#include "anf.h"

void
mat_z_smith WITH_4_ARGS(
	matrix,		mat,
	matrix  *,	mats,
	matrix  *,	t,
	matrix  *,	u
)
/*******************************************************************************
 
Description:
          
	Computes the Smith Normal Form "mats" of a given matrix mat, so that
	 (*)		mats = t * mat * u.
	Note: The matrices t and u will only be computed if they have been
	initialized before calling this routine. This also allows to work
	with already existing transformation matrices belonging to mat. In
	that case t and u will be modified so that (*) is correct for an older
	matrix mat but not for the "input mat".
        If t (or u) is 0 on entry it will not be altered.

  
Calling sequence:
 
        matrix		mat	= input matrix for Smith Normal Form
	matrix *	mats	= Smith Normal Form of matrix mat
	matrix *	t	= left transformation matrix
	matrix *	u	= right transformation matrix


 
History:
 
	92-06-15 JS     zeroing *mats before mat_z_hnf_col_sub
	92-05-29 CO     first version

*******************************************************************************/
{       
	block_declarations;
                                       
/*KW*/	integer_big	h;

	t_handle		z;
	integer_small	n, nt, nu, i, j;
	integer_big     gcd, rep_i, rep_j, dum1, dum2, dum3;
	matrix		hlpmat, hnf_tran;
	t_logical		t_flg, u_flg;


/*******  start with some initialisations  ******/             

                        
	nt = mat_row(mat);
	nu = mat_col(mat);
	n = ( nt > nu )  ?  nu : nt;
                  
	if (*t) t_flg = TRUE;
	else    t_flg = FALSE;
	if (*u) u_flg = TRUE;
	else    u_flg = FALSE;

	z = m_z_str_incref(structure_z);

	hnf_tran = 0; *mats = 0;
         
	/*** init. mats : mat - call by value, mats - call by reference */
	mat_z_hnf_col_sub( z, mat, mats, &hnf_tran, u_flg );

	/*******  modify transformation matrix u, if required  *******/
	if( u_flg )
	{
		hlpmat = *u;
		*u = mat_z_mult( z, *u, hnf_tran );
		mat_delref( z, &hlpmat );
		mat_delref( z, &hnf_tran );
	}
            

/*******  now our loop : ...HNF( HNF( mats^tr )^tr )...  *******/

               
/*KW*/	do
	{

	/*******  transpose *mats  *******/
                                              
		hlpmat = *mats;
		*mats = mat_ring_trans( z, hlpmat );
		mat_delref( z, &hlpmat );

	/*******  HNF  *******/
                                        
		hnf_tran = 0;                         
		hlpmat = 0;
		mat_z_hnf_col_sub( z, *mats, &hlpmat, &hnf_tran, t_flg );
 		mat_delref( z, mats );
 		*mats = hlpmat;

	/*******  modify transformation matrix t, if required  *******/
                                                                  
		if( t_flg )
		{

		/*******  transpose hnf_tran  *******/

			hlpmat = hnf_tran;
			hnf_tran = mat_ring_trans( z, hnf_tran );
			mat_delref( z, &hlpmat );

		/*******  t  <--  hnf_tran * t  *******/

			hlpmat = *t;
			*t = mat_z_mult( z, hnf_tran, *t );
			mat_delref( z, &hlpmat );
			mat_delref( z, &hnf_tran );
		}                                         

/*
... same procedure ... (except of case i = n/2.)
*/                         
	/*******  transpose *mats  *******/
                                             
		hlpmat = *mats;
		*mats = mat_ring_trans( z, *mats );
		mat_delref( z, &hlpmat );


	/*******  HNF  *******/
                                       
		hnf_tran = 0;                         
		hlpmat = 0;
		mat_z_hnf_col_sub( z, *mats, &hlpmat, &hnf_tran, u_flg );
 		mat_delref( z, mats );
		*mats = hlpmat;

	/*******  modify transformation matrix u, if required  *******/
                                                                  
		if( u_flg )
		{                

	/*******  u  <--  u * hnf_tran  *******/

			hlpmat = *u;
			*u = mat_z_mult( z, *u, hnf_tran );
			mat_delref( z, &hlpmat );
			mat_delref( z, &hnf_tran );
		}
/*KW*/	h = 0;
	for(i=1;i<=n;i++) for (j=1;j<=n;j++) if (i!=j) h += mat_elt(*mats,i,j);
	} while(h);

                                        
/*******  divisibility condition  *******/

	for( i = 1; i <= n-1; i++ )
	    for( j = i+1; j <= n ; j++ )
		if ( integer_sign( mat_elt( *mats,i,i ) ) != 0 )
		    if ( integer_sign( mat_elt( *mats,j,j ) ) != 0 )
		    {
	       		integer_quot_rem( mat_elt( *mats,j,j ), mat_elt( *mats,i,i ), &dum1, &dum2 );
			if ( integer_sign( dum2 ) != 0 )                              
                        {

        /* case mats(i,i) does not divide mats(j,j) */

			    integer_delref( dum1 );
			    integer_delref( dum2 );

			    integer_gcd_mult( mat_elt( *mats,i,i ), mat_elt( *mats,j,j ),
						&gcd, &rep_i, &rep_j );
                            
		/*******  modify transformation matrix t, if required  *******/

			    if( t_flg )
			    {
			    	mat_z_row_add( z, *t, i, j, rep_i, 1, nt );
			    	mat_ring_row_swap( z, *t, i, j, 1, nt );
			    	integer_quot_rem( mat_elt( *mats,i,i ), gcd, &dum1, &dum2 );
			    	dum2 = integer_negate( dum1 );
			    	mat_z_row_add( z, *t, i, j, dum2, 1, nt );
			    	integer_delref( dum1 );
			    	integer_delref( dum2 );
			    }

		/*******  modify transformation matrix u, if required  *******/

			    if( u_flg )
			    {
			    	mat_z_col_add( z, *u, j, i, rep_j, 1, nu );
			    	integer_quot_rem( mat_elt( *mats,j,j ), gcd, &dum1, &dum2 );
			    	dum2 = integer_negate( dum1 );
			    	mat_z_col_add( z, *u, i, j, dum2, 1, nu );
			    	integer_delref( dum1 );
			    	integer_delref( dum2 );
			    	mat_z_col_mult( z, *u, j, -1, 1, nu);
			    }

		/*******  mats(i,i) <-- gcd ( mats(i,i), mats(j,j) )  *******/
		/*******  mats(j,j) <-- lcm ( mats(i,i), mats(j,j) )  *******/

			    dum1 = mat_elt( *mats,j,j );
			    dum2 = integer_mult( mat_elt( *mats,i,i ), dum1 );
			    integer_quot_rem( dum2, gcd, &mat_elt( *mats,j,j ), &dum3 );
			    integer_delref( dum1 );
			    integer_delref( dum2 );
			    integer_delref( mat_elt( *mats,i,i ) );
			    mat_elt( *mats,i,i ) = gcd;
			    
			    integer_delref( rep_i );
			    integer_delref( rep_j );
			}
			else  

	/* nothing to do: mats(i,i) divides mats(j,j) */

			{
			    integer_delref( dum1 );
			    integer_delref( dum2 );
			}

		    }
		    else  

	/* case mats(i,i) <> 0 = mats(j,j)  ,i<j  */

		    {
			mat_elt( *mats,j,j ) = mat_elt( *mats,i,i );
			mat_elt( *mats,i,i ) = 0;       

		/*******  modify transformation matrices, if required  *******/

			if( t_flg )
			    mat_ring_row_swap( z, *t, i, j, 1, nu );
			if( u_flg )
			    mat_ring_col_swap( z, *u, i, j, 1, nu );
		    }

	ring_delete(&z);
                               
}
