/*******************************************************************************
  anf_mult_mod.c


  This file contains :

  anf_mult_mod
  anf_z_mult_mod
  anf_mult_z_mod

********************************************************************************/


#include "kant.h"
#include "integer.e"
#include "mat.h"
#include "anf.h"




anf_elt
anf_mult_mod WITH_4_ARGS(
	order,		ord,   
        integer_big,    modul,
	anf_elt,	alpha,
	anf_elt,	beta
)
/*******************************************************************************
Description:

	Multiplication of algebraic numbers over arbitrary orders modula a 
        given integer modul.
	 
	The numbers alpha and beta are multiplied, the result stored in gamma.
	In case the coefficient order is Z anf_z_mult is called.
	The multiplication is done by using the multiplication table of ord.
 
        Note that the function does not care for any denominators.

	So if a non-integer is submitted the result will definitely be a 
        non-integer.

                 
Calling sequence:

        gamma = anf_mult_mod (ord,modul,alpha,beta);

              order             ord       : all arithmetic will be done in this
                                            order.                     
              integer_big       modul     : all operations are done modulo modul
              anf_elt           alpha     : an alg. number of order.
              anf_elt           beta      : dito.
              anf_elt           gamma     : the product of alpha and beta modulo
                                            modul*ord.


 
History:         
	MD 91-11-12        anf_mult ---> anf_mult_mod.
	JS 91-09-18        first version of anf_mult


********************************************************************************/
{
	block_declarations;
 
	anf_elt		gamma;
	integer_small	deg;
        order		ordcoef;
	matrix		mat;
        anf_elt		accu1, accu2, temp1, temp2;	
	integer_small	i, j, k;

        
/*  
    Is one of the parameters a rational integer? 
*/

	if (anf_elt_is_integer(beta))  return anf_mult_z_mod (ord, modul, alpha, beta);
	if (anf_elt_is_integer(alpha)) return anf_mult_z_mod (ord, modul, beta, alpha);
 
/*  
    Is the coefficient order equal Z? 
*/
 
        ordcoef = order_coef_order(ord);
	if (ring_type(ordcoef) == RING_Z) return anf_z_mult_mod (ord, modul, alpha, beta);
  
/* 
    general case
*/
        
	deg = order_rel_degree(ord);
	anf_elt_alloc(gamma, deg);
                                                   
        anf_elt_den(gamma) = 1;
 
       	for (i=1; i<=deg; ++i)
	{
		accu1 = 0;
		for (k=1; k<=deg; ++k)
		{
			accu2 = 0;
			mat = order_mult_matrix(ord, k);
			for (j=1; j<=deg; ++j)
			{      
				temp1 = accu2;
				temp2 = anf_mult_mod (ordcoef, modul, 
                                                      mat_elt(mat, i, j),
						      anf_elt_coef(beta, j));
 				accu2 = anf_add_mod (ordcoef, modul, temp1, temp2);
				anf_elt_delete(ordcoef, &temp1);
				anf_elt_delete(ordcoef, &temp2);
			}
			temp1 = accu1;
			temp2 = anf_mult_mod (ordcoef, modul, accu2, 
                                              anf_elt_coef(alpha, k));
			accu1 = anf_add_mod (ordcoef, modul, temp1, temp2);
			anf_elt_delete(ordcoef, &temp1);
			anf_elt_delete(ordcoef, &temp2);
			anf_elt_delete(ordcoef, &accu2);
		}
		anf_elt_coef(gamma, i) = accu1;
	}
         
         
	return gamma;	
}
 
 
 


anf_elt
anf_z_mult_mod WITH_4_ARGS(
	                   order,		ord,
                           integer_big,		modul,
	                   anf_elt,		alpha,
	                   anf_elt,		beta
)
/*******************************************************************************
Description:
	Multiplication of algebraic numbers over Z modulo a given integer modul.
	The user should call the generic routine anf_mult.
 
	The numbers alpha and beta are multiplied, the result stored in gamma.
	The multiplication is done by using the multiplication table of ord.

        Note that the function does not care for any denominators.
 

Calling sequence:

        gamma = anf_z_mult_mod (ord,modul,alpha,beta);

              order             ord       : all arithmetic will be done in this
                                            order.                     
              integer_big       modul     : all operations are done modulo modul
              anf_elt           alpha     : an alg. number of order.
              anf_elt           beta      : dito.
              anf_elt           gamma     : the product of alpha and beta modulo
                                            modul*ord.

        The speciality of this function is, that we assume order to be defined 
        over Z.

 
History:         
	MD 91-11-12        anf_z_mult ---> anf_z_mult_mod.
	JS 91-09-18        first version of anf_mult


********************************************************************************/
{
	block_declarations;
 
	anf_elt		gamma;
	integer_small	deg;
	matrix		mat;
        integer_big	accu1, accu2, temp1, temp2, mod1, mod2;	
	integer_small	i, j, k;

	if (anf_elt_is_integer(beta))  return anf_mult_z_mod (ord, modul, alpha, beta);
	if (anf_elt_is_integer(alpha)) return anf_mult_z_mod (ord, modul, beta, alpha);
 
	deg = order_rel_degree(ord);
        
	anf_elt_alloc(gamma, deg);
                                                   
        anf_elt_den(gamma) = 1;
 
       	for (i=1; i<=deg; ++i)
	{
		accu1 = 0;
		for (k=1; k<=deg; ++k)
		{
			accu2 = 0;
			mat = order_mult_matrix(ord, k);
			for (j=1; j<=deg; ++j)
			{                                   
                                mod1  = integer_rem (mat_elt (mat,i,j),modul);
                                mod2  = integer_rem (anf_elt_coef (beta,j),modul);

				temp1 = accu2;
				temp2 = modint_mult (modul,mod1,mod2);

 				accu2 = modint_add(modul,temp1, temp2);
                                 
                                integer_delref (mod1);
                                integer_delref (mod2);
				integer_delref (temp1);
				integer_delref (temp2);
			}
			temp1 = accu1;            
                        mod1  = integer_rem (anf_elt_coef (alpha,k),modul);
			temp2 = modint_mult (modul,accu2, mod1);
			accu1 = modint_add (modul,temp1, temp2);

                        integer_delref (mod1);
			integer_delref (temp1);
			integer_delref (temp2);
			integer_delref (accu2);
		}
		anf_elt_coef(gamma, i) = accu1;
	}
         
	return gamma;	
}

 
 
 

anf_elt
anf_mult_z_mod WITH_4_ARGS(
	order,		ord,
        integer_big,    modul,
	anf_elt,	alpha,
	integer_big,	a
)
/*******************************************************************************
Description
	Multiplication of an algebraic number by a rational integer modulo 
        a given integer modul.
	
        This routine is written for arbitrary coefficient orders. If it
	is Z the multiplication is executed straightforward, in the
	other case anf_mult_z is called again componentwise.
 
        Note that the function does not care for any denominators.


Calling sequence:

        gamma = anf_mult_z_mod (ord,modul,alpha,beta);

              order             ord       : all arithmetic will be done in this
                                            order.                     
              integer_big       modul     : all operations are done modulo modul
              anf_elt           alpha     : an alg. number of order.
              integer           beta      : an integer is a special alg. integer.
              anf_elt           gamma     : the product of alpha and beta modulo
                                            modul*ord.

        The speciality of this function is, that we assume one of the mutiplyers
        to be a rational integer.

 
History:                         
        MD 92-03-25        integer_delref (a_mod) added in line 299
        MD 92-02-13        a changed to a_mod
	MD 91-11-12        anf_mult_z ---> anf_mult_z_mod.
	JS 91-09-18        first version of anf_mult


********************************************************************************/
{
	block_declarations;
 
	anf_elt		beta;
	integer_small	deg;
        order		ordcoef;
        integer_big	temp,temp1,temp2,a_mod;	
	integer_small	i;
 
 
/*
     trivial case:
*/                       
        a_mod = integer_rem (a,modul);

	if (anf_elt_is_integer(alpha)) 
        {                
          temp2 = integer_rem (alpha,modul);

          temp  = modint_mult (modul,a_mod,temp2);
                     
          integer_delref (temp2);
          integer_delref (a_mod);

          return temp;
        }
 
	deg = order_rel_degree(ord);
	anf_elt_alloc(beta, deg);
                                                   
        anf_elt_den(beta) = 1;	
        
        ordcoef = order_coef_order(ord);
	if (ring_type(ordcoef) == RING_Z)
        {	
        	for (i=1; i<=deg; ++i)
                {                     
                  temp = integer_rem (anf_elt_coef (alpha,i),modul);
                  anf_elt_coef(beta,i) = modint_mult(modul,temp, a_mod);
                  integer_delref (temp);
                }
	}
	else
	{
        	for (i=1; i<=deg; ++i)
			anf_elt_coef(beta,i) = 
			anf_mult_z_mod (ordcoef, modul, anf_elt_coef(alpha, i), a_mod);
	}
 
        integer_delref (a_mod);
	return beta;	
}
