/******************************************************************************
  anf_mult.c                                                           

  This file contains:
  anf_mult
  anf_z_mult
  anf_mult_z
  anf_con_mult
 
******************************************************************************/
 
#include "kant.h"
#include "integer.e"
#include "real.e"
#include "mat.h"
#include "anf.h"
#include "conv.e"
 
 

anf_elt
anf_mult WITH_3_ARGS(
	order,		ord,
	anf_elt,	alpha,
	anf_elt,	beta
)
/*******************************************************************************
 
Description:
 
	Multiplication of algebraic numbers over arbitrary orders
 
	The numbers alpha and beta are multiplied, the result stored in gamma.
	In case the coefficient order is Z anf_z_mult is called.
	The multiplication is done by using the multiplication table of ord.
	 
	For the reason of speed this routine doesn't care for simplifying 
	coefficients/denominator. Call anf_elt_simplify if denominators could 
	occur.
	So if a non-integer is submitted the result will definitely be a 
        non-integer.
 
 	
Calling sequence:
 
	gamma = anf_mult(ord, alpha, beta);
 
      	order       ord      = t_handle of order 
      	anf_elt     alpha    = first factor     (referring to basis of ord)
      	anf_elt     beta     = second factor    (referring to basis of ord)
      	anf_elt     gamma    = product          (referring to basis of ord)
      
 
History:
 
	92-06-15 JS    order_mult_assure
	91-10-08 JS    minor changes
	91-08-31 JS    written
 
*******************************************************************************/
{
	block_declarations;
 
	anf_elt		gamma;
	integer_small	deg;
        order		ordcoef;
	matrix		mat;
        anf_elt		accu1, accu2, temp1, temp2;	
	integer_small	i, j, k;

        
/*  
    Is one of the parameters a rational integer? 
*/
 
	if (anf_elt_is_integer(beta))  return anf_mult_z(ord, alpha, beta);
	if (anf_elt_is_integer(alpha)) return anf_mult_z(ord, beta, alpha);
 
/*
    Have we to deal with conjugate vectors?
*/
 
	if (anf_elt_is_con(alpha)) return anf_con_mult(ord, alpha, beta);
 
/*  
    Is the coefficient order equal Z? 
*/
 
        ordcoef = order_coef_order(ord);
	if (ring_type(ordcoef) == RING_Z) return anf_z_mult(ord, alpha, beta);
  
/* 
    general case
*/        
        order_mult_assure(ord);
        
	deg = order_rel_degree(ord);
	anf_elt_alloc(gamma, deg);
                                                   
        anf_elt_den(gamma) = integer_mult(anf_elt_den(alpha),anf_elt_den(beta));
 
       	for (i=1; i<=deg; ++i)
	{
		accu1 = 0;
		for (k=1; k<=deg; ++k)
		{
			accu2 = 0;
			mat = order_mult_matrix(ord, k);
			for (j=1; j<=deg; ++j)
			{      
				temp1 = accu2;
				temp2 = anf_mult(ordcoef, mat_elt(mat, i, j),
						 anf_elt_coef(beta, j));
 				accu2 = anf_add(ordcoef, temp1, temp2);
				anf_elt_delete(ordcoef, &temp1);
				anf_elt_delete(ordcoef, &temp2);
			}
			temp1 = accu1;
			temp2 = anf_mult(ordcoef,accu2,anf_elt_coef(alpha, k));
			accu1 = anf_add(ordcoef, temp1, temp2);
			anf_elt_delete(ordcoef, &temp1);
			anf_elt_delete(ordcoef, &temp2);
			anf_elt_delete(ordcoef, &accu2);
		}
		anf_elt_coef(gamma, i) = accu1;
	}
         
         
	return gamma;	
}
 
 
 


anf_elt
anf_z_mult WITH_3_ARGS(
	order,		ord,
	anf_elt,	alpha,
	anf_elt,	beta
)
/*******************************************************************************
 
Description:
 
	Multiplication of algebraic numbers over Z
	The user should call the generic routine anf_mult.
	 
	The numbers alpha and beta are multiplied, the result stored in gamma.
	The multiplication is done by using the multiplication table of ord.
	 
	For the reason of speed this routine doesn't care for simplifying 
	coefficients/denominator. Call anf_simplify if denominators could occur.
 
 
Calling sequence:
 
	gamma = anf_z_mult(ord, alpha, beta);
 
      	order       ord      = t_handle of order which must be Z
      	anf_elt     alpha    = first factor     (referring to basis of ord)
      	anf_elt     beta     = second factor    (referring to basis of ord)
      	anf_elt     gamma    = product          (referring to basis of ord)
      
 
History:
 
	92-06-15 JS    order_mult_assure
	91-09-26 JS    minor changes
	91-08-31 JS    written


*******************************************************************************/
{
	block_declarations;
 
	anf_elt		gamma;
	integer_small	deg;
	matrix		mat;
        integer_big	accu1, accu2, temp1, temp2;	
	integer_small	i, j, k;

        
	if (anf_elt_is_integer(beta))  return anf_mult_z(ord, alpha, beta);
	if (anf_elt_is_integer(alpha)) return anf_mult_z(ord, beta, alpha);
 
        order_mult_assure(ord);
 
	deg = order_rel_degree(ord);
        
	anf_elt_alloc(gamma, deg);
                                                   
        anf_elt_den(gamma) = integer_mult(anf_elt_den(alpha),anf_elt_den(beta));
 
       	for (i=1; i<=deg; ++i)
	{
		accu1 = 0;
		for (k=1; k<=deg; ++k)
		{
			accu2 = 0;
			mat = order_mult_matrix(ord, k);
			for (j=1; j<=deg; ++j)
			{      
				temp1 = accu2;
				temp2 = integer_mult(mat_elt(mat, i, j),
						anf_elt_coef(beta, j));
 				accu2 = integer_add(temp1, temp2);
				integer_delref(temp1);
				integer_delref(temp2);
			}
			temp1 = accu1;
			temp2 = integer_mult(accu2, anf_elt_coef(alpha, k));
			accu1 = integer_add(temp1, temp2);
			integer_delref(temp1);
			integer_delref(temp2);
			integer_delref(accu2);
		}
		anf_elt_coef(gamma, i) = accu1;
	}
         
	return gamma;	
}

 
 
 

anf_elt
anf_mult_z WITH_3_ARGS(
	order,		ord,
	anf_elt,	alpha,
	integer_big,	a
)
/*******************************************************************************
 
Description:
 
	Multiplication of an algebraic number by a rational integer
	This routine is written for arbitrary coefficient orders. If it
	is Z the multiplication is executed straightforward, in the
	other case anf_mult_z is called again componentwise.
 
	For the reason of speed this routine doesn't care for simplifying 
	coefficients/denominator. Call anf_simplify if denominators could occur.
         

Calling sequence:
 
	gamma = anf_mult_z(ord, alpha, beta);
 
      	order       ord      = t_handle of order which must be Z
      	anf_elt     alpha    = first factor     (referring to basis of ord)
      	integer_big beta     = second factor   
      	anf_elt     gamma    = product          (referring to basis of ord)
      
 
History:
 
	91-10-09 JS    minor changes
	91-08-31 JS    written

*******************************************************************************/
{
	block_declarations;
 
	anf_elt		beta;
	integer_small	deg;
        order		ordcoef;
        integer_big	temp;	
	integer_small	i;
 
 
/*
    trivial case:
*/
	if (anf_elt_is_integer(alpha)) return integer_mult(alpha, a);
 
/*
    conjugate vector?
*/
 
	if (anf_elt_is_con(alpha)) return anf_con_mult(ord, a, alpha); 
 
	deg = order_rel_degree(ord);
	anf_elt_alloc(beta, deg);
                                                   
	temp = anf_elt_den(alpha);
        anf_elt_den(beta) = integer_incref(temp);	
        
        ordcoef = order_coef_order(ord);
	if (ring_type(ordcoef) == RING_Z)
        {	
        	for (i=1; i<=deg; ++i)
			anf_elt_coef(beta,i) = 
                        integer_mult(anf_elt_coef(alpha, i), a);
	}
	else
	{
        	for (i=1; i<=deg; ++i)
			anf_elt_coef(beta,i) = 
			anf_mult_z(ordcoef, anf_elt_coef(alpha, i), a);
	}
         
	return beta;	
} 
 
 
 
 
anf_elt
anf_con_mult WITH_3_ARGS(
	order,		ord,
	anf_elt,	alpha,
	anf_elt,	beta
)
/*******************************************************************************
 
 
Description:
 
	Multiplication of two conjugate vectors.
	The user should call the generic routine anf_mult.
 
	The multiplication of the conjugates is done componentwise.
 
 
Calling sequence:
 
	gamma = anf_con_mult(ord, alpha, beta);
 
      	order       ord      = t_handle of order which must be Z
      	anf_elt     alpha    = first factor     (referring to basis of ord)
      	anf_elt     beta     = second factor    (referring to basis of ord)
      	anf_elt     gamma    = product          (referring to basis of ord)
      
 
History:
 
	93-01-15 MJ/KW ralpha deleted
	91-10-11 JS    minor changes
	91-10-01 JS    written
 
*******************************************************************************/
{
	block_declarations;
 
 
	anf_elt		gamma,betacon;
	integer_small	deg, r1, r2, r12, r1pi;
	anf_elt		tempaa;
	t_handle		reals;
	t_real		sq2, ralpha, a1, b1, a2, b2, temp1, temp2, temp3;
	integer_small	i, j;
 
 
/*
    Special case: beta (and alpha?) is integer
*/
 
	if (anf_elt_is_integer(beta)) 
	{	
        	if (anf_elt_is_integer(alpha)) return integer_mult(alpha, beta);
		tempaa = beta;
		beta   = alpha;
		alpha  = tempaa;
	}

       	deg = order_abs_degree(ord);
	r1  = order_r1(ord);
	r2  = order_r2(ord); 
	r12 = r1 + r2;
 
	reals = order_reals(ord);   
 
       	anf_con_alloc(gamma, deg);
 
/*
    Special case: only alpha is integer
*/      
        betacon = anf_elt_con(ord,beta);     

        
	if (anf_elt_is_integer(alpha))
 
        {
		ralpha = conv_int_to_real(reals, alpha);
 
		for (i=1; i<=deg; ++i)
		{
			anf_con(gamma, i) = 
				real_mult(reals, anf_con(betacon, i), ralpha);
		}   
                real_delete(&ralpha);                     
	}
  
	else
 
/*
    General case: multiplication componentwise
*/
 
	{       

		for (i=1; i<=r1; ++i)
		{
			anf_con(gamma, i) = real_mult(reals, anf_con(alpha, i),
							     anf_con(betacon,  i));
		}
		
		sq2 = order_sqrt_2(ord);

		for (i=r1+1, j=r12+1; i<=r12; ++i, ++j)
		{
			a1 = anf_con(alpha, i);
			b1 = anf_con(betacon,  i);
			a2 = anf_con(alpha, j);
			b2 = anf_con(betacon,  j);
 
			temp1 = real_mult(reals, a1, b1);
			temp2 = real_mult(reals, a2, b2);
			temp3 = real_subtract(reals, temp1, temp2);

			anf_con(gamma, i) = real_divide(reals, temp3, sq2);
 
			real_delete(&temp1);
			real_delete(&temp2);
			real_delete(&temp3);
 
			temp1 = real_mult(reals, a1, b2);
			temp2 = real_mult(reals, a2, b1);
			temp3 = real_add(reals, temp1, temp2);

			anf_con(gamma, j) = real_divide(reals, temp3, sq2);
 
			real_delete(&temp1);
			real_delete(&temp2);
			real_delete(&temp3);
		}
	}

        anf_elt_delete(ord,&betacon);
 
	return gamma;
}


