/******************************************************************************
  anf_elt_move.c                                                           
******************************************************************************/
         
#include "kant.h"
#include "integer.e"
#include "anf.h"


anf_elt
anf_elt_move WITH_3_ARGS(
	order,		ord1,
	anf_elt,	alpha,
	order,		ord2
)
/*******************************************************************************
 
Description:
 
	Expressing alpha (given referring to ord1) referring to a basis of ord2.
	This MUST be possible (Responsibility of caller!).
	If ord2 is nontrivial then the output will be a disguised t_handle to an anf_elt 
	but never an integer.
 
	The output of this routine will again be the number alpha, but
	now expressed referring to the basis of ord2.
 
	At the moment the following features are implemented:
 
	(1) alpha is an integer
            The result will be the same integer.
 
        (2) ord1 is equal to ord2 (the handles must be equal)
            alpha will be copied physically   
 
        (3) ord1 is the coefficient order of ord2 (lift up)
 
        (4) ord1 is given via a transition matrix referring to ord2 (lift up)
  
        (5) ord2 is given via a transition matrix referring to ord1 (lift down)
 
        in future:
 
        (6) ord1 is a subfield order of ord2
 
        (7) ord2 is a subfield order of ord1

 
Calling sequence:
 
	beta = anf_elt_move(ord1, alpha, ord2);
 
      	order       ord1  = t_handle of order alpha is given referring to
        anf_elt     alpha = algebraic number to move    
      	order       ord2  = t_handle of order beta will be given referring to
 

History:
 
	92-03-31 JS    just incref if alpha is an integer
	92-03-26 JS    case ord1==ord2 and alpha integer
	91-09-01 JS    first version

*******************************************************************************/
{
	block_declarations;
 
	anf_elt		beta;
	integer_small	deg, degcoef;
        order		ordcoef;
        integer_big	denom;	
	integer_small	onepos;
	integer_small	i;
        
 
/*   Trivial case  */
 
	if(ring_type(ord2) == RING_Z || anf_elt_is_integer(alpha)) 
		return integer_incref(alpha);
 
        deg = order_rel_degree(ord2);
        ordcoef = order_coef_order(ord2);
 
/*   Case ord1=ord2. We copy alpha coefficientwise */
 
        if(ord1 == ord2 && !anf_elt_is_integer(alpha))
        {
	        anf_elt_alloc(beta, deg);
 
                if(ring_type(ordcoef) == RING_Z)
                {
	             for (i=1; i<=deg; ++i)
                        anf_elt_coef(beta, i) = 
			integer_incref(anf_elt_coef(alpha,i));
                }
                else
	        {
		     for (i=1; i<=deg; ++i)
                        anf_elt_coef(beta, i) = 
		        anf_elt_move(ordcoef, anf_elt_coef(alpha, i), ordcoef);
                }
 
                anf_elt_den(beta) = integer_incref(anf_elt_den(alpha));
 
                return beta;
        }
  
	if(ordcoef == ord1)       /* This should be order_equality... */
	{
	        anf_elt_alloc(beta, deg);
 
		onepos = order_one_position(ord2);
		if (onepos <= 0) 
			error_internal("anf_elt_move: One not in basis.");
	
	        for (i=1; i<=deg; ++i)
        	{
              		if (i==onepos) continue;
			anf_elt_coef(beta,i) = 0;
		}
		anf_elt_coef(beta, onepos) = anf_elt_incref(alpha);
		anf_elt_den(beta) 	   = integer_incref(anf_elt_den(alpha));
		if (ring_type (ordcoef) != RING_Z)
                  anf_elt_den(anf_elt_coef(beta, onepos)) = 1;
	}
	else if(order_basis_is_rel(ord2) && order_suborder(ord2) == ord1)
	{
		beta = anf_elt_trans(ord1, alpha, order_invtran(ord2), 1);
	}
	else if(order_basis_is_rel(ord1) && order_suborder(ord1) == ord2)
	{
		beta = anf_elt_trans(ord1, alpha, order_tran(ord1), 
						order_tran_den(ord1));
	}
	else 
	{
             error_internal("anf_elt_move: This constellation not supported.");
	}
 
	return beta;
 
}
