;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald PARSER)


; A top down operator precedence parser 
; with left, nonassociative, and right infix operators.

; Precedence is an integer.  Operators of higher precedence bind
; more tightly.

; Associativity is used to decide what should happen if
; two infix operators have the same precedence.
; If both operators are left associative, the parse succeeds.
; If both operators are right associative, the parse succeeds.
; Otherwise, an error is signaled.

; Possible values for infix associativity.
(define-constant left-assoc 2)
(define-constant right-assoc 1)
(define-constant nonassoc 0)

; A token is either an operator or atomic (number, identifier, etc.)
; A stream is a peekable port providing tokens.

(define-operation (peek stream))

(define-structure-type
  operator
  name
  precedence
  associativity
  null-method;; Op has nothing to its left.
  left-method;; Op has something to its left.
  (((print self port)
    (format port "#{Operator ~a}" (operator-name self))))
  )

(define (make-op name prec assoc null-method left-method)
  (let ((op (make-operator)))
    (set (operator-name op) name)
    (set (operator-precedence op) (or prec default-precedence))
    (set (operator-associativity op) (or assoc default-associativity))
    (set (operator-null-method op) (or null-method default-null-method))
    (set (operator-left-method op) (or left-method default-left-method))
    op))

(define (default-null-method operator stream)
  (if (eq? (operator-left-method operator) default-left-method)
      operator
      (parse-error stream
		   "~a is not a prefix operator" operator)))

(define (null-call token stream)
  (if (operator? token)
      ((operator-null-method token) token stream)
      token))

(define default-left-method '#f)

(define (left-call token left stream)
  ((or (and (operator? token)
	    (operator-left-method token))
       (parse-error stream
		    "~a is not an infix operator" token))
   token
   left
   stream))

(define default-precedence 200)

(define (prec token)
  (if (operator? token)
      (operator-precedence token)
      default-precedence))

(define default-associativity nonassoc)

(define (asso token)
  (if (operator? token)
      (operator-associativity token)
      default-associativity))

(define (delim-error token stream)
  (parse-error stream
	       "~a is an invalid use of a delimiter" token))

(define (extra-paren-err token left stream)
  (ignore token left)
  (parse-error stream "too many right parentheses"))

(define (extra-bracket-err token left stream)
  (ignore token left)
  (parse-error stream "too many right brakets"))

(define (extra-brace-err token left stream)
  (ignore token left)
  (parse-error stream "too many right braces"))

(define (premterm-err token stream)
  (ignore token)
  (parse-error stream "premature termination of input"))

; Parse

(define *parse-debug* '#f)

(define (parse prec-level stream)
  (if *parse-debug* (print-it (list 'parse prec-level)))
  (iterate loop ((translation (null-call (read stream) stream)))
	   (if (< prec-level (prec (peek stream)))
	       (loop (left-call (read stream) translation stream))
	       (block (if *parse-debug* (print-it translation))
		      translation))))

(define (print-it s)
  (print s (standard-output))
  (newline (standard-output)))

;;; Null-method's (also see parse-matchfix).

(define (parse-prefix operator stream)
  (list (operator-name operator)
	(parse (prec operator) stream)))

;;; Left-method's.

; Left infix ignoring associativity checks.
(define (parse-infix operator left stream)
  (list (operator-name operator)
	left
	(parse (prec operator) stream)))

; Left infix with associativity checks.
(define (parse-left-infix operator left stream)
  (let* ((left (parse-infix operator left stream))
	 (next-op (peek stream)))
    (cond ((> (prec operator) (prec next-op)) left)
	  ((= left-assoc (asso next-op))
	   (left-call (read stream) left stream))
	  (else
	   (parse-error
	    stream
	    "Ambiguous parse, ~a not a left associative operator"
	    next-op)))))

; Nonassociative infix.
(define (parse-nonassoc-infix operator left stream)
  (let* ((left (parse-infix operator left stream))
	 (next-op (peek stream)))
    (if (> (prec operator) (prec next-op))
	left
	(parse-error
	 stream
	 "Ambiguous parse, ~a not expected"
	 next-op))))

; Right associative infix.
(define (parse-right-infix operator left stream)
  (list (operator-name operator)
	left
	(parse-right (prec operator) stream)))

(define (parse-right prec-level stream)
  (let* ((translation (parse prec-level stream))
	 (next-op (peek stream)))
    (cond ((> prec-level (prec next-op)) translation)
	  ((= right-assoc (asso next-op))
	   (left-call (read stream) translation stream))
	  (else
	   (parse-error
	    stream
	    "Ambiguous parse, ~a not a right associative operator"
	    next-op)))))

; Nary without associativity checks.
(define (parse-nary operator left stream)
  (cons (operator-name operator) (cons left (prsnary operator stream))))

(define (prsnary operator stream)
  (iterate loop ((l (list (parse (prec operator) stream))))
    (if (eq? operator (peek stream))
	(block (read stream)
	       (loop (cons (parse (prec operator) stream) l)))
	(reverse l))))

; Postfix infix-method
(define (parse-postfix operator left stream)
  (ignore stream)
  (list (operator-name operator) left))

; Null-method for parenthesis matching, with internal commas.
; Kind of a kluge if you ask me.  
(define (parse-matchfix operator stream) ; |x|
  (cons (operator-name operator)
	(prsmatch operator stream)))

(define (prsmatch close-op stream)
  (if (eq? (peek stream) close-op)
      (block (read stream) '())
      (iterate loop ((l (list (parse 10 stream))))
	(if (eq? (peek stream) close-op)
	    (block (read stream) (reverse l))
	    (if (eq? (peek stream) comma-operator)
		(block (read stream)
		       (loop (cons (parse 10 stream) l)))
		(parse-error stream
			     "found ~a instead of a comma or ~a"
			     (read stream)
			     close-op))))))

(define comma-operator (make-op 'comma 10 '#f delim-error '#f))

; Parse error support.

(lset *penpenultimate-token* " ")
(lset *penultimate-token* " ")
(lset *ultimate-token* " ")

(define (update-token-history! token)
  (set *penpenultimate-token*  *penultimate-token*)
  (set *penultimate-token* *ultimate-token*)
  (set *ultimate-token* token))

(define (init-token-history!)
  (set *penpenultimate-token* " ")
  (set *penultimate-token* " ")
  (set *ultimate-token* " "))

(define (parse-error stream format-string . args)
  (format (error-output) "~&~a ~a ==> ~a~%"
	  *penpenultimate-token* *penultimate-token* *ultimate-token*)
  (apply read-error stream format-string args))

; Lexer support:

(define-structure-type
  lexer
  rtab					; Read table
  punctab				; Punctuation table
  keytab)				; Keyword table

(define (make-lexer-table)
  (let ((ltab (make-lexer)))
    (set (lexer-rtab ltab) (make-lexer-read-table))
    (set (lexer-punctab ltab) (make-table))
    (set (lexer-keytab ltab) (make-table))
    ltab))

(define (make-lexer-read-table)
  (make-read-table standard-read-table
		   'operator-precedence-read-table))
				    
(define (lex ltab port)
  (let ((thing (read-object port (lexer-rtab ltab))))
    (update-token-history! thing)
    (cond ((eof? thing)
	   end-of-input-operator)
	  ((symbol? thing)
	   (or (table-entry (lexer-keytab ltab) thing)
	       thing))
	  (else thing))))

(define (set-char-tokenization! rtab char reader term?)
  (set (read-table-entry rtab char)
       (if term?
	   (object reader ((delimiting-read-macro? self) '#t))
	   reader)))

; Keywords

(define (define-keyword ltab name op)
  (set (table-entry (lexer-keytab ltab) name) op))

; Punctuation
;
; Discrimination tree:
; lexnode = (* operator (table-of char (+ lexnode '#f)))
;

(define (define-punctuation ltab string op)
  (let ((end (- (string-length string) 1)))
    (iterate loop ((i 0)
		   (table (lexer-punctab ltab)))
      (let* ((c (string-elt string i))
	     (lexnode
	      (or (table-entry table c)
		  (let ((lexnode
			 (cons (error-operator (substring string 0 (+ i 1)))
			       (make-table))))
		    (set (table-entry table c) lexnode)
		    (if (= i 0)
			(set-char-tokenization! (lexer-rtab ltab)
						c
						(operator-reader lexnode)
						'#t))
		    lexnode))))
	(if (>= i end)
	    (set (car lexnode) op)
	    (loop (+ i 1) (cdr lexnode)))))))

(define (operator-reader lexnode)
  (lambda (port c rtab)
    (ignore c rtab)
    (iterate loop ((lexnode lexnode))
      (let ((nextc (peek-char port)))
	(let ((nextnode (table-entry (cdr lexnode) nextc)))
	  (if nextnode
	      (block (read-char port)
		     (loop nextnode))
	      (car lexnode)))))))

(define (error-operator string)
  (make-op 'invalid-operator '#f '#f
	   (lambda rest
	     (ignore rest)
	     (error "invalid operator ~s" string))
	   '#f))

(define end-of-input-operator
  (make-op "end of input" -1 '#f premterm-err '#f))

(define (port->stream port ltab)
  (let ((peeked? '#f)
	(buffer '())
	(really-read
	 (lambda ()
	   (lex ltab port))))
    (join
      (object '()
	((read self)
	 (if peeked?
	     (block
	       (set peeked? '#f)
	       buffer)
	     (really-read)))
	((peek self)
	 (if peeked?
	     buffer
	     (block
	       (set peeked? '#t)
	       (set buffer (really-read))
	       buffer))))
      port)))

(define (toplevel-parse stream)
  (init-token-history!)
  (if (eq? end-of-input-operator (peek stream))
      (block (read stream) eof)
      (parse -1 stream)))
