;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald (inferences context-sequent))


; 
;
; Contexts are little collections of assumptions to be used by sequents, 
; rules of inference, deduction graphs
;
; Sequents (defined below) have a context and an assertion.

;  Note:  The free variables of a context are all those occurring free in its
;  *assumptions*.  These are interpreted existentially because contexts are
;  essentially used as the left hand sides of sequents.  There may be other
;  free variables in the *entries* of the context, but these arise only when
;  universal quantifiers are stripped.  Hence they are interpreted universally.




; The interface to contexts consists of the following  items:
; 1. context?
; 2. context-assumptions
; 3. (empty-context) (= (the-null-context))
; 4. context-omit-assumption
; 5. context-add-assumption
; 6. context-add-assumptions
; 7. context-absurd?
;
; Context entailment and theory-context also make use of some internal items.  

(define (CONTEXT-CONTAINS-ENTRY? context entry)
  (or (q-same-class? (context-q-classes context) entry truth)
      (let ((c (context-predecessor context)))
	(and c
	     (let ((val (context-contains-entry? c entry)))
	       (if val
		   (install-context-entry context entry))
	       val)))
      (any						;check if predecessors know equality 
       (lambda (a)					;with new assumption
	 (context-exprs-equal? context entry a))
       (context-assumptions context))))

(define (CONTEXT-PASSIVELY-CONTAINS-ENTRY? context entry)
  (q-same-class? (context-q-classes context) entry truth))

(define (CONTEXT-ABSURD? c)
  (context-absurd-flag c))

(define (CONTEXT-ADD-BOUND-VARIABLES c n-bvs)
  (set (context-variables c)
       (set-union n-bvs (context-variables c)))
  (set (context-bound-variables c)
       (set-union n-bvs (context-bound-variables c))))

(define (EQUATE-TO-ALPHA-ROOT context expr)
  (let ((root (expression-alpha-root expr)))
    (or (not (expression? root))
	(eq? expr root)
	(q-equate-exprs
	 (context-q-classes context) (context-q-hashes context)
	 expr root))))

(define (CONTEXT-QUASI-EQUATE-EXPRS context expr1 expr2)
  (context-add-bound-variables
   context
   (set-union
    (bound-variables expr1)
    (bound-variables expr2)))
  (equate-to-alpha-root context expr1)
  (if (not (eq? expr1 expr2))
      (block
	(equate-to-alpha-root context expr2)
	(q-equate-exprs (context-q-classes context)
			(context-q-hashes context)
			expr1 expr2)))
  (let ((root (context-q-root context expr1))
	(theory (context-theory context)))
    (if (or (theory-constants-distinct? theory expr1 expr2)
	    (theory-constants-distinct? theory root expr2)
	    (theory-constants-distinct? theory expr1 root))
	(make-context-absurd context))))

(define (CONTEXT-EQUATE-EXPRS context expr1 expr2)
  (context-quasi-equate-exprs context expr1 expr2)
  (q-make-expr-defined  (context-q-classes context) (context-q-hashes context) expr1)
  (add-context-term-sort context expr1 (expression-sorting expr1))
  (add-context-term-sort context expr2 (expression-sorting expr2))
  (let ((term-sorts (set-union (context-term-defined-sorts context expr1)
			       (context-term-defined-sorts context expr2))))
    (set (table-entry (context-term-sorts context) expr1) term-sorts)
    (set (table-entry (context-term-sorts context) expr2) term-sorts)))

(define (CONTEXT-Q-ROOT-IF-ANY context expr)
  (q-seek-root (context-q-classes context) expr))

(define (CONTEXT-Q-ROOT context expr)
  (let ((q-root (q-seek-root (context-q-classes context) expr)))
    (or q-root expr)))

(define (CONTEXT-WALK-Q-CLASSES proc c)
  (walk-table
   proc
   (context-q-classes c)))

(define (CONTEXT-WALK-ENTRIES 1-proc c)
  (let ((classes (context-q-classes c)))
    (context-walk-q-classes
     (lambda (e q)
       (ignore q)
       (if (q-same-class? classes e truth)
	   (1-proc e)))
     c)))

(define (context-count-q-population context)
  (let ((total 0))
    (context-walk-q-classes
     (lambda (e q)
       (ignore e q)
       (increment total))
     context)
    total))


(define (CONTEXT-PRINT-ENTRIES c)
  (context-walk-entries (lambda (e)(format '#t "~S~%~%" e)) c))

(define (CONTEXT-PRINT-Q-MEMBERS c)
  (context-walk-q-classes
   (lambda (e q)
     (format '#t "~&Entry: ~S~&Class: ~S~&" e q))
   c))
	 

(define (INSTALL-CONTEXT-ENTRY c new-entry)
  (context-equate-exprs c new-entry truth)
;;  (update-context-disjunctive-components c new-entry)
  (return))

;; (define (update-context-disjunctive-components c new-entry)
;;   (if (disjunctive-formula? new-entry)
;;       (let* ((d-cs (disjunctive-components new-entry))
;; 	     (normalized-entry (disjunction-simplifier d-cs))
;; 	     (q-hashes (context-entry-disjuncts c)))
;; 	(walk
;; 	 (lambda (d-c)
;; 	   (q-install-hash-datum q-hashes d-c normalized-entry))
;; 	 d-cs))))

;; (define (walk-possible-matching-disjunctions action-proc c expr)
;;   (q-hashes-walk-possible-matchers action-proc (context-entry-disjuncts c) expr))
  

;; Hopefully this is defunct!
;; 
;; ;;; If selection-proc is a predicate of one argument, and action-proc is a
;; ;;; procedure of two argument expecting a disjunctive-component and a list of
;; ;;; entries, then
;; ;;;
;; ;;; (selectively-walk-disjunctive-components c selection-proc action-proc)
;; ;;; 
;; ;;; applies action-proc to those disjunctive-components/entry-lists for which
;; ;;; selection-proc returns true.  
;; 
;; (define (selectively-walk-disjunctive-components c selection-proc action-proc)
;; ;;
;; ;; let ((context-entry-disjuncts-pairs '()))
;; ;;  
;;     (walk-table
;;      (lambda (d-c entry-list)
;;        (if (selection-proc d-c)
;; 	   (action-proc d-c entry-list)))
;;      (context-entry-disjuncts c)))
;; 
;;     ;; This rigamarole is no longer necessary to avoid changing a table while
;;     ;; walking through it, because of the advent of ribust tables.  (I hope.)
;;     ;; guttman
;;     ;; 
;;     ;;
;;     ;; (walk
;;     ;;  (lambda (pair)
;;     ;;        (let ((d-c (car pair))
;;     ;; 	     (entry-list (cdr pair)))
;;     ;; 	 (action-proc d-c entry-list)))
;;     ;;  context-entry-disjuncts-pairs)

;;(define (CONTEXT-AGE-ENTRIES c)
;;  (set (context-new-entries c) '()))

(define (MAKE-CONTEXT-ABSURD c)
  (set (context-absurd-flag c) '#t))

(define (CONTEXT-THEORY context)
  (let ((th (context-theory-slot context)))
    (if (theory? th)
	th
	the-kernel-theory)))

(define (CONTEXT-TRANSFORM-HANDLER context)
  (theory-transform-handler (context-theory context)))

(define (ADD-CONTEXT-EQUATION c lhs rhs)
  (context-equate-exprs c lhs rhs))

(define (ADD-CONTEXT-QUASI-EQUATION c lhs rhs)
  (context-quasi-equate-exprs c lhs rhs))

(define (ADD-CONTEXT-CONVERGENCE c term)
  (add-context-term-sort c term (expression-sorting term))
  (if (not (necessarily-defined? term))
      (context-equate-exprs c term term)
      '#t))

(define (CONTEXT-EXPRS-EQUAL? context e1 e2)
  (or (and (eq? e1 e2)
	   (context-expr-defined? context e1))
      (let ((q-classes (context-q-classes context)))
	(and (q-same-class? q-classes e1 e2)
	     (q-seek-defined? q-classes e1)))
      (let ((c (context-predecessor context)))
	(and c
	     (let ((val (context-exprs-equal? c e1 e2)))
	       (if val
		   (add-context-equation context e1 e2))
	       val)))))

(define (CONTEXT-EXPRS-QUASI-EQUAL? context e1 e2)
  (or (eq? e1 e2)
      (let ((q-classes (context-q-classes context)))
	(q-same-class? q-classes e1 e2))
      (let ((c (context-predecessor context)))
	(and c
	     (let ((val (context-exprs-QUASI-equal? c e1 e2)))
	       (if val
		   (add-context-QUASI-equation context e1 e2))
	       val)))))

(define (CONTEXT-EXPR-DEFINED? context e1)
  (or (necessarily-defined? e1)
      (q-seek-defined? (context-q-classes context) e1)
      (let ((c (context-predecessor context)))
	(and c
	     (let ((val (context-expr-defined? c e1)))
	       (if val
		   (add-context-convergence context e1))
	       val)))))

(define (CONTEXT-WALK-POSSIBLE-MATCHERS proc context expr)
  (q-hashes-walk-possible-matchers proc (context-q-hashes context) expr))    

(define (CONTEXT-WALK-POSSIBLE-MATCHING-ENTRIES proc context expr)
  (let ((classes (context-q-classes context)))
    (context-walk-possible-matchers
     (lambda (e)
       (if (q-same-class? classes e truth)
	   (proc e)))
     context
     expr)))

(define (CONTEXT-WALK-MATCHES-TO-CONSTRUCTOR-AND-FIRST-LEAD proc context expr)
  (let ((constr (expression-constructor expr))
	(lead   (expression-lead-constant expr)))
  (q-walk-matches-to-constructor-and-first-lead
   (context-q-hashes context)
   proc
   constr
   lead)))

(define (CONTEXT-WALK-ENTRIES-CONSTRUCTOR-AND-FIRST-LEAD proc context constr lead)
  (let ((classes (context-q-classes context)))
    (q-walk-matches-to-constructor-and-first-lead
     (context-q-hashes context)
     (lambda (e)
       (if (q-same-class? classes e truth)
	   (proc e)))
     constr
     lead)))

(define (context-any-matcher? pred context expr)
  (catch found
    (context-walk-possible-matchers
     (lambda (e)
       (if (pred e)
	   (found '#t)))
     context expr)
    '#f))

(define (context-any-matching-entry? pred context expr)
  (let ((classes (context-q-classes context)))
    (context-any-matcher?
     (lambda (e)
       (and (q-same-class? classes e truth)
	    (pred e)))
     context expr)))

(define (CONTEXT-WALK-EQUALITIES proc context)
  (let ((classes (context-q-classes context)))
    (q-walk-matches-to-constructor
     (context-q-hashes context)
     (lambda (e)
       (if (q-same-class? classes e truth)
	   (proc e)))
     equality))) 

(define (CONTEXT-WALK-NEGATED-EQUALITIES-OR-QUASI-EQUALITIES proc context)
  (let ((classes (context-q-classes context)))
    (q-walk-matches-to-constructor
     (context-q-hashes context)
     (lambda (e)
       (if (and (or (equation? (car (expression-components e)))
		    (quasi-equation? (car (expression-components e))))
		(q-same-class? classes e truth))
	   (proc e)))
     negation))) 

(define PRINT-CONTEXTS-VERBOSELY?
  (make-simple-switch 'print-contexts-verbosely? boolean? '#f)) 

(define (ADD-CONTEXT-ENTRIES context new-entries)
  (walk
   (lambda (new-entry)
     (add-context-entry context new-entry))
   new-entries)
  context)

(define (ADD-CONTEXT-ENTRY context new-entry)
  (or (truth? new-entry)
      (context-passively-contains-entry? context new-entry)
      (walk
       (lambda (formula)
	 (add-context-entry-do-addition context formula))
       (cons
	new-entry
	(append
	 (reduce-conjunctions-and-universals
	  (list new-entry)
	  (context-variables context))
	 (let ((conjunctive-component-lists
		(map
		 (lambda (d-c)
		   (conjunctive-components d-c))
		 (disjunctive-components new-entry))))
	   (if (not (null? conjunctive-component-lists))
	       (big-cap conjunctive-component-lists)
	       '()))))))
  context)

(define do-debug?
    (make-simple-switch 'do-debug? boolean? '#f))

(define (ADD-CONTEXT-TERM-SORT context term sort)
  (set (table-entry (context-term-sorts context) term)
       (add-set-element
	sort 
	(table-entry (context-term-sorts context) term)))
  (and (function? term)
       (term-sort-trigger context term sort))
  (return))

(define (term-sort-trigger context op sort)
  (q-walk-matches-to-constructor
   (context-q-hashes context)
   (lambda (app)
     (and (eq? (operator app) op)
	  (walk
	   (lambda (arg arg-sort)
	     (add-context-term-sort context arg arg-sort))
	   (arguments app)
	   (higher-sort-domains sort))))
    apply-operator))


(define (CONTEXT-TERM-DEFINED-SORTS context term)
  (table-entry (context-term-sorts context) term))

(define (ADD-CONTEXT-ENTRY-DO-ADDITION context new-entry)
  (context-add-bound-variables context
			       (bound-variables new-entry))
  (install-context-entry context			;add it, then check 
			 new-entry)			;special cases
  (cond ((context-new-entry-absurd? context new-entry)	;absurdity
	 (make-context-absurd context))
	((quasi-equation? new-entry)
	 (add-context-quasi-equation context
				     (quasi-equation-lhs new-entry)
				     (quasi-equation-rhs new-entry)))
	(else 
	 (select (expression-constructor new-entry)
	   ((equality)
	    (add-context-entry-add-equation context new-entry))
	   ((apply-operator)	
	    (add-context-entry-add-application context new-entry))
	   ((is-defined)
	    (or (necessarily-defined? (convergence-term new-entry))
		(add-context-entry-add-convergence context new-entry)))
	   ((is-defined-in-sort)			
	    (or (necessarily-defined-in-sort? (convergence-term new-entry)
					      (convergence-sort new-entry))
		(add-context-entry-add-sort-convergence context new-entry)))
	   
	   ((negation)					;not
	    (let ((flush (flush-not new-entry)))
	      (or (eq? new-entry flush)
		  (add-context-entry context flush))))

	   ((biconditional)				;biconditional
	    (let ((lhs (expression-lhs new-entry))
		  (rhs (expression-rhs new-entry)))
	      (add-context-equation context lhs rhs)
	      (install-context-entry context (implication lhs rhs))
	      (install-context-entry context (implication rhs lhs))))

	   ((if-form)
	    (let ((comps (expression-components new-entry)))
	      (let ((test    (nth comps 0))
		    (conseq  (nth comps 1))
		    (alt	  (nth comps 2)))
		(install-context-entry
		 context
		 (implication test conseq))
		(install-context-entry
		 context
		 (implication (push-not test) alt))))))))

  (return))

(define (add-context-entry-add-equation context new-entry)
  (let ((lhs (expression-lhs new-entry))
	(rhs (expression-rhs new-entry)))
    (let ((lhs-sort (expression-sorting lhs))
	  (rhs-sort (expression-sorting rhs)))
      (let ((preferred-sort (or (and (sorting-leq lhs-sort rhs-sort)
				     lhs-sort)
				(and (sorting-leq rhs-sort lhs-sort)
				     rhs-sort))))
	(add-context-equation context lhs rhs)
	(walk (lambda (cnv)
		(add-context-entry context cnv))
	      (immediately-consequent-convergences lhs (or preferred-sort lhs-sort)))
	(walk (lambda (cnv)
		(add-context-entry context cnv))
	      (immediately-consequent-convergences rhs (or preferred-sort rhs-sort)))))))
					
(define (add-context-entry-add-application context new-entry)
  (walk
   (lambda (term sort)
     (if (term-or-fn? term)
	 (add-context-entry context (defined-in term sort))))
   (arguments new-entry)
   (higher-sort-domains
    (expression-sorting (operator new-entry))))
  (if (lambda-application? new-entry)
      (add-context-entry context (beta-reduce-recklessly new-entry))))

(define (add-context-entry-add-convergence context new-entry)
  (let ((term
	 (car (expression-components new-entry))))
    (add-context-convergence context term)
    (cond
     ((application? term)
      (let ((known-sorts (cons (expression-sorting (operator term))
			       (context-term-defined-sorts context (operator term)))))
	(walk
	 (lambda (known)
	   (walk
	    (lambda (arg expected-sort)
	      (add-context-entry context (defined-in arg expected-sort)))
	    (arguments term)
	    (higher-sort-domains known)))
	 known-sorts)))
     ((conditional-term? term)
      (if (necessarily-undefined? (conditional-alternative term))
	  (add-context-entry context (conditional-test term)))
      (if (necessarily-undefined? (conditional-consequent term))
	  (add-context-entry context (push-not (conditional-test term))))))
    (walk (lambda (cnv)
	    (add-context-entry context cnv))
	  (immediately-consequent-convergences term (expression-sorting term)))))


(define (add-context-entry-add-sort-convergence context new-entry)
  (let ((term
	 (car (expression-components new-entry)))
	(sort
	 (expression-sorting
	  (cadr (expression-components new-entry)))))
    (add-context-term-sort context term sort)
    (add-context-entry context (is-defined term))
    (walk (lambda (cnv)
	    (add-context-entry context cnv))
	  (immediately-consequent-convergences term sort))))


(define (CONTEXT-NEW-ENTRY-ABSURD? context new-entry)
  (and
   (not (truth? new-entry))
   (or (context-absurd? context)
       (context-contains-entry? context falsehood)
       (falsehood? new-entry)
       (let ((neg-entry (push-not new-entry)))
	 (or (context-contains-entry? context neg-entry)
	     (context-contains-entry? context (expression-alpha-root neg-entry))
	     (mem? alpha-equivalent? neg-entry (context-assumptions context))))
       (and (negated-equation? new-entry)
	    (let* ((equation (car (expression-components new-entry)))
		   (lhs (expression-lhs equation))
		   (rhs (expression-rhs equation)))
	      (context-exprs-equal? context lhs rhs)))
       (and (negated-convergence? new-entry)
	    (let ((term (car (expression-components
			      (car (expression-components new-entry))))))
	      (or (necessarily-defined? term)
		  (context-expr-defined? context term))))
       (and (or (convergence? new-entry)
		(convergence-in-sort? new-entry))
	    (necessarily-undefined? (car (expression-components new-entry))))
       (let ((simp (partially-simplified-form context new-entry)))
	 (and (not (eq? new-entry simp))
	      (< (expression-height simp)
		 (expression-height new-entry))
	      (context-new-entry-absurd? context simp)))
       (let ((root (context-q-root context new-entry)))
	 (and root
	      (< (expression-height root)
		 (expression-height new-entry))
	      (context-new-entry-absurd? context root))))))

(define-structure-type CONTEXT
  
  theory-slot						;containing theory
  assumptions						;full logical content
  q-classes						;table containing eq-classes
  q-hashes						;table of hashes for retrieval 
  partially-simplified					;table of partially simplified forms 
;;;  entry-disjuncts					;table of disjuncts of entries
;;;							;and the entries they belong to
  term-sorts						;table of terms and sorts they are
							;known to be defined in 
  free-variables					;of all assumptions
  bound-variables					;of all entries
  variables						;all of em
  assumption-variables					;variables free and bound in 
							;assumptions only, in that order
  absurd-flag						;'#t if context recognizably
							;inconsistent
  predecessor						;the context this was made by
							;adding assumptions to (if any)
  assumptions-already-simplified?			
  sequents
  
  (((print self port)
    (if (print-contexts-verbosely?)
	(format port "#{IMPS-context ~S~%~A}"
		(object-hash self)
		(assumptions->string assumptions))
	(format port "#{IMPS-context ~S}"
		(object-hash self))))))


(block
  (set (context-theory-slot       (stype-master context-stype)) '#f)
  (set (context-free-variables    (stype-master context-stype)) nil)
  (set (context-bound-variables   (stype-master context-stype)) nil)
  (set (context-variables         (stype-master context-stype)) nil)
  (set (context-absurd-flag       (stype-master context-stype)) '#f)
  (set (context-assumptions-already-simplified?
				  (stype-master context-stype)) '#f)
  (set (context-sequents 	  (stype-master context-stype)) '()))

(define theory-context? context?)

(define-operation sensor)

(define PARTIALLY-SIMPLIFIED-FORM
  (object
      (lambda (context expr)
	(or (table-entry (context-partially-simplified context) expr)
	    (let ((c (context-predecessor context)))
	      (if c
		  (partially-simplified-form c expr)
		  expr))))
    ((setter self)
     (lambda (context expr new-value)
       (cond ((and (truth? new-value)			;add entry?
		   (not (context-contains-entry? context expr)))
	      (add-context-entry context expr)
	      (set (table-entry (context-partially-simplified context)
				expr)
		   new-value))
	     (else
	      (set (table-entry (context-partially-simplified context)
				expr)
		   new-value)))))
    ((sensor self)
     (lambda (context expr)
       (expression? (table-entry (context-partially-simplified context)
				 expr))))))

(define (TRANSITIVE-PARTIALLY-SIMPLIFIED-FORM context expr)
  (let ((simp1 (partially-simplified-form context expr)))
    (if (alpha-equivalent? simp1 expr)
	expr
	(let ((simp2 (transitive-partially-simplified-form context simp1)))
	  (if (not (alpha-equivalent? simp2 simp1))
	      (set (partially-simplified-form context expr) simp2))
	  simp2))))
	  
(lset  *theory-context-table* (settable-alist theory? set-table?))

(define (CLEAR-CONTEXTS)
  (set q-access-total 0)
  (set *theory-context-table* (settable-alist theory? set-table?))
  (return))

(define (max-index theory)
  (let ((table (*theory-context-table* theory))
	(assumptions '()))
    (walk-table
     (lambda (k v)
       (ignore v)
       (set assumptions (set-union k assumptions)))
     table)
    (length (make-set assumptions))))

(define (context-predecessor-chain-length context)
  (let ((pred (context-predecessor context)))
    (if pred (1+ (context-predecessor-chain-length pred)) 1)))

(define (context-subcontext-count context)
  (let ((as (context-assumptions context))
	(table (*theory-context-table* (context-theory context)))
	(count 0))
    (walk-table
     (lambda (asses context)
       (if (subset? asses as)(increment count)))
     table)
    count))
    
(define (context-predecessors&subsets context)
  (return (context-predecessor-chain-length context)
	  (context-subcontext-count context)))

(define (RETRIEVE-CONTEXT theory assumptions)
  (table-entry
   (or (*theory-context-table* theory)
       (block (set (*theory-context-table* theory)
		   (make-set-table (theory-name theory)))
	      (*theory-context-table* theory)))
   assumptions))

(define (THEORY-NULL-CONTEXT theory)
  (or
   (retrieve-context theory nil)
   (let ((c (make-context)))
     (set (context-theory-slot 	         c) theory)
     (set (context-assumptions		 c) nil)
;;     (set (context-entry-disjuncts	 c) (make-table 'entry-disjuncts))
     (set (context-q-classes		 c) (make-table 'q-classes))
     (set (context-q-hashes		 c) (make-two-d-table 'q-hashes))
     (set (context-partially-simplified  c) (make-table 'partially-simplified))
     (set (context-term-sorts		 c) (make-table 'term-sorts))
     (set (context-assumption-variables  c) '())
     (set (context-predecessor		 c) '#f)
     (set (table-entry
	   (*theory-context-table* theory) nil)	 c)
     c)))

(define (EMPTY-CONTEXT) (theory-null-context the-kernel-theory))
(define THE-NULL-CONTEXT EMPTY-CONTEXT)
(define THEORY-EMPTY-CONTEXT theory-null-context)

(define (EMPTY-CONTEXT? context)
  (null? (context-assumptions context)))

(define (CONTEXT-ADD-ASSUMPTIONS context new-assumptions)
  (let ((new-assumptions (delete-set-element truth new-assumptions)))
    (let ((old (context-assumptions context)))
      (receive (really-new all)
	(iterate iter ((really-new  nil)
		       (all	  old)
		       (rest	  new-assumptions))       
	  (cond ((null? rest) (return really-new all))
		((mem? alpha-equivalent? (car rest) all) 
		 (iter really-new
		       all
		       (cdr rest)))
		((context-contains-entry? context (car rest))
		 (iter really-new			;not really new, but 
		       (cons (car rest) all)		;another assumption
		       (cdr rest)))
		(else (iter (cons (car rest) really-new)
			    (cons (car rest) all)
			    (cdr rest)))))
	(let ((theory (context-theory context)))
	  (or
	   (retrieve-context theory all)
	   (let ((c (make-context)))
	     (receive (classes hashes)
	       (initialize-q-tables 'context)
	       (set (context-theory-slot 	c) (context-theory context))
	       (set (context-assumptions 	c) all)
;;	       (set (context-entry-disjuncts	c) (make-table 'entry-disjuncts))
	       (set (context-q-hashes		c) hashes)
	       (set (context-q-classes		c) classes)
	       (set (context-partially-simplified
		     c) (make-table 'partially-simplified))
	       (set (context-term-sorts	c) (make-table 'term-sorts))
	       (set (context-predecessor	c) context)
	       (set (context-free-variables    c) (collect-set free-variables all))
	       (set (context-bound-variables   c) (set-union
						   (context-bound-variables context)
						   (collect-set bound-variables
								really-new)))
	       (set (context-variables		   c) (set-union
						       (context-free-variables    c)
						       (context-bound-variables   c)))
	       (set (context-assumption-variables c)
		    (append (collect-set free-variables all)
			    (collect-set bound-variables all)))
	       (add-context-entries c all)
	       (set (table-entry (*theory-context-table* theory) all) c)
	       c)))))))) 

(define (CONTEXT-ADD-ASSUMPTION context new-assumption)
 (context-add-assumptions context (list new-assumption)))

(define (BUILD-CONTEXT theory assumptions)
  (receive (basis additions)
    (iterate iter ((active assumptions)
		   (inactive nil))
      (cond ((null? active)
	     (return (theory-null-context theory)
		     (reverse! inactive)))
	    ((retrieve-context theory active)
	     =>
	     (lambda (c)
	       (return c (reverse! inactive))))
	    (else
	     (iter (cdr active)
		   (cons (car active) inactive)))))
    (if (null? additions)
	basis
	(context-add-assumptions basis additions))))

(define (CONTEXT->THEORY-CONTEXT theory context)
  (if (eq? theory (context-theory-slot context))
      context
      (let ((asses (context-assumptions context)))
	(build-context theory asses))))

(define (CONTEXT-OMIT-ASSUMPTIONS context omissions)
  (if (null? omissions) context
      (build-context
       (context-theory context)
       (set-difference (context-assumptions context) omissions))))

(define (CONTEXT-OMIT-ASSUMPTION context omission)
  (context-omit-assumptions context (list omission)))

(define (REBINDING-FILTER-CONTEXT formula context)
  (let ((nbvs (newly-bound-variables formula)))
    (let ((suspicious-variables?
	   (lambda (exp-or-context)
	     (non-null-intersection?
	      (free-variables exp-or-context)
	      nbvs))))
      (if (and (binding-expression? formula)
	       (suspicious-variables? context))
	  (context-omit-assumptions
	   context
	   (set-separate				;select bad assumptions 
	    (lambda (assumption)
	      (suspicious-variables? assumption))
	    (context-assumptions context)))
	  context))))

(define REQUANTIFICATION-FILTER-CONTEXT rebinding-filter-context)

(define (REQUANTIFICATION-FILTER-CONTEXT-AND-USING formula context using)
  (let ((suspicious-variables? (lambda (exp-or-context)
				 (non-null-intersection?
				  (free-variables exp-or-context)
				  (newly-bound-variables formula)))))
    (if (not (binding-expression? formula))
	(return context using)
	(return						;context and using 
	 (context-omit-assumptions			;omitting bad assumptions 
	  context
	  (set-separate					
	   (lambda (assumption)
	     (suspicious-variables? assumption))
	   (context-assumptions context)))
	 (set-separate					;omitting bad used formulas 
	  (lambda (used)
	    (not (suspicious-variables? used)))
	  using)))))

(define (REQUANTIFICATION? formula context . using)
  (let ((new-vars (newly-bound-variables formula)))
    (or (not (null-intersection? new-vars (context-free-variables context)))
	(any? 
	 (lambda (f) (not (null-intersection? new-vars (free-variables f))))
	 using))))

(define (REQUANTIFICATION-ON-PATH? formula context path)
  (if (null? path)
      '#f
      (or (requantification? formula context)
	  (receive (new-formula new-path)
	    (host-and-path-after-step formula path)
	    (requantification-on-path? new-formula context new-path)))))
	 
(define (CONTEXT-AND-ADDITIONS-AT-PATH context host path using)
  (if (null? path)
      (return context using)
      (receive (new-context new-using)
	(requantification-filter-context-and-using host context using)
	(receive (new-host new-path)
	  (host-and-path-after-step host path)
	  (context-and-additions-at-path
	   new-context new-host new-path
	   (set-union new-using
		      ((constructor-lc-incrementer (expression-constructor host))
		       (expression-components host)
		       (car path))))))))

(define (LOCAL-CONTEXT-AT-PATH context host path)
  (receive (new-context additions)
    (context-and-additions-at-path context host path nil)
    (context-add-assumptions new-context additions)))

(define (LOCAL-CONTEXT-AT-VIRTUAL-PATH context host path)
  (let* ((paths (expand-virtual-path host path))
	 (contexts (map (lambda (path) (local-context-at-path context host path)) paths)))
    (if contexts
	(build-context
	 (context-theory context)
	 (big-cap (map context-assumptions contexts)))
	(theory-null-context (context-theory context)))))

; A SEQUENT consists of a context and an assertion 

(define-structure-type SEQUENT
  context
  assertion
  free-variables
  bound-variables
  entailment-flag					;'#t if context known to entail assertion 

  (((print self port)
    (format port "#{IMPS-sequent ~S: ~A -> ~A}"
	    (object-hash self)
	    (sequent-context self)
	    (sequent-assertion self)))))

(set (sequent-entailment-flag (stype-master sequent-stype)) '#f)

;; (lset *sequent-table* (make-table '*sequent-table*))


(define (retrieve-sequent context assertion)
  (let ((entries (context-sequents context)))
    (any (lambda (seq)
	   (and (eq? (sequent-assertion seq) assertion)
		seq))
	 entries)))


; The interface to sequents consists of the following items.  

(define (BUILD-SEQUENT context assertion)
  (or (formula? assertion)
      (imps-error "BUILD-SEQUENT: ~A is a non-formula." assertion))
  (or
   (retrieve-sequent context assertion)
   (let ((seq (make-sequent)))
     (set (sequent-context seq) context)
     (set (sequent-assertion seq) assertion)
     (set (sequent-free-variables seq)
	  (set-union (free-variables context)
		     (free-variables assertion)))
     (set (sequent-bound-variables seq)
	  (set-union (bound-variables context)
		     (bound-variables assertion)))
     (push (context-sequents context) seq)
     (if (context-trivially-entails? context assertion)
	 (set (sequent-entailment-flag seq) '#t))
     seq)))

(define (MAKE-SEQUENT-ENTAILED sequent)
  (set (sequent-entailment-flag sequent) '#t)
  (add-context-entry (sequent-context sequent) (sequent-assertion sequent)))

(define (ASSERTION->SEQUENT assertion)
  (build-sequent (the-null-context) assertion))

(define (THEORY-ASSERTION->SEQUENT theory assertion)
  (build-sequent (theory-null-context theory) assertion))

(define (SEQUENT-ASSUMPTIONS seq)
  (context-assumptions
   (sequent-context seq)))

(define (SEQUENT-ADD-ASSUMPTION sequent assumption)
  (build-sequent
   (context-add-assumption (sequent-context sequent) assumption)
   (sequent-assertion sequent)))

(define (SEQUENT-CHANGE-FREE-VARIABLES sequent avoid-vars)
  (let ((avoid-vars (set-intersection avoid-vars (free-variables sequent)))
	(assumptions (sequent-assumptions sequent))
	(assertion (sequent-assertion sequent))
	(theory (context-theory (sequent-context sequent))))
    (build-sequent
     (build-context
      theory 
      (map
       (lambda (a)
	 (expression-change-free-variables a avoid-vars))
       assumptions))
     (expression-change-free-variables assertion avoid-vars))))

(define (SEQUENT-THEORY sequent)
  (context-theory (sequent-context sequent)))

(define (SEQUENT->THEORY-SEQUENT theory sequent)
  (build-sequent (context->theory-context theory (sequent-context sequent))
		 (sequent-assertion sequent)))

(define (sequent->sentence sequent)
  (if (empty-context? (sequent-context sequent))
      (universal-closure (sequent-assertion sequent))
      (let ((hyps (sequent-assumptions sequent)))
	(universal-closure
	 (implication
	  (conjunction-simplifier hyps)
	  (sequent-assertion sequent))))))

(define (sequents-alpha-equivalent? seq1 seq2)
  (and (alpha-equivalent? (sequent-assertion seq1)
			  (sequent-assertion seq2))
       (let ((a2s (sequent-assumptions seq2)))
	 (every?
	  (lambda (a1)
	    (any? (lambda (a2)
		    (alpha-equivalent? a1 a2))
		  a2s))
	(sequent-assumptions seq1)))
       (let ((a1s (sequent-assumptions seq1)))
	 (every?
	  (lambda (a2)
	    (any? (lambda (a1)
		    (alpha-equivalent? a2 a1))
		  a1s))
	  (sequent-assumptions seq2)))))

(define (context-sequent-max-&-average)
  (let ((max-sequents 0)
	(number-of-contexts 0)
	(number-of-sequents 0))
    (walk-table
     (lambda (context subtable)
       (ignore context)
       (increment number-of-contexts)
       (let ((num 0))
	 (walk-table
	  (lambda (assertion sequent)
	    (ignore assertion sequent)
	    (increment num))
	  subtable)
	 (set max-sequents (max max-sequents num))
	 (set number-of-sequents (+ max-sequents num))))       
     *sequent-table* )
    (return
     max-sequents
     (->float (/ number-of-sequents number-of-contexts)))))
