/* $Header: /usr/src/local/imps/src/xdg/RCS/xdg.c,v 1.11 90/11/15 09:22:48 ramsdell Exp $ */

/* Deduction graph displayer for X11 */

/*
   Responds to commands given to stdin.
   See dispatch for the commands handled.
*/

#include <stdio.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>	/* Get standard string definitions. */

#include <X11/Shell.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Viewport.h>	
#include <X11/Xaw/Scrollbar.h>	
#include "Tree.h"
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Label.h>	
#include <X11/Xaw/Cardinals.h>

#include <string.h>
#include "error.h"
#include "store.h"
#include "parse.h"

String fallback_resources[] = { 
    "Xdg.geometry: 600x900",
    "*quit.label: Quit  (q)",
    "*layout.label: Use tree layout  (l)",
    "Xdg*translations: #augment \\n :<Key>q : quit()\\n\
                                    :<Key>l : toggle_layout()",
    NULL,
};

/* Global state */
static Boolean using_trees = False;
static int dg_number = 1;
static data current_dg = NIL;	/* Current displayed deduction graph. */
static data next_dg = NIL;	/* Next one to be displayed. */

#if defined DEBUG
static void
  tell_me_about (w)
    Widget w;
{
  if (NULL == w) warning("Widget is NULL.\n");
  else
    warning("Widget %s is %ssensitive, %srealized, and %smanaged.\n",
	    XtName(w),
	    XtIsSensitive(w)?"":"not ",
      	    XtIsRealized(w)?"":"not ",
	    XtIsManaged(w)?"":"not ");
}
#endif

/*	Function Name: Syntax
 *	Description: Prints a the calling syntax for this function to stdout.
 *	Arguments: app_con - the application context.
 *                 call - the name of the application.
 *	Returns: none - exits tho.
 */

static void 
Syntax(app_con, call)
XtAppContext app_con;
char *call;
{
    XtDestroyApplicationContext(app_con);
    fatal("Usage: %s [file name]\n", call);
}

/*	Function Name: Quit
 *	Description: This function exits the program.
 *	Arguments: w - ** UNUSED **
 *                 call_data - ** UNUSED **
 *                 client_data - ** UNUSED **
 *	Returns: none
 */

static XtCallbackProc
Quit(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
{
    XtDestroyApplicationContext(XtWidgetToApplicationContext(w));
    exit(0);
}

static void
  quit(w, event, params, num_params)
     Widget w; XEvent *event; String *params; Cardinal *num_params;
{
  Quit(w, NULL, NULL);
}

static Widget layout;

static void
  set_layout_label()
{
  Arg args[1];
  XtSetArg(args[0], XtNlabel,
	   (XtArgVal) using_trees
	   ?"Use form layout  (l)"
	   :"Use tree layout  (l)");
  XtSetValues(layout, args, XtNumber(args));
}
  
static XtCallbackProc
Toggle_layout(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
{
  static void display_current_dg();
  using_trees = !using_trees;
  display_current_dg();
  set_layout_label();
}

static void
  toggle_layout(w, event, params, num_params)
     Widget w; XEvent *event; String *params; Cardinal *num_params;
{
  Toggle_layout(w, NULL, NULL);
}

static Widget title;

static void
  set_title()
{
  char buf[1000];  Arg args[1];
  sprintf(buf, "Deduction Graph %d", dg_number);
  XtSetArg(args[0], XtNtitle, (XtArgVal) buf);
  XtSetValues(title, args, XtNumber(args));
}

static Widget			/* Base for displaying a deduction graph. */
  make_base(parent)		/* Other panes could be added here. */
     Widget parent;
{
  Widget w, b, q;
  
  static Arg paned_args[] = {
    { XtNborderWidth, (XtArgVal) 0},
    { XtNshowGrip, (XtArgVal) FALSE},
    { XtNresizeToPreferred, (XtArgVal) TRUE},
  };
  static Arg title_args[] = {
    { XtNborderWidth, (XtArgVal) 0},
  };
  static Arg box_args[] = {
    { XtNshowGrip, (XtArgVal) FALSE},
    { XtNresizeToPreferred, (XtArgVal) TRUE},
  };

  title = parent;
  set_title();
  w = XtCreateManagedWidget("base", panedWidgetClass,
			    parent, paned_args, XtNumber(paned_args));
  b = XtCreateManagedWidget("box", boxWidgetClass,
			    w, box_args, XtNumber(box_args));
  layout = XtCreateManagedWidget("layout", commandWidgetClass,
				 b, NULL, ZERO);
  XtAddCallback(layout, XtNcallback, Toggle_layout, NULL);
  set_layout_label();
  q = XtCreateManagedWidget("quit", commandWidgetClass,
			    b, NULL, ZERO);
  XtAddCallback(q, XtNcallback, Quit, NULL);
  XtInstallAllAccelerators(parent, q);
  return XtCreateManagedWidget("dgbase", panedWidgetClass,
			       w, paned_args, XtNumber(paned_args));
}
    
static Widget			/* Scrolling view port for the graph. */
  make_view(parent)
     Widget parent;
{
  static Arg args[] = {
    { XtNallowHoriz, (XtArgVal) True},
    { XtNallowVert, (XtArgVal) True},
    { XtNmin, (XtArgVal) 100},
  };

  return XtCreateManagedWidget("dgview", viewportWidgetClass,
			       parent, args, XtNumber(args));
}

static void
Sequent(w, client_data, call_data) /* Called when a sequent node */
Widget w;			/* is clicked.  It tells Emacs to */
XtPointer client_data, call_data; /* display the node. */
{
  fprintf(stdout,
	  "(sqn-display %d %d)\n",
	  dg_number,
	  ((int) client_data) - 1);
  fflush(stdout);
}

static char set_dg_number[] = "set-dg-number";

/* 
   Responds to a command of the form:
   (set-dg-number <deduction graph number>)
*/

static int
 do_set_dg_number(d)
     data d;
{
  data a = data_tail(d);
  
  if (!is_pair(a) || !is_number(data_head(a)) || !is_nil(data_tail(a))) {
    warning("Bad arguments to %s.\n", set_dg_number);
    dispose(d);
    return -1;
  }
  dg_number = data_number(data_head(a));
  dispose(d);
  set_title();
  return 0;
}
     
static Widget
  make_node(d, parent, super)
     data d; Widget parent, super;
{
  static Arg args[] = {
    { XtNsuperNode, (XtArgVal) NULL},
    { XtNlabel, (XtArgVal) "empty"},
    { XtNborderWidth, (XtArgVal) 0},
  };
  Widget w;
  char label[1000];

  if (!is_string(d) && !is_number(d)) 
    fatal("Bad deduction graph node as input.\n");

  XtSetArg(args[0], XtNsuperNode, super);

  if (is_string(d)) {
    XtSetArg(args[1], XtNlabel, data_string(d));
    w = XtCreateManagedWidget("rule", labelWidgetClass,
			      parent, args, XtNumber(args));
  }
  else {
    sprintf(label, "Sequent %d", data_number(d));
    XtSetArg(args[1], XtNlabel, (XtArgVal) label);
    w = XtCreateManagedWidget("sequent",commandWidgetClass,
			      parent, args, XtNumber(args));
    XtAddCallback(w, XtNcallback, Sequent, (XtPointer) data_number(d));
  }
  return w;
}

static char sequent[] = "sequent";

Boolean is_sequent(d)		/* Sequent nodes look like: */
     data d;			/* (Sequent <number> <rules>...) */
{
  return
    is_pair(d)
      && is_string(data_head(d))
	&& 0 == strcmp(sequent, data_string(data_head(d)))
	  && is_pair(data_tail(d))
	    && is_number(data_head(data_tail(d)));
}

static void
  make_branch(d, parent, super)
     data d; Widget parent, super;
{
  data dg;
  for(; !is_nil(d); d = data_tail(d))
    if (!is_pair(d)) fatal("Bad deduction graph branch as input.\n");
    else {
      dg = data_head(d);
      if (!is_pair(dg)) (void) make_node(dg, parent, super);
      else
	if (is_sequent(dg))
	  make_branch(data_tail(data_tail(dg)), parent,
		      make_node(data_head(data_tail(dg)), parent, super));
	else
	  make_branch(data_tail(dg), parent,
		      make_node(data_head(dg), parent, super));
    }
}
  
static void			/* Tree for the graph. */
  make_tree(d, parent)
     data d; Widget parent;
{
  static Arg args[] = {
    { XtNuseFormLayout, TRUE }
  };
  
  parent = XtCreateManagedWidget("dgtree", XstreeWidgetClass, 
				 parent,
				 using_trees?NULL:args,
				 using_trees?ZERO:XtNumber(args));
  make_branch(d, parent, NULL);
}

/* Routines for the display of a deduction graph. */

static Widget view;

#define THUMB_ZEROISH 0.005	/* About zero for a thumb */

static float
  get_thumb(w)
     Widget w;
{
  static float thumb, shown;
  static Arg args[] = {
    { XtNshown, (XtArgVal) &shown},
    { XtNtopOfThumb, (XtArgVal) &thumb},
  };

  if (NULL == w) return 0.0;
  XtGetValues(w, args, XtNumber(args));
  if (thumb <= THUMB_ZEROISH) return 0.0;
  if (thumb + shown >= 1.0 - THUMB_ZEROISH) return 1.0;
  return thumb;
}

static void
  set_thumb(w, thumb)
     Widget w;
     float thumb;
{
  float top;
  if (NULL == w || thumb <= THUMB_ZEROISH) return;

  top = thumb;			/* top cannot be an argument. */
  XtCallCallbacks(w, XtNjumpProc, (XtPointer) &top);
}

static void
  display_current_dg()
{
  Widget base;
  float h, v;
  
  /* Get scroll. */
  h = get_thumb(XtNameToWidget(view, "horizontal"));
  v = get_thumb(XtNameToWidget(view, "vertical"));

  base = XtParent(view);
  XtUnmanageChild(view);
  XtUnrealizeWidget(base);
  XtDestroyWidget(view);

  if (!is_nil(next_dg)) {
    dispose(current_dg);
    current_dg = next_dg;
    next_dg = NIL;
  }
  
  view = make_view(base);
  make_tree(current_dg, view);	/* Make new deduction graph. */

  XtSetMappedWhenManaged(view, FALSE);
  XtManageChild(base);

  /* Set scroll. */
  set_thumb(XtNameToWidget(view, "horizontal"), h);
  set_thumb(XtNameToWidget(view, "vertical"), v);
  XtSetMappedWhenManaged(view, TRUE);
}

static char show_deduction_graph[] = "show-deduction-graph";

/* 
   Responds to a command of the form:
   (show-deduction-graph <deduction graph>)
*/

static int
  do_show_deduction_graph (d)
     data d;
{
  data a = data_tail(d);

  if (!is_pair(a) || is_nil(data_head(a)) || !is_nil(data_tail(a))) {
    warning("Bad arguments to %s.\n", show_deduction_graph);
    dispose(d);
    return -1;
  }

  dispose(next_dg);
  next_dg = data_head(a);
  data_head(a) = NIL;
  dispose(d);
  return 0;
}

static int
  dispatch (d)			/* Command dispatch */
     data d;
{
  char *s;

  if (!is_pair(d) || !is_string(data_head(d))) {
    warning("Bad xdg command.\n");
    dispose(d);
    return -1;
  }
  
  s = data_string(data_head(d));

  if (0 == strcmp(s, show_deduction_graph))
    return do_show_deduction_graph(d);

  if (0 == strcmp(s, set_dg_number))
    return do_set_dg_number(d);

  if (0 == strcmp(s, "quit")) exit(0);

  warning("Warning: %s is an unrecognized command.\n", s);
  dispose(d);
  return -1;
}

static void			/* Handles input from stdin. */
  eval(client_data, source, id)
     XtPointer client_data;
     int *source;
     XtInputId *id;
{
  data d; int result;
  XtInputMask mask;

  result = parse(&d);

  if (0 == result) {		/* EOF found. */
    XtRemoveInput(*id);
    fclose(stdin);
    if (!is_nil(next_dg)) display_current_dg();
    return;
  }

  if (1 != result) return;	/* Error parsing input. */

  if (0 != dispatch(d))
    warning("xdg command failed.\n");

  if (is_nil(next_dg)) return;	/* Redisplay when new graph is there. */
  mask = XtAppPending(XtWidgetToApplicationContext(view));

  if (0 == (mask & XtIMAlternateInput))	/* Redisplay when */
    display_current_dg();	/* there is nothing more to eval */
}

int main(argc, argv)
int argc;
char **argv;
{
    XtAppContext app_con;
    Widget toplevel;

    toplevel = XtAppInitialize(&app_con, "Xdg",
			       NULL, ZERO,
			       &argc, argv,
			       fallback_resources,
			       NULL, ZERO);

    if (argc != 1 && argc != 2)		
      Syntax(app_con, argv[0]);

    if (argc == 2 && NULL == freopen(argv[1], "r", stdin))
      fatal("Cannot open %s for reading.\n", argv[1]);

    XtAppAddInput(app_con, fileno(stdin),
		  XtInputReadMask, eval, NULL);

    {
      static XtActionsRec actions[] = {
	{"quit", quit },
	{"toggle_layout", toggle_layout },
      };
      XtAppAddActions(app_con, actions, XtNumber(actions));
    }

    view = make_view(make_base(toplevel));
    current_dg = NIL;
    display_current_dg();
    XtRealizeWidget(toplevel);
    XtAppMainLoop(app_con);
}
