;; Copyright (c) 1990-1994 The MITRE Corporation
;; 
;; Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;;   
;; The MITRE Corporation (MITRE) provides this software to you without
;; charge to use, copy, modify or enhance for any legitimate purpose
;; provided you reproduce MITRE's copyright notice in any copy or
;; derivative work of this software.
;; 
;; This software is the copyright work of MITRE.  No ownership or other
;; proprietary interest in this software is granted you other than what
;; is granted in this license.
;; 
;; Any modification or enhancement of this software must identify the
;; part of this software that was modified, by whom and when, and must
;; inherit this license including its warranty disclaimers.
;; 
;; MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;; OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;; OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;; FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;; SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;; SUCH DAMAGES.
;; 
;; You, at your expense, hereby indemnify and hold harmless MITRE, its
;; Board of Trustees, officers, agents and employees, from any and all
;; liability or damages to third parties, including attorneys' fees,
;; court costs, and other related costs and expenses, arising out of your
;; use of this software irrespective of the cause of said liability.
;; 
;; The export from the United States or the subsequent reexport of this
;; software is subject to compliance with United States export control
;; and munitions control restrictions.  You agree that in the event you
;; seek to export this software or any derivative work thereof, you
;; assume full responsibility for obtaining all necessary export licenses
;; and approvals and for assuring compliance with applicable reexport
;; restrictions.
;; 
;; 
;; COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


;;; Support for *Sequent Nodes* and *Deduction* buffers to display the sequent
;;; nodes of a deduction-graph.

(require 'process-filter)
(provide 'sqns)

;;; (defvar sqn-file (expand-file-name (substitute-in-file-name "$IMPS/../tmp/dg-file.sqn"))
;;;   "*Path to file where Tea writes new sequent node info for sqn-buffers.")
;;; (defvar dg-file (expand-file-name (substitute-in-file-name "$IMPS/../tmp/dg-file.dg"))
;;;   "*Path to file where Tea writes new deduction-graph list for dg-buffers.") 
;;; 


(defvar sqn-display-hook nil)

(defun make-sqn ()
  (make-vector 4 nil))

(defun sqn-start (sqn)
  (aref sqn 0))
(defun sqn-end (sqn)
  (aref sqn 1))
(defun sqn-hash-no (sqn)
  (aref sqn 2))
(defun sqn-grounded-p (sqn)
  (aref sqn 3))

(defun sqn-set-start (sqn new-value)
  (aset sqn 0 new-value))
(defun sqn-set-end (sqn new-value)
  (aset sqn 1 new-value))
(defun sqn-set-hash-no (sqn new-value)
  (aset sqn 2 new-value))
(defun sqn-set-grounded-p (sqn new-value)
  (aset sqn 3 new-value))

;;; A DGR, or deduction graph record, stores all the information about the state
;;; of a particular deduction graph and the sequents it contains.  

(defun make-dgr (sqn-b dg-b dg-hash theory-name)
  (let ((vec (make-vector 8 nil)))
    (aset vec 0 (make-vector 0 nil))	;the sqn-vector
    (aset vec 1 nil)			;the index of the current sqn in sqn-vector 
    (aset vec 2 sqn-b)			;the sqn-buffer
    (aset vec 3 dg-b)			;the dg-buffer
    (aset vec 4 dg-hash)		;the dg-hash number
    (aset vec 5 theory-name)		;the name of the theory
    (aset vec 6 nil)			;the index of the last-inspected sqn
					;in sqn-vector 
    (aset vec 7 nil)			;the vector of cmpn buffers
    vec))

(defun dgr-sqn-vector (dg-r)
  (aref dg-r 0))
(defun dgr-current-sqn (dg-r)
  (aref dg-r 1))
(defun dgr-sqn-buffer (dg-r)
  (aref dg-r 2))
(defun dgr-dg-buffer (dg-r)
  (aref dg-r 3))
(defun dgr-dg-hash (dg-r)
  (aref dg-r 4))
(defun dgr-theory-name (dg-r)
  (aref dg-r 5))
(defun dgr-last-seen-sqn (dg-r)
  (aref dg-r 6))
(defun dgr-cmpn-vector (dg-r)
  (aref dg-r 7))

(defun dgr-set-sqn-vector (dg-r new-vector)
  (aset dg-r 0 new-vector)
  new-vector)
(defun dgr-set-cmpn-vector (dg-r new-vector)
  (aset dg-r 7 new-vector)
  new-vector)
(defun dgr-set-current-sqn (dg-r new-index)
  (if (or (= 0 new-index)
	  (< new-index (length (dgr-sqn-vector dg-r))))
      (aset dg-r 1 new-index)
    (error "dgr-set-current-sqn: index %d bigger than max %d."
	   new-index
	   (1- (length (dgr-sqn-vector dg-r))))))
(defun dgr-set-last-seen-sqn (dg-r new-index)
  (aset dg-r 6 new-index))
(defun dgr-set-cmpn-vector (dg-r new-vector)
  (aset dg-r 7 new-vector)
  new-vector)

(defconst dgr-vector-length 12 "Length to make dgr-vector.")
(defvar dgr-vector (make-vector dgr-vector-length nil)
  "*Vector holding all deduction-graph records." )

(defun dgrv-next-unused ()
  (let ((i 2)
	(len (length dgr-vector)))
    (while (and (not (null (aref dgr-vector i)))
		(< i len))
      (setq i (1+ i)))
    i))

(defun dgrv-fn-at (fn dgrv-index)
  (funcall fn (aref dgr-vector dgrv-index)))

(defun char-no->zero-based-sqn-no (char-no dgr)
  (let* ((vec (dgr-sqn-vector dgr))
	 (i   0)
	 (len (length vec))
	 (found '()))
    (while (and (not found) (< i len))
      (let ((sqn (aref vec i)))
	(if (and (<= (marker-position (sqn-start sqn)) char-no)
		 (<= char-no (marker-position (sqn-end sqn))))
	    (setq found t)
	  (setq i (1+ i)))))
    (and (< i len) i)))

(defun dgrv-initialize-dgr (index sqn-b dg-b dg-hash theory-name)
  (let ((cmpn-vector (and (vectorp (aref dgr-vector index))
			  (dgrv-fn-at 'dgr-cmpn-vector index))))
    (and (vectorp cmpn-vector)		; if necessary, 
	 (mapcar			; clean out old computation-nodes
	  (function
	   (lambda (cmpn-buff)
	     (if (bufferp cmpn-buff)
		 (kill-buffer cmpn-buff))))
	  cmpn-vector)))
  (let ((dgr (make-dgr sqn-b dg-b dg-hash theory-name)))
    (aset dgr-vector index dgr)
    dgr))

(defun sqn-adjoin-to-vector (dgrv-index sqn)
  (let ((new (vconcat (dgrv-fn-at 'dgr-sqn-vector dgrv-index) (list sqn))))
    (dgrv-fn-at
     (function
      (lambda (dgr)
	(dgr-set-sqn-vector dgr new)))
      dgrv-index)
    (dgrv-fn-at 'dgr-sqn-vector dgrv-index)))

(defun sqn-reset-vector (dgrv-index)
  (dgrv-fn-at
   (function
    (lambda (dgr)
      (dgr-set-sqn-vector dgr (make-vector 0 nil))))
   dgrv-index)
  (dgrv-fn-at 'dgr-sqn-vector dgrv-index))

(defun sqn-set-current-sqn (dgrv-index new-value)
  (dgrv-fn-at
   (function
    (lambda (dgr)
      (dgr-set-last-seen-sqn
       dgr
       (dgr-current-sqn dgr))))
   dgrv-index)
  (dgrv-fn-at
   (function
    (lambda (dgr)
      (dgr-set-current-sqn dgr new-value)))
   dgrv-index))

(defun hash-no-to-sqn-index (dgrv-index hash-no)
  (let ((sqn-vector (dgrv-fn-at 'dgr-sqn-vector dgrv-index)))
    (let ((i 0)
	  (len (length sqn-vector)))
      (while (and (< i len)
		  (let ((ith-hash-no (sqn-hash-no (aref sqn-vector i))))
		    (and (numberp ith-hash-no)
			 (not (= hash-no ith-hash-no)))))
	(setq i (1+ i)))
      (if (< i len)
	  i
	nil))))

(defun sqn-index-to-hash-no (dgrv-index i)
  (let ((sqn-vector (dgrv-fn-at 'dgr-sqn-vector dgrv-index)))
    (sqn-hash-no (aref sqn-vector i))))

(defun sqn-current-hash-no (dgrv-index)
  (dgrv-fn-at
   (function
    (lambda (dgr)
      (sqn-hash-no (aref (dgr-sqn-vector dgr)
			 (dgr-current-sqn dgr)))))
   dgrv-index))

(defun sqn-find (dgrv-index hash)
  (let ((index (hash-no-to-sqn-index dgrv-index hash)))
    (and index 
	 (aref (dgrv-fn-at 'dgr-sqn-vector dgrv-index) index))))

(defun sqn-make-grounded (dgrv-index hash)
  (let ((sqn (sqn-find dgrv-index hash)))
    (if sqn
	(sqn-set-grounded-p sqn t)
      (error "sqn-make-grounded: Bad hash %d" hash))))




(defvar dg-number 1
  "The index into the dgr-vector for the deduction-graph in the current buffer.")
(make-variable-buffer-local 'dg-number)


(defun sqn-display (dgrv-index index)
  (let ((sqn-buffer (dgrv-fn-at 'dgr-sqn-buffer dgrv-index))
	(sqn-vector (dgrv-fn-at 'dgr-sqn-vector dgrv-index)))
    (cond ((not (numberp index))
	   (error "sqn-display: bad index %s" index))
	  ((< index 0)
	   (error "index too small"))
	  ((>= index (length sqn-vector))
	   (error "index too large"))
	  (t t))
    (set-buffer sqn-buffer)
    (let ((sqn (aref sqn-vector index)))
      (widen)
      (narrow-to-region (sqn-start sqn)
			(sqn-end sqn))
      (if (not (null sqn-display-hook))
	  (run-hooks 'sqn-display-hook)
	(goto-char (point-max))
	(re-search-backward assumption-assertion-separator-pattern))
      (sqn-set-current-sqn dgrv-index index)
      (setq mode-line-process
	    (format "Sqn %d%s of %d"
		    (sqn-hash-no sqn)
		    (if (sqn-grounded-p sqn)
			" (Grounded)"
		      "")
		    (length sqn-vector))))))

(defun sqn-select (dgrv-index one-based-index)
  "Display the ONE-BASED-INDEXth sqn in the sqn buffer."
  (interactive (list dg-number
		     (read-minibuffer "Sequent Node number (1-based): ")))
  (sqn-display dgrv-index (1- one-based-index)))

(defun sqn-display-jump (dgrv-index one-based-index)
  "Display the ONE-BASED-INDEXth (prefix arg) sqn in the sqn buffer."
  (interactive (list dg-number
		     (prefix-numeric-value current-prefix-arg)))
  (sqn-select dgrv-index one-based-index))

(defun sqn-redisplay-last-seen (dgrv-index)
  "Display the sqn most recently viewed before this one (if any)."
  (interactive (list dg-number))
  (let ((last-seen (dgrv-fn-at 'dgr-last-seen-sqn dgrv-index)))
    (sqn-display dgrv-index last-seen)))

(defun sqn-display-max (dgrv-index) 
  "Display the maximum sqn in the sqn buffer."
  (interactive (list dg-number))
  (sqn-select dgrv-index
	      (length (dgrv-fn-at 'dgr-sqn-vector dgrv-index))))
  
(defun sqn-redisplay (dgrv-index)
  "Display the current sqn in the sqn buffer."
  (interactive (list dg-number))
  (sqn-display dgrv-index
	       (dgrv-fn-at 'dgr-current-sqn dgrv-index)))
      
(defun sqn-display-next (dgrv-index)
  "Display the next sqn in the sqn buffer."
  (interactive (list dg-number))
  (sqn-display dgrv-index
	       (1+ (dgrv-fn-at 'dgr-current-sqn dgrv-index))))

(defun sqn-display-previous (dgrv-index)
  "Display the previous sqn in the sqn buffer."
  (interactive (list dg-number))
  (sqn-display dgrv-index
	       (1- (dgrv-fn-at 'dgr-current-sqn dgrv-index))))

(defun sqn-display-hash (dgrv-index hash)
  "Display the sqn with hash-no HASH in the sqn buffer."
  (interactive (list dg-number
		     (read-minibuffer "Hash-number of sqn to display: ")))
  (sqn-display dgrv-index
	       (hash-no-to-sqn-index dgrv-index hash)))

(defun sqn-select-sqn-at-pos (dgrv-index here)
  "Display the sqn occupying the given position (point, interactively)
in the sqn buffer."
  (interactive (list dg-number (point)))
  (sqn-display
   dgrv-index
   (char-no->zero-based-sqn-no
    here
    (aref dgr-vector dgrv-index))))


(defun sqn-search-forward-regexp (search-string)
  (interactive "sRegexp search-string: ")
  (widen)
  (re-search-forward search-string nil nil)
  (sqn-select-sqn-at-pos dg-number (point)))


(defun sqn-add (dgrv-index hash-no grounded text)
  (let ((sqn-buffer (dgrv-fn-at 'dgr-sqn-buffer dgrv-index)))
    (save-excursion
      (set-buffer sqn-buffer)
      (let ((buffer-read-only nil))
	(widen)
	(if (sqn-find dgrv-index hash-no)
	    (let ((sqn (sqn-find dgrv-index hash-no)))
	      (sqn-set-grounded-p sqn grounded))
	  (let ((new-sqn (make-sqn)))
	    (goto-char (point-max))
	    (sqn-set-start new-sqn (point-marker))
	    (insert text)
	    (sqn-set-end new-sqn
			 (set-marker (make-marker) (point-max)))
	    (sqn-set-hash-no new-sqn hash-no)
	    (sqn-set-grounded-p new-sqn grounded)
	    (sqn-adjoin-to-vector dgrv-index new-sqn)))))))

(defun sqn-add-from-file (dgrv-index sqn-file)
  "Add new sqns to sqn-vector from printed representations in the sqn-file,
Assumes that the file contains a number of items of the form:
Hash-no Grounded Text \^J\^L"
  (let ((tmp-buffer (get-buffer-create " dg-file.sqn")))
    (set-buffer tmp-buffer)
    (erase-buffer)
    (insert-file (expand-file-name (substitute-in-file-name sqn-file)))
    (let ((start (point-min))
	  end)
      (goto-char start)
      (while (re-search-forward page-delimiter nil t)
	(setq end (match-beginning 0))
	(let* ((str (buffer-substring start end))
	       (first-read (read-from-string str))
	       (second-read (read-from-string str (cdr first-read))))
	  (sqn-add dgrv-index (car first-read) (car second-read)
		   (substring str (cdr second-read))))
	(setq start (1+ end))
	(goto-char start)))
    (sqn-redisplay dgrv-index)))


;;These two fns added by jt, 12-07-90.

;;;(defun configure-window-and-attempt-sqn-display (dgrv-index index)
;;;  (set-window-configuration *imps-window-configuration*)
;;;  (condition-case
;;;      v
;;;      (sqn-display dgrv-index (hash-no-to-sqn-index dgrv-index index))
;;;    (error)))

(defun configure-window-and-attempt-sqn-display (dgrv-index index)
  (set-window-configuration *imps-window-configuration*)
  (condition-case
      v
      (sqn-display dgrv-index (hash-no-to-sqn-index dgrv-index index))
    (error (sleep-for 1) (configure-window-and-attempt-sqn-display dgrv-index index))))

(defun start-up-buffer-display 
	(dgrv-index sqn-buffer-name dg-buffer-name dg-hash theory-string)
	(dgrv-initialize-dgr
	 dgrv-index 
	 (get-buffer-create sqn-buffer-name) 
	 (get-buffer-create dg-buffer-name) 
	 dg-hash 
	 theory-string)
	(set-buffer dg-buffer-name)
	(dg-mode dgrv-index)
	(set-buffer  sqn-buffer-name)
	(sqn-mode dgrv-index)
	(pop-to-buffer sqn-buffer-name)
	(pop-to-buffer dg-buffer-name)
	(setq *imps-window-configuration* (current-window-configuration)))

(defun number-list-to-string (nl)
  (mapconcat
   (function 
    (lambda (n)
      (format "%d" n)))
   nl
   ", "))
