#include <unistd.h>
#include <stdlib.h>
#include <math.h>
#include "defines.h"
#include "mem_alloc.h"
#include "main.h"
#include "files.h"

export int max_x, max_y;

static Node **vplace;
static void set_node_incount( Node *list);
static void select_start_nodes( Node *fromlist, Node **tolist);
static void vertical_place( Node **startlist);
static void dfs_vplace( Node *node, int y);
static void uncycle_graph();
static void vert_compact();
static void reduce_density();
static void horizontal_place( int leftbound);
static void horizontal_assign( int startrow, int endrow, int f, int b);
static void hor_move_node( Node *node, int inweight, int outweight);
static void hor_distrib_row( Node *row);
static int column_compar( Node **n1, Node **n2);
static void horizontal_shift( int leftbound);
static void insert_route_nodes();
static void insert_rout_node( Node *nn, int y, Node *orig);

export void place_graph()
{
	Node *startlist;
	int leftbound = 0;
	int overall_max_y = 0;

	set_node_incount(firstnode);

	while (firstnode) /* still unplaced nodes left */
	{	/* maybe more then one cycle needed when */
		/* there are disconnected cyclic subgraphs */

		if (debuglevel >= 3)
			verbose( "Start placement: leftbound = %d\n",
				leftbound);

		startlist = NULL;
		select_start_nodes( firstnode, &startlist);
		vertical_place( &startlist);

		if (max_y > overall_max_y) overall_max_y = max_y;
		/* remove loops from graph, allows easier formulation */
		/* of horizontal placement */
		uncycle_graph();

		vert_compact();
		reduce_density();

		if (routedges)
			insert_route_nodes();

		/* place each next graph to the right */
		horizontal_place( leftbound);
		leftbound = DIST + max_x;
	}

	max_y = overall_max_y;
}

/********* local functions ******/

static void set_node_incount( Node *list)
{
	Node *node;
	Edge *edge;
	int n;

	for (node = list; node; node = node->next)
	{	if (node->flag == -1) /* artificial 'super' node */
			n = 99999999;
		else
		{	n = 0;
			for (edge=node->inedges; edge; edge=edge->next_inedge)
				n++;
			if (node->super) /* don't count artificial in-edge */
				n--;
		}
		node->incount = n;
		if (debuglevel >= 4)
			verbose( "node %s incount = %d\n", node->name, n);
	}

	if (splitchar) /* the 'super-node's might be in use */
	for (node = list; node; node = node->next)
	{	if (!node->super) continue;

		if (node->super->incount > node->incount)
			node->super->incount = node->incount;
	}
}

static void select_start_nodes( Node *fromlist, Node **tolist)
{
	int min_incount, cnt;
	Node *node, *nextnode;

	min_incount = fromlist->incount;
	for (node = fromlist; node; node = node->next)
		if (node->incount < min_incount)
				min_incount = node->incount;
	
	if (debuglevel>=4)
		verbose( "select_start_list: take nodes with incount==%d\n",
			min_incount);

	for (node = fromlist; node; node = nextnode)
	{	nextnode = node->next;
		cnt = node->super ? node->super->incount : node->incount;
		if (cnt == min_incount)
		{	/* add this node */
			{	unlink_node( node);
				link_node( tolist, node);
				if (debuglevel>=5)
					verbose( "node %s on startlist\n",
						node->name);
			}
		}
	}
}

static void vertical_place( Node **startlist)
{
	if (vplace)
	{	/* clear (previous) placement */
		int i;
		for (i=0; i<n_nodes; i++)
			vplace[i] = NULL;
	} else
	{	/* array to hold list of nodes for each vertical pos */
		vplace = newarray( n_nodes, Node *);
	}

	max_y = 0;
	while (*startlist)
		dfs_vplace( *startlist, 0);
}

static void dfs_vplace( Node *node, int y)
{	/* node->flag==0: untouched, ==1: now on path, ==2: placed */
	Edge *edge, *se;
	Node *n;

	
	if (node->flag==1 || node->super && node->super->flag==1)
	{	if (debuglevel>=5)
			verbose( "dfs_vplace( node %s, y=%d): on path!\n",
				node->name, y);
		return;  /* cycle detected */
	}

	if (node->flag == 2 && node->y >= y)
	{	if (debuglevel>=5)
			verbose( "dfs_vplace( node %s, y=%d): is on %d\n",
				node->name, y, node->y);
		return;  /* is already correctly placed */
	}

	/* now perform placement */
	if (y > max_y) max_y = y;

	if (debuglevel>=4)
		verbose( "dfs_vplace:( node %s move to y=%d)\n",
			node->name, y);


	if (node->super)
	{	if (node->super->y != y)
		{	node->super->y = y;
			unlink_node( node->super);
			link_node( &vplace[y], node->super);
		}
		if (debuglevel>=5)
			verbose( "dfs_vplace: move super-node '%s'\n",
				node->super->name);
		node->super->flag = 1;
		for (se=node->super->outedges; se; se=se->next_outedge)
		{	n = se->to;
			n->y = y;
			n->flag = 1;
			if (debuglevel>=5)
				verbose( "dfs_vplace: '%s' glues "
					 "'%s' to y=%d\n",
					 node->name, n->name, y);
			unlink_node( n); /* remove node from its list */
			link_node( &vplace[y], n);
			for (edge=n->outedges; edge; edge=edge->next_outedge)
				dfs_vplace( edge->to, y+1);
			n->flag = 2;
		}
		node->super->flag = -1;
	} else
	{	node->y = y;
		node->flag = 1;
		unlink_node( node); /* remove node from its list */
		link_node( &vplace[y], node);
		for (edge = node->outedges; edge; edge = edge->next_outedge)
			dfs_vplace( edge->to, y+1);
		node->flag = 2;
	}
}

static void vert_compact()
{	/* move nodes down to compact the graph */
	int y, high;
	Node *n, *next;
	Edge *e;

	for (y=max_y-1; y>=0; y--)
	for (n=vplace[y]; n; n=next)
	{	next = n->next;
		if (n->super || !n->outedges) continue;

		high = max_y;
		for( e=n->outedges; e; e = e->next_outedge)
			if (e->to->y < high) high = e->to->y;

		high--;
		if (high > y) /* can move node down */
		{	n->y = high;
			unlink_node( n); /* remove node from its list */
			link_node( &vplace[high], n);
		}
	}
}

static void insert_route_nodes()
{	/* insert nodes in edges that span more than 1 y-coordinate */
	int y;
	Node *n, *nn;
	Edge *e, **ep;

	if (debuglevel>=3) verbose("Insert_route_nodes()\n");

	/* handle 'normal' top->down edges */
	for (y=0; y<=max_y-2; y++)
	for (n=vplace[y]; n; n=n->next)
	{	if (n->super) continue;

		if (debuglevel >=5) verbose( "Check node %s\n",
					n->name?n->name:"?");
		nn = NULL;
		for (ep=&(n->outedges); e=*ep;)
		{      	if (e->to->y > y+1 && !e->reversed)
			{	/* move this edge to new node down */
				if (!nn) nn = newstruct(Node);
				*ep = e->next_outedge;
				e->next_outedge = nn->outedges;
				nn->outedges = e;
				e->from = nn;
				if (!mergeedges)
				{	 insert_rout_node( nn, y+1, n);
					 nn = NULL;
			        }
		       } else /* check next out-edge */
				ep = &(e->next_outedge);
		}

		/* properly introduce a single new node for merged edges */
		if (nn) insert_rout_node( nn, y+1, n);
	}

	/* handle 'bottom-up' edges, which were earlier transformed in */
	/* 'normal' top-down edges with the 'reversed' sign on */
	for (y=max_y; y>=2; y--)
	for (n=vplace[y]; n; n=n->next)
	{	if (n->super) continue;

		nn = NULL;
		for (ep=&(n->inedges); e=*ep;)
		{	if (e->from->y < y-1 && e->reversed)
			{	/* move this edge to new node up */
				if (!nn) nn = newstruct(Node);
				*ep = e->next_inedge;
				e->next_inedge = nn->inedges;
				nn->inedges = e;
				e->to = nn;
				if (!mergeedges)
				{	 insert_rout_node( nn, y-1, n);
					 nn = NULL;
			        }
		       } else /* check next in-edge */
				ep = &(e->next_outedge);
		}

		/* properly introduce a single new node for merged edges*/
		if (nn) insert_rout_node( nn, y-1, n);
	}
}

static void insert_rout_node( Node *nn, int y, Node *orig)
{
	Edge *e;

      	if (debuglevel>=4)
       		verbose("New node at y=%d, for node %s at y=%d\n",
	       		y, orig->name?orig->name:"?", orig->y);
       	nn->y = y;
	link_node( vplace+y, nn);
	e = newstruct(Edge);
	link_edge( e, orig, nn);
} 

static void reduce_density()
{	/* move nodes up if this reduces the total wire length */
	/* i.e. the node has more up- than down-edges */
	int y, low, n_down, n_up;
	Node *n, *next;
	Edge *e;

	for (y=1; y<max_y; y++)
	for (n=vplace[y]; n; n=next)
	{	next = n->next;
		if (n->super || !n->inedges) continue;

		low = 0;
		n_down = n_up = 0;
		for( e=n->outedges; e; e = e->next_outedge)
			n_down++;
		for( e=n->inedges; e; e = e->next_inedge)
		{	if (e->from->y > low) low = e->from->y;
			n_up++;
		}

		low++;
		if (low < y && n_down <= n_up) /* can move node up */
		{	n->y = low;
			unlink_node( n); /* remove node from its list */
			link_node( &vplace[low], n);
		}
	}
}

static void uncycle_graph()
{	/* reverse all edges which point upwards (from high to low y) */
	int y;
	Node *from, *to, *next;
	Edge *edge, *nextedge;

	for (y=0; y<=max_y; y++)
	for (from=vplace[y]; from; from=next)
	{	next = from->next;
		for( edge=from->outedges; edge; edge = nextedge)
		{	nextedge = edge->next_outedge;
			to = edge->to;
			if (from->flag == -1)
			{	/* remove edges from 'super'nodes */
				unlink_edge( edge);
			} else if (to->y < y)
			{	/* reverse edge */
				if (debuglevel >= 3)
					verbose( "reverse edge from "
					 	"%s to %s\n",
					 	from->name, to->name);
				unlink_edge( edge);
				link_edge( edge, to, from);
				edge->reversed = True;
			}
		}
		if (from->flag == -1) unlink_node( from);
	}
}

static void horizontal_place( int leftbound)
{
	horizontal_assign( 0, max_y+1, 1, 0);
	horizontal_assign( max_y-1 , -1, 1, 0);
	horizontal_assign( 1, max_y+1, 2, 1);
	horizontal_assign( max_y/2 , -1, 2, 1);
	horizontal_assign( max_y/2+1 , max_y+1, 2, 1);

	horizontal_shift( leftbound);
}

static void horizontal_assign( int startrow, int endrow,
		int forwardweight, int backweight)
{
	Node *node;
	int i;
	int inc = (startrow<=endrow) ? 1 : -1;

	if (debuglevel>=3)
		verbose( "horizontal_assign: rows %d to %d, weight %d and %d\n",
			startrow, endrow, forwardweight, backweight);

	for (i=startrow; i!=endrow; i+=inc)
	{	if (i<0 || i>max_y) continue;

		for (node=vplace[i]; node; node=node->next)
			if (inc > 0)
				hor_move_node( node, forwardweight, backweight);
			else	hor_move_node( node, backweight, forwardweight);
		hor_distrib_row( vplace[i]);
	}
}

static void hor_move_node( Node *node, int inweight, int outweight)
{
	Edge *edge;
	int no = 0;
	int sum = 0;
	int w;

	if (inweight)
	for (edge=node->inedges; edge; edge=edge->next_inedge)
	{	w = inweight;
		if (inweight > 1 && edge->to->y - edge->from->y > 1)
			w--; /* decrease weight of long edges */
		no += w;
		sum += w * edge->from->x;
		if (debuglevel >= 8)
			verbose( "   due to node %s: no=%d, sum=%d\n",
				edge->from->name, no, sum);
	}

	if (outweight)
	for (edge=node->outedges; edge; edge=edge->next_outedge)
	{	w = outweight;
		if (outweight > 1 && edge->to->y - edge->from->y > 1)
			w--; /* decrease weight of long edges */
		no += w;
		sum += w * edge->to->x;
		if (debuglevel >= 8)
			verbose( "   due to node %s: no=%d, sum=%d\n",
				edge->to->name, no, sum);
	}

	sum += (sum>0) ? no/2 : -no/2; /* forces nice rounding */

	if (no>0) node->x = sum/no;

	if (debuglevel >= 4)
		verbose( "node %s moved to x=%d\n", node->name, node->x);
}

static void hor_distrib_row( Node *row)
{
	static Node **hplace;
	Node *node;
	int i, w, lastcol, midx, midi, bound;

	if (!row) return;
	if (!hplace) hplace = newarray( n_nodes, Node *);

	/* copy row-list into array */
	for( w=0, node=row; node; node=node->next, w++)
		hplace[w] = node;

	if (w>1)
	{
		/* sort array */
		qsort( (void *)hplace, w, sizeof( Node *), 
		      (int (*)(const void *, const void *))column_compar);

		/* reassign column numbers to legal values */
		/* find central x coordinate */
		midx = (hplace[0]->x + hplace[w-1]->x)/2;
		/* and find corresponding (almost) central node */
		for (midi=0; hplace[midi]->x < midx; midi++);
		for (i=midi; i<w && hplace[i]->x <= midx; i++);
		midi = (midi + i)/2;

		/* Note that nodes introduced to just route edges */
		/* over multiple levels, are placed closer together */
		/* 1. from center to right */
		lastcol = midx - ((w&1) ? DIST/2 : 0);
		/*
		   if (midi > 0)
		   {	node = hplace[midi-1];
		        lastcol = node->x + (node->name ? DIST/2 : 0);
		   } else	lastcol = midx - DIST;
		   */
		for (i=midi; i<w; i++)
		{	node = hplace[i];
			bound = lastcol + (node->name ? (DIST+1)/2 : 1);
			if (node->x < bound) node->x = bound;
			if (debuglevel>=5) verbose("  node[%d] '%s' to x=%d\n",
						   i, node->name, node->x);
			lastcol = node->x + (node->name ? DIST/2 : 1);
		}
		/* 2. from center to left */
		node = hplace[midi]; /* just on/over the center, was placed */
		lastcol = node->x - (node->name ? DIST/2 : 1);
		for (i=midi-1; i>=0; i--)
		{	node = hplace[i];
			bound = lastcol - (node->name ? (DIST+1)/2 : 1);
			if (node->x > bound) node->x = bound;
			if (debuglevel>=5) verbose("  node[%d] '%s' to x=%d\n",
						   i, node->name, node->x);
			lastcol = node->x - (node->name ? DIST/2 : 1);
		}
	}
	if (debuglevel >= 3)
	{	verbose( "Distrib_row: y=%d: nodes moved to", hplace[0]->y);
		for (i=0; i<w; i++)
			verbose( " %s=%d", hplace[i]->name, hplace[i]->x);
		verbose( "\n");
	}
}

static int column_compar( Node **n1, Node **n2)
{
	int diff = (*n1)->x - (*n2)->x;

	if (diff < 0) return( -1);
	else if (diff > 0) return( 1);
	else return( 0);
}

static void horizontal_shift( int leftbound)
{
	int min, max, x, y, shift;
	Node *node;

	min = max = vplace[0]->x;

	for (y=0; y<=max_y; y++)
	for (node=vplace[y]; node; node=node->next)
	{	x = node->x;
		if (x < min) min = x;
		else if (x > max) max = x;
	}

	shift = leftbound - min;

	for (y=0; y<=max_y; y++)
	for (node=vplace[y]; node; node=node->next)
		node->x += shift;
	
	max_x = max + shift;
}
