/****************************************************************************
	Memory allocation functions.

	Gathered here to keep track of total amount of allocated memory,
	and add check for succesful completion,
	which saves many (identical) lines throughout the program,

	See also myfree() and newstruct() in defines.h
	If you often want to alloc and free structs of one type,
  	you should add dedicated routines here to keep,
	those in a dedicated bucket in stead of really free-ing them.
****************************************************************************/
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/times.h>
#include "defines.h"
#include "main.h"

static char *lowbound, *highbound;

/**** obtain an uninitialised amount of memory ****/
export void *mymalloc( int size)
{
	char *p = (char *)(malloc( size));

	if (!p) error( "Memory allocation error!"
			" (current amount is %d kbyte)\n",
			(highbound - lowbound)>>10);

	if (p < lowbound || !highbound) lowbound = p;
	if (p+size > highbound) highbound = p+size;

	return( (void *)p);
}

/**** obtain an amount of memory initialised to 0 ****/
export void *mycalloc( int n, int size)
{
	char *p = (char *)(calloc( n, size));

	if (!p) error( "Memory allocation error!"
			" (current amount is %d kbyte)\n",
			(highbound - lowbound)>>10);

	if (p < lowbound || !highbound) lowbound = p;
	if (p+size > highbound) highbound = p+size;

	return( (void *)p);
}

/**** resize an array ***********************************/
export void *myrealloc( void *old, int size)
{
	char *s;

	s = (char *)(realloc( old, size));

	if (!s) error( "Memory allocation error!"
			" (current amount is %d kbyte)\n",
			(highbound - lowbound)>>10);

	if (s < lowbound || !highbound) lowbound = s;
	if (s+size > highbound) highbound = s+size;

	return( (void *)s);
}

/**** free() memory again for later use *****************************
      The version with all lower-case chars myfree() is a #define,
      which should be used throughout the program in stead of
      directly referencing this Myfree()
*********************************************************************/
export void MyFree( void *p)
{
	free( p);
}

/**** make a copy of a string in newly allocated memory **************/
/**** memory is actually allocated in larger chunks for efficiency ***/
/**** NOTE: you cannot free() these strings !!!!!!      **************/
export char *newstring( char *s)
{
	static char *buf;
	static int bufsize;
	int l;
	char *p;

	l = strlen(s) + 1;
	if (bufsize < l) /* create fresh buffer */
	{	bufsize = 1<<11;
		while (bufsize < l) bufsize <<= 1;
		buf = (char *)mymalloc( bufsize);
	}

	/* now take string from buffer */
	p = buf;
	strcpy( p, s);
	bufsize -= l;
	buf += l;

	return( p);
}

export void print_resources()
{
	struct tms timesbuf;
	long clk_tck;

	verbose( "Amount of memory allocated: %d kbyte.\n",
		(highbound-lowbound)>>10);

	clk_tck = sysconf( _SC_CLK_TCK );
	times( &timesbuf);
	verbose( "CPU time used: user %.2f, sys %.2f seconds\n",
		((float)timesbuf.tms_utime)/clk_tck,
		((float)timesbuf.tms_stime)/clk_tck);
}
