/*----------------------------------------------------------------------------
    MatVec.h	simple matrix vector structures, Mar 5, 1992

    written by Andreas Hohmann, ZIB, Hohmann@sc.ZIB-Berlin.DE
-----------------------------------------------------------------------------*/

typedef Real* RealVec;
typedef Int* IntVec;
typedef RealVec* RealMat;
typedef RealVec** RealTen;

/*----------------------------------------------------------------------------
    allocation procedures
-----------------------------------------------------------------------------*/

extern RealVec NewRealVec(Int l, Int h);
extern IntVec NewIntVec(Int l, Int h);

extern RealVec* NewRealPtrVec(int l, int h);
extern RealMat* NewRealPtrMat(int rl, int rh, int cl, int ch);

extern RealMat NewRealMat(Int rl, Int rh, Int cl, Int ch);
extern RealMat NewRealLowMat(Int rl, Int rh, Int cl, Int ch);
extern RealMat NewRealUpMat(Int rl, Int rh, Int cl, Int ch);

extern RealTen NewRealTen(Int lMin, Int lMax, Int mMin, Int mMax,
						Int nMin, Int nMax);
extern RealTen NewRealLowTen(Int lMin, Int lMax, Int mMin, Int mMax,
						Int nMin, Int nMax);
extern RealMat RealSubMat(RealMat mOld, Int rl, Int rh, Int cl, Int ch,
                                        Int rlNew, Int clNew);

/*----------------------------------------------------------------------------
    deallocation procedures
-----------------------------------------------------------------------------*/

extern void FreeRealVec(RealVec, Int nMin, Int nMax);
extern void FreeIntVec(IntVec, Int nMin, Int nMax);
extern void FreeRealMat(RealMat, Int mMin, Int mMax, Int nMin, Int nMax);
/***extern void FreeRealLowMat(RealMat, Int rl, Int rh, Int cl, Int ch);
extern void FreeRealUpMat(RealMat, Int rl, Int rh, Int cl, Int ch);
use FreeRealMat ***/

extern void FreeRealTen(RealTen, Int lMin, Int lMax, Int mMin, Int mMax,
				Int nMin, Int nMax);
extern void FreeRealSubMat(RealMat, Int rl, Int rh);
extern void FreeRealPtrVec(RealVec *x, Int nMin, Int nMax);
extern void FreeRealPtrMat(RealMat *m, Int mMin, Int mMax, Int nMin, Int nMax);

/*----------------------------------------------------------------------------
    utility procedures
-----------------------------------------------------------------------------*/

extern Real Product(RealVec, RealVec, Int nMin, Int nMax);
extern Real Distance(RealVec, RealVec, Int nMin, Int nMax);
extern Real Cosin(RealVec v, RealVec w, Int nl, Int nh);
extern Real SquaredNorm(RealVec v, Int nl, Int nh);
extern Real Norm(RealVec, Int nMin, Int nMax);
extern Real SquaredDistance(RealVec v, RealVec w, Int nl, Int nh);
extern Real MaxNorm(RealVec v, Int nl, Int nh);

extern Real ScaledSquaredDistance(RealVec v, RealVec w, RealVec scale, Int nl, Int nh);
extern Real ScaledDistance(RealVec x, RealVec y, RealVec scale, Int nMin, Int nMax);
extern Real ScaledSquaredNorm(RealVec v, RealVec scale, Int nl, Int nh) ;
extern Real ScaledNorm(RealVec x, RealVec xScale, Int nMin, Int nMax);
extern Real ScaledProduct(RealVec v, RealVec w, RealVec scale, Int nl, Int nh);
extern Real ScaledCosin(RealVec v, RealVec w, RealVec scale, Int nl, Int nh);
extern Real ScaledMaxNorm(RealVec v, RealVec scale, Int nl, Int nh);

extern void FillRealVec(Real x, Int l, Int h, RealVec v);
extern void SwapRealVecs(RealVec v, RealVec w, Int nl, Int nh);
extern void TurnRealVec(RealVec v, Int nl, Int nh);
extern void CopyIntVec(IntVec from, Int nMin, Int nMax, RealVec to);
extern void CopyRealVec(RealVec from, Int nMin, Int nMax, RealVec to);
extern void Normalize(RealVec v, Int nl, Int nh);

extern void SwapColumns(RealMat a, Int j, Int k, Int ml, Int mh);

extern void ScalarVecMult(RealVec x, Real lambda, Int n, RealVec y);
extern void RealVecAdd(RealVec, RealVec, Int n, RealVec);
extern void RealVecSubtract(RealVec, RealVec, Int n, RealVec);

/*----------------------------------------------------------------------------
    matrix vector operations
-----------------------------------------------------------------------------*/

extern void MatVecMult(RealMat a, RealVec x, Int m, Int n, RealVec b);
extern void SymMatVecMult(RealMat a, RealVec x,  Int n, RealVec b);
extern void MatTransVecMult(RealMat a, RealVec x, Int m, Int n, RealVec b);

/*----------------------------------------------------------------------------
    copy procedures
-----------------------------------------------------------------------------*/

extern void CopyRealMat(RealMat from, Int nrl, Int nrh, Int ncl, Int nch, RealMat to);
extern void CopyRealUpMat(RealMat from, Int nrl, Int nrh, Int ncl, Int nch, RealMat to);
extern void CopyRealLowMat(RealMat from, Int nrl, Int nrh, Int ncl, Int nch, RealMat to);

/*----------------------------------------------------------------------------
    matrix matrix operations
-----------------------------------------------------------------------------*/

extern void RealMatMatMult(RealMat, RealMat, Int m, Int n, Int k, RealMat);
extern void RealMatAdd(RealMat, RealMat, Int m, Int n, RealMat);
extern void Transpose(RealMat a, Int m, Int n, RealMat b);

/*----------------------------------------------------------------------------
    forward and backward substitutions
-----------------------------------------------------------------------------*/

extern void SolveRealUpMat(RealMat A, RealVec b, Int n, RealVec x);
extern void SolveTransposedRealUpMat(RealMat A, RealVec b, Int n, RealVec x);
extern void SolveRealLowMat(RealMat A, RealVec b, Int n, RealVec x);
extern void SolveTransposedRealLowMat(RealMat A, RealVec b, Int n, RealVec x);

/*----------------------------------------------------------------------------
    transform a vector into a matrix
-----------------------------------------------------------------------------*/

extern RealMat VectorMatrix(RealVec z, Int rh, Int ch);
extern void FreeVectorMatrix(RealMat W, Int rh, Int ch);

/*----------------------------------------------------------------------------
    the following procedures realize the orthogonal projections on the
    subspace given by the ON-basis t[1..q][1..n] and its orthogonal
    complement, respectively. If P is the projection onto the subspace
    span(t[1],...,t[q]), we get

    Project:		xOut = P xIn
    OrthoProject:	xOut = (I-P) xIn

    The vectors t[1],...,t[q] have to be orthonormal with respect to the
    scaled product given by scale[1..n].
-----------------------------------------------------------------------------*/

extern void Project(RealMat t, RealVec xIn, RealVec scale, Int q, Int n, RealVec xOut);
extern void OrthoProject(RealMat t, RealVec xIn, RealVec scale, Int q, Int n, RealVec xOut);

/*----------------------------------------------------------------------------
    input/output procedures
-----------------------------------------------------------------------------*/

extern void sprintfRealVec(char *str, char *format, RealVec x, Int nMin, Int nMax); 
extern void printfRealVec(char *format, RealVec x, Int nMin, Int nMax); 
extern void printIntVec(IntVec x, Int nMin, Int nMax); 
extern void fprintfRealVec(FILE *f, char *format, RealVec x, Int nMin, Int nMax);
extern void fprintIntVec(FILE *f, IntVec x, Int nMin, Int nMax);

extern char *PrintRealVec(char *format, RealVec x, Int nMin, Int nMax);
extern char *PrintIntVec(IntVec x, Int nMin, Int nMax);

extern void sprintfRealMat(char *str, char *format, RealMat x, Int mMin, Int mMax, Int nMin, Int nMax); 
