;;; This file contains the definition of values 
;;; and other associated expressions.
;;;
;;;
;;; Derek Lindner  buddha@theory.lcs.mit.edu
;;; Justin Liu     dondon@theory.lcs.mit.edu
;;; Brian So       brianso@theory.lcs.mit.edu
;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                 ;;;
;;; Definition of VALUES and associated expressions ;;;
;;;                                                 ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (value? exp)
  (or (letrec-value? exp)
      (letrec-free-value? exp)))

(define (letrec-free-value? exp)
  (or (lambda-value? exp)
      (known-constant? exp)
      (quote? exp)
      (pair-value? exp)
      (string? exp)))

(define (lambda-value? exp)
  (lambda? exp))

(define (known-constant? exp)
  (or (nil-builtin? exp)
      (number? exp)
      (boolean? exp)
      (functional-constant? exp)
      (side-effect-constant? exp)
      (equal? exp undefined-value)))

(define (quotestable-value? exp)
  (or (nil-builtin? exp)
      (number? exp)
      (boolean? exp)
      (equal? exp undefined-value)
      (bracketed-identifier? exp)
      (string? exp)))

(define nil-builtin '())          ;REVISED

(define (nil-builtin? exp)             ;REVISED
  (equal? exp nil-builtin))

(define (nonnull-constant? exp)         ;REVISED
  (and (known-constant? exp)
       (not (nil-builtin? exp))))

(define (pair-value? exp)
  (or (cons-value? exp)
      (nonnull-list-value? exp)))

(define (nonnull-list-value? exp)
  (and (listexp? exp) 
       (> (length exp) 1)
       (all-letrec-free-values? (rest exp))))

(define (cons-value? exp)
  (and (consexp? exp)
       (= (length exp) 3)
       (letrec-free-value? (cadr exp))
       (nonlist-letrec-free-value? (caddr exp))))

(define (nonlist-letrec-free-value? exp)
  (or (nonnull-constant? exp)
      (lambda? exp)
      (quote? exp)
      (string? exp)
      (cons-value? exp)))

(define (letrec-value? exp)
  (and (letrec? exp)
       (all-bindings-letrec-free-values? (bindings-letc exp))
       (value? (body-letc exp))))

(define (all-values? exp)
  (if (null? exp)
      #t
      (and (value? (car exp))
	   (all-values? (cdr exp)))))

; ALL-BINDINGS-LETREC-FREE-VALUES? takes a list of bindings 
; of the form ((var init) (var init) ... ) and
; returns #TRUE only if all the init portions 
; of the bindings are letrec-free-values.

(define (all-bindings-letrec-free-values? bindings)
  (cond ((null? bindings)
	 #t)
	(else (and (letrec-free-value? (first-binding-init bindings))
		   (all-bindings-letrec-free-values? (cdr bindings))))))

; KEYWORDS

(define (keyword? word)
  (member word keywords))

(define keywords '(lambda if cond let letrec quote and or begin))
;REVISE: LOAD IMPORT ABORT CONTINUATION

; VARIABLES

(define (variable?
 exp)
  (and (symbol? exp)
       (not (bracketed-identifier? exp))
       (not (keyword? exp))
       (not (equal? exp 'define))))

; CONSTANTS

(define undefined-value '*value-not-specified*)

(define (functional-constant? exp)
  (memq exp functional-constants))

(define (side-effect-constant? exp)
  (memq exp side-effect-constants))

(define init-functional-constants
  '(<<+>> <<->> <<*>> <</>> <<=>> <<_<_>> <<_>_>> <<_>=_>> <<_<=_>> <<1+>> <<-1+>> 
      <<quotient>> <<sqrt>> <<expt>> <<round>> <<abs>> <<gcd>> <<max>> <<min>>
      <<zero?>> <<positive?>> <<negative?>> <<odd?>> <<even?>>
      <<exp>> <<log>> <<sin>> <<cos>> <<tan>> <<asin>> <<acos>> <<atan>>
      <<not>>
      <<equal?>>
      <<boolean?>> <<symbol?>> <<char?>> <<pair?>> <<number?>> <<procedure?>>       
      <<car>> <<cdr>> <<cons>> <<list>>
      <<list?>> <<null?>> <<length>> <<append>> <<reverse>> <<list-tail>> <<list-ref>>    
      <<member>> <<assoc>> <<delete>>
      <<apply>> <<map>>
      <<string?>> <<make-string>> <<string>> <<string-length>> <<string-ref>> 
      <<string=?>> <<string<?>> <<string>?>> <<string<=?>> <<string>=?>>
      <<string-ci=?>> <<string-ci<?>> <<string-ci>?>> <<string-ci<=?>> <<string-ci>=?>>
      <<substring>> <<string-append>> <<string->list>> <<list->string>>
      <<number->string>> <<symbol->string>>))

(define functional-constants (copy init-functional-constants))

(define init-side-effect-constants
  '(<<display>>  <<newline>> <<error>>))
                                        ;REVISE: include LOAD and IMPORT

(define side-effect-constants (copy init-side-effect-constants))

(define (all-letrec-free-values?  exp)
 (if (null? exp)
      #t
      (and (letrec-free-value? (car exp))
	   (all-letrec-free-values? (cdr exp)))))

(define (false? exp)
  (or (eq? exp #f)
      (nil-builtin? exp)))
