(*  :Title:	Supporting routines for Manipulating Multirate Signals  *)

(*  :Authors:	Brian Evans, James McClellan  *)

(*
    :Summary:   Provide functions which are common in multirate DSP but
		not standard in Mathematica.  Functions like detecting
		sampling matrices and the modulo of a vector and a
		sampling matrix.  This package also implements some
		of the ideas from the Geometry of Numbers.
 *)

(*  :Context:	SignalProcessing`Support`Multirate`  *)

(*  :PackageVersion:  2.4	*)

(*
    :Copyright:	Copyright 1989-1991 by Brian L. Evans
		Georgia Tech Research Corporation

	Permission to use, copy, modify, and distribute this software
	and its documentation for any purpose and without fee is
	hereby granted, provided that the above copyright notice
	appear in all copies and that both that copyright notice and
	this permission notice appear in supporting documentation,
	and that the name of the Georgia Tech Research Corporation,
	Georgia Tech, or Georgia Institute of Technology not be used
	in advertising or publicity pertaining to distribution of the
	software without specific, written prior permission.  Georgia
	Tech makes no representations about the suitability of this
	software for any purpose.  It is provided "as is" without
	express or implied warranty.
 *)

(*  :History:	multirate signal processing  *)

(*  :Keywords:	downsampling, sampling matrix, upsampling *)

(*
    :Source:         J. Kovacevic and M. Veterli, "Perfect Reconstruction
		Filter Banks with Rational Sampling Rates in One- and
		Two-dimensions," in {Proc. SPIE Conference on Visual
		Communications and Image Processing} (Philadelphia, PA),
		pp. 1258-1268, November 1989.

		     A. Kaufmann and A. Henry-Labordere, {Integer and
		Mixed Progamming: Theory and Applications}, Academic Press,
		New York, 1977.

		     R. Bamberger, {The Directional Filter Bank:  A Multirate
		Filter Bank for the Directional Decompostion of Images},
		Georgia Tech Ph. D. Thesis, 1990.
 *)

(*  :Warning:	*)

(*  :Mathematica Version:  1.2 or 2.0  *)

(*  :Limitation:  *)

(*  :Discussion:  *)

(*
    :Functions:	BezoutNumbers
		CommutableResamplingMatricesQ
		DistinctCosetVectors
		IntegerMatrixEuclid --  not debugged
		IntegerVectorQ
		NormalizeSamplingMatrix
		ResamplingMatrix
		ResamplingMatrixMod
		SmithNormalForm
		SmithFormSameU
		SmithFormSameV
		SmithOrderedForm
		SmithReducedForm
 *)
 

If [ TrueQ[ $VersionNumber >= 2.0 ],
     Off[ General::spell ];
     Off[ General::spell1 ] ]


(*  B E G I N     P A C K A G E  *)

BeginPackage[ "SignalProcessing`Support`Multirate`",
	      "SignalProcessing`Support`SigProc`",
	      "SignalProcessing`Support`SupCode`" ]

(*  U S A G E     I N F O R M A T I O N  *)

BezoutNumbers::usage =
	"BezoutNumbers[a, b] gives the Bezout numbers of integers a and b. \
	That is, it returns {lambda, mu} so that lambda a + mu b == gcd(a,b)."

CommutableResamplingMatricesQ::usage =
	"CommutableResamplingMatricesQ[downsamplingmat, upsamplingmat] \
	returns True if the two resampling operations are commutable."

DistinctCosetVectors::usage =
	"DistinctCosetVectors[resmat] returns a sorted list of all of the \
	distinct coset vectors associated with the resampling matrix resmat. \
	DistinctCosetVectors[resmat, U^-1, Lambda, V^-1] finds all of the \
	distinct coset vectors of resmat from its Smith Form  U Lambda V. \
	This is carried by finding the distinct coset vectors of the diagonal \
	matrix Lambda, mapping them by U^-1, and then taking each vector \
	modulo the resampling matrix resmat. \
	The returned coset vectors are not sorted." 

IntegerVectorQ::usage =
	"IntegerVectorQ[arg] returns True if arg is a vector whose
	elements are integers."

NormalizeSamplingMatrix::usage =
	"NormalizeSamplingMatrix[sampmat] will decompose the sampling matrix \
	sampmat into a diagonal matrix D and a normalized sampling matrix V. \
	The results are returned as { D, V }."

ResamplingMatrix::usage =
	"ResamplingMatrix[arg] returns True if arg is a square matrix \
	whose elements are integers and whose determinant is non-zero. \
	It returns False if arg is not a square matrix. \
	It also returns False if arg is a square matrix with one element \
	being a number that is not an integer. \
	Otherwise, it returns the conditions for which the square matrix \
	would be a resampling matrix."

ResamplingMatrixMod::usage =
	"ResamplingMatrixMod[n, N] implements the integer vector n \
	modulo a sampling matrix N which is defined as \
	n - N . Floor[ N^-1 . n ]. \
	The operation will be carried out if and only if n is an integer \
	vector, N is (or could be) a sampling matrix, and \
	N . n is a valid operation."

SmithNormalForm::usage =
	"SmithNormalForm[A] returns a list of three matrices \
	U^-1, D, and V^-1 such that the m x n resampling matrix A \
	equals U D V where U (m x m) and V (n x n) are integer matrices \
	with determinants of +1 or -1 and D is a diagonal matrix \
	of integer components such that |det(D)| = |det(A)|. \
	The algorithm follows Kaufmann's {Integer and Mixed Programming}. \
	Note that this factorization is not unique. \
	SmithNornamlForm supports a Dialogue option. \
	If True or All, SmithNormalForm will show immediate steps. \
	See also SmithOrderedForm and SmithReducedForm."

SmithFormSameU::usage =
	"SmithFormSameU[smithfun, m1, m2] computes the Smith Form of the \
	integer matrix m1 so that m1 = u1 d1 v1. \
	The routine then uses u2 = u1 to decompose m2. \
	The first argument specifies which decomposition routine should \
	be used to decompose m1 (e.g., SmithNormalForm). \
	The routine returns {cond, {u1^-1, d1, v1^-1}, {u2^-1, d2, v2^-1}}, \
	where cond is True of False depending on whether or not matrices \
	m1 and m2 can be decomposed using the same U."

SmithFormSameV::usage =
	"SmithFormSameV[smithfun, m1, m2] computes the Smith Form of the \
	integer matrix m1 so that m1 = u1 d1 v1. \
	The routine then uses v2 = v1 to decompose m2. \
	The first argument specifies which decomposition routine should \
	be used to decompose m1 (e.g., SmithNormalForm). \
	The routine returns {cond, {u1^-1, d1, v1^-1}, {u2^-1, d2, v2^-1}}, \
	where cond is True of False depending on whether or not matrices \
	m1 and m2 can be decomposed using the same U."

SmithOrderedForm::usage =
	"SmithOrderedForm[A] returns a list of three matrices \
	U, D, and V such that the m x n resampling matrix A \
	equals U D V where U (m x m) and V (n x n) are integer matrices \
	with a determinant of 1 and D is a diagonal matrix \
	of integer components such that det(D) = det(A). \
	The components along the diagonal of D are sorted by absolute value. \
        The diagonal elements are positive except for possibly the \
	last one whose sign will be the sign of det(A). \
	Note that this factorization is unique if none of the diagonal \
	elements are equal in absolute value. \
	Also note that the starting point is the result of the Smith \
	Normal Form decomposition. \
	Like SmithNormalForm, SmithOrderedForm supports a Dialogue option. \
	If True or All, SmithNormalForm will show immediate steps. \
	See also SmithNormalForm and SmithReducedForm."

SmithReducedForm::usage =
	"SmithReducedForm[A] returns a list of three matrices \
	U, D, and V such that the m x n resampling matrix A \
	equals U D V where U (m x m) and V (n x n) are integer matrices \
	with a determinant of +1 or -1 and D is a diagonal matrix \
	of positive integers such that |det(D)| = |det(A)|. \
	Each component down the diagonal is a factor of the next one. \
	As a result, the diagonal elements are sorted down the diagonal. \
	The factorization is not unique. \
	Note that the starting point for this algorithm is the Smith \
	Ordered Form. \
	Like SmithOrderedForm, SmithReducedForm supports a Dialogue option. \
	If True or All, SmithNormalForm will show immediate steps. \
	See also SmithNormalForm and SmithOrderedForm."

(*  E N D     U S A G E     I N F O R M A T I O N  *)


(*
IntegerMatrixEuclid::usage =
	"IntegerMatrixEuclid[A, B] performs Euclid's algorithm on integer \
	matrices A and B. \
	Before the iteration begins, matrices with negative determinants \
	are adjusted so that they have positive determinants which adjusts \
	the result as well. \
	In each iteration, the condition A = B is checked; if true, \
	then the matrix is returned. \
	Then, A and B checked for singularity (return the zero matrix if so). \
	In each iteration, U is set to the one with the largest determinant, \
	L to the one with the smaller determinant. \
	Then, the quotient is Q = Floor[ U L^-1 ] and the remainder is
	R = U - Q L. \
	Euclid's algorithm is then called with arguments L and R."
 *)


Begin[ "`Private`" ]


(*  M E S S A G E S  *)

CommutableResamplingMatricesQ::invalid =
	"At least one of the arguments is not a resampling matrix."

DistinctCosetVectors::notresmat =
	"The lone argument is not a resampling matrix: ``"
DistinctCosetVectors::zerodet = "Resampling matrix has zero determinant."

IntegerMatrixEuclid::notcompatible =
	"The matrices are not square matrices having the same dimension"
IntegerMatrixEuclid::notres1 =
	"The first matrix is not a resampling matrix: ``"
IntegerMatrixEuclid::notres2 =
	"The second matrix is not a resampling matrix: ``"

SmithNormalForm::invalid =
	"The determinants of U^-1 and V^-1 should both be one."


(*  S U P P O R T I N G     R O U T I N E S  *)

(* eucliditerate *)
eucliditerate[a_, b_, ident_] :=
	Block [	{deta, detb, lower, quot, rem, return = a, upper},

		While [ a != b,
			deta = Det[a];
			detb = Det[b];

			If [ deta < 2 || detb < 2,
			     return = ident; Break[] ];

			If [ deta < detb,
			     lower = a; upper = b,
			     lower = b; upper = a ];
			quot = Floor[ upper . Inverse[l] ];
			rem = u - quot . l;

			a = l;
			b = rem ];

		return ]

(*  mymod  *)
SetAttributes[mymod, {Listable}]
mymod[x_] := Mod[x, 1]

(*  onefun  *)
onefun[x__] := 1

(*  vectorgcd   *)
vectorgcd[ x_ ] := Apply[GCD, x]



(* FindMinMatrixElement *)
FindMinMatrixElement[mat_List] :=
	Block [ {item},
		item = myabs[ Flatten[mat] ];
		Join[ Position[ mat, item ], Position[ mat, -item ] ] ]

(* intdivide --  kludge for Quotient primitive  *)
intdivide[n_, m_] := Sign[n] Sign[m] Quotient[Abs[n], Abs[m]]

(* myabs *)
myabs[x_List] := Min[Abs[Select[x, nonzero]]]

(* nonzero *)
nonzero[x_] := ( x != 0 )

(* TransitionMatrix *)
TransitionMatrix[r1_, r2_, dim_] :=
		TransitionMatrix[r1, r2, dim, IdentityMatrix[dim]]
TransitionMatrix[r1_, r2_, dim_, mat_] :=
	Block [	{resmat, temp},
		resmat = mat;
		temp = resmat[[r1]];
		resmat[[r1]] = resmat[[r2]];
		resmat[[r2]] = temp;
		resmat ]


(* SmithFormA --  returns non-zero minimum value, left transition *)
(*   matrix, and right transition matrix of the current matrix.   *)
SmithFormA[ curmat_, dim_, m_, n_, identm_, identn_ ] :=
	Block [	{col, i, minvalue, row, workmat},
		workmat = Table[ Take[ curmat[[i]], {1 + dim, n} ],
				 {i, 1 + dim, m} ];
		{row, col} = First[FindMinMatrixElement[workmat]];
		{ curmat[[row + dim]][[col + dim]],
		  TransitionMatrix[dim + 1, row + dim, m, identm],
		  TransitionMatrix[dim + 1, col + dim, n, identn] } ]

(* SmithFormB --  returns the left and right subtraction matrices *)
SmithFormB[ curmat_, dim_, minvalue_, m_, n_, identm_, identn_ ] :=
	Block [	{a, i, j, submatleft, submatright},

		submatleft = identm;
		For [ i = dim + 2, i <= m, i++,
		      a = curmat[[i, 1 + dim]];
		      If [ ! SameQ[a, 0],
	        	   submatleft[[i, 1+dim]] = intdivide[-a, minvalue]]];

		submatright = identn;
		For [ j = dim + 2, j <= n, j++,
		      a = curmat[[1 + dim, j]];
		      If [ ! SameQ[a, 0],
	 		   submatright[[1+dim, j]] = intdivide[-a, minvalue]]];

		{ submatleft, submatright } ]

(* SmithFormEndTest --  end if the elements in the first column  *)
(*   and first row of the working matrix except (1,1) are zero.  *)
SmithFormEndTest[curmat_, dim_, m_, n_] :=
	Block [ {firstrowcollist},

		firstrowcollist = Join[ Take[ curmat [[dim+1]], {dim+2, m} ],
					Take[ Transpose[curmat] [[dim + 1]],
					      {dim + 2, n} ] ];

		SameQ[First[firstrowcollist], 0] &&
		  Apply[Equal, firstrowcollist] ]

(* SmithFormTrace *)
SmithFormTrace[ dialogue_, str_, mat_ ] :=
	If [ dialogue || SameQ[dialogue, All],
	     Print[ ]; Print[ str ]; Print[ MatrixForm[mat] ] ]

SmithFormTrace[ dialogue_, str_ ] :=
	If [ dialogue || SameQ[dialogue, All], Print[ ]; Print[ str ] ]

SmithFormTrace2[ dialogue_, str_, mat_ ] :=
	If [ SameQ[dialogue, All],
	     Print[ ];
	     Print[ "  ", str, " subtraction matrix:" ];
	     Print[ "  ", MatrixForm[mat] ] ]


(*  E X P O R T E D     R O U T I N E S  *)

(*  BezoutNumbers  *)
BezoutNumbers[a_Integer, b_Integer] := BezoutNumbers[a, b, GCD[a, b]]

BezoutNumbers[a_Integer, b_Integer, gcd_Integer] :=
	Block [ {lamdba, mu = 0},

		If [ (! IntegerQ[a/gcd]) || (! IntegerQ[b/gcd]),
		     Return[Infinity, Infinity] ];

		While [ ! IntegerQ[lambda],
			mu++;
			lambda = ( gcd - mu b ) / a ];
		{ lambda, mu } ]

(*  CommutableResamplingMatricesQ, adapted from [Kovacevic & Vetterli, p. 4] *)
(*  A cascade of a downsampling by D1 and upsampling by D2 is commutable if  *)
(*  1. The matrix product is commutable, i.e.   D1 . D2 == D2 . D1	     *)
(*  2. The sets A and B are equivalent where				     *)
(*									     *)
(*                                      t -1				     *)
(*                  A = { exp(-2 pi j (D )   k), k in LatD1 }		     *)
(*                                      1				     *)
(*									     *)
(*                                     t   t -1				     *)
(*                  B = { exp(-2 pi j D  (D )   n), n in LatD1 }	     *)
(*                                     2   1				     *)
(*									     *)
(*  The points in the lattice associated with D1 (LatD1) are none other      *)
(*  than the set of fundamental cosets.					     *)

Options[ CommutableResamplingMatricesQ ] := { Dialogue -> False }

CommutableResamplingMatricesQ[ D1_, D2_, op___ ] :=
	Block [	{anglesofA, anglesofB, cosets, dialogue, D1tinv, vectors},

		(*  check arguments  *)
		If [ ! ( ResamplingMatrix[D1] && ResamplingMatrix[D2] ),
		     Return[Message[CommutableResamplingMatricesQ::invalid]] ];

		(*  check if dialogue is enabled  *)
		dialogue = Replace[ Dialogue,
				    ToList[op] ~Join~
				    Options[CommutableResamplingMatricesQ] ];

		(*  Condition #1  *)
		If [ ! SameQ[D1 . D2, D2 . D1], Return[False] ];

		(*  Condition #2  *)
		cosets = DistinctCosetVectors[D1];
		D1tinv = Inverse[ Transpose[D1] ];
		vectors = D1tinv . Transpose[cosets];
		anglesofA = Union[Transpose[mymod[vectors]]];
		anglesofB = Union[Transpose[mymod[Transpose[D2] . vectors]]];

		If [ dialogue,
		     Print["angles associated with the downsampling matrix:"];
		     Print[2 Pi anglesofA];
		     Print["angles associated with both resampling matrices:"];
		     Print[2 Pi anglesofB] ];

		SameQ[anglesofA, anglesofB] ]

(*  DistinctCosetVectors--  resmat is a dim x dim matrix  *)
DistinctCosetVectors[ resmat_?ResamplingMatrix ] :=
	Apply[DistinctCosetVectors, Prepend[SmithNormalForm[resmat], resmat]]

DistinctCosetVectors[ resmat_ ] :=
	Message[ DistinctCosetVectors::notresmat, resmat ]

DistinctCosetVectors[ resmat_, uinv_, lambda_, vinv_ ] :=
	Block [	{absdet, cosets, curvertex, diagvector, dim,
		 invresmat, lowervertex, uppervertex, zerovector},

		absdet = Abs[Det[resmat]];
		If [ SameQ[absdet, 0],
		     MyMessage[ DistinctCosetVectors::zerodet ] ];

		dim = Apply[Min, Dimensions[resmat]];
		zerovector = Apply[Table, {0, {dim}}];
		cosets = { zerovector };

		If [ SameQ[absdet, 1], Return[cosets] ];

		invresmat = Inverse[resmat];

		diagvector = Table[ lambda[[i,i]], {i, 1, dim} ];
		uppervertex = Abs[ diagvector ];
		lowervertex = curvertex = zerovector;

		(*  enumerate all points in the rectangular prism	  *)
		(*    delimited by the column vectors of the diagonal     *)
		(*    matrix lambda, map them ny U^-1, and then modulo    *)
		(*    each one with the original resampling matrix resmat *)
		cosets = Table[ curvertex = IncList[ curvertex,
						     lowervertex,
						     uppervertex ];
				ResamplingMatrixMod[ uinv . curvertex,
				  		     resmat, invresmat ],
				{absdet - 1} ];

		Prepend[cosets, zerovector] ]

(* IntegerMatrixEuclid *)
IntegerMatrixEuclid[a_, b_] := Message[ IntegerMatrixEuclid::notcompatible ] /;
	Dimensions[a] != Dimensions[b] && SameQ[Apply[Equal, Dimensions[a]]]

IntegerMatrixEuclid[a_, b_] := Message[IntegerMatrixEuclid::notres1, a] /;
	TrueQ[ ResamplingMatrixQ[a] ]

IntegerMatrixEuclid[a_, b_] := Message[IntegerMatrixEuclid::notres2, b] /;
	TrueQ[ ResamplingMatrixQ[b] ]

IntegerMatrixEuclid[a_, b_] :=
	Block [	{anegflag, adja, adjb, bnegflag, gcf, n, negmat},
		n = First[ Dimensions[a] ];
		negmat = ident = IdentityMatrix[n];
		negmat[[1,1]] = -1;

		anegflag = ( Det[a] < 0 );
		bnegflag = ( Det[b] < 0 );
		adja = If [ anegflag, negmat . a, a ];
		adjb = If [ bnegflag, negmat . b, b ];

		gcf = eucliditerate[adja, adjb, ident];

		If [ Xor[anegflag, bnegflag], negmat . gcf, gcf ] ]

(*  IntegerVectorQ  *)
IntegerVectorQ[arg_] := VectorQ[arg] && Apply[And, Map[IntegerQ, arg]]

(*  NormalizeSamplingMatrix  *)
NormalizeSamplingMatrix[ sampmat_ ] :=
	Block [	{diag},
		diag = Map[ vectorgcd, sampmat ];
		{ DiagonalMatrix[diag], mat / diag } ]

(*  ResamplingMatrixQ  *)
intcond[a_Integer] := True
intcond[a_?NumberQ] := False

intcond/: Format[ intcond[a_] ] := StringForm[ "Head[``] == Integer", a ]

ResamplingMatrix[a_] :=
	MatrixQ[ a ] && Apply[ SameQ, Dimensions[a] ] &&
	Apply[ And, Map[intcond, Flatten[a]] ] && ( Det[a] != 0 )

(*  ResamplingMatrixMod  *)
ResamplingMatrixMod[n_?IntegerVectorQ, sampmat_?MatrixQ] :=
	ResamplingMatrixMod[n, sampmat, Inverse[sampmat] ] /;
	( ! SameQ[ ResamplingMatrix[sampmat], False ] ) &&
	Apply[ SameQ, Prepend[Dimensions[sampmat], Length[n]] ]

ResamplingMatrixMod[n_, sampmat_, invsampmat_] :=
	n - ( sampmat . Floor[ invsampmat . n ] )


(*  S M I T H     D E C O M P O S I T I O N     R O U T I N E S  *)

(* SmithNormalForm *)
Options[ SmithNormalForm ] := { Dialogue -> False }

SmithNormalForm[ mat_?MatrixQ, op___ ] :=
	Block [	{curmat, dialogue, dim, endflag, identm, identn,
		 m, n, oplist, r, submatleft, submatright, transmatleft,
		 transmatright, uinv, vinv },

		oplist = ToList[op] ~Join~ Options[SmithNormalForm];
		dialogue = Replace[Dialogue, oplist];

		curmat = mat;
		{m, n} = Dimensions[mat];
		uinv = identm = IdentityMatrix[m];
		vinv = identn = IdentityMatrix[n];
		r = Min[m, n];
			
		For [ dim = 0, dim < r - 1, dim++,

		      If [ dialogue || SameQ[dialogue, All],
			   Print[]; Print[ "Iteration #", dim + 1 ]; Print[] ];

		      endflag = False;
		      While [ ! endflag,

			      (* part a: move min element to (1,1) *)

			      { minvalue, transmatleft, transmatright } =
				SmithFormA[curmat, dim, m, n, identm, identn];
			      curmat = transmatleft . curmat . transmatright;

			      SmithFormTrace[dialogue, "Part A", curmat];

			      (* part b: eliminate elements on first row/col *)

			      { submatleft, submatright } =
				SmithFormB[ curmat, dim, minvalue,
					    m, n, identm, identn ];
			      curmat = submatleft . curmat . submatright;

			      SmithFormTrace[dialogue, "Part B"];
			      SmithFormTrace2[dialogue, "left", submatleft];
			      SmithFormTrace2[dialogue, "right", submatright];
			      SmithFormTrace[dialogue, " ", curmat];

			      (* update U^-1 and V^-1 *)
			      uinv = submatleft . transmatleft . uinv;
			      vinv = vinv . transmatright . submatright;

			      (* check for the ending condition *)
			      endflag = SmithFormEndTest[curmat, dim, m, n] ] ];

	{uinv, curmat, vinv} ]			

(* SmithOrderedForm --  sort the diagonal matrix *)
Options[SmithOrderedForm] := Options[SmithNormalForm]

SmithOrderedForm[ mat_List, op___ ] :=
	Block [	{d, dim, i, identm, identn, j, jend, m, n, r,
		 temp, utransmat, vtransmat, uinv, vinv},

		{m, n} = Dimensions[mat];

		(* First, find its Smith Normal Form decomposition *)
		{ uinv, d, vinv } = SmithNormalForm[ mat, op ];

		(* Factor out the signs of the diagonal elements *)
		(*   the diagonal values will all be positive    *)
		(*   propagate the sign info to U^-1		 *)
		Clear[i];
		uinv = Sign[ Table[ Take[d[[i]], m], {i, 1, m} ] ] . uinv;
		d = Abs[d];

		(* Sort the elements along the diagonal *)
		utransmat = identm = IdentityMatrix[m];
		vtransmat = identn = IdentityMatrix[n];
		r = Min[m, n];
		For [ i = 1, i <= r, i++,
		      jend = r - i;
		      For [ j = 1, j <= jend, j++,
			    If [ d[[j,j]] > d[[j+1,j+1]],
				 temp = d[[j,j]];
			      	 d[[j,j]] = d[[j+1,j+1]];
			      	 d[[j+1,j+1]] = temp;

			      	 temp = utransmat[[j]];
			      	 utransmat[[j]] = utransmat[[j+1]];
			      	 utransmat[[j+1]] = temp;

			      	 temp = vtransmat[[j]];
			      	 vtransmat[[j]] = vtransmat[[j+1]];
			      	 vtransmat[[j+1]] = temp ] ] ];

		(* We switched rows instead of columns, so transpose it *)
		vtransmat = Transpose[vtransmat];

		(* We have changed  U D V  to  U Tu^-1 Tu D Tv Tv^-1 V	*)
		(*   Dnew = Tu D Tv  which was done in the sort above	*)
		(*   Unew = U Tu^-1  ==>  Unew^-1 = Tu U^-1		*)
		(*   Vnew = Tv^-1 V  ==>  Vnew^-1 = V^-1 Tv		*)
		uinv = utransmat . uinv;
		vinv = vinv . vtransmat;

		(* return U^-1, D, and V^-1 as a list *)
		{ uinv, d, vinv } ]

(* SmithReducedForm *)
(* -- A is the original matrix and A = U D V  *)
(* -- D has only positive values              *)
(* -- start with        U^-1 A V^-1   =   D   *)
(* -- iterate with    G U^-1 A V^-1 H = G D H *)
Options[SmithReducedForm] := Options[SmithNormalForm]

SmithReducedForm[ mat_List, op___ ] :=
	Block [	{adj, d, di, gcd, g, h, i, identm, identn,
		 lambda, lastd, lasti, lcm, m, mu, n, r, uinv, vinv},

		(*  Find the dimensions of the matrix *)
		{ m, n } = Dimensions[mat];
		r = Min[m, n];
		identm = IdentityMatrix[m];
		identn = IdentityMatrix[n];

		(*  Put the matrix in Smith Ordered Form  *)
		{ uinv, d, vinv } = SmithOrderedForm[mat, op];

		(*  Convert the diagonal  *)
		lastd = d[[1,1]];
		lasti = 1;
		For [ i = 2, i <= r, i++,
		      di = d[[i,i]];
		      gcd = GCD[lastd, di];
		      If [ gcd == lastd, lastd = di; Continue[] ];

		      g = identm;
		      h = identn;
		      { lambda, mu } = BezoutNumbers[lastd, di, gcd];
		      lcm = LCM[lastd, di];

		      (* define g matrix *)
		      g[[lasti, lasti]] = 1;
		      g[[lasti, i]] = 1;
		      g[[i, lasti]] = -mu di / gcd;
		      g[[i, i]] = lambda lastd / gcd;

		      (* define h matrix *)
		      h[[lasti, lasti]] = lambda;
		      h[[lasti, i]] = -di / gcd;
		      h[[i, lasti]] = mu;
		      h[[i, i]] = lastd / gcd;

		      (* adjust diagonal elements *)
		      d[[lasti, lasti]] = gcd;
		      d[[i,i]] = lastd = lcm;

		      (* update uinv and vinv *)
		      uinv = g . uinv;
		      vinv = vinv . h;

		      lasti = i ];

		{ uinv, d, vinv } ]

(* SmithFormSameV *)
SmithFormSameV[ smithfun_, m1_, m2_, op___] :=
	Block [	{d1, d2, diag, u1inv, u2, u2d2, u2inv, v1inv, v2inv},

		{u1inv, d1, v1inv} = smithfun[ m1, op ];

		v2inv = v1inv;
		u2d2 = m2 . v2inv;			(* U2 D2 = M2 V2^-1 *)
		diag = Map[vectorgcd, Transpose[u2d2]];	(* D2 is the gcd of *)
		d2 = DiagonalMatrix[diag];		(*   cols of U2 D2  *)
		u2 = u2d2 . DiagonalMatrix[1/diag];	(* U2 = U2 D2 D2^-1 *)
		u2inv = Inverse[u2];

		{Abs[Det[u2inv]] == 1, {u1inv, d1, v1inv}, {u2inv, d2, v2inv}} ]

(* SmithFormSameU *)
SmithFormSameU[ smithfun_, m1_, m2_, op___ ] :=
	Block [	{d1, d2, diag, u1inv, u2inv, u2invd2, v1inv, v2inv},

		{u1inv, d1, v1inv} = smithfun[ m1, op ];

		u2inv = u1inv;
		d2v2 = u2inv . m2;			(* D2 V2 = U2^-1 M2 *)
		diag = Map[vectorgcd, d2v2];		(* D2 is the gcd of *)
		d2 = DiagonalMatrix[diag];		(*   rows of D2 V2  *)
		v2 = DiagonalMatrix[1/diag] . d2v2;	(* V2 = D2^-1 D2 V2 *)
		v2inv = Inverse[v2];

		{Abs[Det[v2inv]] == 1, {u1inv, d1, v1inv}, {u2inv, d2, v2inv}} ]


(*  E N D     P A C K A G E  *)

End[]
EndPackage[]

If [ TrueQ[ $VersionNumber >= 2.0 ],
     On[ General::spell ];
     On[ General::spell1 ] ]


(*  H E L P     I N F O R M A T I O N  *)

Block [	{newfuns},
	newfuns = 
	  { BezoutNumbers,		CommutableResamplingMatricesQ,
	    DistinctCosetVectors,	IntegerMatrixEuclid,
	    IntegerVectorQ,		NormalizeSamplingMatrix,
	    ResamplingMatrix,		ResamplingMatrixMod,
	    SmithFormSameU,		SmithFormSameV,
	    SmithNormalForm,		SmithOrderedForm,
	    SmithReducedForm };
	Combine[ SPfunctions, newfuns ];
	Apply[ Protect, newfuns ] ]


(*  E N D     M E S S A G E  *)

Print["Functions supporting multirate signal manipulations have been loaded."]
Null

