#ifndef IGEMATRIX_H
#define IGEMATRIX_H
#pragma once 

/*
 *	Declarations for int general matricies.
 *
 *	Copyright (C) 1988, 1989.
 *
 *	Dr. Thomas Keffer
 *	Rogue Wave Associates
 *	P.O. Box 85341
 *	Seattle WA 98145-1341
 *
 *	Permission to use, copy, modify, and distribute this
 *	software and its documentation for any purpose and
 *	without fee is hereby granted, provided that the
 *	above copyright notice appear in all copies and that
 *	both that copyright notice and this permission notice
 *	appear in supporting documentation.
 *	
 *	This software is provided "as is" without any
 *	expressed or implied warranty.
 *
 *
 *	@(#)IGEMatrix.h	2.2	9/26/89
 */

/*
 *	This class is derived from class IntVec.  Data is stored
 *	FORTRAN style: by columns.
 *
 *	Defining the preprocessor directive "BOUNDS_CHECK" will invoke
 *	bounds checking.
 */

#include "IntVec.h"

class IGEMatrix : public IntVec {
  int ncols;			// Number of columns
  int nrows;			// Number of rows
protected:
  void 			assertColRange(int);
  void 			assertRowRange(int);
  void			assertRowCol(const IGEMatrix&);
  void			assertLength(const IntVec&);
  void			assertSquare();
  void			assertProduct(const IGEMatrix&);
  void			assertProduct(const IntVec&);
public:
  IGEMatrix();
  IGEMatrix(int rows, int cols);
  IGEMatrix(int rows, int cols, int initval);
  IGEMatrix(const int* dat, int, int);  // Copy of dat will be made
  IGEMatrix(const IntVec& v, int, int); // Reference to v will be made
  IGEMatrix(const IGEMatrix& m);	   // Reference to m will be made

  int*			data()		{return IntVec::data();}
  int			cols();
  int			rows();

  IGEMatrix&		reference(IGEMatrix& m); // Reference self to m
  IGEMatrix		deepCopy();	// copy of self with distinct instance variables 
  IGEMatrix		copy()		{return deepCopy();} // Synonym for deepCopy()
  void			deepenShallowCopy();	// Guarantee that references==1:
  void			resize(unsigned,unsigned);

  void			scanFrom(istream& s);	// Format 2 x 2 [ 3 2 1 4 ]
  void			printOn(ostream& s);
  void			readFrom(istream&);
  void			storeOn(ostream&);
  void			readFrom(fileDescTy&);
  void			storeOn(fileDescTy&);

  IntVec		operator[](int j);	// Return a col as a slice
  IntVec		col(int j);		// Return a col as a slice
  IntVec		row(int i);		// Return a row as a slice
  IntVec		diagonal(int idiag=0);	// Return a diagonal as a slice
  int&			operator()(int i, int j); // Subscripting

// Math functions
  IGEMatrix		product(const IGEMatrix&); // Inner product
  IntVec		product(const IntVec&);

// Assignment operators --- self must be same size as m
  IGEMatrix&		operator=(const IGEMatrix& m);
  IGEMatrix&		operator=(int);
  IGEMatrix&		operator+=(const IGEMatrix& m);
  IGEMatrix&		operator+=(int);
  IGEMatrix&		operator-=(const IGEMatrix& m);
  IGEMatrix&		operator-=(int);
  IGEMatrix&		operator*=(const IGEMatrix& m);
  IGEMatrix&		operator*=(int);
//IGEMatrix&		operator/=(const IGEMatrix& m);
//IGEMatrix&		operator/=(int);

// Increment/decrement operators
  IGEMatrix&		operator++();
  IGEMatrix&		operator--();

// Friendly arithmetic operators; Notice that operator* is an element-by-
// element multiply, NOT a matrix multiply.
  friend IGEMatrix	operator-(const IGEMatrix&);	// Unary minus
  friend IGEMatrix	operator+(const IGEMatrix&);	// Unary plus
  friend IGEMatrix	operator*(const IGEMatrix&, const IGEMatrix&);
//friend IGEMatrix	operator/(const IGEMatrix&, const IGEMatrix&);
  friend IGEMatrix	operator+(const IGEMatrix&, const IGEMatrix&);
  friend IGEMatrix	operator-(const IGEMatrix&, const IGEMatrix&);
  friend IGEMatrix	operator*(const IGEMatrix&, int);
  friend IGEMatrix	operator*(int, const IGEMatrix&);
//friend IGEMatrix	operator/(const IGEMatrix&, int);
//friend IGEMatrix	operator/(int, const IGEMatrix&);
  friend IGEMatrix	operator+(const IGEMatrix&, int);
  friend IGEMatrix	operator+(int, const IGEMatrix&);
  friend IGEMatrix	operator-(const IGEMatrix&, int);
  friend IGEMatrix	operator-(int, const IGEMatrix&);

};

// Other (related) declarations:
ostream&		operator<<(ostream& s, const IGEMatrix& m);
IGEMatrix		transpose(const IGEMatrix&);

/******************* I N L I N E S **************************/

Inline int IGEMatrix::cols() { return ncols;}
Inline int IGEMatrix::rows() { return nrows;}
Inline void IGEMatrix::deepenShallowCopy(){IntVec::deepenShallowCopy();}
Inline IGEMatrix operator+(const IGEMatrix& m)		{ return m; }
Inline IGEMatrix operator*(int d, const IGEMatrix& m){ return m*d; }
Inline IGEMatrix operator+(int d, const IGEMatrix& m){ return m+d; }

// Return a column
Inline IntVec IGEMatrix::operator[](int j){
#if BOUNDS_CHECK
  assertColRange(j);
#endif
  return IntVec::slice(j*nrows,nrows,1);
}

Inline IntVec IGEMatrix::col(int j){	// Same as above
#if BOUNDS_CHECK
  assertColRange(j);
#endif
  return IntVec::slice(j*nrows,nrows,1);
}

Inline IntVec IGEMatrix::row(int i){
#if BOUNDS_CHECK
  assertRowRange(i);
#endif
  return IntVec::slice(i, ncols, nrows);
}

Inline IntVec IGEMatrix::diagonal(int i){
  register int iabs=ABSOLUTE(i);
#if BOUNDS_CHECK
  assertSquare();
  assertRowRange(iabs);
#endif
  return IntVec::slice(i>0 ? i*nrows : iabs, nrows-iabs, nrows+1);
}

Inline int& IGEMatrix::operator()(int i, int j){
#if BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return IntVec::operator()(j*nrows+i);
}

#endif IGEMATRIX_HXX
