#ifndef FGEMATRIX_H
#define FGEMATRIX_H
#pragma once 

/*
 *	Declarations for float precision general matricies.
 *
 *	Copyright (C) 1988, 1989.
 *
 *	Dr. Thomas Keffer
 *	Rogue Wave Associates
 *	P.O. Box 85341
 *	Seattle WA 98145-1341
 *
 *	Permission to use, copy, modify, and distribute this
 *	software and its documentation for any purpose and
 *	without fee is hereby granted, provided that the
 *	above copyright notice appear in all copies and that
 *	both that copyright notice and this permission notice
 *	appear in supporting documentation.
 *	
 *	This software is provided "as is" without any
 *	expressed or implied warranty.
 *
 *
 *	@(#)FGEMatrix.h	2.2	9/26/89
 */

/*
 *	This class is derived from class FloatVec.  Data is stored
 *	FORTRAN style: by columns.
 *
 *	Defining the preprocessor directive "BOUNDS_CHECK" will invoke
 *	bounds checking.
 */

#include "FloatVec.h"

class FGEMatrix : public FloatVec {
  int ncols;			// Number of columns
  int nrows;			// Number of rows
protected:
  void 			assertColRange(int);
  void 			assertRowRange(int);
  void			assertRowCol(const FGEMatrix&);
  void			assertLength(const FloatVec&);
  void			assertSquare();
  void			assertProduct(const FGEMatrix&);
  void			assertProduct(const FloatVec&);
public:
  FGEMatrix();
  FGEMatrix(int rows, int cols);
  FGEMatrix(int rows, int cols, float initval);
  FGEMatrix(const float* dat, int, int);  // Copy of dat will be made
  FGEMatrix(const FloatVec& v, int, int); // Reference to v will be made
  FGEMatrix(const FGEMatrix& m);	   // Reference to m will be made

  float*		data()		{return FloatVec::data();}
  int			cols();
  int			rows();

  FGEMatrix&		reference(FGEMatrix& m); // Reference self to m
  FGEMatrix		deepCopy();	// copy of self with distinct instance variables 
  FGEMatrix		copy()		{return deepCopy();} // Synonym for deepCopy()
  void			deepenShallowCopy();	// Guarantee that references==1:
  void			resize(unsigned,unsigned);

  void			scanFrom(istream& s);	// Format 2 x 2 [ 3 2 1 4 ]
  void			printOn(ostream& s);
  void			readFrom(istream&);
  void			storeOn(ostream&);
  void			readFrom(fileDescTy&);
  void			storeOn(fileDescTy&);

  FloatVec		operator[](int j);	// Return a col as a slice
  FloatVec		col(int j);		// Return a col as a slice
  FloatVec		row(int i);		// Return a row as a slice
  FloatVec		diagonal(int idiag=0);	// Return a diagonal as a slice
  float&		operator()(int i, int j); // Subscripting

// Math functions
  FGEMatrix		product(const FGEMatrix&); // Inner product
  FloatVec		product(const FloatVec&);

// Assignment operators --- self must be same size as m
  FGEMatrix&		operator=(const FGEMatrix& m);
  FGEMatrix&		operator=(float);
  FGEMatrix&		operator+=(const FGEMatrix& m);
  FGEMatrix&		operator+=(float);
  FGEMatrix&		operator-=(const FGEMatrix& m);
  FGEMatrix&		operator-=(float);
  FGEMatrix&		operator*=(const FGEMatrix& m);
  FGEMatrix&		operator*=(float);
  FGEMatrix&		operator/=(const FGEMatrix& m);
  FGEMatrix&		operator/=(float);

// Increment/decrement operators
  FGEMatrix&		operator++();
  FGEMatrix&		operator--();

// Friendly arithmetic operators; Notice that operator* is an element-by-
// element multiply, NOT a matrix multiply.
  friend FGEMatrix	operator-(const FGEMatrix&);	// Unary minus
  friend FGEMatrix	operator+(const FGEMatrix&);	// Unary plus
  friend FGEMatrix	operator*(const FGEMatrix&, const FGEMatrix&);
  friend FGEMatrix	operator/(const FGEMatrix&, const FGEMatrix&);
  friend FGEMatrix	operator+(const FGEMatrix&, const FGEMatrix&);
  friend FGEMatrix	operator-(const FGEMatrix&, const FGEMatrix&);
  friend FGEMatrix	operator*(const FGEMatrix&, float);
  friend FGEMatrix	operator*(float, const FGEMatrix&);
  friend FGEMatrix	operator/(const FGEMatrix&, float);
  friend FGEMatrix	operator/(float, const FGEMatrix&);
  friend FGEMatrix	operator+(const FGEMatrix&, float);
  friend FGEMatrix	operator+(float, const FGEMatrix&);
  friend FGEMatrix	operator-(const FGEMatrix&, float);
  friend FGEMatrix	operator-(float, const FGEMatrix&);

};

// Other (related) declarations:
ostream&		operator<<(ostream& s, const FGEMatrix& m);
FGEMatrix		transpose(const FGEMatrix&);

/******************* I N L I N E S **************************/

Inline int FGEMatrix::cols() { return ncols;}
Inline int FGEMatrix::rows() { return nrows;}
Inline void FGEMatrix::deepenShallowCopy(){FloatVec::deepenShallowCopy();}
Inline FGEMatrix operator+(const FGEMatrix& m)		{ return m; }
Inline FGEMatrix operator*(float d, const FGEMatrix& m){ return m*d; }
Inline FGEMatrix operator+(float d, const FGEMatrix& m){ return m+d; }

// Return a column
Inline FloatVec FGEMatrix::operator[](int j){
#if BOUNDS_CHECK
  assertColRange(j);
#endif
  return FloatVec::slice(j*nrows,nrows,1);
}

Inline FloatVec FGEMatrix::col(int j){	// Same as above
#if BOUNDS_CHECK
  assertColRange(j);
#endif
  return FloatVec::slice(j*nrows,nrows,1);
}

Inline FloatVec FGEMatrix::row(int i){
#if BOUNDS_CHECK
  assertRowRange(i);
#endif
  return FloatVec::slice(i, ncols, nrows);
}

Inline FloatVec FGEMatrix::diagonal(int i){
  register int iabs=ABSOLUTE(i);
#if BOUNDS_CHECK
  assertSquare();
  assertRowRange(iabs);
#endif
  return FloatVec::slice(i>0 ? i*nrows : iabs, nrows-iabs, nrows+1);
}

Inline float& FGEMatrix::operator()(int i, int j){
#if BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return FloatVec::operator()(j*nrows+i);
}

#endif FGEMATRIX_HXX
