/*
 * Support functions for the units(3) library.
 */

/*LINTLIBRARY*/

#ifndef lint
    static char rcsid[] = "$Id: lib.c,v 1.8 1991/11/19 17:49:34 steve Exp $";
    static char afsid[] = "$__Header$";
#endif

#include <udposix.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <assert.h>
#include <search.h>
#include <stddef.h>		/* for ptrdiff_t */
#include "udalloc.h"
#include "uderrmsg.h"
#include "udunits.h"

extern int	UtLineno;		/* input-file line index */
extern int	UnitNotFound;		/* parser didn't find unit */
extern char	*buffer;                /* input-string buffer */
extern char	*pointer;               /* position in input buffer */
extern utUnit	*FinalUnit;		/* fully-parsed specification */

#define	DUPSTR(s)	strcpy((char*)UD_ALLOC(strlen(s)+1, char), s)
#define	ABS(a)		((a) < 0 ? -(a) : (a))
#define	MIN(a,b)	((a) < (b) ? (a) : (b))
#define	MAX(a,b)	((a) > (b) ? (a) : (b))

typedef struct {
    char	*name;
    int		nchr;
    int		HasPlural;
    utUnit	unit;
} UnitEntry;

typedef struct {
    char	*name;		/* prefix string (e.g. "milli") */
    UtFactor    factor;		/* corresponding multiplying factor */
    short       nchar;		/* size of prefix string excluding EOS */
} PrefixEntry;

/*
 *  Prefix table in the order required by the prefix-entry comparison function.
 *  The names are Based on ANSI/IEEE Std 260-1978 (a.k.a. ANSI Y10.19-1969) -- 
 *  reaffirmed 1985.  The names must be unique.
 *
 *  NB: The short-prefix symbol corresponding to the prefix "micro" has been 
 *  changed here from the standard one (the Greek letter "mu") to the symbol 
 *  "u".
 */
#define PE(name, factor)    {name, factor, sizeof(name)-1}
static PrefixEntry	PrefixTable[]   = {
    PE("E",     1e18),
    PE("G",     1e9),
    PE("M",     1e6),
    PE("P",     1e15),
    PE("T",     1e12),
    PE("a",     1e-18),
    PE("atto",  1e-18),
    PE("c",     1e-2),
    PE("centi", 1e-2),
    PE("d",     1e-1),
    PE("da",    1e1),
    PE("deca",  1e1),	/* Spelling according to "ISO 2955: Information
			 * processing -- Representation of SI and other units
			 * in systems with limited character sets". */
    PE("deci",  1e-1),
    PE("deka",  1e1),	/* Spelling according to "ASTM Designation: E 380 - 85:
			 * Standard for METRIC PRACTICE" and "ANSI/IEEE Std
			 * 260-1978 (Reaffirmed 1985): IEEE Standard Letter 
			 * Symbols for Units of Measurement". */
    PE("exa",   1e18),
    PE("f",     1e-15),
    PE("femto", 1e-15),
    PE("giga",  1e9),
    PE("h",     1e2),
    PE("hecto", 1e2),
    PE("k",     1e3),
    PE("kilo",  1e3),
    PE("m",     1e-3),
    PE("mega",  1e6),
    PE("micro", 1e-6),
    PE("milli", 1e-3),
    PE("n",     1e-9),
    PE("nano",  1e-9),
    PE("p",     1e-12),
    PE("peta",  1e15),
    PE("pico",  1e-12),
    PE("tera",  1e12),
    PE("u",     1e-6),
    NULL
};

static UnitEntry	*root		= NULL;
static int	initialized	= 0;	/* module initialized = no */
static int	NumberBaseUnits	= 0;	/* number of base units */
static char	UtBuf[512];		/* input units-file buffer */
static char	BaseName[UT_MAXNUM_BASE_QUANTITIES][UT_NAMELEN];
static FILE	*UtFile;		/* input units-file */


/*
 *  Indicate whether or not the given unit-structure is a scalar.
 *
 *  This function returns:
 *	1	unit-structure is a scalar;
 *	0	unit-structure is not a scalar.
 */
    static int
IsaScaler(up)
    register utUnit	*up;
{
    register		status	= 1;	/* return status = yes */

    if (up->origin != 0) {
	status	= 0;
    } else {
	int	iquan	= 0;

	while (iquan < UT_MAXNUM_BASE_QUANTITIES)
	    if (up->power[iquan++] != 0) {
		status	= 0;
		break;
	    }
    }

    return status;
}


/*
 *  Clear a unit structure by setting it to the dimensionless identity
 *  unit structure.
 *
 *  This function returns a pointer to the unit structure.
 */
    utUnit*
utClear(unit)
    utUnit	*unit;
{
    register	iquan;

    unit->origin	= 0.0;
    unit->factor	= 1.0;

    for (iquan = 0; iquan < UT_MAXNUM_BASE_QUANTITIES; ++iquan)
	unit->power[iquan] = 0;

    return unit;
}


/*
 *  Copy a unit structure.
 *
 *  This function returns a pointer to the destination unit structure.
 */
    utUnit*
utCopy(source, dest)
    const utUnit	*source;
    utUnit		*dest;
{
    *dest	= *source;

    return dest;
}


/*
 *  Set the exponent of the given base quantity to unity.
 *
 *  This function returns:
 *	 NULL	error;
 *	!NULL	success (points to the input unit structure).
 */
    utUnit*
utSetPower(unit, position)
    utUnit	*unit;
    int		position;
{
    utUnit	*result	= NULL;
    static char	me[]	= "utSetPower";	/* this function's name */

    if (position < 0 || position >= UT_MAXNUM_BASE_QUANTITIES) {
	udadvise("%s: %d is an invalid quantity index.  Valid range is 0-%d",
	    me, UT_MAXNUM_BASE_QUANTITIES);
    } else {
	unit->power[position]	= 1;
	result	= unit;
    }

    return result;
}


/* 
 *  Shift the origin of a "unit" structure.
 *
 *  This function returns a pointer to the destination unit structure.
 */
    utUnit*
utShift(source, amount, result)
    utUnit     *source;
    double      amount;
    utUnit	*result;
{
    utCopy(source, result);
    result->origin  += amount*result->factor;

    return result;
}


/* 
 *  Scale a "unit" structure.
 *
 *  This function returns a pointer to the destination unit structure.
 */
    utUnit*
utScale(source, factor, result)
    utUnit     *source;
    double      factor;
    utUnit	*result;
{
    utCopy(source, result);
    result->factor	*= factor;
    result->origin	*= factor;

    return result;
}


/* 
 *  Multiply two unit-structures.
 *
 *  This function returns
 *	NULL			failure;
 *	pointer to `result'	success.
 */
    utUnit*
utMultiply(term1, term2, result)
    utUnit     *term1, *term2, *result;
{
    utUnit	*res	= NULL;
    static char	me[]	= "utMultiply";

    if ((!IsaScaler(term1) && term2->origin != 0) || 
	(!IsaScaler(term2) && term1->origin != 0)) {

	udadvise("%s: Can't multiply units with non-zero origins", me);
    } else {
	int         iquan;

	result->factor	= term1->factor * term2->factor;
	result->origin	= term1->origin == 0
			    ? term2->origin
			    : term1->origin;

	for (iquan = 0; iquan < UT_MAXNUM_BASE_QUANTITIES; ++iquan)
	    result->power[iquan]    = term1->power[iquan] + term2->power[iquan];

	res	= result;
    }

    return res;
}


/*
 *  Form the reciprocal of an internal unit specification.
 *
 *  This function returns
 *	NULL			failure;
 *	pointer to `result'	success.
 */
    utUnit*
utInvert(source, result)
    utUnit     *source, *result;
{
    utUnit	*res	= NULL;
    static char	me[]	= "utInvert";

    if (source->origin != 0) {
	udadvise("%s: Can't invert a unit with a non-zero origin", me);
    } else {
	int         iquan;

	result->factor	= 1./source->factor;
	result->origin	= 0.0;

	for (iquan = 0; iquan < UT_MAXNUM_BASE_QUANTITIES; ++iquan)
	    result->power[iquan]  = -source->power[iquan];

	res	= result;
    }

    return res;
}


/* 
 *  Divide two unit-structures.
 *
 *  This function returns
 *	NULL			failure;
 *	pointer to `result'	success.
 */
    utUnit*
utDivide(numer, denom, result)
    utUnit     *numer, *denom, *result;
{
    utUnit	*res	= NULL;
    static char	me[]	= "utDivide";

    if ((!IsaScaler(numer) && denom->origin != 0) ||
	(!IsaScaler(denom) && numer->origin != 0)) {

	udadvise( "%s: Can't divide units with non-zero origins", me);
    } else {
	int         iquan;

	result->factor	= numer->factor / denom->factor;
	result->origin	= numer->origin;

	for (iquan = 0; iquan < UT_MAXNUM_BASE_QUANTITIES; ++iquan)
	    result->power[iquan]    = numer->power[iquan] - denom->power[iquan];

	res	= result;
    }

    return res;
}


/* 
 *  Raise a unit-structure to a given power.
 *
 *  This function returns
 *	NULL			failure;
 *	pointer to `result'	success.
 */
    utUnit*
utRaise(source, power, result)
    utUnit     *source;
    int         power;
    utUnit	*result;
{
    utUnit	*res	= NULL;
    static char	me[]	= "utRaise";

    if (source->origin != 0) {
	udadvise("%s: Can't exponentiate a unit with a non-zero origin", me);
    } else {
	int         iquan;

	result->factor	= pow(source->factor, (double)power);
	result->origin	= 0.0;

	for (iquan = 0; iquan < UT_MAXNUM_BASE_QUANTITIES; ++iquan)
	    result->power[iquan]    = source->power[iquan] * power;

	res	= result;
    }

    return res;
}


/*
 *  Open a units-file.
 *
 *  This function returns:
 *	 0		success
 *	UT_ENOFILE	couldn't open file
 */
    static int
OpenUnits(filename)
    const char	*filename;
{
    int		status;			/* return status */
    static char	me[]	= "OpenUnits";	/* this function's name */

    if ((UtFile	= fopen(filename, "r")) == NULL) {
	uderror("fopen");
	udadvise("%s: Couldn't open units-file \"%s\"", me, filename);
	status	= UT_ENOFILE;
    } else {
	UtLineno	= 0;
	UtBuf[sizeof(UtBuf)-1]	= 0;
	status	= 0;
    }

    return status;
}


/*
 * Close a units file.
 *
 * This function returns void.
 */
    static void
CloseUnits()
{
    if (UtFile != NULL)
	(void)fclose(UtFile);
}


/*
 *  Decode a unit specification.
 *
 *  This function returns:
 *	 0		success
 *	UT_ESYNTAX	syntax error
 *	UT_EUNKNOWN	unknown specification
 */
    static int
DecodeUnit(spec, unit)
    char	*spec;
    utUnit	*unit;
{
    /* Remove leading whitespace */
    while (isspace(*spec))
	++spec;

    pointer = buffer    = spec;

    (void)utClear(FinalUnit = unit);

    UnitNotFound	= 0;
    return utParseSpec() == 0 ? 0 : UnitNotFound ? UT_EUNKNOWN : UT_ESYNTAX;
}


/*
 *  Scan (decode) the next entry in the units file.
 *  If the specification is for a base unit, then the returned unit-structure
 *  is cleared except that the value of the next available exponent slot is 
 *  set to one.  A base unit is the unit for a fundamental (base) physical
 *  quantity (e.g. the unit `meter' for the base physical quantity `length').
 *
 *  This function returns:
 *	0		success
 *	UT_EOF		end-of-file encountered
 *	UT_EIO		I/O error
 *	UT_ESYNTAX	syntax error
 *	UT_EUNKNOWN	unknown specification
 *	UT_EALLOC	allocation failure
 */
    static int
ScanUnit(name, sizename, unit, HasPlural)
    char	*name;			/* name buffer */
    int		sizename;		/* size of name buffer */
    utUnit	*unit;			/* unit specification */
    int		*HasPlural;		/* specification has a plural form? */
{
    int		status	= 0;		/* return status = success */
    static char	me[]	= "ScanUnit";	/* this function's name */

    for (;;) {
	char		*cp;
	static char	WhiteSpace[]	= " \t";

	++UtLineno;
	if (fgets(UtBuf, (int)sizeof(UtBuf), UtFile) == NULL) {
	    if (feof(UtFile)) {
		status	= UT_EOF;
	    } else {
		uderror("fgets");
		status	= UT_EIO;
	    }
	    break;
	}
	
	if (UtBuf[strlen(UtBuf)-1] != '\n') {
	    udadvise("%s: Input-line longer than %ul-byte buffer",
		me, (unsigned long)sizeof(UtBuf));
	    status	= UT_ESYNTAX;
	    break;
	}

	/* Truncate at comment character */
	if ((cp = strchr(UtBuf, '#')) != NULL)
	    *cp	= 0;

	/* Trim trailing whitespace */
	for (cp = UtBuf + strlen(UtBuf); cp > UtBuf; --cp)
	     if (!isspace(cp[-1]))
		break;
	*cp	= 0;

	/* Find first "black" character */
	cp	= UtBuf + strspn(UtBuf, WhiteSpace);

	if (*cp != 0) {
	    int		n	= strcspn(cp, WhiteSpace);
	    char	buf[512];

	    assert(sizeof(buf) > strlen(UtBuf));
	    assert(sizename > strlen(UtBuf));

	    strncpy(name, cp, n)[n]	= 0;

	    cp	+= n;
	    cp	+= strspn(cp, WhiteSpace);
	    n	= strcspn(cp, WhiteSpace);
	    strncpy(buf, cp, n)[n]	= 0;

	    if (strcmp(buf, "P") == 0) {
		*HasPlural	= 1;
	    } else if (strcmp(buf, "S") == 0) {
		*HasPlural	= 0;
	    } else {
		udadvise("%s: Invalid single/plural indicator \"%s\"", 
		    me, buf);
		status	= UT_ESYNTAX;
		break;
	    }

	    cp	+= n;
	    cp	+= strspn(cp, WhiteSpace);
	    (void)strcpy(buf, cp);

	    if (buf[0] == 0) {
		(void)utClear(unit);
		if (utSetPower(unit, NumberBaseUnits) == NULL) {
		    udadvise("%s: Couldn't set base unit #%d", 
			me, NumberBaseUnits);
		    status	= UT_EALLOC;
		} else {
		    (void)strncpy(BaseName[NumberBaseUnits], name, 
				  UT_NAMELEN-1);
		    ++NumberBaseUnits;
		}
	    } else {
		if ((status = DecodeUnit(buf, unit)) != 0) {
		    udadvise("%s: Couldn't decode \"%s\" definition \"%s\"", 
			me, name, buf);
		}
	    }
	    break;
	}					/* if not a layout line */
    }						/* input-line loop */

    if (status != 0 && status != UT_EOF)
	udadvise("%s: Error occurred at line %d", me, UtLineno);

    return status;
}


/*
 * Read and decode the entries in the units-file, adding them to the units-
 * table.
 *
 * This function returns:
 *	0		success
 *	UT_ENOFILE	no units-file
 *	UT_ESYNTAX	syntax error
 *	UT_EUNKNOWN	unknown specification
 *	UT_EALLOC	allocation failure
 *	UT_EIO		I/O error
 */
    static int
ReadUnits(path)
    const char	*const path;
{
    int		status;				/* return status */
    static char	me[]	= "ReadUnits";		/* this function's name */

    if ((status = OpenUnits(path)) == 0) {
	for (;;) {
	    int		HasPlural;
	    char	name[512];
	    utUnit	unit;

	    if ((status = ScanUnit(name, sizeof(name), &unit, 
				   &HasPlural)) == UT_EOF) {
		status	= 0;
		break;
	    } else if (status == 0) {
		if ((status = utAdd(name, HasPlural, &unit)) != 0) {
		    udadvise("%s: Couldn't add \"%s\" to units-table",
			me, name);
		    break;
		}
	    } else {
		udadvise("%s: Couldn't read units-file \"%s\"", me, path);
		break;
	    }
	}

	CloseUnits();
    }						/* units-file opened */

    return status;
}


/*
 *  Initialize the units(3) package.
 *
 *  This function returns:
 *	0		success
 *	UT_ENOFILE	no units-file
 *	UT_ESYNTAX	syntax error in units-file
 *	UT_EUNKNOWN	unknown specification in units-file
 *	UT_EIO		units-file I/O error
 *	UT_EALLOC	allocation failure
 */
    int
utInit(path)
    const char	*path;
{
    int		status;

    if (path == NULL || path[0] == 0)
	path	= DEFAULT_PATH;

    status	= ReadUnits(path);

    if (status == 0) {
	initialized	= 1;
    } else {
	utTerm();
    }

    return status;
}


/*
 *  Decode a unit specification.
 *
 *  This function returns:
 *	0		success
 *	UT_ENOINIT	the package hasn't been initialized
 *	UT_EUNKNOWN	unknown specification
 *	UT_ESYNTAX	syntax error
 */
    int
utScan(spec, up)
    char	*spec;
    utUnit     *up;
{
    int         status  = 0;                    /* return status = success */
    static char	me[]	= "utScan";		/* this function's name */

    if (!initialized) {
	udadvise("%s: Package hasn't been initialized", me);
	status	= UT_ENOINIT;
    } else {
	status	= DecodeUnit(spec, up);
    }

    return status;
}


/*
 *  Encode a unit-structure into a formatted unit-secification.
 *
 *  This function returns:
 *	0		success
 *	UT_ENOINIT	the package hasn't been initialized
 *	UT_EINVALID	the unit-structure is invalid
 *
 *  On error, the string argument is set to the nil-pointer.
 */
    int
utPrint(unit, s)
    register utUnit	*unit;
    char		**s;
{
    int			status;
    static char		me[]	= "utPrint";

    if (!initialized) {
	udadvise("%s: Package hasn't been initialized", me);
	*s	= NULL;
	status	= UT_ENOINIT;

    } else {
	if (unit->factor == 0.0) {
	    *s		= NULL;
	    status	= UT_EINVALID;

	} else {
	    register		iquan;
	    register char	*buf	= UtBuf;

	    *buf	= 0;

	    /*	Print the scale factor if it's non-unity. */
	    if (unit->factor != 1.0) {
		(void)sprintf(buf, "%g ", unit->factor);
		buf	+= strlen(buf);
	    }

	    /*	Append the dimensions in terms of base quanitities. */
	    for (iquan = 0; iquan < UT_MAXNUM_BASE_QUANTITIES; ++iquan) {
		if (unit->power[iquan] != 0) {
		    if (unit->power[iquan] == 1) {
			(void)sprintf(buf, "%s ", BaseName[iquan]);
		    } else {
			(void)sprintf(buf, "%s%d ", BaseName[iquan],
				      unit->power[iquan]);
		    }
		    buf	+= strlen(buf);
		}
	    }

	    /*	Append the origin-shift if it's non-zero. */
	    if (unit->origin != 0.0) {
		(void)sprintf(buf, "@ %g ", unit->origin);
		buf	+= strlen(buf);
	    }

	    /*	Remove trailing space. */
	    if (buf > UtBuf)
		buf[-1]	= 0;

	    *s		= UtBuf;
	    status	= 0;
	}					/* unit-structure is valid */
    }						/* package is initialized */

    return status;
}


/*
 *  Compare two strings.
 *
 *  This routine is used instead of strncmp(3) in order to obtain greater
 *  control over the returned value.
 *
 *  This function returns the index of the first, non-matching character or
 *  the length of the strings if all characters match.
 */
    static ptrdiff_t
mystrcmp(s1, s2)
    register const char	*s1, *s2;
{
    register const char	*start	= s1;

    for (; *s1 != 0 && *s2 != 0 && *s1 == *s2; ++s1, ++s2)
	continue;

    return s1 - start;
}


/*
 *  Compare two nodes in the units-table by comparing their names.
 *
 *  This function returns:
 *	<0	first argument less than second
 *	=0	first and second arguments are equal
 *	>0	first argument greater than second
 */
    static int
CompareNodes(node1, node2)
    const voidp	node1;
    const voidp	node2;
{
    register const char	*name1	= ((UnitEntry*)node1)->name;
    register const char	*name2	= ((UnitEntry*)node2)->name;
    register		status	= name1[0] - name2[0];	/* quick comparison */

    if (status == 0) {
	/*
	 * Quick comparison failed.  Perform a long comparison.
	 */
	status	= mystrcmp(name1, name2);
	status	= name1[status] - name2[status];
    }

    return status;
}


/*
 *  Compare a target and a node in the units-table by comparing their names.
 *  Allow the target name to be the plural of the node name if appropriate.
 *
 *  This function returns:
 *	<0	first argument less than second
 *	=0	first and second arguments are equal
 *	>0	first argument greater than second
 */
    static int
FindNodes(node1, node2)
    const voidp		node1;
    const voidp		node2;
{
    register const char		*name1	= ((UnitEntry*)node1)->name;
    register const char		*name2	= ((UnitEntry*)node2)->name;
    register			status	= name1[0] - name2[0];
						/* quick comparison */

    if (status == 0) {
	/*
	 * Quick comparison failed.  Perform a long comparison.
	 */
	int	i	= mystrcmp(name1, name2);

	status	= name1[i] - name2[i];

	if (status == 's') {
	    UnitEntry	*n1	= (UnitEntry*)node1;
	    UnitEntry	*n2	= (UnitEntry*)node2;

	    if (n2->HasPlural && i == n2->nchr && n2->nchr+1 == n1->nchr)
		status	= 0;
	}
    }

    return status;
}


/*
 *  Create a new node.
 *
 *  This function returns:
 *	!=NULL	success
 *	==NULL	failure
 */
    static UnitEntry*
CreateNode(name, HasPlural, unit)
    const char		*name;
    int			HasPlural;
    const utUnit	*unit;
{
    int		nchr	= strlen(name);
    UnitEntry	*node	= NULL;
    static char	me[]	= "NewNode";		/* this function's name */

    if (nchr+1 > UT_NAMELEN) {
	udadvise("%s: The name \"%s\" is too long", me, name);
    } else {
	node	= UD_ALLOC(1, UnitEntry);

	if (node == NULL) {
	    udadvise("%s: Couldn't allocate new entry", me);
	} else {
	    if ((node->name = DUPSTR(name)) == NULL) {
		udadvise("%s: Couldn't duplicate name", me);
		FREE(node);
	    } else {
		node->nchr	= strlen(node->name);
		node->HasPlural	= HasPlural;
		utCopy(unit, &node->unit);
	    }
	}
    }

    return node;
}


/*
 *  Destroy a node.
 *
 *  This function returns void.
 */
    static void
DestroyNode(node)
    UnitEntry	*node;
{
    if (node != NULL) {
	if (node->name != NULL)
	    FREE(node->name);
	FREE(node);
    }
}
    

/*
 *  Add a unit-structure to the units-table.
 *
 *  This function returns:
 *	0		success
 *	UT_EALLOC	memory allocation failure
 */
    int
utAdd(name,  HasPlural, unit)
    const char		*name;
    int			HasPlural;
    const utUnit	*unit;
{
    int		status	= 0;			/* return status = success */
    UnitEntry	*nodep	= CreateNode(name, HasPlural, unit);
    static char	me[]	= "utAdd";		/* this function's name */

    if (nodep != NULL) {
	UnitEntry	**found	= (UnitEntry**)tsearch((voidp)nodep, 
						       (voidp*)&root, 
						       CompareNodes);

	if (found == NULL) {
	    udadvise("%s: Couldn't expand units-table", me);
	    status	= UT_EALLOC;
	} else if (*found != nodep) {
	    udadvise("%s: Replacing unit \"%s\"", me, name);
	    DestroyNode(*found);
	    *found	= nodep;
	}
    }

    return status;
}


/*
 *  Find the entry in the units-table corresponding to a given name.  
 *  If an entry isn't found, try again using the singular form, if
 *  appropriate.
 *
 *  This function returns:
 *	 NULL	entry not found;
 *	!NULL	entry found.
 */
    static UnitEntry*
FindUnit(name)
    const char	*name;
{
    UnitEntry	node;
    UnitEntry	**found;

    node.name	= (char*)name;
    node.nchr	= strlen(name);

    found	= (UnitEntry**)tfind((voidp)&node, (voidp*)&root, FindNodes);

    if (found == NULL) {
	/*
	 * Not found.  If appropriate, try again with singular form.
	 */
	if (node.nchr > 1 && node.name[node.nchr-1] == 's') {
	    char	buf[UT_NAMELEN];

	    assert(sizeof(buf) > node.nchr-1);

	    node.name	= strncpy(buf, name, --node.nchr);
	    node.name[node.nchr]	= 0;

	    found	= (UnitEntry**)tfind((voidp)&node, (voidp*)&root, 
					     FindNodes);
	}
    }

    return found == NULL ? NULL : *found;
}


/*
 *  Find the entry in a prefix-table corresponding to a possible
 *  prefix.  A linear-search of the prefix-table is performed.
 *  An attempt is made to insure that the longest, possible, matching
 *  entry is returned.
 *
 *  NB: A binary-search of the table is not possible because, for example,
 *  the prefix-entry for the input specification "mzmeters" (where "mz" is 
 *  a made-up prefix) would be after the entry "micro", but the prefix-entry 
 *  corresponding to the specification "mm" (i.e. "m") lies before "micro".
 *  Thus, the binary-search comparison-function can't indicate which direction
 *  to go.
 *
 *  This function returns:
 *	 NULL	not found
 *	!NULL	found
 */
    static PrefixEntry*
FindPrefix(spec)
    const char	*spec;
{
    PrefixEntry			*found	= NULL;
    register PrefixEntry	*entry;

    for (entry = PrefixTable; entry->name != NULL; ++entry) {
	register			status;

	if (entry->name[0] - spec[0] < 0 ||
		(status = strncmp(entry->name, spec, entry->nchar)) < 0)
	    continue;

	if (status > 0)
	    break;

	if (found == NULL || found->nchar < entry->nchar)
	    found	= entry;
    }

    return found;
}


/*
 *  Return the unit-structure corresponding to a unit-specification.
 *
 *  NB:
 *	It is permissible for the specification to consist solely
 *	of a prefix (e.g "milli").
 *
 *	An empty specification returns a dimensionless, unity unit-structure.
 *
 *	On failure, the output unit-structure is unmodified.
 *
 *  This function returns:
 *	0		found (the output unit-structure is set).
 *	UT_ENOINIT	the units-table hasn't been initialized
 *	UT_EUNKNOWN	not found
 */
    int
utFind(spec, up)
    const char	*spec;
    utUnit	*up;
{
    int		status	= 0;		/* return status = found */
    UnitEntry	*entry	= NULL;
    double	factor	= 1;
    static char	me[]	= "utFind";	/* this function's name */

    if (root == NULL) {
	udadvise("%s: Units-table hasn't been initialized", me);
	status	= UT_ENOINIT;
    } else {
	while (*spec != 0) {
	    PrefixEntry	*PrefixEnt;

	    /*
	     *  See if the specification is an isolated unit (e.g. "meter").
	     *  We're done if it is.
	     */
	    if ((entry = FindUnit(spec)) != NULL)
		break;

	    /*
	     *  See if the specification has a multiplying prefix.  If 
	     *  so, then use the prefix's dimensionless value, skip 
	     *  over the prefix characters, and rescan.
	     */

	    if ((PrefixEnt = FindPrefix(spec)) != NULL) {
		factor	*= PrefixEnt->factor;
		spec	+= strlen(PrefixEnt->name);
		continue;
	    }
	    
	    status	= UT_EUNKNOWN;
	    break;
	}					/* while something to decode */
    }						/* units-table is initialized */

    if (status == 0)
	(void)utScale(entry == NULL ? utClear(up) : &entry->unit,
		      factor, up);

    return status;
}


/*
 *  Compute the conversion factor between two unit-structures.
 *
 *  This function returns:
 *	0		success.
 *	UT_ENOINIT	the units-table hasn't been initialized
 *	UT_EINVALID	a structure is invalid
 *	UT_ECONVERT	the structures are not convertable
 */
    int
utConvert(from, to, slope, intercept)
    utUnit	*from;
    utUnit	*to;
    double	*slope, *intercept;
{
    int		status	= 0;
    static char	me[]	= "utConvert";

    if (!initialized) {
	udadvise("%s: Package hasn't been initialized", me);
	status	= UT_ENOINIT;
    } else {
	if (from->factor == 0.0 || to->factor == 0.0) {
	    status	= UT_EINVALID;
	} else {
	    register	iquan;

	    for (iquan = 0; iquan < UT_MAXNUM_BASE_QUANTITIES; ++iquan)
		if (from->power[iquan] != to->power[iquan]) {
		    status	= UT_ECONVERT;
		    break;
		}

	    if (status == 0) {
		*slope		= from->factor/to->factor;
		*intercept	= (from->origin - to->origin)/to->factor;
	    }
	}
    }						/* package is initialized */

    return status;
}


/*
 *  Free allocated nodes.
 */
    static void
NodeWalker(node, order, level)
    UnitEntry	**node;
    VISIT	order;
    /*ARGSUSED*/
    int		level;
{
    if (order == postorder || order == leaf)
	DestroyNode(*node);
}


/*
 *  Terminate use of this package.
 *
 *  This function returns void.
 */
    void
utTerm()
{
    if (root != NULL) {
	twalk((voidp)root, NodeWalker);
	FREE(root);
	root		= NULL;
    }
    initialized	= 0;
}
