/*LINTLIBRARY*/

#include "udposix.h"		/* get NeedFunctionPrototypes defs */
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <stdlib.h>
#include <limits.h>		/* for _POSIX_PATH_MAX */

#include "udalloc.h"
#include "uderrmsg.h"
#include "cmdline.h"
#include "valuestring.h"
#include "udpath.h"
#include "udres.h"
#include "udrm.h"
#include "dectab.h"
#include "stringlist.h"

#ifndef PATH_MAX
#   define PATH_MAX	_POSIX_PATH_MAX
#endif

#ifndef lint
    static char	rcsid[]	= "$Id: udres.c,v 2.22 1991/07/31 20:30:35 steve Exp $";
    static char	afsid[]	= "$__Header$";
#endif

extern char	*udstrdup();
extern char	*guisevers();		/* Returns GUISE version-string */

/* What character separates the filename from the path? */
#ifdef	vms
#   define SEP		']'
#   define CWD		"[]"
#   define LISTSEP 	','	/* ??? */
#else
#   define SEP		'/'
#   define CWD		"."
#   define LISTSEP	':'
#endif	/* vms undefined */

/*
 * Return zero if unacceptable as a character in a resource name or class.
 * NB: Periods are allowed in order to keep the ".out" in "a.out".
 */
#define isResChar(c) \
	(isalnum((c)) || (c) == '_' || (c) == '-' || (c) == '.')

#define MAX(a,b)	((a) > (b) ? (a) : (b))
#define MIN(a,b)	((a) < (b) ? (a) : (b))
#define	ABS(x)		((x) < 0 ? -(x) : (x))

static UrmDatabase	uRdb;			/* resource database */
static int		ErrorHappened;		/* An error occurred? */
static int		DoStore;		/* Store parameters? */
static char		DefaultuName[]	= "<unknown>";	/* default progname */
static char		*uName		= DefaultuName;	/* program name */
static char		*uClass		= NULL;	/* program class */
static DtPtr		dectab		= NULL;	/* Decode-table */


/*
 * returns a newly allocated string which is "str1.str2"
 */

    static char*
dotcat( str1, str2 )
    char *str1;		/* first string */
    char *str2;		/* second string */
{
#   define DOT '.'
    char	*new;
    int		len1 = 0;
    int		len2 = 0;
    static char	me[]	= "dotcat";

    if( str1 != NULL )
	len1 = strlen(str1);

    if( str2 != NULL )
	len2 = strlen(str2);

    if ((new = UD_ALLOC(len1+len2+2, char)) == NULL) {
	udadvise("%s: Couldn't allocate output buffer.", me);
    } else {
	if( str1 != NULL )
	    (void)strcpy(new, str1);

	new[len1]		= DOT;
	new[len1 +1]	= 0;

	if( str2 != NULL )
	    (void)strcat(new, str2);
/*
#	ifndef NDEBUG
	(void)fprintf(stderr,"\tdotncat %s\n", new);
#	endif
*/
    }

    return new;
}


/*
 * Return zero if acceptable as resource name or class
 */

    static int
badname( str )
     char *str;
{
    int len = 0;

    if(str == NULL || *str == 0)
	return ~0;

    for(len = 0; *str != 0; len++, str++ ) {
	if(!isResChar(*str))
	    return ~0;
	if(len >= PATH_MAX )
	    return ~0;
    }

    return 0;
}



    static int
MergeFileDatabase(filename)
     char *filename;
{
    int		success	= 0;
    UrmDatabase fileDB;

    assert(filename != NULL);
    
    if (UrmGetFileDatabase(filename, &fileDB)) {
	UrmMergeDatabases(fileDB, &uRdb);
	success	= 1;
    }

    return success;
}



    static char *
catfname( dir, name, suf )
    char	*dir;		/* Needs to have modifiable space at the end */
    char	*name;
    char	*suf;
{
    if( dir == NULL || *dir == 0 || name == NULL || *name == 0 )
	return NULL;
#ifndef vms
    {
	char *cp;

	cp = dir + strlen(dir);
	if(*(cp-1) != SEP) {
	    *cp++ = SEP;
	    *cp = 0;
	}
    }
#endif
    (void)strcat(dir, name);
    (void)strcat(dir, suf);
/*
#ifndef NDEBUG
    (void)fprintf(stderr,"\t\tcatfname  %s\n", dir);
#endif
*/
    return dir;
}



    static void
clrfname( path )
    char	*path;		/* Needs to be modifiable */
{
    char *cp;

    if ((cp = strrchr(path, SEP)) == NULL)
	return;

    *(cp +1 ) = 0;
}


/*
 *	Merge resources specified by the environmental variables UDRESPATH
 *	and UDRESOURCES.  If it exists, UDRESPATH contains a colon-separated
 *	list of resource-files and/or directories.  The files and directories
 *	of UDRESPATH are merged in the reverse order in which they appear
 *	(i.e. they're merged right to left).  Any components of UDRESPATH that
 *	do not exist are simply ignored.  If a directory name appears, it is 
 *	treated as if it were the two resource files "<name>.res" and
 *	"<class>.res" in that directory, where <name> and <class> are the name
 *	and class, respectively, of this program as specified to udinit().  If
 *	UDRESPATH doesn't exist, then the current working directory is used.
 *
 *	UDRESOURCES, if it exists, contains a string-specification of one or
 *	more resources in the manner of the X routine XrmGetStringDatabase().
 */

    static int
merge_resource_dbs()
{
    bool	success	= NO;		/* Routine status */
    char	*cp;
    char	*pathlist;
    static char	me[]	= "merge_resource_dbs";

    pathlist	= getenv(UDRESPATH);
    if (pathlist == NULL)
	    pathlist = CWD;

    /* 
     *	We may need to add some characters on the end, so make a copy 
     */
    if ((cp = UD_ALLOC(strlen(pathlist)+PATH_MAX, char)) == NULL) {
	udadvise("%s: Couldn't allocate buffer.", me);
    } else {
	(void)sprintf(cp,"%c%s", LISTSEP, pathlist);

	pathlist	= cp;

	while((cp = strrchr(pathlist,LISTSEP)) != NULL) {
	    int		type, access;

	    cp++;

	    /* The following is algorithmically slow */
	    if (udpath(cp, &type, &access)) {
		if (type == UD_FREG) {
		    (void)MergeFileDatabase(cp);

		} else if (type == UD_FDIR) {
		    (void)MergeFileDatabase(catfname(cp, uClass, URES_SUFFIX));
		    clrfname(cp);
		    (void)MergeFileDatabase(catfname(cp, uName, URES_SUFFIX));
		}
	    }

	    *(cp-1) = 0;
	}

	FREE(pathlist);

	/* Add the resources specified in the environmental variable
	 * UDRESOURCES to the resource database
	 */
	{
	    char	*EnvResString	= getenv("UDRESOURCES");

	    if (EnvResString == NULL) {
		success	= YES;

	    } else {
		UrmDatabase	EnvDb;
		
		if (!UrmGetStringDatabase(EnvResString, &EnvDb)) {
		    udadvise("%s: Couldn't create environmental database.",
			me);
		} else {
		    UrmMergeDatabases(EnvDb, &uRdb);
		    success	= YES;
		}
	    }
	}
    }

    return success;
}


/*
 * WHAT:   Return a string containing the type of value decoded by the given
 *	   function.
 *
 * HOW:    Ask the converter-function to return a mnemonic string.  If it 
 *	   doesn't, then use the generic mnemonic string.
 *
 * INPUT:  Pointer to a decode-function;
 *
 * OUTPUT: Pointer to a string.
 */

    static char*
ValueTypeString(funct)
    UdConverter	*funct;
{
    char	*vs;
    static char	def[]	= "val";
    static char	me[]	= "ValueTypeString";

    if (funct == NULL) {
	vs	= udstrdup(def);
    } else {
	char	*string;

    	if (!(*funct)((VOIDP)&string, (VOIDP)NULL, UD_MNEMONIC)) {
	    vs	= udstrdup(def);
	} else {
	    if ((vs = VsJoin(1, &string)) == NULL)
		udadvise("%s: Couldn't create value-string.", me);
	}
    }

    return vs;
}


/*
 * WHAT:   Print a usage message.
 *
 * HOW:    Use the keyed-parameter and positional-parameter tables to print
 *	   a usage statement on the standard output.
 *
 * OUTPUT: Prints a message on the standard output.
 */

    void
udusage()
{
    int		errmode	= uderrmode(UD_VERBOSE);
    DtPtr	dt	= dectab;
    bool	title	= NO;
    bool	UserKeyedParameters	= NO;
    static char	me[]	= "udusage";

#   define PUTS(x)	(void)fputs(x, stderr)
#   define PUT(x)	(void)fputc(x, stderr)
#   define TAB()	PUT('\t')
#   define NL()		PUT('\n')

    if (dt == NULL) {
	udadvise("%s: Decode-table not initialized", me);
    } else {
	DtReset(dt);

	NL();

	(void)fprintf(stderr, "Usage for program \"%s\":\n", uName);

	PUTS("\tKeyed-parameters:\n");

	while (DtNextKey(dt)) {
	    int	nval	= DtNval(dt);
	    int	isopt	= !DtRequired(dt);

	    TAB(); TAB();

	    if (isopt)
		PUT('[');

	    PUT('-');
	    PUTS(DtName(dt));

	    if (nval != 0) {
		char	*type	= ValueTypeString(DtFunct(dt));
		int		optval	= nval == UDARB || nval < 0;

		PUT(' ');

		if (optval)
		    PUT('[');

		PUTS(type);

		if (nval == UDARB || nval < -1)
		    PUT('[');

		if (nval == UDARB || nval < -1 || nval > 1)
		    PUTS(",...");

		if (nval == UDARB || nval < -1)
		    PUT(']');

		if (optval)
		    PUT(']');

		if (nval == UDARB) {
		    PUTS("\tArbitrary number of values");
		} else if (nval < -1) {
		    (void)fprintf(stderr, "\tUp to %d values", -nval);
		} else if (nval > 1) {
		    (void)fprintf(stderr, "\tExactly %d values", nval);
		}

		(void)FREE(type);
	    }					/* if non-boolean parameter */

	    if (isopt)
		PUT(']');

	    NL();
	    UserKeyedParameters	= YES;
	}					/* keyed-parameter loop */

	if (UserKeyedParameters)
	    NL();
	PUTS("\t\t[-name string]\n");
	PUTS("\t\t[-usage]\n");
	PUTS("\t\t[-udversion]\n");
	PUTS("\t\t[-udrm name:value[,...]\tArbitrary number of values]\n");

	title	= NO;

	while (DtNextPos(dt)) {
	    int	nval	= DtNval(dt);
	    char	*type	= ValueTypeString(DtFunct(dt));
	    int	isopt	= !DtRequired(dt);


	    if (!title) {
		NL();
		TAB();
		PUTS("Positional-parameters:\n");
		title	= YES;
	    }

	    TAB(); TAB();
	    PUTS(DtName(dt));
	    PUT(':');
	    TAB();

	    if (isopt)
		PUT('[');

	    PUTS(type);

	    if (nval == UDARB || nval < -1 || nval > 1)
		PUTS(" ...");

	    if (nval == UDARB) {
		PUTS("\tArbitrary number of values");
	    } else if (nval < -1) {
		(void)fprintf(stderr, "\tUp to %d values", -nval);
	    } else if (nval > 1) {
		(void)fprintf(stderr, "\tExactly %d values", nval);
	    }

	    if (isopt)
		PUT(']');

	    NL();
	    (void)FREE(type);
	}					/* positional-parameter loop */

	NL();

	DtReset(dt);

	(void)uderrmode(errmode);
    }
}


/*
 * WHAT    Store command-line specified resources into the resource database.
 *
 * HOW     Call the X resource-manager storing routine using the
 *	   values from the command-line as the resource name/value strings.
 *
 * INPUT   Because this routine appears to be a regular decode-function, the
 *	   parameters are the string representation, the binary destination,
 *	   and the mode.  Note that not all of the parameters are used.
 *
 * OUTPUT  Success flag: 0 => failure; 1 => success.
 */

/*ARGSUSED*/
    static int
CvtUdrm(str, bin, mode)
    VOIDP	str;
    VOIDP	bin;
    int		mode;
{
    if (mode == UD_TO_BINARY) {
	assert(str != NULL);
	UrmPutLineResource(&uRdb, (char*)str);
    }

    return 1;
}


/*
 * WHAT    Decode a command-line-specified program name.
 *
 * HOW     Strip-off any leading path specification, check that the result is
 *	   a valid resource component, and then modify the global program-name 
 *	   variable.
 *
 * INPUT   Because this routine appears to be a regular decode-function, the
 *	   parameters are the string representation, the binary destination,
 *	   and the mode.  Note that not all of the parameters are used.
 *
 * OUTPUT  The number of successfully decoded vaues.
 */

    static int
CvtName(str, bin, mode)
    VOIDP	str;
    VOIDP	bin;
    int		mode;
{
    int		status	= 0;		/* Routine status = failure */
    static char	me[]	= "CvtName";	/* This routine's name */

    if (UdString(str, bin, mode)) {

	if (mode == UD_MNEMONIC) {
	    *(char**)str	= "ProgName";

	} else if (mode == UD_TO_BINARY) {
	    char	*buf	= udstrdup((char*)str);	/* name buffer */
	    char	*name;				/* actual name */

	    /* Strip-off prefix */
	    if ((name = strrchr(buf, SEP)) == NULL) {
		name	= buf;
	    } else {
		++name;
	    }

	    /* Strip-off suffix */
	    {
		char	*cp	= strchr(name, '.');

		if (cp != NULL)
		    *cp	= 0;
#if vms
		if ((cp = strchr(name, ';')) != NULL)
		    *cp	= 0;
#endif
	    }

	    if (badname(name)) {
		udadvise("%s: %s \"%s\" %s", me, "Program name", name,
		    "is not acceptable as a resource component.");
	    } else {
		if (uName != DefaultuName)
		    (void)FREE(uName);
		if ((name = udstrdup(name)) == NULL) {
		    udadvise("%s: Couldn't copy name.", me);
		} else {
		    uName	= name;
		    status	= 1;
		}
	    }

	    (void)FREE(buf);
	}
    }

    (void)uderrname(uName);

    return status;
}


/*
 * WHAT:   Set the program-name and the program-class name.
 *
 * HOW:    Handle the program name by using the static function `name', which
 *	   is used for `decoding' a command-line program-name specification.
 *	   If the class name is not given, then capitalize the first letter of 
 *	   the program name.
 *
 * INPUT:  Program name and (possibly empty) program-class name.
 *
 * OUTPUT: Success flag: 0 => failure; 1 => success.
 */

    static int
SetNames(ProgName, ClassName)
    char	*ProgName;	/* name of program */
    char	*ClassName;	/* name of program class.  NULL => program
				 * name with 1st letter capitalized */
{
    int		status	= 0;		/* routine status = failure */
    static char	me[]	= "SetNames";	/* this routine's name */

    assert(ProgName != NULL);

    /*
     *	The following line causes the SunOS 4.1 lint(1) to generate a warning
     *	about "possible pointer alignment problem.  This may safely be
     *	ignored.
     */
    if (CvtName((VOIDP)ProgName, (VOIDP)NULL, UD_TO_BINARY)) {
	if (ClassName == NULL || ClassName[0] == 0) {
	    uClass	= udstrdup(uName);
	    assert(uClass != NULL);
	    if (islower(uClass[0]))
		uClass[0]	= toupper(uClass[0]);
	} else {
	    uClass	= udstrdup(ClassName);
	    assert(uClass != NULL);
	}

	if (badname(uClass)) {
	    udadvise("%s: %s \"%s\" %s", me, "Class name", uClass,
		"is not acceptable as a resource component.");
	    (void)FREE(uClass);
	    uClass	= NULL;
	} else {
	    status	= 1;
	}
    }

    if (!status) {
	(void)FREE(uName);
	uName	= NULL;
    }

    return status;
}


/*
 * WHAT:   Return the number of values for the given item.
 *
 * HOW:    Compare the desired number with the potential number and work 
 *	   something out.
 *
 * INPUT:  The number of desired vaues and the number of potential values.
 *
 * OUTPUT: The maximum number of values obtainable.  Less than zero implies 
 *	   too few values.
 */

    static int
NumVal(nwant, nhave)
    int		nwant;		/* desired number of values */
    int		nhave;		/* potential number of values */
{
    int		nval;			/* the number of values */

    assert(nhave >= 0);

    if (nwant == UDARB) {			/* if arbitrary-number */
	nval	= nhave;
    } else if (nwant > 0) {			/* if fixed-number */
	if (nwant > nhave) {			/* if too few values */
	    nval	= -nhave;
	} else {
	    nval	= nwant;
	}
    } else {					/* if maximum-number */
	nval	= MIN(nhave, -nwant);
    }

    return nval;
}


/*
 * WHAT:   Return the number of values for the current item.
 *
 * HOW:    Compare the desired number (from the decode table) with the 
 *	   potential number (from the command-line object) and work something 
 *	   out.
 *
 * INPUT:  Valid pointers to valid decode-table and command-line objects.
 *
 * OUTPUT: The maximum number of values obtainable.  Less than zero implies 
 *	   too few values.
 */

    static int
Nval(dt, cl)
    DtPtr	dt;		/* Decode-table */
    ClPtr	cl;		/* Command-line object */
{
    int		nwant;			/* desired number of values */
    int		nval;			/* the number of values */
    int		nhave;			/* potential number of values */
    static char	me[]	= "Nval";	/* this routine's name */

    assert(dt != NULL);
    assert(cl != NULL);

    nwant	= DtNval(dt);
    nhave	= ClNval(cl);

    assert(nhave >= 0);

    if (nwant == UDARB || nwant < 0) {
	/* 
	 * Arbitrary or up-to-maximum number of values wanted.
	 */
	int	nleave	= DtNleave(dt);		/* number to leave behind */
	int	nleft	= ClNleft(cl);		/* Max-number values left */
	int	navail	= nleft - nleave;	/* Max-number available */

	/*
	 * NB: For positional-parameters, `nleft' can be greater than `nhave'
	 * due to intervening keyed-parameters.
	 */

	assert(nleave >= 0);
	assert(nleft >= 0);
	assert(nleft >= nhave);

	nval	= navail <= 0
		    ? 0
		    : nwant == UDARB
			? MIN(navail,  nhave)
			: MIN(navail, -nwant);

    } else if (nwant >= 0) {
	/*
	 * Fixed number of values wanted.
	 */
	if (nwant > nhave) {
	    udadvise("%s: Too few values for parameter \"%s\".", 
		me, DtName(dt));
	    nval	= -nhave;
	} else {
	    nval	= nwant;
	}
    }

    return nval;
}


/*
 * WHAT:   Convert the given values.
 *
 * HOW:    Use the given converter-function repeatedly to convert the given
 *	   values between a string representation and a binary representation.
 *
 * INPUT:  The desired number of values, the converter-function, the string
 *	   representation area, the binary representation area, the size of
 *	   a binary object, and the conversion mode.
 *
 * OUTPUT: Success flag: 0 => failure; 1 => success.
 */

    static int
convert(nval, funct, str, bin, BinSize, mode)
    int		nval;		/* number of values to convert:
				 *     UDARB => arbitrary number of values */
    UdConverter	*funct;		/* Converter-function */
    char	**str;		/* String-representation of values */
    VOIDP	bin;		/* Binary-representation of values.  NULL => 
				 * don't.
				 * For fixed-number and max-number, `where' =
				 * location of user-supplied storage.  For
				 * arbitrary number, `where' = location of
				 * user-supplied pointer to space malloc'ed by
				 * this routine */
    int		BinSize;	/* Size of a binary representation */
    int		mode;		/* Conversion mode */
{
    int		i;
    int		status	= 1;		/* Routine status = success */

    i	= 0;
    do {
	VOIDP	cp;

	if (nval == 0) {
	    cp	= NULL;
	} else if (mode == UD_TO_BINARY) {
	    /*
	     *	The following line causes the SunOS 4.1 lint(1) to generate a 
	     *	warning about "possible pointer alignment problem.  This may 
	     *	safely be ignored.
	     */
	    cp	= (VOIDP)str[i];
	} else {
	    cp	= (VOIDP)(str + i);
	}

	/*
	 *	The following line causes the SunOS 4.1 lint(1) to generate a 
	 *	warning about "possible pointer alignment problem.  This may 
	 *	safely be ignored.
	 */
	if (!(*funct)(cp, bin == NULL 
				? bin
				: (VOIDP)((char*)bin+i*BinSize), 
	    mode)) {

	    status	= 0;
	    break;
	}
	++i;
    } while (--nval > 0);

    return status;
}


/*
 * WHAT:   Encode the given values.
 *
 * HOW:    Use the given converter-function repeatedly to encode the given
 *	   values from a binary representation to a string representation.
 *
 * INPUT:  The desired number of values, the converter-function, the location
 *	   of an array of string pointers and the binary values.
 *
 * OUTPUT: Success flag: 0 => failure; 1 => success.  The string pointers will
 *	   point to the string representation of the binary values.
 */

    static int
encode(nval, funct, str, bin)
    int		nval;		/* number of values to encode */
    UdConverter	*funct;		/* Converter-function */
    char	**str;		/* String-representation of values.  Storage
				 * to hold the actual strings will be allocated
				 * by this routine; the pointers will go into
				 * `str'. */
    VOIDP	bin;		/* Binary-representation of values.  NULL => 
				 * don't actually decode: just check. */
{
    int		status	= 1;		/* Routine status = success */
    int		BinSize	= (*funct)((VOIDP)NULL, (VOIDP)NULL, UD_BIN_SIZE);

    assert(nval >= 0);
    assert(funct != NULL);
    assert(!(nval > 0 && str == NULL));
    assert(!(nval > 0 && bin == NULL));

    status	= convert(nval, funct, str, bin, BinSize, UD_TO_STRING);

    return status;
}


/*
 * WHAT:   Decode the given values.
 *
 * HOW:    Use the given converter-function repeatedly to decode the given
 *	   values from a string representation to a binary representation.
 *
 * INPUT:  The desired number of values, the converter-function, the string
 *	   representation area and the binary representation area.
 *
 * OUTPUT: Success flag: 0 => failure; 1 => success.
 */

    static int
decode(nval, funct, str, bin)
    int		nval;		/* number of values to decode:
				 *     <0 => allocate storage for values */
    UdConverter	*funct;		/* Converter-function */
    char	**str;		/* String-representation of values */
    VOIDP	bin;		/* Binary-representation of values.  NULL => 
				 * don't actually decode: just check.  For 
				 * an arbitrary number of values, this routine
				 * will allocate storage and place a pointer
				 * to that storeage in *bin.  Otherwise, bin
				 * is assumed to point to sufficient storage. 
				 */
{
    int		status	= 0;		/* Routine status = failure */
    int		BinSize	= (*funct)((VOIDP)NULL, (VOIDP)NULL, UD_BIN_SIZE);
    char	*buf	= NULL;
    VOIDP	BinPtr;
    static char	me[]	= "decode";

    assert(funct != NULL);
    assert(!(nval > 0 && str == NULL));

    if (nval < 0 && bin != NULL) {
	/*
	 *	The following line causes the SunOS 4.1 lint(1) to generate a 
	 *	warning about "possible pointer alignment problem.  This may 
	 *	safely be ignored.
	 */
	if ((BinPtr = (VOIDP)(buf = UD_ALLOC(-nval*BinSize, char))) == NULL) {
	    udadvise("%s: Couldn't allocate %u-element binary-buffer.", me,
		-nval);
	} else {
	    /*
	     *	The following line causes the SunOS 4.1 lint(1) to generate a 
	     *	warning about "possible pointer alignment problem.  This may 
	     *	safely be ignored.
	     */
	    (void)(*funct)((VOIDP)buf, bin, UD_POINTER);
	    nval	= -nval;
	}
    } else {
	BinPtr	= bin;
    }

    if (!convert(nval, funct, str, BinPtr, BinSize, UD_TO_BINARY)) {
	if (buf != NULL)
	    (void)FREE(buf);
    } else {
	status	= 1;
    }

    return status;
}


/*
 * WHAT    Store a resource value-string in the resource database.
 *
 * HOW     Call the X resource-manager storing routine using the given storage
 *	   key with the program name prepended.
 *
 * INPUT   Partial storage-key and resource value-string.
 *
 * OUTPUT  Success flag: 0 => failure; 1 = success.
 */

    int
udputres(key, string)
    char	*key;		/* Partial storage-key.  Program name will be
				 * prepended. */
    char	*string;	/* Resource values-string */
{
    int		errmode	= uderrmode(UD_VERBOSE);
    int		status	= 0;		/* Routine status = failure */
    static char	me[]	= "udputres";	/* This routine's name */

    if (key == NULL) {
	udadvise("%s: Nil storage-key.", me);
    } else {
	char	*StoreKey;		/* Resource storage-key */

	if ((StoreKey = dotcat(uName, key)) == NULL) {
	    udadvise("%s: Couldn't form storage-key.", me);
	} else {
	    UrmPutStringResource(&uRdb, StoreKey, string);
	    (void)FREE(StoreKey);
	    status	= 1;
	}
    }

    ErrorHappened	|= !status;

    (void)uderrmode(errmode);

    return status;
}


/*
 * WHAT    Store the binary values of a resource in the resource database.
 *
 * HOW     Convert the binary values into a list of strings, Catenate the list
 *	   of strings together into a resource value-string, and then call the
 *	   resource storing routine `udputres()'.
 *
 * INPUT   Partial storage-key, list of binary values, converter function.
 *
 * OUTPUT  Success flag: 0 => failure; 1 => success.
 */

    int
udputval(key, nval, funct, where)
    char	*key;		/* Partial storage-key.  Program name will be
				 * prepended. */
    int		nval;		/* Number of values */
    UdConverter	*funct;		/* Converter function */
    VOIDP	where;		/* Source of the binary values */
{
    int		errmode	= uderrmode(UD_VERBOSE);
    int		status	= 0;		/* Routine status = failure */
    static char	me[]	= "udputval";	/* This routine's name */

    if (nval <= 0) {
	udadvise("%s: Non-positive number-of-values.", me);
    } else {
	char	**list;			/* Vector of strings */

	if ((list = SlCreate(nval)) == NULL) {
	    udadvise("%s: Couldn't allocate string pointers.");
	} else {
	    if (!encode(nval, funct, list, where)) {
		udadvise("%s: Couldn't convert values.");
	    } else {
		char	*string	= VsJoin(nval, list);

		if (string == NULL) {
		    udadvise("%s: Couldn't join strings.");
		} else {
		    status	= udputres(key, string);
		    (void)FREE(string);
		}
	    }
	    SlFree(nval, list);
	}
    }

    ErrorHappened	|= !status;

    (void)uderrmode(errmode);

    return status;
}


/*
 * WHAT:   Check (i.e. verify) the current parameter as to data-type and number
 *	   of values.
 *
 * HOW:    Perform a pseudo-decode on the parameter using the decode function 
 *	   from the decode table.
 *
 * INPUT:  Valid pointers to valid decode-table and command-line objects.
 *
 * OUTPUT: Success flag: 0 => not valid; 1 => valid.
 *
 * NOTES   If the requested number of values to be checked is zero, then the
 *	   routine always returns "valid".
 */

    static int
RightType(nval, dt, cl)
    int		nval;		/* number of values */
    DtPtr	dt;		/* decode-table */
    ClPtr	cl;		/* command-line object */
{
    int		status	= 1;		/* routine status = valid */
    UdConverter	*funct;			/* Converter-function */
    static char	me[]	= "RightType";	/* this routine's name */

    assert(nval >= 0);
    assert(dt != NULL);
    assert(cl != NULL);

    if ((funct = DtFunct(dt)) != NULL) {
	/*
	int	*count	= DtCount(dt);
	*/

	if (!decode(DtNval(dt) == UDARB 
			? -nval 
			: nval,
		    funct,
		    ClValues(cl),
		    (VOIDP)NULL)) {

	    udadvise("%s: Couldn't decode parameter \"%s\".", me, DtName(dt));
	    status	= 0;
	}

	/*
	if (count != NULL)
	    *count	= nval;
	*/
    }

    return status;
}


/*
 * WHAT:   Store the current parameter in the resource database.
 *
 * HOW:    Catenate the program name and the decode-table name of the
 *	   current parameter to form a resource name.  Use the external
 *	   function `udputres()' to store the value-strings returned by the
 *	   command-line object into the database.
 *
 * INPUT:  A valid decode-table and a valid command-line object.
 *
 * OUTPUT: Succes flag: 0 => failure; 1 => success.
 */

    static int
store(nval, dt, cl)
    int		nval;		/* Number of values to store. 0 <=> boolean */
    DtPtr	dt;		/* Decode-table */
    ClPtr	cl;		/* Command-line object */
{
    char	*key;			/* Resource storage-key */
    int	status	= 0;			/* Routine status = failure */
    static char	me[]	= "store";	/* This routine's name */

    assert(nval >= 0);
    assert(dt != NULL);
    assert(cl != NULL);

    key	= DtName(dt);

    if (nval == 0) {				/* If no values */
	status	= udputres(key, DtFunct(dt) == UdBool ? "on" : "");
    } else {
	char	*resource	= VsJoin(nval, ClValues(cl));

	if (resource == NULL) {
	    udadvise("%s: Couldn't form resource value.", me);
	} else {
	    status	= udputres(key, resource);
	    (void)FREE(resource);
	}
    }

    return status;
}
    

/*
 * WHAT:   Parse the command-line into the resource database.
 *
 * HOW:    Create a decode-table object and a command-line object.  
 *	   First sequentially process all keyed parameters -- checking and 
 *	   storing them in the database.  Then sequentially process the 
 *	   positional parameters -- checking and storing them in the database.  
 *	   Don't stop if an error occurs: processes all parameters.
 *
 * INPUT:  A valid command-line table and valid command-line arguments.
 *
 * OUTPUT: Success flag: 0 => failure; 1 => success.
 */

    static int
ParseCmdLine(dt, argc, argv)
    DtPtr	dt;		/* Decode-table */
    int		*argc;		/* Number of words */
    char	**argv;		/* Command-line word-list */
{
    int		status	= 0;			/* routine status = failure */
    ClPtr	cl;				/* command-line object */
    static char	me[]	= "ParseCmdLine";	/* this routine's name */

    assert(dt != NULL);
    assert(argc != NULL);
    assert(*argc >= 0);
    assert(!(*argc > 0 && argv == NULL));

    cl	= ClCreate(*argc, argv);

    if (cl == NULL) {
	udadvise("%s: Couldn't make command-line object.", me);
    } else {
	status	= 1;

	while (ClNextKey(cl))
	    if (DtKeyFind(dt, ClName(cl))) {
		int	nval	= Nval(dt, cl);

		if (nval < 0) {
		    nval	= -nval;
		} else {
		    if (DtFunct(dt) == UdBool && nval == 1)
			nval	= 0;
		    status	&= RightType(nval, dt, cl) &&
			(DoStore ? store(nval, dt, cl) : 1);
		}
		(void)ClSkip(cl, nval);
	    }

	while (ClNextPos(cl) && DtPosFind(dt)) {
	    int	nval	= Nval(dt, cl);

	    if (nval < 0) {
		nval	= -nval;
	    } else if (nval > 0) {
		status	&= RightType(nval, dt, cl) &&
		    (DoStore ? store(nval, dt, cl) : 1);
	    }
	    if (ClSkip(cl, nval) == NO) {
		udadvise(
		    "%s: Error skipping values of positional-parameter \"%s\".",
		    me, DtName(dt));
	    }
	}

	status	&= !DtWasError(dt) && !ClWasError(cl);

	*argc	= ClFree(cl);
    }

    return status;
}


/*
 * WHAT:   Return the value-string associated with a resource.
 *
 * HOW:    Create a name-key and class-key and then search the resource 
 *	   database for a matching entry.  Return the associated value.
 *	   If a nil partial class-key is specified, then a partial class-key 
 *	   will be automatically generated from the partial name-key by
 *	   capitalizing the first letter.  Before searching, the program name
 *	   will be prepended to the name-key as will the program class to the
 *	   class-key.
 *
 * INPUT:  A valid partial resource name-key and an optional partial class-key.
 *	   The keys can be simple names or a fully qualified lists of names
 *	   separated by periods.  They shall, however, omit the first 
 *	   component.
 *
 * OUTPUT: Either the value-string of the corresponding resource or a nil 
 *	   pointer.  The value-string resides in malloc'ed space, which may be
 *	   freed by the user.  A nil pointer can mean either an error occurred,
 *	   or that the resource doesn't exist.  Use the routine udreserr() to
 *	   distinguish.
 */

    char*
udgetres(name, class)
    char	*name;		/* Resource name.  This can be a simple option 
				 * name or a fully qualified list of names 
				 * separated by ".".  The program name will be 
				 * prepended before searching. */
    char	*class;		/* Resource class (excluding program class).
				 * NULL <=> use resource name with first letter
				 * capitalized */
{
    int		errmode	= uderrmode(UD_VERBOSE);
    char	*ValueString	= NULL;	/* returned resouce value-string =
					 * not found */
    static char	me[]	= "udgetres";	/* this routine's name */

    if (name == NULL) {
	udadvise("%s: Nil resource name.", me);

    } else {
	char		*NameKey, *ClassKey;

	if ((NameKey = dotcat(uName, name)) == NULL) {
	    udadvise("%s: Couldn't create name-key.", me);
	    ErrorHappened	= 1;
	} else {
	    if (class == NULL) {
		char	*cp;

		class	= udstrdup(name);
		assert(class != NULL);

		/* Capitalize first letter of each component. */
		for (cp = class; ;) {
		    if (islower(*cp))
			*cp	= toupper(*cp);
		    if ((cp = strchr(cp, '.')) == NULL) {
			break;
		    } else {
			++cp;
		    }
		}

		ClassKey	= dotcat(uClass, class);
		(void)FREE(class);
	    } else {
		ClassKey	= dotcat(uClass, class);
	    }

	    if (ClassKey == NULL) {
		udadvise("%s: Couldn't create class-key.", me);
		ErrorHappened	= 1;
	    } else {
		char		*type	= NULL;
		UrmValue	value;

		if (UrmGetResource(uRdb, NameKey, ClassKey, &type, &value))
		    if ((ValueString = udstrdup((char*)value.addr)) == NULL) {
			udadvise("%s: Couldn't allocate value-buffer.", me);
			ErrorHappened	= 1;
		    }

		(void)FREE(ClassKey);
	    }
	    (void)FREE(NameKey);
	}
    }

    (void)uderrmode(errmode);

    return ValueString;
}


/*
 * WHAT:   Return the values of a resource.
 *
 * HOW:    Use the external function `udgetres' to obtain the resource value-
 *	   string.  Then decode the resource value-string into the given 
 *	   destination.
 *
 * INPUT:  The partial name of the resource, an optional class name, the
 *	   desired number of values, the decoding function, the
 *	   destination for the values, the destination of the number of
 *	   decoded values.
 *
 * OUTPUT: Success flag: 0 => resource is unavailable; 1 => success.
 *	   On success, the destination will either hold the decoded values, or 
 *	   (if the resource may have an arbitrary number of values) a pointer 
 *	   to the decoded values.  Optional return of the number of
 *	   values decoded.
 */

    int
udgetval(name, class, nwant, funct, where, count)
    char	*name;		/* Resource name (excluding program name) */
    char	*class;		/* Resource class (excluding program class).
				 * NULL <=> use resource name with first letter
				 * capitalized */
    int		nwant;		/* Desired number of values:
				 *     UDARB <=> arbitrary number of values;
				 *     else fixed number of values.
				 * For a scalar boolean resource, `nwant' must
				 * be 1. */
    UdConverter	*funct;		/* Converter-function */
    VOIDP	where;		/* Destination for values.  For resources with
				 * a fixed or maximum number of values, `where'
				 * is the location of user-supplied storage.
				 * For resources with an arbitrary number of
				 * values, `*where' will receive the location
				 * of malloc'ed space containing the values. */
    int		*count;		/* Number of values decoded. NULL => exact
				 * number expected. */
{
    int		errmode	= uderrmode(UD_VERBOSE);
    int		success	= 0;		/* Return status = failure */
    static char	me[]	= "udgetval";	/* This routine's name */

    if (funct == NULL) {
	udadvise("%s: Nil decode-function.", me);
	ErrorHappened	= 1;

    } else if (where == NULL) {
	udadvise("%s: Nil value-destination.", me);
	ErrorHappened	= 1;

    } else if (nwant != UDARB && nwant < 0) {
	udadvise(
	    "%s: Negative number-of-values requested for resource \"%s\".",
	    me, name);
	ErrorHappened	= 1;

    } else {
	char	*ValueString	= udgetres(name, class);

	if (nwant != UDARB && count != NULL)
	    nwant	= -nwant;		/* means |nwant| or fewer */

	if (ValueString != NULL) {
	    if (ValueString[0] == 0) {
		success	= nwant <= 0 || nwant == UDARB;
		if (count != NULL)
		    *count	= 0;

	    } else {
		int	nhave;
		char	**ValStrings	= VsSplit(ValueString, &nhave);

		if (ValStrings == NULL) {
		    udadvise("%s: Couldn't split resource \"%s\".", me, name);
		    ErrorHappened	= 1;
		} else {
		    int	num	= NumVal(funct == UdBool && nwant == 0 
					    ? 1 
					    : nwant,
					nhave);

		    if (num < 0) {
			udadvise("%s: Resource \"%s\" has too few values.", me,
			    name);
			ErrorHappened	= 1;
		    } else if (!decode(nwant == UDARB ? -num : num, funct, 
			ValStrings, where)) {

			udadvise("%s: Couldn't decode resource \"%s\".", 
			    me, name);
			ErrorHappened	= 1;
		    } else {
			if (count != NULL)
			    *count	= num;
			success	= 1;
		    }

		    VsFree(nhave, ValStrings);
		}
	    }
	}					/* Resource exists */
    }

    (void)uderrmode(errmode);

    return success;
}


/*
 * WHAT	   Retrieve resource-values for appropriate entries of a decode-table.
 *
 * OUTPUT  Success flag: 0 => failure; 1 => success.  It is not an error for a
 *	   resource not to exist.  The resource destination fields (i.e. the
 *	   binary values and the number of values) will be set if the pointers
 *	   are non-nil.
 */

    static int
retrieve(dt)
    DtPtr	dt;		/* Decode-table */
{
    int		success	= 1;		/* Routine status = success */
    static char	me[]	= "retrieve";	/* This routine's name */

    while (DtNextPar(dt)) {
	VOIDP	where		= DtWhere(dt);

	if (where != NULL) {
	    int		nval	= DtNval(dt);

	    if (!udgetval(	DtName(dt), 
				(char*)NULL,
				nval == UDARB
				    ? UDARB
				    : ABS(nval),
				DtFunct(dt),
				where,
				DtCount(dt))) {

		if (ErrorHappened) {
		    success	= 0;
		    break;
		} else if (DtRequired(dt)) {
		    udadvise("%s: Required parameter \"%s\" not found.", 
			me, DtName(dt));
		    success	= 0;
		}
	    }
	}
    }

    return success;
}


/*
 * WHAT:   Parse a word-vector into the resource database.
 *
 * HOW:    Parse the word-vector and store any recognized parameters in 
 *	   the database -- overriding any already there.
 *
 * INPUT:  Application-defined keyword- and positional-parameter tables, 
 *	   and a vector of words to be parsed.
 *
 * OUTPUT: Success flag: 1 => success; 0 => error.
 */

    static int
parse(KeyTab, PosTab, argc, argv, dt)
    UdKeyEntry	*KeyTab;	/* command-line keyed-parameter table */
    UdPosEntry	*PosTab;	/* command-line positional-parameter table */
    int		*argc;		/* number of words in command-line */
    char	**argv;		/* words of command-line */
    DtPtr	*dt;		/* decode-table */
{
    int		errmode	= uderrmode(UD_VERBOSE);
    int		status	= 0;		/* routine status = failure */
    static char	me[]	= "parse";	/* this routine's name */

    *dt	= NULL;

    if (argc == NULL || *argc < 0 || (*argc > 0 && argv == NULL)) {
	udadvise("%s: Invalid word-vector.", me);

    } else {
	*dt	= DtCreate(KeyTab, PosTab);

	if (*dt == NULL) {
	    udadvise("%s: Couldn't make decode-table.", me);

	} else {
	    /* Process the word-vector */
	    if (ParseCmdLine(*dt, argc, argv)) {
		DtReset(*dt);
		status	= retrieve(*dt);
	    }
	}
    }						/* valid word-vector */

    (void)uderrmode(errmode);

    return status;
}


/*
 * WHAT:   Create a Unidata application resource database.
 *
 * HOW:    Create the database.  Store default resources, which are specified 
 *	   in files, into the database, then parse the command line and store 
 *	   any recognized parameters in the database -- overriding the default 
 *	   resources.
 *
 * INPUT:  Application-defined command-line table, invocation arguments,
 *	   program name, and program-class name.
 *
 * OUTPUT: Success flag: 1 => success; 0 => error.
 */

    int
udinit(KeyTab, PosTab, argc, argv, ProgName, ClassName)
    UdKeyEntry	*KeyTab;	/* command-line keyed-parameter table */
    UdPosEntry	*PosTab;	/* command-line positional-parameter table */
    int		*argc;		/* number of words in command-line */
    char	**argv;		/* words of command-line */
    char	*ProgName;	/* name of program. NULL => use argv[0] */
    char	*ClassName;	/* name of program class.  NULL => use program
				 * name with 1st letter capitalized */
{
    int		errmode	= uderrmode(UD_VERBOSE);
    int		status	= 0;		/* routine status = failure */
    static char	me[]	= "udinit";	/* this routine's name */

    if (ProgName == NULL && (*argc == 0 || argv[0] == NULL)) {
	udadvise("%s: Nil program name.", me);

    } else if (!SetNames(ProgName == NULL ? argv[0] : ProgName, ClassName)) {
	udadvise("%s: Couldn't set program/class names.", me);

    } else {
	if (!UrmInitialize()) {
	    udadvise("%s: Couldn't initialize resource manager.", me);

	/*
	 * NB: The database is created using a non-empty resource-string
	 * because some X resource-managers fail when given an empty string.
	 */
	} else if (!UrmGetStringDatabase("ZZZ:ZZZ", &uRdb)) {
	    udadvise("%s: Couldn't create database.", me);

	} else {
	    DtPtr		dt;
	    static UdKeyTab	NameTab	= {
		{"name",	1,	CvtName,	(VOIDP)&uName},
		NULL
	    };

	    /* Hide program name */
	    --*argc;
	    ++argv;

	    ErrorHappened	= 0;
	    DoStore		= 1;

	    /* Set the program name before anything else */
	    if (!parse(NameTab, (UdPosEntry*)NULL, argc, argv, &dt)) {
		udadvise("%s: Couldn't parse \"name\" parameter.", me);
		DtFree(dt);

	    } else {
		/* Process other "internal" parameters */
		static int		PrintUsage	= 0;
		static int		PrintVersion	= 0;
		static UdKeyTab	MyTab	= {
		    {"udrm",		UDARB,	CvtUdrm},
		    {"udversion",	0,	UdBool, (VOIDP)&PrintVersion},
		    {"usage",		0,	UdBool, (VOIDP)&PrintUsage},
		    NULL
		};

		if (!parse(MyTab, (UdPosEntry*)NULL, argc, argv, &dt)) {
		    udadvise("%s: Couldn't parse internal parameters.", me);
		    DtFree(dt);

		} else {
		    if (PrintVersion)
			udadvise("GUISE version: %s", guisevers());

		    DtFree(dectab);

		    if (PrintUsage) {
			if ((dectab = DtCreate(KeyTab, PosTab)) == NULL) {
			    udadvise("%s: Couldn't make decode-table.", me);
			} else {
			    udusage();
			}

		    } else {
			/* Merge the resource-files into the database */
			if (!merge_resource_dbs()) {
			    udadvise("%s: Couldn't merge resource-files", 
				me);
			} else {
			    /* Process the user's parameters */
			    if (parse(KeyTab, PosTab, argc, argv, &dectab)) {
				status	= 1;
			    }
			}
		    }				/* not a "usage" request */
		}				/* Internal parameters 
						 * parsed */
	    }					/* "name" parameter
						 * parsed */

	    /* Uncover program name */
	    ++*argc;
	}					/* database created */
    }						/* program/class names set */

    (void)uderrmode(errmode);

    return status;
}


/*
 * WHAT	   Indicate whether or not an error occurred.
 *
 * OUTPUT  True if and only if an error occurred.
 *
 * HOW	   Return the relevant, static variable.
 */

    int
udreserr()
{
    return ErrorHappened;
}
