// This may look like C code, but it is really -*- C++ -*-

// <copyright>
// 
// Copyright (c) 1994
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
// 
// </copyright>


//<file>
//
// Name:       getargs.h
//
// Purpose:    GetArguments: processing commandline arguments
//
// Created:    18 Feb 1994   Michael Pichler
//
// Modified:   18 Feb 1994   Michael Pichler
//
//
//</file>


#ifndef hg_utils_getargs_h
#define hg_utils_getargs_h



// OptionStruct - description of a commandline option

struct OptionStruct
{
  const char* option;                   // command line option (e.g. "-help"), last one: nil
  int tag;                              // symbolic constant for that option
};



//<class>
//
// Name:       GetArguments
//
// Purpose:    processing commandline arguments
//
//
//</class>


// usage (example):
// enum { OptHelp, ... };
// OptionStruct options[] = { { "-h", OptHelp }, ..., { nil, 0 } };
// GetArguments getargs(argc, argv, options);
// while (getargs.more())
// { switch (getargs.getNext())
//   { case <OptionTag>:
//     ... set a flag or process getargs.nextArg()
//     case GetArguments::UnknownOption: ...
//     default: ... process getargs.currentArg()
//   }
// }


class GetArguments
{
  public:
    GetArguments(                       // constructor
      int argc,                         //   no. of command line arguments
      char** argv,                      //   array of command line arguments
      const OptionStruct* options       //   array of valid options, must end with nil
    );

    int more()                          // check for further unprocessed args
    { return cur_+1 < argc_; }

    int getNext();                      // get next argument: if the argument matches
                                        //   one of the options the corresponding tag
                                        //   is returned; other return values see below

    enum                                // special return values of getNext:
    { UnknownOption = -1,               //   argument not in options, beginning with '-'
      OtherArgument = -2,               //   other argument
      NoMoreArgs = -3                   //   can not happen when more() is true
    };

    char* programName()                 // program name (as stored in argv [0])
    { return argv_ [0]; }

    char* currentArgString();           // get current argument

    char* nextArgString();              // get next argument (nil if no more)

  private:
    int argc_;
    char** argv_;
    const OptionStruct* options_;
    int cur_;

}; // class GetArguments



inline char* GetArguments::currentArgString()
{
  return (cur_ < argc_) ? argv_[cur_] : 0;
}


inline char* GetArguments::nextArgString()
{
  return (++cur_ < argc_) ? argv_[cur_] : 0;
}



#endif
