/****************************************************************************/
/*                                                                          */
/*  VolVis is a volume visualization system for investigating, manipulating */
/*  and rendering geometric and volumetric data.                            */
/*                                                                          */
/*  Copyright (C) 1993 by the Research Foundation of the State University   */
/*                            of New York                                   */
/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 1, or (at your option)     */
/*  any later version.                                                      */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this program; if not, write to the Free Software             */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/*                                                                          */
/*  For information on VolVis, contact us at:                               */
/*                                                                          */
/*                volvis@cs.sunysb.edu                         (email)      */
/*                                                                          */
/*                Lisa Sobierajski & Ricardo Avila             (US Mail)    */
/*                Department of Computer Science                            */
/*                State University of New York at Stony Brook               */
/*                Stony Brook, New York  11794-4400                         */
/*                                                                          */
/****************************************************************************/




/*
 *                  File: MOTIF_nav_ct_ui.c
 *                Author: Hui Chen 
 *                  Date: 03/05/92
 *           Description: Motif Windowing Routines Of The Navigator.
 *			  This file contains: create_nav_ct_window()
 *					      nav_CreateTransformType()
 *					      nav_CreateCoordType()
 *					      nav_CreateGlobalOption()
 *					      nav_CreateReferenceType()
 *					      nav_CreateOtherPoint()
 *					      nav_CreateTranAmount()
 *					      nav_CreateMenuButton()
 * 
 *			  They are the routines for creating the control
 *			  window of the navigator.
 *  Modification History:
 *
 *         who?         when?           why?
 *    -----------------------------------------------------------
 *
 */

# include <stdio.h>
# include <string.h>
# include <X11/Intrinsic.h>
# include <X11/Shell.h>
# include <X11/Xlib.h>
# include <X11/Xutil.h>
# include <Xm/Xm.h>
# include <Xm/RowColumn.h>
# include <Xm/Frame.h>
# include <Xm/Form.h>
# include <Xm/ToggleBG.h>
# include <Xm/ToggleB.h>
# include <Xm/Text.h>
# include <Xm/PushB.h>
# include <Xm/PushBG.h>
# include <Xm/LabelG.h>
# include <Xm/CascadeB.h>
# include <Xm/BulletinB.h>
# include <Xm/FileSB.h>
# include <Xm/SelectioB.h>
# include <Xm/DialogS.h>
# include <Xm/MessageB.h>

#include "C_volvis.h"
#include "C_navigator.h"
#include "MOTIF_windefs.h"
#include "C_help.h"

/* Global Variables For Motif Windowing */
extern Widget           vv_toplevel;	 /* VolVis toplevel Shell Widget */
extern Widget 		nav_ct_bbdialog; /* The parent widget for the */
					 /* navigator control window  */
extern C_WindowDefs	window_defs;	 /* window appearance definitions */


C_NavCWInfo	nav_cw_info; /* contains the control window global variables*/

/*
 *        Procedure Name: create_nav_ct_window
 *          Return Value: none
 *       Input Variables: none
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: window_defs     - window appearance definitions 
 *                        vv_toplevel     - VolVis toplevel Shell Widget
 *                        nav_ct_bbdialog - The parent widget of control window 
 *			  nav_cw_info	  - The control window global variable
 *           Description: This procedure creates the label Widgets of the
 *                        navigator control window and calls the routines
 *			  to create all the children Widgets.
 */

void nav_create_ct_window()
{
	/***    Internal Routines  ***/
	void nav_CreateTransformType();
	void nav_CreateCoordType();
	void nav_CreateGlobalOption();
	void nav_CreateReferenceType();
	void nav_CreateOtherPoint();
	void nav_CreateTranAmount();
	void nav_CreateMenuButton();

	/***    Local Variables     ***/
	Widget flabel[4];
	Arg args[20];
	XmString string;
	int n;

	/***    initialize control global selections     ***/
	nav_cw_info.nav_global.nav_mode = C_TRANSLATE;
        nav_cw_info.nav_global.nav_coord = C_LOCAL_COORD;
        nav_cw_info.nav_global.nav_ref = C_ORIGIN_REF;
        nav_cw_info.nav_global.nav_option = 0;

	/***** Create the control window bulletin board *****/
        n=0;
        XtSetArg( args[n], XmNautoUnmanage, False ); n++;
        string = XmStringCreate( "Navigator Control Window", 
				  XmSTRING_DEFAULT_CHARSET);
        XtSetArg( args[n], XmNdialogTitle, string ); n++;
	C_Set_Color_Dialog( args, n, window_defs );
        nav_ct_bbdialog = XmCreateBulletinBoardDialog( vv_toplevel,
                       "Navigator Control Window", args, n );


	/***** Create a frame for transformation selection *****/
	n = 0;
	XtSetArg( args[n], XmNx, 10 ); n++;
	XtSetArg( args[n], XmNy, 10 ); n++;
	flabel[0] = XmCreateLabelGadget(nav_ct_bbdialog, 
				"Transformation:", args, n);
	XtManageChild(flabel[0]);

	nav_CreateTransformType(nav_ct_bbdialog);


	/***** Create a frame for coordinate system selection *****/
	n = 0;
	XtSetArg( args[n], XmNx, 10 ); n++;
	XtSetArg( args[n], XmNy, 60 ); n++;
	flabel[1] = XmCreateLabelGadget(nav_ct_bbdialog, 
				"Coordinate sys:", args, n);
	XtManageChild(flabel[1]);

	nav_CreateCoordType(nav_ct_bbdialog);

	/**** Create a option menu for global coordinate system selection ****/
	nav_CreateGlobalOption(nav_ct_bbdialog);


	/***** Create a frame for reference point selection *****/
	n = 0;
	XtSetArg( args[n], XmNx, 10 ); n++;
	XtSetArg( args[n], XmNy, 110 ); n++;
	flabel[2] = XmCreateLabelGadget(nav_ct_bbdialog, 
				"Reference point:", args, n);
	XtManageChild(flabel[2]);

	nav_CreateReferenceType(nav_ct_bbdialog);

	/***** Create a frame for other reference point selection *****/
	nav_CreateOtherPoint(nav_ct_bbdialog);


	/***** Create a frame for transform amount input *****/
	n = 0;
	XtSetArg( args[n], XmNx, 10 ); n++;
	XtSetArg( args[n], XmNy, 165 ); n++;
	flabel[3] = XmCreateLabelGadget(nav_ct_bbdialog, 
				"Transform amount:", args, n);
	XtManageChild(flabel[3]);

	nav_CreateTranAmount(nav_ct_bbdialog);

	/***** Create a frame for menu buttons *****/
	nav_CreateMenuButton(nav_ct_bbdialog);
}


/*
 *        Procedure Name: nav_CreateMenuButton
 *          Return Value: none
 *       Input Variables: parent	  - The parent widget of control window 
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: window_defs     - window appearance definitions 
 *           Description: This procedure creates the "Accept", "Undo", 
 *			  "Close", "Help" buttons for the 
 *                        navigator control window and adds the
 *                        activation call back function to them. 
 */
 
void nav_CreateMenuButton(parent)
Widget parent;
{
	/***    External Routines  ***/
	extern void nav_accept_proc();
	extern void nav_undo_proc();
	extern void nav_close_proc();
	extern void help_callback();

	/***    Local Variables     ***/
	Widget rc;
	Widget accept;
	Widget undo;
	Widget cancel;
	Widget help;
	Arg args[20];
	XmString string;
	static char *menu_name[4] = {"Accept", "Undo", "Close", "Help"};


	int n;

	n = 0;
	XtSetArg( args[n], XmNx, 10 ); n++;
	XtSetArg( args[n], XmNy, 335 ); n++;
	XtSetArg( args[n], XmNheight, 40 ); n++;
	XtSetArg( args[n], XmNspacing, 100 ); n++;
	XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
	C_Set_Color_RowColumn( args, n, window_defs );	
	rc = XmCreateRowColumn (parent, "rc", args, n);
	XtManageChild(rc);

	n = 0;
	C_Set_Color_Button( args, n, window_defs );
	accept = XmCreatePushButton(rc, menu_name[0], args, n);
	XtManageChild(accept);
	XtAddCallback(accept, XmNactivateCallback, nav_accept_proc, 0);

	undo = XmCreatePushButton(rc, menu_name[1], args, n);
	XtManageChild(undo);
	XtAddCallback(undo, XmNactivateCallback, nav_undo_proc, 0);

	cancel = XmCreatePushButton(rc, menu_name[2], args, n);
	XtManageChild(cancel);
	XtAddCallback(cancel, XmNactivateCallback, 
		      nav_close_proc, nav_ct_bbdialog );

	help = XmCreatePushButton(rc, menu_name[3], args, n);
	XtManageChild(help);
	XtAddCallback(help, XmNactivateCallback, help_callback, 
			C_NAV_MANUAL_HELP );
	
}


/*
 *        Procedure Name: nav_CreateTransformType
 *          Return Value: none
 *       Input Variables: parent	  - The parent widget of control window 
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: window_defs     - window appearance definitions 
 *			  nav_cw_info	  - The control window global variable
 *           Description: This procedure creates a radio box which contains
 *			  transformation type selection buttons   
 *                        and adds the activation call back function to them. 
 */

void nav_CreateTransformType(parent) 
Widget parent;
{
	/***    External Routines  ***/
	extern void nav_set_mode();

	/***    Local Variables     ***/
	Widget frame;
	Widget radiobox;
	Widget toggle[3];
	Arg args[20];
	XmString string;
	static char *toggle_name[3] = {"Translate", "Rotate", "Scale"};

	int n;


	/***** the frame *****/

	n = 0;
    	XtSetArg (args[n], XmNshadowType, XmSHADOW_ETCHED_IN);  n++;
   	XtSetArg (args[n], XmNx, 140);  n++;
   	XtSetArg (args[n], XmNy, 10);  n++;
	XtSetArg( args[n], XmNheight, 40 ); n++;
	C_Set_Color_Frame( args, n, window_defs );
    	frame = XmCreateFrame (parent, "frame", args, n);
    	XtManageChild (frame);

	/***** the frame encloses a RadioBox *****/

	n = 0;
	XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
	C_Set_Color_RadioBox( args, n, window_defs );
	radiobox = XmCreateRadioBox (frame, "radiobox", args, n);
	XtManageChild (radiobox);

	/*****  Create Children  ********************************************/

	n = 0;
	string = XmStringCreate (toggle_name[0], XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_ToggleButton( args, n, window_defs );
	toggle[0] = XtCreateWidget (toggle_name[0], xmToggleButtonWidgetClass,
                                    radiobox, args, n);
	XmStringFree (string);
	XtAddCallback (toggle[0], XmNvalueChangedCallback, nav_set_mode, 
		       C_TRANSLATE);

	n = 0;
	string = XmStringCreate (toggle_name[1], XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_ToggleButton( args, n, window_defs );
	toggle[1] = XtCreateWidget (toggle_name[1], xmToggleButtonWidgetClass,
                                    radiobox, args, n);
	XmStringFree (string);
	XtAddCallback (toggle[1], XmNvalueChangedCallback, nav_set_mode, 
		       C_ROTATE);

	n = 0;
	string = XmStringCreate (toggle_name[2], XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_ToggleButton( args, n, window_defs );
	toggle[2] = XtCreateWidget (toggle_name[2], xmToggleButtonWidgetClass,
                                    radiobox, args, n);
	XmStringFree (string);
	XtAddCallback (toggle[2], XmNvalueChangedCallback, nav_set_mode, 
		       C_SCALE);

	n = 0;
	XtSetArg( args[n], XmNset, TRUE ); n++;
	XtSetValues (toggle[nav_cw_info.nav_global.nav_mode], args, n); 

	XtManageChildren (toggle, 3);

}

/*
 *        Procedure Name: nav_CreateCoordType
 *          Return Value: none
 *       Input Variables: parent	  - The parent widget of control window 
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: window_defs     - window appearance definitions 
 *			  nav_cw_info	  - The control window global variable
 *           Description: This procedure creates a radio box which contains
 *			  coordinate system selection buttons   
 *                        and adds the activation call back function to them. 
 */

void nav_CreateCoordType(parent) 
Widget parent;
{
	/***    External Routines  ***/
	extern void nav_set_coord();

	/***    Local Variables     ***/
	Widget frame;
	Widget radiobox;
	Arg args[20];
	XmString string;
	static char *toggle_name[2] = {"Local", "Global-"};

	int n;


	/***** the frame *****/

	n = 0;
    	XtSetArg (args[n], XmNshadowType, XmSHADOW_ETCHED_IN);  n++;
   	XtSetArg (args[n], XmNx, 140);  n++;
   	XtSetArg (args[n], XmNy, 60);  n++;
	XtSetArg( args[n], XmNheight, 40 ); n++;
	C_Set_Color_Frame( args, n, window_defs );
    	frame = XmCreateFrame (parent, "frame", args, n);
    	XtManageChild (frame);

	/***** the frame encloses a RadioBox *****/

	n = 0;
	XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
	C_Set_Color_RadioBox( args, n, window_defs );
	radiobox = XmCreateRadioBox (frame, "radiobox", args, n);
	XtManageChild (radiobox);

	/*****  Create Children  *****/

	n = 0;
	string = XmStringCreate (toggle_name[0], XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_ToggleButton( args, n, window_defs );
	nav_cw_info.nav_local_toggle = XtCreateWidget (toggle_name[0], 
				xmToggleButtonWidgetClass,
                                    radiobox, args, n);
	XmStringFree (string);
	XtAddCallback (nav_cw_info.nav_local_toggle, XmNvalueChangedCallback, 
			nav_set_coord, C_LOCAL_COORD);

	n = 0;
	string = XmStringCreate (toggle_name[1], XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_ToggleButton( args, n, window_defs );
	nav_cw_info.nav_global_toggle = XtCreateWidget (toggle_name[1], 
				xmToggleButtonWidgetClass,
                                    radiobox, args, n);
	XmStringFree (string);
	XtAddCallback (nav_cw_info.nav_global_toggle, XmNvalueChangedCallback, 
			nav_set_coord, C_GLOBAL_COORD);

	n = 0;
	XtSetArg( args[n], XmNset, TRUE ); n++;
	XtSetValues (nav_cw_info.nav_local_toggle, args, n); 

	XtManageChild (nav_cw_info.nav_local_toggle);
	XtManageChild (nav_cw_info.nav_global_toggle);

	if (nav_cw_info.nav_global.nav_mode == C_SCALE) {
	    XtSetSensitive(nav_cw_info.nav_global_toggle, False);
	}

}

/*
 *        Procedure Name: nav_CreateGlobalOption
 *          Return Value: none
 *       Input Variables: parent	  - The parent widget of control window 
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: window_defs     - window appearance definitions 
 *			  nav_cw_info	  - The control window global variable
 *           Description: This procedure creates a option menu for 
 *			  global coordinate system selection which contains
 *			  all the coorinates in the "world", 
 *                        and adds the activation call back function to them. 
 */

void nav_CreateGlobalOption(parent)
Widget parent;
{
	/***    External Routines  ***/
	extern void nav_global_option();

	/***    External Variables     ***/
	extern C_World  world;

	/***    Local Variables     ***/
	Arg args[10];
	XmString string;

	int num_global_options;
	int n;
	int i;

	/***** initialization *****/
	num_global_options = world.num_volumes + 2;

	/***** Create a Row Column in bboard for global selection ********/
	n = 0;
   	XtSetArg (args[n], XmNx, 315);  n++;
   	XtSetArg (args[n], XmNy, 58);  n++;
	C_Set_Color_RowColumn( args, n, window_defs );
	nav_cw_info.nav_global_rc = XmCreateRowColumn(parent, "rc", args, n); 
	XtManageChild (nav_cw_info.nav_global_rc);

	/***** Create a pulldown menu for global selection *************/

	n = 0;
	C_Set_Color_PullDown( args, n, window_defs );
	nav_cw_info.nav_pulldown = XmCreatePulldownMenu(
			nav_cw_info.nav_global_rc, "pulldown", args, n); 

	n = 0;
	string = XmStringCreate ("World", XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	nav_cw_info.nav_options[0] = XmCreatePushButtonGadget(
				nav_cw_info.nav_pulldown, "World", args, n);
	XmStringFree (string);
	XtAddCallback (nav_cw_info.nav_options[0], XmNactivateCallback, 
		       nav_global_option, 0);

	n = 0;
	string = XmStringCreate ("View", XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	nav_cw_info.nav_options[1] = XmCreatePushButtonGadget(
				nav_cw_info.nav_pulldown, "View", args, n);
	XmStringFree (string);
	XtAddCallback (nav_cw_info.nav_options[1], XmNactivateCallback, 
		       nav_global_option, 1);

	for (i=2; i<world.num_volumes+2; i++)
	{ 
	    n = 0;
	    string = XmStringCreate (world.volume[i-2]->volume_name, 
					XmSTRING_DEFAULT_CHARSET);
	    XtSetArg (args[n], XmNlabelString, string); n++;
	    nav_cw_info.nav_options[i] = XmCreatePushButtonGadget(
					nav_cw_info.nav_pulldown, 
				   world.volume[i-2]->volume_name, args, n);
	    XmStringFree (string);
	    XtAddCallback (nav_cw_info.nav_options[i], XmNactivateCallback, 
		           nav_global_option, i);
	}
/******
	for (i=world.num_volumes+1; i<num_global_options; i++)
	{ 
	    n = 0;
	    string = XmStringCreate (
			world.light[i-1-world.num_volumes]->light_name, 
					XmSTRING_DEFAULT_CHARSET);
	    XtSetArg (args[n], XmNlabelString, string); n++;
	    nav_cw_info.nav_options[i] = XmCreatePushButtonGadget(
					nav_cw_info.nav_pulldown, 
			world.light[i-1-world.num_volumes]->light_name, 
			 args, n);
	    XmStringFree (string);
	    XtAddCallback (nav_cw_info.nav_options[i], XmNactivateCallback, 
		           nav_global_option, i);
	}
******/

	XtManageChildren (nav_cw_info.nav_options, num_global_options);
	
	/***** Create a option menu and attach the pulldown menu ********/

	n = 0;
	string = XmStringCreate (">", XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	XtSetArg (args[n], XmNsubMenuId, nav_cw_info.nav_pulldown); n++;
	XtSetArg (args[n], XmNmenuHistory, 
	    nav_cw_info.nav_options[nav_cw_info.nav_global.nav_option]); n++;
	C_Set_Color_OptionMenu( args, n, window_defs );
	nav_cw_info.nav_option_menu = XmCreateOptionMenu(
			nav_cw_info.nav_global_rc, "option_menu", args, n);

	XtManageChild(nav_cw_info.nav_option_menu);

	if (nav_cw_info.nav_global.nav_mode == C_SCALE) { 
	    XtSetSensitive(nav_cw_info.nav_global_rc, False);
	}
}

void nav_ct_update_opt()
{
	extern void nav_global_option();

	extern C_World	world;

        Arg             args[10];       /* Argument List */
        int             n;              /* Argument Count */
        int             loop;
        XmString        string;
        int             i;


        if (nav_ct_bbdialog)
        {
	    /* Destroy Existing Widgets */
            for( loop=2; loop<C_MAX_VOLUMES+2; loop++)
            {
                if (nav_cw_info.nav_options[loop])
                {
			XtDestroyWidget( nav_cw_info.nav_options[loop] );
                        nav_cw_info.nav_options[loop] = NULL;
                }
            }

	    /* Create New Widgets For Every Volume */
	    for (i=2; i<world.num_volumes+2; i++)
	    { 
	        n = 0;
	        string = XmStringCreate (world.volume[i-2]->volume_name, 
					XmSTRING_DEFAULT_CHARSET);
	        XtSetArg (args[n], XmNlabelString, string); n++;
	        nav_cw_info.nav_options[i] = XmCreatePushButtonGadget(
					nav_cw_info.nav_pulldown, 
				   world.volume[i-2]->volume_name, args, n);
	        XmStringFree (string);
                XtManageChild( nav_cw_info.nav_options[i] );

	        XtAddCallback (nav_cw_info.nav_options[i], XmNactivateCallback, 
		           nav_global_option, i);
	    }

	    if( nav_cw_info.nav_global.nav_option > world.num_volumes+1 )
		nav_cw_info.nav_global.nav_option = 0;

	    n = 0;
	    XtSetArg (args[n], XmNmenuHistory, 
	       nav_cw_info.nav_options[nav_cw_info.nav_global.nav_option]); n++;
	    XtSetArg (args[n], XmNsubMenuId, nav_cw_info.nav_pulldown); n++;
	    XtSetValues( nav_cw_info.nav_option_menu, args, n );
	}
}


/*
 *        Procedure Name: nav_CreateReferenceType
 *          Return Value: none
 *       Input Variables: parent	  - The parent widget of control window 
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: window_defs     - window appearance definitions 
 *			  nav_cw_info	  - The control window global variable
 *           Description: This procedure creates a radio box which contains
 *			  reference point selection buttons   
 *                        and adds the activation call back function to them. 
 */

void nav_CreateReferenceType(parent) 
Widget parent;
{
	/***    External Routines  ***/
	extern void nav_set_ref();

	/***    Local Variables     ***/
	Widget frame;
	Widget toggle[3];
	Arg args[20];
	XmString string;
	static char *toggle_name[3] = {"Origin", "Center", "Other->"};

	int n;


	/***** the frame *****/

	n = 0;
    	XtSetArg (args[n], XmNshadowType, XmSHADOW_ETCHED_IN);  n++;
   	XtSetArg (args[n], XmNx, 140);  n++;
   	XtSetArg (args[n], XmNy, 110);  n++;
	XtSetArg( args[n], XmNheight, 40 ); n++;
	C_Set_Color_Frame( args, n, window_defs );
    	frame = XmCreateFrame (parent, "frame", args, n);
    	XtManageChild (frame);

	/***** the frame encloses a RadioBox *****/

	n = 0;
	XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
	C_Set_Color_RadioBox( args, n, window_defs );
	nav_cw_info.nav_ref_radiobox = 
		XmCreateRadioBox (frame, "radiobox", args, n);
	XtManageChild (nav_cw_info.nav_ref_radiobox);

	/*****  Create Children  *******************************************/

	n = 0;
	string = XmStringCreate (toggle_name[0], XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_ToggleButton( args, n, window_defs );
	toggle[0] = XtCreateWidget (toggle_name[0], xmToggleButtonWidgetClass,
                                    nav_cw_info.nav_ref_radiobox, args, n);
	XmStringFree (string);
	XtAddCallback (toggle[0], XmNvalueChangedCallback, nav_set_ref, 
		       C_ORIGIN_REF);

	n = 0;
	string = XmStringCreate (toggle_name[1], XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_ToggleButton( args, n, window_defs );
	toggle[1] = XtCreateWidget (toggle_name[1], xmToggleButtonWidgetClass,
                                    nav_cw_info.nav_ref_radiobox, args, n);
	XmStringFree (string);
	XtAddCallback (toggle[1], XmNvalueChangedCallback, nav_set_ref, 
		       C_CENTER_REF);

	n = 0;
	string = XmStringCreate (toggle_name[2], XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_ToggleButton( args, n, window_defs );
	toggle[2] = XtCreateWidget (toggle_name[2], xmToggleButtonWidgetClass,
                                    nav_cw_info.nav_ref_radiobox, args, n);
	XmStringFree (string);
	XtAddCallback (toggle[2], XmNvalueChangedCallback, nav_set_ref, 
		       C_OTHER_REF);

	n = 0;
	XtSetArg( args[n], XmNset, TRUE ); n++;
	XtSetValues (toggle[nav_cw_info.nav_global.nav_ref], args, n); 

	XtManageChildren (toggle, 3);

	if (nav_cw_info.nav_global.nav_mode == C_TRANSLATE) {
	    XtSetSensitive(nav_cw_info.nav_ref_radiobox, False);
	}

}

/*
 *        Procedure Name: nav_CreateOtherPoint
 *          Return Value: none
 *       Input Variables: parent	  - The parent widget of control window 
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: window_defs     - window appearance definitions 
 *			  nav_cw_info	  - The control window global variable
 *           Description: This procedure creates a input text area used 
 *			  to specify the position of a reference point 
 *			  for transformation, 
 *                        and adds the activation call back function to them. 
 */

void nav_CreateOtherPoint(parent) 
Widget parent;
{
	Widget frame;
	Widget label;
	Widget other_label[3];
	Arg args[20];
	XmString string;
	static char *other_name[3] = {"X:", "Y:", "Z:"};
	char str[100];

	int n;


	/***** the frame *****/

	n = 0;
    	XtSetArg (args[n], XmNshadowType, XmSHADOW_ETCHED_IN);  n++;
   	XtSetArg (args[n], XmNx, 425);  n++;
   	XtSetArg (args[n], XmNy, 110);  n++;
	XtSetArg( args[n], XmNheight, 120 ); n++;
	C_Set_Color_Frame( args, n, window_defs );
    	frame = XmCreateFrame (parent, "frame", args, n);
    	XtManageChild (frame);

	/***** the frame encloses a bulletin board *****/

	n = 0;
	C_Set_Color_Bboard( args, n, window_defs );
	nav_cw_info.nav_bb = XmCreateBulletinBoard (frame, "bb", args, n);
	XtManageChild (nav_cw_info.nav_bb);


	/*****  Create Children  *******/
	
	n = 0;
   	XtSetArg (args[n], XmNx, 5);  n++;
   	XtSetArg (args[n], XmNy, 5);  n++;
	XtSetArg( args[n], XmNheight, 35 ); n++;
	string = XmStringCreate ("World Position", XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_Label( args, n, window_defs );
	label = XmCreateLabel (nav_cw_info.nav_bb, "World Position", args, n);
	XmStringFree (string);

	XtManageChild (label);

	n = 0;
   	XtSetArg (args[n], XmNx, 5);  n++;
   	XtSetArg (args[n], XmNy, 45);  n++;
	XtSetArg( args[n], XmNheight, 35 ); n++;
	string = XmStringCreate (other_name[0], XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_Label( args, n, window_defs );
	other_label[0] = XmCreateLabel( nav_cw_info.nav_bb, 
					other_name[0], args, n);
	XmStringFree (string);

	n  = 0;
   	XtSetArg (args[n], XmNx, 30);  n++;
   	XtSetArg (args[n], XmNy, 45);  n++;
	XtSetArg( args[n], XmNwidth, 100 ); n++;
	XtSetArg( args[n], XmNheight, 35 ); n++;
/****
	sprintf (str, "%f", nav_other_x);
	XtSetArg( args[n], XmNvalue, str ); n++;
****/
	XtSetArg( args[n], XmNvalue, "0.0" ); n++;
	C_Set_Color_Text( args, n, window_defs );
	nav_cw_info.nav_other_text[0] = 
		XmCreateText (nav_cw_info.nav_bb, "x", args, n);

	n = 0;
   	XtSetArg (args[n], XmNx, 5);  n++;
   	XtSetArg (args[n], XmNy, 80);  n++;
	XtSetArg( args[n], XmNheight, 35 ); n++;
	string = XmStringCreate (other_name[1], XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_Label( args, n, window_defs );
	other_label[1] = XmCreateLabel( nav_cw_info.nav_bb, 
					other_name[1], args, n);
	XmStringFree (string);

	n = 0;
   	XtSetArg (args[n], XmNx, 30);  n++;
   	XtSetArg (args[n], XmNy, 80);  n++;
	XtSetArg( args[n], XmNwidth, 100 ); n++;
	XtSetArg( args[n], XmNheight, 35 ); n++;
	XtSetArg( args[n], XmNvalue, "0.0" ); n++;
	C_Set_Color_Text( args, n, window_defs );
	nav_cw_info.nav_other_text[1] = 
		XmCreateText( nav_cw_info.nav_bb, "y", args, n);

	n = 0;
   	XtSetArg (args[n], XmNx, 5);  n++;
   	XtSetArg (args[n], XmNy, 115);  n++;
	XtSetArg( args[n], XmNheight, 35 ); n++;
	string = XmStringCreate (other_name[2], XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_Label( args, n, window_defs );
	other_label[2] = 
		XmCreateLabel (nav_cw_info.nav_bb, other_name[2], args, n);
	XmStringFree (string);

	n = 0;
   	XtSetArg (args[n], XmNx, 30);  n++;
   	XtSetArg (args[n], XmNy, 115);  n++;
	XtSetArg( args[n], XmNwidth, 100 ); n++;
	XtSetArg( args[n], XmNheight, 35 ); n++;
	XtSetArg( args[n], XmNvalue, "0.0" ); n++;
	C_Set_Color_Text( args, n, window_defs );
	nav_cw_info.nav_other_text[2] = 
		XmCreateText (nav_cw_info.nav_bb, "z", args, n);

	XtManageChildren (other_label, 3);
	XtManageChildren (nav_cw_info.nav_other_text, 3);

	if (nav_cw_info.nav_global.nav_mode == C_TRANSLATE)
	{
	    XtSetSensitive(nav_cw_info.nav_bb, False);
	}

}

/*
 *        Procedure Name: nav_CreateTranAmount
 *          Return Value: none
 *       Input Variables: parent	  - The parent widget of control window 
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: window_defs     - window appearance definitions 
 *			  nav_cw_info	  - The control window global variable
 *           Description: This procedure creates a input text area used 
 *			  to specify the transfer amounts, 
 *                        and adds the activation call back function to them. 
 */

void nav_CreateTranAmount(parent) 
Widget parent;
{
	Widget frame;
	Widget bb;
	Arg args[20];
	XmString string;
	char str[200];

	int n;


	/***** the frame *****/

	n = 0;
    	XtSetArg (args[n], XmNshadowType, XmSHADOW_ETCHED_IN);  n++;
   	XtSetArg (args[n], XmNx, 10);  n++;
   	XtSetArg (args[n], XmNy, 190);  n++;
	XtSetArg( args[n], XmNheight, 135 ); n++;
	C_Set_Color_Frame( args, n, window_defs );
    	frame = XmCreateFrame (parent, "frame", args, n);
    	XtManageChild (frame);

	/***** the frame encloses a bulletin board *****/

	n = 0;
	C_Set_Color_Bboard( args, n, window_defs );
	bb = XmCreateBulletinBoard (frame, "bb", args, n);
	XtManageChild (bb);


	/*****  Create Children  *****/

	/* initialize the label names according to the type of transformation */

	switch (nav_cw_info.nav_global.nav_mode) {
	    case C_TRANSLATE :
			nav_cw_info.nav_label_name[0] = "X unit:";
			nav_cw_info.nav_label_name[1] = "Y unit:";
			nav_cw_info.nav_label_name[2] = "Z unit:";
			break;
	    case C_ROTATE :
			nav_cw_info.nav_label_name[0] = "X angle:";
			nav_cw_info.nav_label_name[1] = "Y angle:";
			nav_cw_info.nav_label_name[2] = "Z angle:";
			break;
	    case C_SCALE :
			nav_cw_info.nav_label_name[0] = "X factor:";
			nav_cw_info.nav_label_name[1] = "Y factor:";
			nav_cw_info.nav_label_name[2] = "Z factor:";
			break;
	}


	n = 0;
   	XtSetArg (args[n], XmNx, 10);  n++;
   	XtSetArg (args[n], XmNy, 5);  n++;
	string = XmStringCreate(nav_cw_info.nav_label_name[0], 
				XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_Label( args, n, window_defs );
	nav_cw_info.nav_label[0] = XmCreateLabel(bb, 
				nav_cw_info.nav_label_name[0], args, n);
	XmStringFree (string);

	n  = 0;
   	XtSetArg (args[n], XmNx, 100);  n++;
   	XtSetArg (args[n], XmNy, 5);  n++;
	XtSetArg( args[n], XmNheight, 35 ); n++;
	XtSetArg( args[n], XmNvalue, "0.0" ); n++;
	C_Set_Color_Text( args, n, window_defs );
	nav_cw_info.nav_text[0] = XmCreateText (bb, "x", args, n);

	n = 0;
   	XtSetArg (args[n], XmNx, 10);  n++;
   	XtSetArg (args[n], XmNy, 50);  n++;
	string = XmStringCreate(nav_cw_info.nav_label_name[1], 
				XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_Label( args, n, window_defs );
	nav_cw_info.nav_label[1] = XmCreateLabel(bb, 
				nav_cw_info.nav_label_name[1], args, n);
	XmStringFree (string);

	n = 0;
   	XtSetArg (args[n], XmNx, 100);  n++;
   	XtSetArg (args[n], XmNy, 50);  n++;
	XtSetArg( args[n], XmNheight, 35 ); n++;
	XtSetArg( args[n], XmNvalue, "0.0" ); n++;
	C_Set_Color_Text( args, n, window_defs );
	nav_cw_info.nav_text[1] = XmCreateText (bb, "y", args, n);

	n = 0;
   	XtSetArg (args[n], XmNx, 10);  n++;
   	XtSetArg (args[n], XmNy, 90);  n++;
	string = XmStringCreate(nav_cw_info.nav_label_name[2], 
				XmSTRING_DEFAULT_CHARSET);
	XtSetArg (args[n], XmNlabelString, string); n++;
	C_Set_Color_Label( args, n, window_defs );
	nav_cw_info.nav_label[2] = XmCreateLabel(bb, 
				nav_cw_info.nav_label_name[2], args, n);
	XmStringFree (string);

	n = 0;
   	XtSetArg (args[n], XmNx, 100);  n++;
   	XtSetArg (args[n], XmNy, 90);  n++;
	XtSetArg( args[n], XmNheight, 35 ); n++;
	XtSetArg( args[n], XmNvalue, "0.0" ); n++;
	C_Set_Color_Text( args, n, window_defs );
	nav_cw_info.nav_text[2] = XmCreateText (bb, "z", args, n);

	XtManageChildren (nav_cw_info.nav_label, 3);
	XtManageChildren (nav_cw_info.nav_text, 3);

}


