/****************************************************************************/
/*                                                                          */
/*  VolVis is a volume visualization system for investigating, manipulating */
/*  and rendering geometric and volumetric data.                            */
/*                                                                          */
/*  Copyright (C) 1993 by the Research Foundation of the State University   */
/*                            of New York                                   */
/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 1, or (at your option)     */
/*  any later version.                                                      */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this program; if not, write to the Free Software             */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/*                                                                          */
/*  For information on VolVis, contact us at:                               */
/*                                                                          */
/*                volvis@cs.sunysb.edu                         (email)      */
/*                                                                          */
/*                Lisa Sobierajski & Ricardo Avila             (US Mail)    */
/*                Department of Computer Science                            */
/*                State University of New York at Stony Brook               */
/*                Stony Brook, New York  11794-4400                         */
/*                                                                          */
/****************************************************************************/



#include <stdio.h>
#include <fcntl.h>
#include "C_volvis.h"
#include <gl/gl.h>
#include <gl/device.h>
#include <gl/addrs.h>
#include <gl/cg2vme.h>
#include <get.h>


#define WINDOW_MODE		0
#define FULL_SCREEN_MODE	1

#define	MAX_IMAGES	500
#define STOPPED         1
#define LOOPING         2
#define MOUSEPOS        3
#define FORBACK         4
#define SWINGING        5

#define FORWARD         0
#define BACK            1

#define	LEFT		1
#define	RIGHT		2
#define	VPH		491
#define	VPW		1200

#define	TIME_INCREMENT	0.05

int		current_frame;
unsigned long 	*left_images[MAX_IMAGES];
unsigned long 	*right_images[MAX_IMAGES];
int		width, height;
int		depth;
int		num_images;
int		mode;
int		direction;
int		left_startx, left_starty;
int		right_startx, right_starty;
int		left_endx, left_endy;
int		right_endx, right_endy;

double		delay_time;

float		fgetmousex();

main(argc, argv)
int argc;
char *argv[];
{
	int		fd;
	int		loop;
	char		final_left[256];
	char		final_right[256];
	unsigned char	*red_image;
	unsigned char	*green_image;
	unsigned char	*blue_image;
	unsigned char	*rptr;
	unsigned char	*gptr;
	unsigned char	*bptr;
	unsigned char	temp_char;
	unsigned long	*iptr;
	int		zoom_factor;
	int		image_start;
	int		image_limit;
	int		screen_mode;
	int		menu;
	int		status;
	short		val;
	C_Image		image;
	int		w, h;

	char		*msg[200];

	extern void	C_wait();

	if(argc != 5)
	{
       	    C_message("Usage: sanimate base_name zoom image_start image_limit");
	    exit(0);
	}

	zoom_factor = atoi( argv[2] );

	image_start = atoi( argv[3] );

	image_limit = atoi( argv[4] );

	screen_mode = FULL_SCREEN_MODE;

	/* Open The First Image To Get Width & Height Info */
	sprintf(final_left,  "%s.L.%d.img", argv[1], image_start );	
	sprintf(final_right, "%s.R.%d.img", argv[1], image_start );	

	status = C_read_image_from_file( &image, final_left );
	if( status == C_ERROR )
	{
		sprintf( msg, "Problem Opening %s", final_left );
		C_error_message( msg );
		exit( -1 );
	}

	width = image.width;
	height = image.height;

	/* Free Up First Image Memory */
	free( image.red );
	free( image.green );
	free( image.blue );

	switch( screen_mode )
	{
		case FULL_SCREEN_MODE:
			left_startx = 
				(getgdesc(GD_XPMAX) - (width*zoom_factor) ) / 4;
			left_starty = (491 - (height*zoom_factor) ) / 2;

			right_startx = 
				(getgdesc(GD_XPMAX) - (width*zoom_factor) ) / 4;
			right_starty = 532 + left_starty;

			if( left_startx < 0  || left_starty < 0 )
			{
				C_error_message("Try a smaller zoom factor!");
				exit(0);
			}
			break;

		case WINDOW_MODE:
			left_startx = 1;
			left_starty = 1;
			right_startx = 1;
			right_starty = 1;
			break;
	}

	left_endx = left_startx + width - 1;
	left_endy = left_starty + height - 1;

	right_endx = right_startx + width - 1;
	right_endy = right_starty + height - 1;

	prefposition( 0, getgdesc( GD_XPMAX ), 0, getgdesc( GD_YPMAX ) );

	winopen( "Sanimate" );
	RGBmode();
	doublebuffer();
	gconfig();

	RGBcolor( 0, 0, 0 );

	clear();
	swapbuffers();
	clear();

	num_images = 0;

	rectzoom( 2.0 * (float) zoom_factor, (float) zoom_factor );

	setmonitor( STR_RECT );

	while ( ((fd = open(final_left, O_RDONLY ) ) >= 0) && 
		(num_images < MAX_IMAGES) &&
		(num_images < image_limit))
	{
		close(fd);

		/************************************/
		/* Read And Transfer Left Eye Image */
		/************************************/
		status = C_read_image_from_file( &image, final_left );
		if( status == C_ERROR )
		{
			sprintf( msg, "Problem Opening %s", final_left );
			C_error_message( msg );
			exit( -1 );
		}

		rptr = image.red;
		gptr = image.green;
		bptr = image.blue;

		left_images[num_images] = (unsigned long *)
			malloc( width * height * sizeof(long) );
		if( left_images[num_images] == NULL )
		{
			sprintf(msg, "Could NOT Allocate Memory For Image: %d", 
				num_images );
			C_error_message( msg );
			exit( -2 );
		}

		for( h = 0; h < height; h++ )
		{
		   iptr = left_images[num_images] + (height - h - 1) * width;

		   for( w = 0; w < width; w++ )
		   {
			*(iptr++) =   (*(rptr++)) +
				     ((*(gptr++))<<8) +
				     ((*(bptr++))<<16);
		   }
		}

		free( image.red );
		free( image.green );
		free( image.blue );

		/*************************************/
		/* Read And Transfer Right Eye Image */
		/*************************************/
		status = C_read_image_from_file( &image, final_right );
		if( status == C_ERROR )
		{
			sprintf( msg, "Problem Opening %s", final_right );
			C_error_message( msg );
			exit( -1 );
		}

		rptr = image.red;
		gptr = image.green;
		bptr = image.blue;

		right_images[num_images] = (unsigned long *)
			malloc( width * height * sizeof(long) );
		if( right_images[num_images] == NULL )
		{
			sprintf(msg, "Could NOT Allocate Memory For Image: %d", 
				num_images );
			C_error_message( msg );
			exit( -2 );
		}

		for( h = 0; h < height; h++ )
		{
		   iptr = right_images[num_images] + (height - h - 1) * width;

		   for( w = 0; w < width; w++ )
		   {
			*(iptr++) =   (*(rptr++)) +
				     ((*(gptr++))<<8) +
				     ((*(bptr++))<<16);
		   }
		}

		free( image.red );
		free( image.green );
		free( image.blue );

		num_images++;

		sprintf(final_left, "%s.L.%d.img", argv[1], 
				(num_images+image_start));
		sprintf(final_right, "%s.R.%d.img", argv[1], 
				(num_images+image_start));

	}

    	qdevice(LEFTMOUSE);
    	qdevice(MIDDLEMOUSE);
    	qdevice(MENUBUTTON);
    	qdevice(INPUTCHANGE);
    	qdevice(LEFTARROWKEY);
    	qdevice(RIGHTARROWKEY);

	menu = defpup("animate %t|stop|loop|swing|mousepos|for/back|slower|faster|exit");

	delay_time = 0.0;

	current_frame = 0;

	mode = LOOPING;

	direction = FORWARD;

/****
	pixmode( PM_TTOB, TRUE );
****/

	while( 1 )
	{
            if(mode == LOOPING) 
	    {
                while(!qtest())
                    step();
            }
            if(mode == SWINGING) 
	    {
                while(!qtest())
                    swing();
            }
	    switch(qread(&val))
	    {
		case LEFTARROWKEY:
			if(val)
			{
				direction = BACK;
				step();
			}
			break;
		case RIGHTARROWKEY:
			if(val)
			{
				direction = FORWARD;
				step();
			}
			break;
		case MIDDLEMOUSE:
			if(val)
			{
				direction = FORWARD;
				while(getbutton(MIDDLEMOUSE))
					step();
			}
			break;
		case LEFTMOUSE:
			if(val)
			{
				if(mode == MOUSEPOS)
				{
				    while(getbutton(LEFTMOUSE))
				      gotoframe((int)(num_images*fgetmousex()));
                    		} 
				else 
				{
				    direction = BACK;
				    while(getbutton(LEFTMOUSE))
					step();
				}
			}
			break;

		case MENUBUTTON:
			if(val)
			{
				switch(dopup(menu))
				{
                        		case 1:
                         	   		mode = STOPPED;
                            			break;
                        		case 2:
                            			mode = LOOPING;
                            			break;
                        		case 3:
                            			mode = SWINGING;
                            			break;
                        		case 4:
                            			mode = MOUSEPOS;
                            			break;
                        		case 5:
                            			mode = FORBACK;
                            			break;
					case 6:		/* Slower */
						delay_time += TIME_INCREMENT;
						break;
					case 7:		/* Faster */
						delay_time -= TIME_INCREMENT;
						if( delay_time < 0.0 )
							delay_time = 0.0;
						break;
					case 8:
						setmonitor( HZ60 );
						free_memory();
						exit();
                    		}
                	}

			break;

		}
					
	}

}

display_frame( frame_number )
{
	/* Delay Image Display */
	if( delay_time != 0.0 )
		C_wait( delay_time );

	lrectwrite(left_startx, left_starty, left_endx, left_endy, 
		right_images[frame_number]);
	lrectwrite(right_startx, right_starty, right_endx, right_endy, 
		left_images[frame_number]);
}

free_memory()
{
	int	loop;

	for( loop=0; loop<num_images; loop++ )
	{
		free( left_images[loop] );
		free( right_images[loop] );
	}
}

back()
{
    current_frame--;
    if(current_frame<0)
        current_frame = num_images-1;
    display_frame(current_frame);
    swapbuffers();
}

forward()
{
    current_frame++;
    if(current_frame >= num_images)
        current_frame = 0;
    display_frame(current_frame);
    swapbuffers();
}

step()
{
    if(direction == FORWARD)
        forward();
    else
        back();
}

swing()
{

    if(current_frame == 0)
        direction = FORWARD;
    if(current_frame == num_images-1)
        direction = BACK;
    step();
}

gotoframe(n)
int n;
{
    if(n >= num_images)
        n = num_images-1;
    if(n<0)
        n = 0;
    if(n!=current_frame) {
        current_frame = n;
        display_frame(current_frame);
        swapbuffers();
    }
}


float fgetmousex()
{
    long xorg, yorg;
    long xsize, ysize;

    getsize(&xsize,&ysize);
    getorigin(&xorg,&yorg);
    return ((float)getvaluator(MOUSEX)-xorg)/(float)xsize;
}

C_message( string )
char	*string;
{
	printf("%s\n", string );
}

C_error_message( string )
char	*string;
{
	printf("ERROR: %s\n", string );
}
