/*
 * Copyright (c) 1990,1,2 Mark Nitzberg
 * and President and Fellows of Harvard College
 * All rights reserved.
 * 
 * pscurves -- make curve list into postscript
 * 
 * pscurves [ file ...] produces file.ps for each file.
 * 
 * with no args, reads stdin, writes stdout
 */
#include "edgemap.h"

char    Usage[] = "\
Usage: %s [-l] [edgemap_file ...]\n\
	converts edgemap file to postscript; produces `edgemap_file.ps'\n\
	-l  label contours by contour number starting at 0\n";

int     LabelContours = FALSE;

main(argc, argv)
int     argc;
char  **argv;
{
    char    c;
    extern char *optarg;
    extern int optind;

    PgmName = argv[0];

    while ((c = getopt(argc, argv, "l")) != -1) {

	switch (c) {

	case '?':		/* getopt() arg not in list */
	    fprintf(stderr, Usage, PgmName);
	    exit(1);

	case 'l':
	    LabelContours = TRUE;
	    break;

	}
    }

    if (optind < argc) {
	while (optind < argc) {
	    PSCurves(argv[optind]);
	    optind++;
	}
    } else
	PSCurves(NULL);

    return Status;
}

PSCurves(file)
char   *file;
{
    char    buf[512];
    PointList *curve;
    FILE   *fin = stdin, *fout = stdout;
    int     i;

    CurrentFile = file;
    CurrentLine = 1;

    if (file != NULL) {
	if ((fin = fopen(file, "r")) == NULL) {
	    fprintf(stderr,
		    "%s: no access to `%s', not processed\n",
		    PgmName, file);
	    Status = 1;
	    return;
	}
	sprintf(buf, "%s.ps", file);
	if ((fout = fopen(buf, "w")) == NULL) {
	    fprintf(stderr,
		    "%s: can't create `%s'; not processed\n",
		    PgmName, buf);
	    Status = 1;
	    return;
	}
    }
    if (ReadContours(fin) == FALSE) {
	if (file != NULL)
	    fclose(fin);
	Status = 1;
	return;
    }
    PostScriptHeader(fout, Width, Height);

    for (i = 0; i < NContours; i++) {
	curve = Contours[i].p;
	DrawPostScriptCurve(fout, curve);
	if (LabelContours)
	    LabelCurve(fout, curve, i);
    }

    PostScriptTrailer(fout);
    if (file != NULL) {
	fclose(fin);
	fclose(fout);
	fprintf(stderr, "%s: Wrote postscript to %s\n",
		PgmName, buf);
    }
    /* reset */
    FreeContours();
}



#define	MAXHEIGHT (9.0*72)
#define MAXWIDTH (6.5*72)

PostScriptHeader(stream, width, height)
FILE   *stream;
real    width, height;
{
    real    scale;

    /* postscript header */
    fprintf(stream, "%%!\n");

    /* postscript points per pixel: either 6.5" wide or 9" high */
    scale = MIN(MAXWIDTH / width, MAXHEIGHT / height);

    /* bounding box with 1/2-pix cushion */
    fprintf(stream, "%%%%BoundingBox: %g %g %g %g\n\n",
	    306 - (width + .5) * scale / 2, 396 - (height + .5) * scale / 2,
	    306 + (width + .5) * scale / 2, 396 + (height + .5) * scale / 2);

    fprintf(stream, "\
%% font for labelling\n\
/Times-Roman findfont [%g 0 0 %g 0 0] makefont setfont\n", 8 / scale, -8 / scale);

    if (LabelContours)
	fprintf(stream, "\
%% Arrow draws an arrow:\n\
%% tailx taily tipx tipy tailthick tipthick headlength Arrow -\n\
/arrowdict 14 dict def\n\
arrowdict begin\n\
 /mtrx matrix def\n\
end\n\
/Arrow\n\
  { arrowdict begin\n\
     /headlength exch def\n\
     /halfheadthickness exch 2 div def\n\
     /halfthickness exch 2 div def\n\
     /tipy exch def /tipx exch def\n\
     /taily exch def /tailx exch def\n\
\n\
     /dx tipx tailx sub def\n\
     /dy tipy taily sub def\n\
     /arrowlength dx dx mul dy dy mul add\n\
       sqrt def\n\
     /angle dy dx atan def\n\
     /base arrowlength headlength sub def\n\
     /savematrix mtrx currentmatrix def\n\
\n\
     tailx taily translate\n\
     angle rotate\n\
\n\
     0 halfthickness neg moveto\n\
     base halfthickness neg lineto\n\
     base halfheadthickness neg lineto\n\
     arrowlength 0 lineto\n\
     base halfheadthickness lineto\n\
     base halfthickness lineto\n\
     0 halfthickness lineto\n\
     closepath\n\
\n\
     savematrix setmatrix\n\
    end\n\
  } def\n");


    fprintf(stream, "\
%% x y C draws a small filled circle at (x,y)\n\
%% x y O draws a less small hollow circle at (x,y)\n\
%% x y theta F draws a small filled circle and needle at (x,y) w/angle theta\n\
%% x y theta H draws a less small hollow circle and needle\n\
\n\
/R2 .5 def\n\
\n\
/C { newpath 0.15 0 360 arc closepath fill } def\n\
\n\
/O { newpath 0.3 0 360 arc closepath stroke } def\n\
\n\
/F { /theta exch def /y exch def /x exch def\n\
	theta cos R2 mul /dx exch def\n\
	theta sin R2 mul /dy exch def\n\
	x y C newpath x dx add y dy add moveto\n\
	dx -2 mul dy -2 mul rlineto stroke } def\n\
\n\
/H { /theta exch def /y exch def /x exch def\n\
	theta cos R2 mul /dx exch def\n\
	theta sin R2 mul /dy exch def\n\
	x y O newpath x dx add y dy add moveto\n\
	dx -2 mul dy -2 mul rlineto stroke } def\n");

    fprintf(stream, "%% origin at center of page\n306 396 translate\n\n");

    fprintf(stream, "%% scale pixels to points; flip y axis\n%g %g scale\n\n",
	    scale, -scale);

    fprintf(stream, "%% line width = 0.1 point\n%g setlinewidth\n\n",
	    0.1 / scale);

    fprintf(stream, "%% origin at `center' of upper left pixel\n%g %g translate\n\n",
	    -(real) width / 2.0 + 0.5, -(real) height / 2.0 + 0.5);

    fprintf(stream, "%% The contours:\n");
}

DrawPostScriptCurve(stream, curve)
FILE   *stream;
PointList *curve;
{
    int     i;

    if (curve->n < 1)
	return;

    /*
     * for each point, draw a line segment & a little dot ignore curvature
     * for the time being
     */

    if (curve->boundary) {
	/* the endpoints of non-closed curves get hollow circles */
	fprintf(stream, "%g %g %s\n",
		curve->x[0], curve->y[0],
		curve->closed ? "C" : "O");
	fprintf(stream, "%g %g %s\n",
		curve->x[curve->n - 1], curve->y[curve->n - 1],
		curve->closed ? "C" : "O");

	/* all points in between get filled circles */
	for (i = 1; i < curve->n - 1; i++)
	    fprintf(stream, "%g %g C\n",
		    curve->x[i], curve->y[i]);
    } else {
	/* the endpoints of non-closed curves get hollow circles */
	fprintf(stream, "%g %g %g %s\n",
		curve->x[0], curve->y[0], DEG(curve->theta[0]),
		curve->closed ? "F" : "H");
	fprintf(stream, "%g %g %g %s\n",
		curve->x[curve->n - 1], curve->y[curve->n - 1],
		DEG(curve->theta[curve->n - 1]),
		curve->closed ? "F" : "H");

	/* all points in between get filled circles */
	for (i = 1; i < curve->n - 1; i++)
	    fprintf(stream, "%g %g %g F\n",
		    curve->x[i], curve->y[i], DEG(curve->theta[i]));
    }

    /* draw a curves through all the points */
    fprintf(stream, "gsave 0 setlinewidth newpath %g %g moveto\n",
	    curve->x[0], curve->y[0]);

    /* connect them */
    for (i = 1; i < curve->n; i++)
	fprintf(stream, "%g %g lineto\n",
		curve->x[i], curve->y[i]);

    if (curve->closed)
	fprintf(stream, "closepath ");

    if (curve->contin)
	fprintf(stream, "[ .1 .2 ] 0 setdash ");

    fprintf(stream, "stroke grestore\n");
}

LabelCurve(stream, curve, i)
FILE   *stream;
PointList *curve;
int     i;
{
    /* put 0 at the head, 1 at the tail, and i in the middle */
    int     n = curve->n;
    real    x, y;
    real    sint, cost, norm;

    /* print the contour number i in the middle */
    x = curve->x[n / 2] - curve->x[n / 2 - 1];
    y = curve->y[n / 2] - curve->y[n / 2 - 1];
    if (x * x + y * y <= 0) {
	x = curve->x[n / 2 + 1] - curve->x[n / 2];
	y = curve->y[n / 2 + 1] - curve->y[n / 2];
	if (x * x + y * y <= 0)
	    return;
    }
    norm = sqrt(x * x + y * y);
    sint = y / norm;
    cost = x / norm;

    /* let x,y be center of character */
    x = curve->x[n / 2] + 3 * (-sint);
    y = curve->y[n / 2] + 3 * cost;

    fprintf(stream,
	    "%g %g moveto\n\
(%d) stringwidth 2 div neg exch 2 div neg exch rmoveto (%d) show\n",
	    x, y, i, i);

    /* tailx taily tipx tipy tailthick tipthick headlength Arrow - */
    fprintf(stream,
	    "newpath %g %g %g %g %g %g %g Arrow fill\n",
	    x + 2 * cost, y + 2 * sint,
	    x + 3.5 * cost, y + 3.5 * sint,
	    0.2, 0.4, 0.3);
}

PostScriptTrailer(stream)
FILE   *stream;
{
    /* postscript header */
    fprintf(stream, "showpage\n");
}
