/***************************************************************************
***	HVISION.H
***
***	Copyright 1987 by the Massachusetts Institute of Technology and by
***	the President and Fellows of Harvard College.
***	All rights reserved.
***
***	This software is provided AS IS.  M.I.T., Harvard, and the authors do
***	not guarantee or assume any responsibility for the performance of this
***	software, though every effort has been made to insure its reliability.
***	The HVision package is licensed only to specific sites. Licensees may
***	NOT distribute any portion of this software outside of the site
***	specified in the license. For more information, contact the
***	Office For Patents, Copyrights, and Licensing, Harvard University.
***
***	Peter W. Hallinan	3/88	more modifications
***	Mark Nitzberg		11/87	added macros
***	Peter W. Hallinan	11/87	modifications
***	Gaile G. Gordon		6/87	creation
***
***************************************************************************/

#ifdef TEST
#include <stdlib.h>
#include <math.h>		/* so we dont forget and get frustrating
				 * errors */
typedef unsigned char u_char;
typedef unsigned short u_short;
typedef unsigned int u_int;

#endif

#ifndef TEST
#include <sys/types.h>		/* because of x windows */
#include <math.h>		/* so we dont forget and get frustrating
				 * errors */
extern char *malloc();		/* no more stupid compiler warnings */
extern char *calloc();		/* no more stupid compiler warnings */

#endif

extern double atof();		/* no more stupid compiler warnings */

#include <string.h>		/* so we dont forget and get frustrating
				 * errors */

/***************************************************************************
***	critical definitions for images
***	pixeltypes and bandorders must be greater than zero
***************************************************************************/
#define HSIF_ID		"hsi\n"	/* a unique 4-byte identifier */

#define MAX_VAR		30	/* number of user-definable variables */

#define BINARY		0	/* char pixel type */
#define ONEBYTE		1	/* char pixel type */
#define UTWOBYTE	2	/* unsigned short pixel type */
#define STWOBYTE	3	/* signed short pixel type */
#define INTPIX		4	/* integer pixel type */
#define REALPIX		5	/* floating point pixel type */
#define DBLPIX		6	/* double pixel type */
#define BYTES_PER_TYPE	"\1\1\2\2\4\4\10"	/* octal, indexed by types
						 * above */

#define SEQUENTIAL	0	/* b0(p0), b0(p1), ..., b0(pn), b1(p0), ... */
#define INTERLEAVED	1	/* b0(p0), b1(p0), ..., bm(p0), b0(p1), ... */
/* where bi = ith band, pj = jth pixel */

/***************************************************************************
***	rename/abbreviate C types
***
***	the first 3 are commented out because they appear in sys/types.h
***************************************************************************/
/* typedef unsigned char u_char; */
/* typedef unsigned short u_short; */
/* typedef unsigned int u_int; */
typedef float real;

/***************************************************************************
***	define a complex type that can be used to get at complex numbers
***	stored in a 2-band, REALPIX, INTERLEAVED image
***************************************************************************/
typedef struct t_complex {
    real    c_re, c_im;
} complex;

/***************************************************************************
***	the definition of an image
***************************************************************************/
typedef struct t_IMAGE {
    char    imagetype[4];	/* must be HSIF_ID */
    char   *name;		/* symbolic name of image */
    int     height;		/* height of a band in pixels */
    int     width;		/* width of a band in pixels */
    int     numbands;		/* number of bands in image */
    int     bandorder;		/* the order in which bands are stored */
    int     pixeltype;		/* internal representation of a pixel */
    int     bytesize;		/* size of image (all bands) in bytes */
    int     comlen;		/* length of comment */
    char   *comment;		/* comment string */
    int     var[MAX_VAR];	/* set of user-definable variables */
    u_char *im;			/* ptr to 1d data stream */
    u_char **twod;		/* ptr to array of row pointers */
    u_char ***threed;		/* ptr to array of bands */
} IMAGE;

/***************************************************************************
***	definition of a vector list
***
***	Type can only be one of the legal pixel types.
***	No matter what the type is, z values (pixel values)
***	are stored as reals.
***************************************************************************/
typedef struct t_VLIST {
    int     numv;		/* number of vectors in list */
    int     type;		/* pixel type */
    int    *x;			/* x grid coordinate */
    int    *y;			/* y grid coordinate */
    real   *z;			/* z coordinate or pixel value */
}       VLIST;

/***************************************************************************
***	legal kernel ids
***************************************************************************/
#define K_EMPTY		0
#define K_HVISION	1
#define K_MORPH		2

/***************************************************************************
***	define a hvision kernel
***
***	A kernel is really just a small image, usually 1-band, sequential.
***	Kernel elements lie on a grid having only integer valued coordinates
***	but can have any type of value themselves.
***	Kernels may be decomposed by hand into subkernels which can be stored
***	within the kernel.
***************************************************************************/
typedef struct t_KERNEL {
    IMAGE  *image;		/* kernel is stored as an image */
    int     numsubk;		/* number of subkernels */
    struct t_KERNEL **subk;	/* array of subkernels */
} KERNEL;

/***************************************************************************
***	sample point or pixel that is part of an edge
***************************************************************************/
typedef struct t_LINK {
    real    x, y;		/* coordinates of a sample point */
    real    aux1, aux2;		/* temporary storage */
    real    aux3, aux4;		/* temporary storage */
    real    aux5, aux6;		/* temporary storage */
    struct t_LINK *prev;	/* forward pointer */
    struct t_LINK *next;	/* backward pointer */
}       LINK, *linkp;

/***************************************************************************
***	structure describing a sampled curve or edge
***************************************************************************/
typedef struct t_EDGE {
    int     edgeid;		/* number of links in the edge */
    int     linknum;		/* number of links in the edge */
    int     edgetype;		/* see "edge types" below */
    int     beg, end;		/* labelling numbers of 1st & last link */
    int     left, right;	/* labelling numbers of left & right regions */
    LINK   *beglink;		/* pointer to 1st link structure */
    LINK   *endlink;		/* pointer to last link structure */
    struct t_EDGE *prev;	/* forward pointer */
    struct t_EDGE *next;	/* backward pointer */
}       EDGE, *edgep;

/***************************************************************************
***	edge types for edgetype member of edge structure
***************************************************************************/
#define LOOP		0	/* closed loop */
#define BORDER		1	/* edge on image border */
#define TRIPLE_TRIPLE	2	/* edge from triple pt to triple pt */
#define TRIPLE_CRACK	3	/* edge from triple pt to crack tip */
#define TRIPLE_BORDER	4	/* edge from triple pt to border */
#define CRACK_TRIPLE	5	/* etc.			 */
#define CRACK_CRACK	6
#define CRACK_BORDER	7
#define BORDER_TRIPLE	8
#define BORDER_CRACK	9
#define BORDER_BORDER	10

/***************************************************************************
***	error allowance of the polygonal approximations to the given image
***************************************************************************/
#define ERR_OK		2.5

/***************************************************************************
***	string names for each pixeltype and bandorder
***	also, string name for image with no given name
***************************************************************************/
#define NAME_BINARY		"Binary"
#define NAME_ONEBYTE		"OneByte"
#define NAME_UTWOBYTE		"UTwoByte"
#define NAME_STWOBYTE		"STwoByte"
#define NAME_INTPIX		"IntPix"
#define NAME_REALPIX		"RealPix"
#define NAME_DBLPIX		"DoublePix"
#define NAME_SEQUENTIAL		"Sequential"
#define NAME_INTERLEAVED	"Interleaved"

#define NAME			(image->name?image->name:"NoName")

/***************************************************************************
***	minimum and maximum pixel values for each type
***
***	Note that floating point bounds are reduced by 2.5 each so that
***	map_range and other functions won't cause an exception when
***	they compute the distance between the maximum and minimum values.
***************************************************************************/
#include <values.h>
#define MIN_BINARY	(0)
#define MAX_BINARY	(1)
#define MIN_ONEBYTE	(0)
#define MAX_ONEBYTE	(255)
#define MIN_UTWOBYTE	(0)
#define MAX_UTWOBYTE	(0xffff)
#define MIN_STWOBYTE	(-(MAXSHORT)-1)
#define MAX_STWOBYTE	(MAXSHORT)
#define MIN_INTPIX	(-(MAXINT)-1)
#define MAX_INTPIX	((MAXINT)-100)
#define MIN_REALPIX	(-(MAXFLOAT/2.5))
#define MAX_REALPIX	(MAXFLOAT/2.5)
#define MIN_DBLPIX	(-(MAXDOUBLE/2.5))
#define MAX_DBLPIX	(MAXDOUBLE/2.5)

/***************************************************************************
***	the usual macros
***************************************************************************/

#ifndef TRUE
#define TRUE	1
#endif

#ifndef FALSE
#define FALSE	0
#endif

#ifndef NULL
#define NULL	0
#endif

#ifndef SQ
#define SQ(a)	((a)*(a))
#endif

#ifndef LOWER
#define LOWER(c) (((c)<'a')?((c)-'A'+'a'):(c))
#endif

#define MIN(a,b)	(((a)<(b))?(a):(b))
#define MAX(a,b)	(((a)>(b))?(a):(b))

#ifndef ABS
#define ABS(a)		(((a)>0)?(a):-(a))
#endif

#ifndef PI
#define PI		3.141592653589793
#endif

#ifndef TWOPI
#define TWOPI		6.283185307179586
#endif

#ifndef SWAP
#define SWAP(t,a,b)	{t temp; temp=(a);(a)=(b);(b)=temp;}
#endif

#define REAL_EPSILON	1.0e-10
#define ZERO(x)		((ABS(x)<REAL_EPSILON)?TRUE:FALSE)

/***************************************************************************
***	rounding
***************************************************************************/
#define ROUND(x)	((int)(.5 +(x)))	/* round positive real only */

/*
 * round via (n-0.5,n+0.5] -> n for negative integers (-0.5,+0.5)   -> 0
 * [n-0.5,n+0.5) -> n for positive integers
 */
#define RRND(x)		(((x)<0.0)?((x)-0.5):((x)+0.5))

/*
 * round via [n-0.5,n+0.5) -> n for all integers #define RRND(x)
 * ((x)<0.0)? ((((int)(x))<=(x)+0.5)?x:x-1): ((x)+0.5))
 */

#define IRND(x)		((int)RRND(x))

/***************************************************************************
***	return values for functions
***************************************************************************/
#define I_SUCCESS	1	/* must be positive */
#define I_ERROR		((int)0)/* must be zero */
#define I_QUIT		-1	/* must be negative */

/***************************************************************************
***	defines for various library routines
***************************************************************************/
#define INPLACE		TRUE	/* used by assign_work */
#define COPYSOURCE	-1	/* must be negative - used by assign_work */
#define ALL_BANDS	-1	/* must be negative - used by many functions */
#define CLIP_OUTPUT	1
#define NORM_OUTPUT	2
#define PI_PAD		1
#define	PI_POWER2	2
#define PI_FFT		1
#define	PI_COPY		2

/***************************************************************************
***	macros that return info about an image
***
***	You can use any of these except I_BYTESPER on the left-hand side
***	of an assignment statement.
***************************************************************************/
#define I_TYPE(i)	((i)->pixeltype)
#define I_ROWS(i)	((i)->height)
#define I_COLS(i)	((i)->width)
#define I_BANDS(i)	((i)->numbands)
#define I_ORDER(i)	((i)->bandorder)
#define I_BYTESPER(i)	BYTES_PER_TYPE[I_TYPE(i)]	/* #bytes per pixel */

/***************************************************************************
***	short macros to access individual macros
***
***	WARNING!!!!
***	These are not the things to use for fast point transforms.
***	All the macros except C_PIX are for a ONE BAND IMAGE ONLY.
***	C_PIX only works on a complex (2-band, REALPIX, INTERLEAVED) image.
***************************************************************************/
#define PIXEL(I,i,j,t)	((t **)((I)->twod))[i][j]
#define B_PIX(I,i,j)	PIXEL(I,i,j,u_char)
#define US_PIX(I,i,j)	PIXEL(I,i,j,u_short)
#define SS_PIX(I,i,j)   PIXEL(I,i,j,short)
#define INT_PIX(I,i,j)  PIXEL(I,i,j,int)
#define R_PIX(I,i,j)	PIXEL(I,i,j,real)
#define C_PIX(I,i,j)	PIXEL(I,i,j,complex)
#define D_PIX(I,i,j)	PIXEL(I,i,j,double)

/***************************************************************************
***	real and imaginary parts of a pixel in a complex image
***************************************************************************/
#define RVAL(i,y,x)	C_PIX(i,y,x).c_re
#define IVAL(i,y,x)	C_PIX(i,y,x).c_im

/***************************************************************************
***	row pointers
***
***	WARNING!
***	These work for ONE BAND or COMPLEX images ONLY.
***************************************************************************/
#define ROW(I,i,t)	((t *)(I)->twod[i])
#define B_ROW(I,i)	ROW(I,i,u_char)
#define U_ROW(I,i)	ROW(I,i,u_short)
#define S_ROW(I,i)	ROW(I,i,short)
#define I_ROW(I,i)	ROW(I,i,int)
#define R_ROW(I,i)	ROW(I,i,real)
#define C_ROW(I,i)	ROW(I,i,complex)
#define D_ROW(I,i)	ROW(I,i,double)

/***************************************************************************
***	the i/o functions - change these at your peril
***************************************************************************/
extern int (*gets_action) ();
extern int (*write_action) ();
extern int (*error_action) ();
extern int (*abort_action) ();
extern int (*warning_action) ();

#define igets		(*gets_action)
#define iprintf		(*write_action)
#define ierror		(*error_action)
#define iabort		(*abort_action)
#define iwarning	(*warning_action)

/***************************************************************************
***	useful predicates
***************************************************************************/
#define IS_GREYSCALE(i)	(((i)->pixeltype==ONEBYTE)&&((i)->numbands==1))
#define IS_REAL(i)	(((i)->pixeltype==REALPIX)&&((i)->numbands==1))
#define IS_COMPLEX(i)	(((i)->pixeltype==REALPIX)&&\
			 ((i)->numbands==2)&&\
			 ((i)->bandorder==INTERLEAVED))
#define IS_RGBCOLOR(i)	(((i)->pixeltype==ONEBYTE)&&((i)->numbands==3))
#define LEGAL_BAND(I,b)	(((0<=b)&&(b<(I)->numbands))||(b==ALL_BANDS))
#define LEGAL_PVAL(I,v)	(hvLegalPval((I),((real)(v))))
#define LEGAL_PPOS(I,i,j) ((0<=(i)) && ((i)<(I)->height)&&\
			   (0<=(j)) && ((j)<(I)->width))
#define SAME_HW(I,J)	((I)->width==(J)->width&&(I)->height==(J)->height)
#define SAME_NB(I,J)	((I)->numbands==(J)->numbands)
#define SAME_SIZE(I,J)	((I)->width==(J)->width&&(I)->height==(J)->height&&\
			 (I)->numbands==(J)->numbands)
#define SAME_ORDER(I,J)	((I)->bandorder==(J)->bandorder)
#define SAME_TYPE(I,J)	((I)->pixeltype==(J)->pixeltype)

/***************************************************************************
***	useful abbreviations of calls to hvision routines
***************************************************************************/
#define MAKE_GREYSCALE(h,w)	hvMakeImage((h),(w),1,SEQUENTIAL,ONEBYTE)
#define MAKE_RGBCOLOR(h,w)	hvMakeImage((h),(w),3,INTERLEAVED,ONEBYTE)
#define MAKE_COMPLEX(h,w)	hvMakeImage((h),(w),2,INTERLEAVED,REALPIX)

/***************************************************************************
***	binary operators expected by pix_op1 & pix_op2
***************************************************************************/
#define BOP_AND		0
#define BOP_OR		1
#define BOP_XOR		2
#define BOP_ADD		3
#define BOP_SUB		4
#define BOP_MUL		5
#define BOP_DIV		6
#define BOP_DIS		7
#define BOP_MIN		8
#define BOP_MAX		9

/***************************************************************************
***	orders that an application program can send to gterm
***************************************************************************/

#ifdef MTERM
#define MTERM_INIT		-1
#define MTERM_MENU		-2
#define MTERM_HIGH		-3
#define MTERM_NORM		-4
#endif

#define GTERM_QUIT		0
#define GTERM_CLOSE		0	/* synonym for GTERM_QUIT */
#define GTERM_VIEW		1
#define GTERM_DOTS		2
#define GTERM_LINES		3
#define GTERM_CIRCLE		4
#define GTERM_BLOCK		5
#define GTERM_MOVE		6
#define GTERM_VALUES		7
#define GTERM_M_ANYRECT		8
#define GTERM_M_BDRECT		9
#define GTERM_M_BDCIRC		10
#define GTERM_M_FIXRECT		11
#define GTERM_M_ANYSQ		12
#define GTERM_M_2N2MRECT	13
#define GTERM_M_LINE		14
#define GTERM_M_2NSQ		15
#define GTERM_M_DOT		16
#define GTERM_M_CIRCLE		17
#define GTERM_FB_ON		18
#define GTERM_FB_OFF		19
#define GTERM_FB_RPIXEL		20
#define GTERM_FB_RROW		21
#define GTERM_FB_WPIXEL		22
#define GTERM_FB_WROW		23
#define GTERM_IMMORTALITY_ON	24
#define GTERM_IMMORTALITY_OFF	25
#define GTERM_M_ELLIPSE		26
#define GTERM_DELETE		27
#define GTERM_FB_RCOL		28
#define GTERM_FB_WCOL		29
#define GTERM_CDOTS		30
#define GTERM_CLINES		31
#define GTERM_Q_NAME		32
#define GTERM_ERASE_ON		33
#define GTERM_ERASE_OFF		34
#define GTERM_M_VRECT		35
#define GTERM_XHAIRS		36
#define GTERM_CXHAIRS		37
#define GTERM_VIEWCOLOR         38
#define GTERM_GETCMAP           39
#define GTERM_DIE_WHEN_DONE	40
#define GTERM_ABORT		41
#define GTERM_VALUES_OFFSET	42
#define GTERM_NOP		43
#define GTERM_M_FILLRECT	44
#define GTERM_M_COPYRECT	45
#define GTERM_FBVIEW		46
#define GTERM_M_EDIT		47
#define GTERM_M_ODDSQ		48
#define GTERM_CCIRCLE		49
#define GTERM_SAVE_MAP		50
#define GTERM_RESTORE_MAP	51
#define GTERM_FRES_INIT		52
#define GTERM_FRES_END		53
#define GTERM_FRES_ON		54
#define GTERM_FRES_OFF		55
#define GTERM_SRES_INIT		56
#define GTERM_SRES_END		57
#define GTERM_SRES_ON		58
#define GTERM_SRES_OFF		59
#define GTERM_GRAB		60
#define GTERM_UNGRAB		61
#define GTERM_SUBIMAGE		62
#define GTERM_M_2CIRCLE		63
#define GTERM_ICONIFY_ON	64
#define GTERM_ICONIFY_OFF	65
#define GTERM_M_IVPAIR		66
#define GTERM_CRECT		67
#define GTERM_CUSTOM_VALUES	68

/***************************************************************************
***	return values from gterm recv_* commands
***************************************************************************/
#define R_ERROR		0
#define R_VALUE		1
#define R_NOVALUE	2

/***************************************************************************
***	legal flags to pass to hvPlot2d()
***************************************************************************/
#define PLOT2D_LINE	0
#define PLOT2D_CONNECT	1
#define PLOT2D_CONFIX	2

/***************************************************************************
***	convolution flags
***************************************************************************/
#define SLOW_CONVOLVE	0
#define FFT_CONVOLVE	1
#define INT_CONVOLVE	2

/***************************************************************************
***	function prototypes
***************************************************************************/
int	_hvAbbrevStr(char *, char *);
int	hvAccess(char *);
int	hvActOnImages(int (*)());
IMAGE	*hvAffine(IMAGE *, IMAGE *, double, double, double, double, double, double, int, int, int, int);
char	*hvAlloc(int);
EDGE	*hvAppendEdge(EDGE **, EDGE *, int);
LINK	*hvAppendExistingLink(EDGE *, LINK *, int);
LINK	*hvAppendLink(EDGE *, int, int);
int	hvAppendVlist(int, VLIST *, char *);
int	hvAppendVlistVA(int, VLIST *, char *, ...);
IMAGE	*hvAssignDest(IMAGE *, IMAGE *, IMAGE *);
IMAGE	*hvAssignWork(IMAGE *, IMAGE *, int, ...);
IMAGE	*hvAverage(IMAGE *, IMAGE *, int, int);
void	_hvBSData(IMAGE *, int);
void	_hvBSHdr(char *);
int	hvBSRead(int, unsigned char *, int, int, int, int);
void	_hvBSRealRead(unsigned char *);
void	_hvBSRealWrite(unsigned char *);
void	_hvBSVlist(VLIST *, int);
int	hvBSWrite(int, unsigned char *, int, int, int, int);
IMAGE	*hvBandPass(IMAGE *, IMAGE *, double, double);
int	hvBeepOff(void);
int	hvBeepOn(void);
IMAGE	*hvBilInt(IMAGE *, IMAGE *, double, double);
IMAGE	*hvBilIntToSize(IMAGE *, IMAGE *, int, int);
IMAGE	*hvBilinearInt(IMAGE *, IMAGE *, double, double);
int	hvBinQuickSort(unsigned char *, int);
IMAGE	*hvBinarize(IMAGE *, IMAGE *, int, double, double, double);
IMAGE	*hvBlowDown(IMAGE *, IMAGE *);
IMAGE	*hvBlowUp(IMAGE *, IMAGE *);
int	hvBoxDot(IMAGE *, IMAGE *, double, int, int);
IMAGE	*hvBuildComplex(IMAGE *, int, IMAGE *, int, IMAGE *);
int	hvBytesPerPixel(int);
IMAGE	*hvCanny(IMAGE *, IMAGE *, int, int, int);
int	_hvCatchSignals(void);
void	hvCbreakOff(void);
void	hvCbreakOn(void);
int	hvCeilPowerOf2(int);
int	hvCleanImagelist(void);
int	hvCleanVL(VLIST **, int);
IMAGE	*hvCleanWork(IMAGE *, IMAGE *, IMAGE *);
int	hvCleanWorklist(void);
int	hvCloseImageStream(struct _iobuf *);
IMAGE	*hvCompose(IMAGE **, int, int, int, int, int, int, double);
IMAGE	*hvComposePyramid(IMAGE **, int, int, double, int, int);
IMAGE	*hvComposeTower(IMAGE **, int, int, int, int, double, int);
IMAGE	*hvConCom(IMAGE *, IMAGE *, int, int, int *);
IMAGE	*hvConComAll(IMAGE *, IMAGE *, int, int *);
IMAGE	*hvConComSkip(IMAGE *, IMAGE *, int, int *, int);
IMAGE	*hvConvert(IMAGE *, IMAGE *, int);
IMAGE	*hvConvertZ(IMAGE *, IMAGE *, int);
IMAGE	*hvConvolve(IMAGE *, IMAGE *, int, KERNEL *, int);
IMAGE	*_hvConvolve(IMAGE *, IMAGE *, int, KERNEL *, int, int);
IMAGE	*hvCopyContents(IMAGE *, IMAGE *);
EDGE	*hvCopyEdge(EDGE *, int);
int	_hvCopyHeader(IMAGE *, IMAGE *, int, int);
IMAGE	*hvCopyImage(IMAGE *);
KERNEL	*hvCopyKernel(KERNEL *);
int	hvCopyLineData(GRAPH *, int, float *, float *);
LINK	*hvCopyLink(LINK *, LINK *, LINK *);
int	hvCopyList(IMAGE **, IMAGE **, int, int);
IMAGE	*hvCopyRegion(IMAGE *, IMAGE *, int, int, int, int, int, int, int, int);
int	hvCopyToWork(IMAGE *, IMAGE *, IMAGE *, int);
double	hvCorrelate(IMAGE *, IMAGE *, IMAGE *);
int	hvCountLinks(EDGE *);
IMAGE	*hvCrossCorrelate(IMAGE *, IMAGE *, IMAGE *, IMAGE *, int);
int	_hvDefaultAbort(char *, char *, ...);
void	_hvDefaultAnyError(char *, char *, char *, int (*)(), int);
int	_hvDefaultError(char *, char *, ...);
int	_hvDefaultGets(char *, int);
int	_hvDefaultPrintf();
int	_hvDefaultWarning(char *, char *, ...);
IMAGE	*hvDither(IMAGE *, IMAGE *);
IMAGE	*hvDog(IMAGE *, IMAGE *, double, double, int, int);
int	hvDrawBlock(IMAGE *, int, int, int, int, int, double, int);
int	hvDrawBorder(IMAGE *, int, int, int, double);
int	hvDrawCircle(IMAGE *, int, int, int, int, double, int);
int	hvDrawCircleInVlist(VLIST *, double, double, double, int);
int	hvDrawEdge(IMAGE *, EDGE *, int);
int	hvDrawGNoise(IMAGE *, double, double);
int	hvDrawGScale(IMAGE *);
int	hvDrawGauSpline(IMAGE *, int, double);
int	hvDrawGaussian(IMAGE *, int, double, double);
int	hvDrawGrid(IMAGE *, int, int, int, int, int, double);
int	hvDrawLine(IMAGE *, int, int, int, int, int, double);
int	hvDrawParabolaInVlist(VLIST *, double, double, double, double, double, double, int);
int	hvDrawRectInVlist(VLIST *, int, int, int, int, int);
int	hvDrawText(IMAGE *, int, int, int, char *, int, int, int);
int	hvDrawWNoise(IMAGE *, int, double, double, int);
int	hvDumpGraph(GRAPH *);
int	hvDumpHdr(IMAGE *, int, int);
int	hvDumpLine(line *, int);
int	hvDumpMinMax(IMAGE *, char *);
int	hvEdgeIntoVlist(EDGE *, VLIST *, int *);
VLIST	*hvEdgeToVlist(EDGE *);
IMAGE	*hvEditBorder(IMAGE *, IMAGE *, int, int, int, int, int);
int	_hvEnterImage(IMAGE *);
IMAGE	*hvExactCopy(IMAGE *);
int	hvExistImage(IMAGE *, char *, char *);
int	hvExpandPyramid(IMAGE **, IMAGE **, int, int, int, double);
IMAGE	*hvExtractBand(IMAGE *, int);
IMAGE	*_hvExtremaDown(IMAGE *, IMAGE *, int);
IMAGE	*hvFFT(IMAGE *, IMAGE *, int);
int	hvFFT1d(complex *, int, double, int, int, int);
int	hvFFT2d(complex *, int, int, int, int);
void	hvFastCircle(int, int, int, int (*)());
void	hvFastLine(int, int, int, int, int (*)());
int	hvFillPolygon(IMAGE *, VLIST *, double);
void	_hvFindAxisRanges(double, double, float *, float *, float *);
int	hvFindEdgeBBox(EDGE *, LINK **, LINK **, LINK **, LINK **);
IMAGE	*hvFindImage(char *);
int	hvFlagOn(int, int);
int	hvFlagSet(int *, int, int);
int	hvFloorPowerOf2(int);
int	hvFree(char *);
int	hvFreeEdge(EDGE *, int);
int	hvFreeEdgelist(EDGE *);
int	hvFreeGraph(GRAPH *);
int	_hvFreeHeader(IMAGE *);
int	hvFreeImage(IMAGE *);
int	hvFreeImagelist(int);
int	hvFreeKernel(KERNEL *);
int	hvFreeLink(LINK *);
int	hvFreeLinklist(LINK *);
int	hvFreeObjSet(OBJSET *);
int	hvFreeObject(OBJ *);
int	_hvFreeThreed(IMAGE *);
int	hvFreeVlist(VLIST *);
int	hvFreeWorklist(int);
int	hvFullImagelist(void);
int	hvFullWorklist(void);
int	hvFuncText(int (*)(), int, int, char *, int, int);
double	hvGRand(void);
IMAGE	*hvGauSpline(IMAGE *, IMAGE *, double, int);
IMAGE	*hvGaussian(IMAGE *, IMAGE *, double, int);
char	*hvGetBOName(int);
unsigned char *hvGetBot(IMAGE *, int);
IMAGE	*hvGetField(char *, char *, char *);
void	_hvGetFullParts(float *, int *, int *);
int	hvGetInc(IMAGE *, int);
int	hvGetObjScale(double, double, double);
int	hvGetOpt(int *, char ***, int *, char *);
double	hvGetPTMax(int);
double	hvGetPTMin(int);
char	*hvGetPTName(int);
void	_hvGetParts(float *, int *, int *);
double	hvGetPixel(IMAGE *, int, int, int);
int	hvGetPyrOffsets(IMAGE **, int, int, int, int, int, int *, int *);
int	hvGetPyrWinDim(IMAGE **, int, int, int, int, int *, int *);
unsigned char *hvGetTop(IMAGE *, int);
IMAGE	*hvGidgi(IMAGE *, IMAGE *);
int	hvGmaField(IMAGE *, IMAGE **, IMAGE **);
int	hvGmxyField(IMAGE *, IMAGE **, IMAGE **, IMAGE **);
int	hvGradList(IMAGE **, IMAGE **, int, int);
int	hvGradient(IMAGE *, IMAGE **, IMAGE **, IMAGE **, int);
int	hvGradient1(IMAGE *, IMAGE **, IMAGE **, IMAGE **, int);
GRAPH	*hvGraphHistogram(VLIST *, int, double, double);
GRAPH	*hvGraphXSection(VLIST *, int, int, int, int, int);
IMAGE	*hvGreyscaleToComplex(IMAGE *, IMAGE *);
IMAGE	*hvHaralick(IMAGE *, IMAGE *);
int	hvHasBands(IMAGE *, int, char *, char *);
int	hvHasHeight(IMAGE *, int, char *, char *);
int	hvHasOrder(IMAGE *, int, char *, char *);
int	hvHasType(IMAGE *, int, char *, char *);
int	hvHasWidth(IMAGE *, int, char *, char *);
IMAGE	*hvHessian(IMAGE *, int);
IMAGE	*hvHistEqual(IMAGE *, IMAGE *, int);
VLIST	*hvHistogram(IMAGE *, int);
int	_hvIgnoreSignals(void);
int	hvImageAccounting(void);
int	hvImageAccountingOff(void);
int	hvImageAccountingOn(void);
KERNEL	*hvImageToKernel(IMAGE *);
LINK	*hvInLinklist(EDGE *, LINK *);
int	hvInObjSet(char *, OBJSET **, int);
int	hvIncXLineData(GRAPH *, int, double, double);
int	hvInitBeep(int (*)());
int	hvInitIO(int (*)(), int (*)(), int (*)(), int (*)(), int (*)());
int	hvInitImagelist(int);
int	hvInitMore(int, int (*)());
int	hvInitPause(int (*)());
void	hvInitSignalUsage(void (*)(), void (*)());
void	hvInitSignals(void (*)());
int	hvInitWorklist(int);
int	hvInsertBand(IMAGE *, int, IMAGE *, int);
int	hvIntQuickSort(int *, int);
int	hvIntSqrt(int);
IMAGE	*hvInvert(IMAGE *, IMAGE *, int);
int	hvIsBeepOn(void);
int	_hvIsCVDec(char *);
int	_hvIsCVSun(char *);
int	hvIsCbreakOn(void);
int	_hvIsGIF87a(char *);
int	hvIsMoreOn(void);
int	_hvIsOldHSIF(char *);
int	_hvIsOldHips(char *);
int	_hvIsPCEye(unsigned char *);
int	_hvIsPipe(struct _iobuf *);
int	hvIsPowerOf2(int);
int	_hvIsSunRaster(char *);
IMAGE	*hvKernelToImage(KERNEL *);
IMAGE	*hvLaplacian(IMAGE *, IMAGE *);
int	hvLegalPval(IMAGE *, double);
IMAGE	*hvLevel(IMAGE *, IMAGE *, double, unsigned char);
LINK	**hvLinkCom(IMAGE *, int);
int	hvLoadPyrWin(IMAGE **, IMAGE *, int, int, int *, int *);
int	hvLoadPyrWinByFunc(IMAGE **, IMAGE *, int, int, int *, int *, int (*)());
IMAGE	*hvLocalStat(IMAGE *, int);
int	hvLogImage(IMAGE *, int);
int	hvLowestRes(IMAGE *, double, int, int);
int	hvMRLRange(IMAGE *, int *, int *, double);
IMAGE	*hvMagnitude(IMAGE *, IMAGE *, int);
int	_hvMakeDataspace(IMAGE *);
EDGE	*hvMakeEdge(EDGE *, EDGE *);
double	_hvMakeFloat(int, int);
GRAPH	*hvMakeGraph(int, int);
GRAPH	*hvMakeGraphVA();
IMAGE	*_hvMakeHeader(int, int, int, int, int);
IMAGE	*hvMakeImage(int, int, int, int, int);
KERNEL	*hvMakeKernel(int, int, int, int, int);
LINK	*hvMakeLink(LINK *, LINK *, int, int);
OBJSET	*hvMakeObjSet(int, int);
OBJ	*hvMakeObject(int, int, int, int, int, ...);
char	*_hvMakePostPatternFromData(int, int, int, int);
GRAPH	*hvMakeSEGraphVA();
int	_hvMakeThreed(IMAGE *);
VLIST	*hvMakeVlist(int, int);
char	*_hvMakeXPatternFromData(int, int, int, int);
IMAGE	*hvMapRange(IMAGE *, IMAGE *, int, double, double, double, double);
int	hvMarkWorklist(void);
IMAGE	*hvMaskHistEqual(IMAGE *, IMAGE *, int (*)());
int	hvMaskXCor(IMAGE *, IMAGE *, IMAGE *, IMAGE *, double);
int	hvMatchAll(IMAGE *, IMAGE *, char *);
int	hvMatchBands(IMAGE *, IMAGE *, char *);
int	hvMatchHeight(IMAGE *, IMAGE *, char *);
int	hvMatchOrder(IMAGE *, IMAGE *, char *);
int	hvMatchSize(IMAGE *, IMAGE *, char *);
int	hvMatchType(IMAGE *, IMAGE *, char *);
int	hvMatchWidth(IMAGE *, IMAGE *, char *);
IMAGE	*hvMaxDown(IMAGE *, IMAGE *);
IMAGE	*hvMedian(IMAGE *, IMAGE *, KERNEL *);
EDGE	*hvMergeEdges(EDGE *, EDGE *);
IMAGE	*hvMinDown(IMAGE *, IMAGE *);
int	hvMinMax(IMAGE *, int, float *, float *);
int	hvMoreOff(void);
int	hvMoreOn(void);
IMAGE	*hvMorphClose(IMAGE *, IMAGE *, KERNEL *, int);
IMAGE	*hvMorphOpen(IMAGE *, IMAGE *, KERNEL *, int);
int	hvMultiResList(IMAGE *, IMAGE **, IMAGE **, int *, int *, int, int, int, double, int, int);
void	hvNegate(IMAGE *);
int	hvNewStylePyramid(void);
IMAGE	*hvNitShi(IMAGE *, int, int, double, double);
VLIST	*hvNonMaxSup(IMAGE *, int, int, int, double, int *);
double	hvNormImage(IMAGE *, int, double);
int	hvNormList(IMAGE **, int, int);
IMAGE	*hvNormalize(IMAGE *, IMAGE *, int, double, double);
int	hvNullImage();
int	hvNumbers(IMAGE *, struct _iobuf *, int, char *);
IMAGE	*hvOldComposePyramid(IMAGE **, int, int, double);
int	hvOldStylePyramid(void);
int	hvOneByteQuickSort(unsigned char *, int);
IMAGE	*hvOnebandToComplex(IMAGE *, IMAGE *, int);
struct	_iobuf *hvOpenImageStream(char *);
char	*_hvOutStr(char *);
unsigned char *_hvPackBinary(IMAGE *);
IMAGE	*hvPadImage(IMAGE *, IMAGE *, int);
int	hvParseBandOrder(char *);
int	_hvParseGraph(int, GRAPH *);
int	hvParsePixelType(char *);
int	_hvPatternIsNotLegal(int, int, int, int);
char	hvPause(int);
int	hvPercentile(IMAGE *, int, double, float *);
IMAGE	*hvPhase(IMAGE *, IMAGE *);
IMAGE	*hvPixFun(IMAGE *, IMAGE *, int, int);
IMAGE	*hvPixOp1(IMAGE *, IMAGE *, int, int, double, int);
IMAGE	*hvPixOp2(IMAGE *, IMAGE *, IMAGE *, int, int, int);
IMAGE	*hvPixRepVal(IMAGE *, IMAGE *, int, double, double);
IMAGE	*hvPlot2d(VLIST *, int, int, int, int);
IMAGE	*hvPlot2db(VLIST *, double, double, int, int, int, int);
IMAGE	*hvPlotv(VLIST *, int, int, int);
EDGE	*hvPolyLine(EDGE *, double, double, int);
int	hvPopState(IMAGE *, IMAGE *);
int	hvPostGraph(GRAPH *, char *, char *, int);
int	_hvPrepareGraph(GRAPH *, int (*)());
LINK	*hvPrependLink(EDGE *, int, int);
int	_hvPrintArray(struct _iobuf *, IMAGE *, int, int, int, int);
void	hvPrintEdge(EDGE *, int);
void	hvPrintLink(LINK *);
int	hvPrintNumbers(IMAGE *, char *, int, char *);
IMAGE	**hvPrism(IMAGE *);
int	hvPushState(IMAGE *, IMAGE **, int, int, int, int, char *);
int	hvPyramidLevel(IMAGE *, double);
int	hvQueryHistogram(float *, float *);
IMAGE	*hvQuickSort(IMAGE *, IMAGE *, int);
IMAGE	*hvReadAnyImageFP(struct _iobuf *, char *);
EDGE	*hvReadAsciiEdgeFP(struct _iobuf *, int);
EDGE	*hvReadAsciiEdgelist(char *, int);
GRAPH	*hvReadAsciiGraph(char *);
GRAPH	*hvReadAsciiGraphFD(int);
LINK	*hvReadAsciiLinkFP(struct _iobuf *, int);
GRAPH	*hvReadBinaryGraph(char *);
GRAPH	*hvReadBinaryGraphFD(int);
IMAGE	*_hvReadCVDec(struct _iobuf *);
IMAGE	*_hvReadCVSun(struct _iobuf *);
char	hvReadChar(void);
IMAGE	*_hvReadGIF87a(struct _iobuf *);
IMAGE	*hvReadImage(char *);
IMAGE	*hvReadImage4(char *, char *, char *, int);
IMAGE	*hvReadImageBand(char *, int *);
IMAGE	*hvReadImageFD(int);
IMAGE	*hvReadImageFP(struct _iobuf *);
IMAGE	*hvReadImageHdr(char *);
IMAGE	*hvReadImageVA();
int	hvReadIntoImage(IMAGE *, char *);
int	hvReadIntoImageFD(IMAGE *, int);
int	hvReadIntoImageFP(IMAGE *, struct _iobuf *);
int	hvReadIntoImageVA(IMAGE *, char *, ...);
int	hvReadIntoNullImageVA(IMAGE **, char *, ...);
int	hvReadIntoRegion(IMAGE *, char *, int, int);
int	hvReadIntoRegionFD(IMAGE *, int, int, int);
KERNEL	*hvReadKernel(char *);
IMAGE	*_hvReadLisp(struct _iobuf *);
int	hvReadList(IMAGE **, int, int *, int *, char *, char *);
int	hvReadObjs(char *, OBJSET **, int, double, double, double, char *);
IMAGE	*_hvReadOldHSIF(struct _iobuf *);
IMAGE	*_hvReadOldHips(struct _iobuf *);
IMAGE	*_hvReadPCEye(struct _iobuf *);
IMAGE	*hvReadRaw(char *, int, int, int, int, int);
IMAGE	*_hvReadSunRaster(struct _iobuf *);
VLIST	*hvReadVlist(char *);
int	hvReadVlistFP(struct _iobuf *, VLIST **);
int	hvReadVlistList(VLIST **, int, char *);
int	hvReadVlistListVA(VLIST **, int, char *, ...);
VLIST	*hvReadVlistVA();
int	hvRealQuickSort(float *, int);
IMAGE	*hvRealpartToGreyscale(IMAGE *, IMAGE *);
IMAGE	*hvRecombine(IMAGE *, IMAGE *, IMAGE *);
int	hvRedirectIO(int, int, int, int);
int	_hvRemoveImage(IMAGE *);
int	_hvRepRead(int, char *, int);
void	hvRestoreSigHandler(void);
int	hvReverseEdge(EDGE *);
IMAGE	*hvRot90(IMAGE *, IMAGE *);
int	hvSSQuickSort(short *, int);
int	hvSameData(IMAGE *, IMAGE *, int);
int	hvSameLink(LINK *, LINK *);
int	hvSameProp(IMAGE *, IMAGE *);
int	_hvSameStr(char *, char *);
IMAGE	*hvSample(IMAGE *, IMAGE *, int, int);
double	hvScaleObjDim(double, double, int);
int	hvScaleVlist(VLIST *, double);
int	hvSetAbortOnError(int);
int	hvSetComment(IMAGE *, char *, int);
int	hvSetErrorFunctions(int (*)(), int (*)(), int (*)());
int	hvSetErrorPrefixes(char *, char *, char *);
int	hvSetGraphLegend(GRAPH *, int);
int	hvSetGraphSize(GRAPH *, int, int, int, int, int, int);
int	hvSetGraphTitleVA();
int	hvSetGraphXLabelVA();
int	hvSetGraphYLabelVA();
int	hvSetLineBoxSize(GRAPH *, int, int);
int	hvSetLineColor(GRAPH *, int, int);
int	hvSetLineData(GRAPH *, int, float *, float *);
int	hvSetLineLegendVA();
int	hvSetLineThickness(GRAPH *, int, double);
int	hvSetLineType(GRAPH *, int, int);
int	hvSetLinkNum(EDGE *);
int	hvSetName(IMAGE *, char *, int);
void	hvSetSignal(int, void (*)());
int	hvSetState(IMAGE *, int, int);
int	hvShiftVlist(VLIST *, int, int);
IMAGE	*hvShrinkImage(IMAGE *, int, int);
int	hvSigFreeSendOff(void);
int	hvSigFreeSendOn(void);
void	hvSignalHandler(int, int, struct sigcontext *);
int	hvSimpleXCor(IMAGE *, IMAGE *, IMAGE *, double);
IMAGE	*hvSmooth(IMAGE *, IMAGE *, int);
int	hvSmoothField(IMAGE *, double, int);
IMAGE	*hvSobel(IMAGE *, IMAGE *, int);
int	hvSortImagelist(void);
int	hvSortRIIbyR(int, float *, int *, int *);
int	hvSortVlistZ(VLIST *);
int	hvSqrtImage(IMAGE *);
int	hvStatistics(IMAGE *, int, float *, float *, float *, float *, float *, float *);
int	hvStripIm(char *);
char	*hvStripPath(char *);
int	hvStripSuffix(char *, char *);
int	hvStripSuffixes(char *);
int	hvStripToStem(char *, char *);
int	hvStripZ(char *);
IMAGE	*hvSuperImpose(IMAGE *, IMAGE *, IMAGE *, IMAGE *, int);
void	_hvSw2b(unsigned char *);
void	_hvSw4b(unsigned char *);
int	hvTextBBox(int, int, char *, int, int *, int *, int *, int *);
IMAGE	*hvThinBlob(IMAGE *, IMAGE *, double, double, int, int);
IMAGE	*hvThreshold(IMAGE *, IMAGE *, int, double, double, double, double);
IMAGE	*hvToggleBO(IMAGE *, IMAGE *);
int	hvTowerOp1(IMAGE **, IMAGE **, int, int, int, int, int);
int	hvTowerOp2(IMAGE **, IMAGE **, int, int, int, int);
EDGE	*hvTraceEdges(IMAGE *, int);
IMAGE	*hvTraceRegions(IMAGE *, EDGE **, int *, int);
IMAGE	*hvTranspose(IMAGE *, IMAGE *);
int	hvTwiddle(char *, char *);
int	hvTwiddleExpand(char *, char *);
int	hvUSQuickSort(unsigned short *, int);
char	*_hvUniqueName(void);
void	_hvUnpackBinary(IMAGE *);
IMAGE	*hvVarEdge(IMAGE *, int, double);
EDGE	*hvVlistToEdge(VLIST *);
int	hvWeakMemSOR(IMAGE *, IMAGE **, IMAGE **, double, double, int, int, int, int, int);
IMAGE	*hvWindow(IMAGE *, IMAGE *, int);
int	hvWorkAccounting(void);
int	hvWriteAsciiEdgeFP(EDGE *, struct _iobuf *, int);
int	hvWriteAsciiEdgelist(EDGE *, char *, int);
int	hvWriteAsciiGraph(GRAPH *, char *);
int	hvWriteAsciiGraphVA(GRAPH *, char *, ...);
int	hvWriteAsciiLinkFP(LINK *, struct _iobuf *);
int	hvWriteBinaryGraph(GRAPH *, char *);
int	hvWriteBinaryGraphFD(GRAPH *, int);
int	hvWriteBinaryGraphVA(GRAPH *, char *, ...);
int	hvWriteImage(IMAGE *, char *);
int	hvWriteImage4(IMAGE *, char *, char *, char *, int);
int	hvWriteImageBandFD(IMAGE *, int, int);
int	hvWriteImageFD(IMAGE *, int);
int	hvWriteImageVA();
int	hvWriteKernel(KERNEL *, char *);
int	hvWriteList(IMAGE **, int, int, char *, char *);
int	hvWriteVlist(VLIST *, char *);
int	hvWriteVlistFP(VLIST *, struct _iobuf *);
int	hvWriteVlistVA(VLIST *, char *, ...);
IMAGE	*hvXConvolve(IMAGE *, IMAGE *, int, KERNEL *, int);
VLIST	*hvXSection(IMAGE *, int, int, int, int, int);
IMAGE	*hvZeroX(IMAGE *, IMAGE *);
